#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>

#include <dfb_types.h>

#include <directfb.h>
#include <directfb_keyboard.h>

#include <core/coredefs.h>
#include <core/coretypes.h>
#include <core/input.h>
#include <core/system.h>

#include <direct/debug.h>
#include <direct/mem.h>
#include <direct/messages.h>
#include <direct/thread.h>
#include <direct/util.h>

#include <misc/conf.h>
#include <misc/util.h>

#include <core/input_driver.h>


DFB_INPUT_DRIVER( cetvfb_input )


D_DEBUG_DOMAIN( CETVFBInput, "CETVFB/Input", "CETVFB System Input" );

typedef enum {
     RS_IDLE,
     RS_START,
     RS_REPEAT
} RepeatState;

/*
 * declaration of private data
 */
typedef struct {
     CoreInputDevice         *device;

     DirectThread            *thread;

     DFBInputDeviceKeySymbol  last_symbol;
     int                      last_key_code;

     bool                     quit_repeater;
     pthread_cond_t           repeat_cond;
     pthread_mutex_t          repeat_lock;
     RepeatState              repeat_state;
} JAGInputData;

static JAGInputData *gData;

/*
 * Translates a TV Command into a DirectFB keycode.
 */

static void
send_symbol( int code, DFBInputDeviceKeySymbol symbol, bool down )
{
     DFBInputEvent event;

     if (symbol == DIKS_NULL)
          return;

     event.type       = down ? DIET_KEYPRESS : DIET_KEYRELEASE;
     event.flags      = DIEF_KEYSYMBOL | DIEF_KEYCODE | DIEF_MODIFIERS;
     event.modifiers  = DIMM_HYPER;
     event.key_symbol = symbol;
     event.key_code   = code;

     dfb_input_dispatch( gData->device, &event );

}

static void *
repeater_thread( DirectThread *thread, void *arg )
{
     struct timespec  timeout;
     JAGInputData    *data = arg;

     pthread_mutex_lock( &data->repeat_lock );

     while (!data->quit_repeater) {
          switch (data->repeat_state) {
               case RS_IDLE:
                    pthread_cond_wait( &data->repeat_cond, &data->repeat_lock );
                    break;

               case RS_REPEAT:
                    send_symbol( data->last_key_code, data->last_symbol, true );

                    /* fall through */

               case RS_START:
                    direct_util_get_monotonic_pthread_timeout(
                                        &timeout,
                                        0, // seconds
                                        (data->repeat_state == RS_REPEAT) ? 50000000 : 300000000 // nanoseconds
                            );

                    if (pthread_cond_timedwait( &data->repeat_cond, &data->repeat_lock, &timeout ) == ETIMEDOUT)
                         if (data->repeat_state == RS_START)
                              data->repeat_state = RS_REPEAT;

                    break;
          }

     }

     pthread_mutex_unlock( &data->repeat_lock );

     return NULL;
}

void
DirectFBJAGInputNotify( int          code,
						unsigned int command,
                        bool         down )
{
     D_DEBUG_AT( CETVFBInput, "%s( 0x%x, %s )\n", __FUNCTION__, command, down ? "down" : "up" );
//     printf( "%s( 0x%x, %s )\n", __FUNCTION__, command, down ? "down" : "up" );
	
     if (!gData)
          return;

     pthread_mutex_lock( &gData->repeat_lock );

     gData->last_symbol   = command;
	 gData->last_key_code = code;
		  
     send_symbol( gData->last_key_code, gData->last_symbol, down );


     gData->repeat_state = /*down ? RS_START : */RS_IDLE;

     if (!down)
	 {
          gData->last_symbol   = DIKS_NULL;
          gData->last_key_code = -1;
	 }

     pthread_cond_signal( &gData->repeat_cond );

     pthread_mutex_unlock( &gData->repeat_lock );

}

/*
 * Return the number of available devices.
 * Called once during initialization of DirectFB.
 */
static int
driver_get_available()
{
     return 1;
}

/*
 * Fill out general information about this driver.
 * Called once during initialization of DirectFB.
 */
static void
driver_get_info( InputDriverInfo *info )
{
     /* fill driver info structure */
     snprintf ( info->name,
                DFB_INPUT_DRIVER_INFO_NAME_LENGTH, "JAG Input Driver" );
     snprintf ( info->vendor,
                DFB_INPUT_DRIVER_INFO_VENDOR_LENGTH, "Denis Oliver Kropp" );

     info->version.major = 0;
     info->version.minor = 5;
}

/*
 * Open the device, fill out information about it,
 * allocate and fill private data, start input thread.
 * Called during initialization, resuming or taking over mastership.
 */
static DFBResult
driver_open_device( CoreInputDevice  *device,
                    unsigned int      number,
                    InputDeviceInfo  *info,
                    void            **driver_data )
{
     JAGInputData *data;
	 
     /* fill info */
     snprintf( info->desc.name, DFB_INPUT_DEVICE_DESC_NAME_LENGTH, "Philips JAG Input" );
     snprintf( info->desc.vendor, DFB_INPUT_DEVICE_DESC_VENDOR_LENGTH, "Philips" );

     info->desc.caps = DICAPS_KEYS;
     info->desc.type = DIDTF_REMOTE;

     /* allocate and fill private data */
     data = D_CALLOC( 1, sizeof(JAGInputData) );

     data->device = device;

     direct_util_monotonic_pthread_cond_init( &data->repeat_cond );
     pthread_mutex_init( &data->repeat_lock, NULL );

     data->thread = direct_thread_create( DTT_INPUT, repeater_thread, data, "Key Repeater" );

     /* set private data pointer */
     *driver_data = data;


     /* DIRTY */
     gData = data;

     return DFB_OK;
}


/*
 * Fetch one entry from the kernel keymap.
 */
static DFBResult
driver_get_keymap_entry( CoreInputDevice           *device,
                         void                      *driver_data,
                         DFBInputDeviceKeymapEntry *entry )
{
     return DFB_UNSUPPORTED;
}

/*
 * End thread, close device and free private data.
 */
static void
driver_close_device( void *driver_data )
{
     JAGInputData *data = driver_data;

     data->quit_repeater = true;

     pthread_cond_signal( &data->repeat_cond );

     direct_thread_join( data->thread );
     direct_thread_destroy( data->thread );

     /* free private data */
     D_FREE( data );
}

