/*
   (c) Copyright 2000-2002  convergence integrated media GmbH.
   (c) Copyright 2002-2004  convergence GmbH.

   All rights reserved.

   Written by Denis Oliver Kropp <dok@directfb.org>,
              Andreas Hundt <andi@fischlustig.de>,
              Sven Neumann <neo@directfb.org> and
              Ville Syrjl <syrjala@sci.fi>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

  (Information filled in by ChangeSynergy:)
            %name: cetvfb.c %
         %version: brg45mgr#RR13.2.5 %
   %date_modified: Thu Apr 07 10:23:23 2011 %
      %derived_by: devcetv %

*/

#include <config.h>

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <fcntl.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <directfb.h>

#include <fusion/arena.h>
#include <fusion/shmalloc.h>

#include <core/core.h>
#include <core/coredefs.h>
#include <core/coretypes.h>
#include <core/layers.h>
#include <core/palette.h>
#include <core/surface.h>
#include <core/surface_pool.h>
#include <core/system.h>

#include <gfx/convert.h>

#include <misc/conf.h>

#include <direct/messages.h>


#include "cetvfb.h"
#include "cetvfb_surfacepool.h"
#include "primary.h"
#include "callbacks.h"

#include <core/core_system.h>


/* copy paste reuse from shmemipcdrv interface, which is otherwise not exported to our process */
#define IOCTL_IPC_SET_PAGES_UNCACHED   _IOR('S', 0x12, int)
#define IOCTL_IPC_CACHE_FLUSH          _IOR('S', 0x13, int)
#define IOCTL_IPC_CACHE_INVALIDATE     _IOR('S', 0x14, int)

#define D_DEBUG_AT(x,s...) printf (s)

DFB_CORE_SYSTEM( cetvfb )

D_DEBUG_DOMAIN( CETVFB_System, "CETVFB/System", "Core System CETVFB" );

/**********************************************************************************************************************/

DFBCETVFB          *dfb_cetvfb        = NULL;
CoreDFB            *dfb_cetvfb_core   = NULL;
FusionWorld        *dfb_cetvfb_world  = NULL;

/**********************************************************************************************************************/

static void
CETVFB_Exit()
{
    /* TODO: destroy memspace */
}

static FusionCallHandlerResult
dfb_cetvfb_call_handler( int           caller,
                         int           call_arg,
                         void         *call_ptr,
                         void         *ctx,
                         unsigned int  serial,
                         int          *ret_val )
{
     switch (call_arg) {
          case CETVFB_SET_REGION:
               *ret_val = dfb_cetvfb_set_region_handler( call_ptr );
               break;

          case CETVFB_UPDATE_SCREEN:
               *ret_val = dfb_cetvfb_update_screen_handler( call_ptr );
               break;

          case CETVFB_SET_PALETTE:
               *ret_val = dfb_cetvfb_set_palette_handler( call_ptr );
               break;

          case CETVFB_REMOVE_REGION:
               *ret_val = dfb_cetvfb_remove_region_handler( call_ptr );
               break;

          case CETVFB_TMML_ALLOCATE:
               *ret_val = dfb_cetvfb_allocate_handler( call_ptr );
               break;

          case CETVFB_TMML_DEALLOCATE:
               *ret_val = dfb_cetvfb_deallocate_handler( call_ptr );
               break;

          case CETVFB_SET_NAT_MOTION_WDW:
               *ret_val = dfb_cetvfb_set_nat_motion_handler( call_ptr );
               break;

          case CETVFB_SET_LEVEL:
               *ret_val = dfb_cetvfb_set_level_handler( call_ptr );
               break;
               
          case CETVFB_GET_LEVEL:
               *ret_val = dfb_cetvfb_get_level_handler( call_ptr );
               break;

          default:
               D_BUG( "unknown call" );
               *ret_val = DFB_BUG;
               break;
     }

     return FCHR_RETURN;
}

static void RegisterScreenFunctions( void )
{
     CoreScreen          *screen;

     /* register screen functions */
     screen = dfb_screens_register( NULL, NULL, &cetvfbPrimaryScreenFuncs );

	 /* register layer functions for the 3 layers,
        but only if the layers are present in the 'ini' file.
        In this way, it is possible to only use 2 layers on boards without back-end. */
     D_ASSERT( dfb_config != NULL );
     if( dfb_config->layers[0].init ) { dfb_layers_register( screen, (void*) 0, &cetvfbPrimaryLayerFuncs ); }
     if( dfb_config->layers[1].init ) { dfb_layers_register( screen, (void*) 1, &cetvfbPrimaryLayerFuncs ); }
     if( dfb_config->layers[2].init ) { dfb_layers_register( screen, (void*) 2, &cetvfbPrimaryLayerFuncs ); }
}


/* For the 444MB memory map, NXP has split the physical memory.on the internal memory bus between MIPS and memory controller.
  RAM          Physical Address    Virtual Address
  0-256MB      0-256MB             0-256MB
  256-512MB    512MB-768MB         256-512MB

  This means that we now need to mmap anything above 256MB differently.  Physical address is 256MB larger than virtual address.

  This memory hole is defined in the /proc/pnx8xxx/mem_start_hole0, mem_end_hole0 and mem_size_hole0

  This is done to keep MMIO physical at 0x1BE00000.

  For the final solution, MMIO will be moved and the memory hole disappears.  This MapSharedMemoryAreaTemporaryFixup can be removed then and
  calls can be replaced by MapSharedMemoryArea. (remove physoffset from MapSharedMemoryArea)
*/

#define PROC_PATH                   "/proc/pnx8xxx/"
#define PROC_MEM_START_HOLE0        PROC_PATH "mem_start_hole0"
#define PROC_MEM_END_HOLE0          PROC_PATH "mem_end_hole0"
#define MAX_PROC_PATH_SIZE          128

static void GetPnxProcItem(char* filename, unsigned int* item)
{
    char procname[MAX_PROC_PATH_SIZE];
    FILE* f;
    int n;

    //n = snprintf(procname, MAX_PROC_PATH_SIZE, PROC_PATH "%s", filename);
    n = snprintf(procname,MAX_PROC_PATH_SIZE, "%s", filename);
    D_ASSERT( n > -1 );
    D_ASSERT( n < MAX_PROC_PATH_SIZE );

    f = fopen(procname, "r");
    D_ASSERT( f != NULL );
    
    if ( f != NULL )
    {
        n = fscanf(f, "%x", item);
        D_ASSERT( n != EOF );

        fclose(f);
    }

    return;
}

static DFBResult MapSharedMemoryArea(CETVFBMemArea* area, int prot, int cached, off_t physoffset);

static DFBResult MapSharedMemoryAreaTemporaryFixup(CETVFBMemArea* area, int prot, int cached)
{
    static unsigned int holestart = 0;
    static unsigned int holeend = 0;
    static unsigned int holesize = -1;
    DFBResult result = DFB_OK;
    
    GetPnxProcItem(PROC_MEM_START_HOLE0, &holestart);
    GetPnxProcItem(PROC_MEM_END_HOLE0, &holeend);
    holesize = holeend - holestart;

    if (holesize != 0) {
        if ((unsigned int)area->start >= holestart)
        {
            //after hole
            result = MapSharedMemoryArea(area, prot, cached, (off_t)holesize);
        } 
        else 
        {
            if ((unsigned int)area->start + area->length <= holestart)
            {
                //before hole
                result = MapSharedMemoryArea(area, prot, cached, 0);
            }
            else 
            {
                //hole is inside area, need to split in 2 mmaps.
                CETVFBMemArea area2;
                area2.start = area->start;
                area2.length = holestart-(unsigned int)area->start;
                result = MapSharedMemoryArea(&area2, prot, cached, 0);
                if (result == DFB_OK)
                {
                    area2.start = (void*)holestart;
                    area2.length = area->length - area2.length;
                    result = MapSharedMemoryArea(&area2, prot, cached, (off_t)holesize);
                }
            }
        }

    } else {
        // no memory hole, so workaround not needed
        result = MapSharedMemoryArea(area, prot, cached, 0);
    }
    return result;
}


//static DFBResult MapSharedMemoryArea(CETVFBMemArea* area, int prot, int cached)
static DFBResult MapSharedMemoryArea(CETVFBMemArea* area, int prot, int cached, off_t physoffset)
{
    int   fd      = 0;
    void *address = 0;
    int   ShmemDriver = -1;
    int   retval;    
    if( area->length != 0 ) /* if the length is 0, the area will be considered empty */
    {
        /* we need the kernel to put mem to cached/uncached, so we have a kernel module for that */
        ShmemDriver = open( "/dev/cachectl", O_RDWR );
        if( ShmemDriver == -1 )
        {
            D_PERROR( "System/CETVFB: Couldn't open /dev/cachectl!\n" );
            return DFB_INIT;
        }
        
        fd = open( "/dev/mem", O_RDWR );
        if (fd < 0) {
            printf( "System/CETVFB: Couldn't open /dev/mem! errno=%d\n", errno);
            return DFB_INIT;
        }
        
        //address = mmap( area->start, area->length, prot, MAP_FIXED | MAP_SHARED, fd, (off_t)area->start );
        address = mmap( area->start, area->length, prot  ,MAP_FIXED|  MAP_SHARED, fd, (off_t) area->start + (off_t) physoffset );
        D_ASSERT( address != MAP_FAILED );
        if( address != area->start )
        {
            printf( "System/CETVFB: Couldn't mmap to required address %p! errno=%d\n", area->start, errno );
            return DFB_INIT;
        }
        if (cached)
        {
            /* kernel be default makes anything which is mapped above linux memory as uncached.
               mprotect has side effect that it if pagetable needs to change, the memory will become cached.
               By splitting pagetable in 2 parts, one part will become cached.  By doing split second time, the other part will become cached.
        
               (This workaround code assumes that cached implies prot = PROT_READ | PROT_WRITE!)
            */
            retval = mprotect( (void*)area->start, 4*1024, PROT_READ );  // force split up of shared memory vma in 2 parts (one readonly, one read/write)
            if( retval )
            {
                perror("mprotect 1 failure\n" );
            }
            
            retval = mprotect( (void*)area->start, 4*1024, PROT_READ | PROT_WRITE); // set first vma back to read/write
            if( retval )
            {
                perror("mprotect 1 failure\n" );
            }
           
            
            
            
            retval = mprotect( 4*1024 + (char*)area->start, area->length - 4*1024, PROT_READ );  // force split up of shared memory vma in 2 parts (one readonly, one read/write)
            if( retval )
            {
                perror("mprotect 1 failure\n" );
            }
            
            retval = mprotect( 4*1024 + (char*)area->start, area->length - 4*1024, PROT_READ | PROT_WRITE); // set first vma back to read/write
            if( retval )
            {
                perror("mprotect 1 failure\n" );
            }
           
        }
        else
        {
            ioctl( ShmemDriver, IOCTL_IPC_SET_PAGES_UNCACHED, &area );
        }

        close(fd);
        close(ShmemDriver);
    }

    return DFB_OK;
}

/*
 * This function maps the shared memory (this includes all video memory!) to the same physical address.
 * It's done for 3 ranges: cached, non-cached, and read-only (BIS).
 * CALLER must make sure these ranges are on cache boundaries.
 */
static DFBResult MapSharedMemory( void )
{
     DFBResult result;
     
     D_DEBUG_AT( CETVFB_System, "%s() enter\n", __FUNCTION__ );

     CETVFBMemArea area = {0};

     /* map the complete range of shared memory.
      * NOT only the DirectFB memory!
      * This requirement has been given to the DirectFB library, ONLY for slaves
      * As per requirement, Phys == Virt, so we put a hard check on that.
      */

     /* clever is to map BIS first:
      * later mappings will overwrite first, so we start with the 'tightest' mappings
      * to prevent problems if a border is not aligned with a cache page
      */

     /* map BIS area */
     //area = dfb_cetvfb->bis;
     //D_DEBUG_AT( CETVFB_System, "attempting to map BIS at: %p - length %x\n", area.start, area.length );
     //result = MapSharedMemoryAreaTemporaryFixup(&area, PROT_READ, 0);
     //if (result != DFB_OK) return result;

     /* map NON-CACHED area */
     area = dfb_cetvfb->uncached;
     D_DEBUG_AT( CETVFB_System, "attempting to map NON-CACHED at: %p - length %x\n", area.start, area.length );
     result = MapSharedMemoryAreaTemporaryFixup(&area, PROT_READ | PROT_WRITE, 0);
     if (result != DFB_OK) return result;

     /* map CACHED area */
     area = dfb_cetvfb->cached;
     D_DEBUG_AT( CETVFB_System, "attempting to map CACHED at: %p - length %x\n", area.start, area.length );
     result = MapSharedMemoryAreaTemporaryFixup(&area, PROT_READ | PROT_WRITE, 1);
     if (result != DFB_OK) return result;

     /* done */
     D_DEBUG_AT( CETVFB_System, "%s() exit\n", __FUNCTION__ );

     return DFB_OK;
}

/**********************************************************************************************************************/

static void
system_get_info( CoreSystemInfo *info )
{
     info->type = CORE_ANY;
     info->caps = CSCAPS_ACCELERATION;

     snprintf( info->name, DFB_CORE_SYSTEM_INFO_NAME_LENGTH, "CETVFB" );
}

static DFBResult
system_initialize( CoreDFB *core, void **data )
{
     FusionWorld         *world;
     FusionSHMPoolShared *pool;

     D_DEBUG_AT( CETVFB_System, "%s()\n", __FUNCTION__ );

     D_ASSERT( dfb_cetvfb == NULL );

     /**** end of loading module ***/
     world = dfb_core_world( core );
     pool  = dfb_core_shmpool( core );

     dfb_cetvfb = (DFBCETVFB*) SHCALLOC( pool, 1, sizeof(DFBCETVFB) );
     if (!dfb_cetvfb) {
          D_ERROR( "DirectFB/CETVFB: Couldn't allocate shared memory!\n" );
          return DFB_NOSYSTEMMEMORY;
     }

     dfb_cetvfb->shmpool = pool;
     dfb_cetvfb_core     = core;
     dfb_cetvfb_world    = world;

     {
          CETVFBCallBackTable  cetvfbfct = {0};
          
          CETVFBMemArea uncached = {0};
          CETVFBMemArea cached   = {0};
          CETVFBMemArea bis      = {0};

          /* callbacks must be present, or this will fail */
          cetvfbfct = dfb_cetvfb_get_callbacks( );
          
          D_ASSERT( cetvfbfct.GetSharedMemoryLocation != 0 );
          cetvfbfct.GetSharedMemoryLocation( &uncached, &cached, &bis );

          dfb_cetvfb->uncached = uncached;
          dfb_cetvfb->cached   = cached;
          dfb_cetvfb->bis      = bis;

          dfb_cetvfb->shmAcquire = cetvfbfct.MemoryAcquire;
          dfb_cetvfb->shmRelease = cetvfbfct.MemoryRelease;
     }

     fusion_skirmish_init( &dfb_cetvfb->lock, "CETVFB System", world );
     fusion_call_init( &dfb_cetvfb->call, dfb_cetvfb_call_handler, NULL, world );

     RegisterScreenFunctions();

     fusion_arena_add_shared_field( dfb_core_arena( core ), "CETVFB", dfb_cetvfb );

     dfb_surface_pool_initialize( core, &cetvfbSurfacePoolFuncs, &dfb_cetvfb->tmml_surface_pool );

     *data = dfb_cetvfb;

     return DFB_OK;
}

static DFBResult
system_join( CoreDFB *core, void **data )
{
     void       *field;
     DFBResult   rval   = DFB_OK;

     D_DEBUG_AT( CETVFB_System, "%s()\n", __FUNCTION__ );

     D_ASSERT( dfb_cetvfb == NULL );

     fusion_arena_get_shared_field( dfb_core_arena( core ), "CETVFB", &field );

     dfb_cetvfb       = field;
     dfb_cetvfb_core  = core;
     dfb_cetvfb_world = dfb_core_world( core );

     rval = MapSharedMemory();
     if( rval != DFB_OK ) { return rval; }

     RegisterScreenFunctions();

     dfb_surface_pool_join( core, dfb_cetvfb->tmml_surface_pool, &cetvfbSurfacePoolFuncs );

     *data = dfb_cetvfb;

     return DFB_OK;
}

static DFBResult
system_shutdown( bool emergency )
{
     D_DEBUG_AT( CETVFB_System, "%s()\n", __FUNCTION__ );

     D_ASSERT( dfb_cetvfb != NULL );

     fusion_call_destroy( &dfb_cetvfb->call );

     fusion_skirmish_prevail( &dfb_cetvfb->lock );

     dfb_surface_pool_destroy( dfb_cetvfb->tmml_surface_pool );

     CETVFB_Exit();

     fusion_skirmish_destroy( &dfb_cetvfb->lock );

     SHFREE( dfb_cetvfb->shmpool, dfb_cetvfb );

     dfb_cetvfb       = NULL;
     dfb_cetvfb_core  = NULL;
     dfb_cetvfb_world = NULL;

     return DFB_OK;
}

static DFBResult
system_leave( bool emergency )
{
     D_DEBUG_AT( CETVFB_System, "%s()\n", __FUNCTION__ );

     D_ASSERT( dfb_cetvfb != NULL );

     dfb_cetvfb       = NULL;
     dfb_cetvfb_core  = NULL;
     dfb_cetvfb_world = NULL;

     return DFB_OK;
}

static DFBResult
system_suspend()
{
     D_DEBUG_AT( CETVFB_System, "%s()\n", __FUNCTION__ );

     return DFB_UNIMPLEMENTED;
}

static DFBResult
system_resume()
{
     D_DEBUG_AT( CETVFB_System, "%s()\n", __FUNCTION__ );

     return DFB_UNIMPLEMENTED;
}

static volatile void *
system_map_mmio( unsigned int    offset,
                 int             length )
{
     D_DEBUG_AT( CETVFB_System, "%s()\n", __FUNCTION__ );

     return NULL;
}

static void
system_unmap_mmio( volatile void  *addr,
                   int             length )
{
     D_DEBUG_AT( CETVFB_System, "%s()\n", __FUNCTION__ );
}

static int
system_get_accelerator()
{
     D_DEBUG_AT( CETVFB_System, "%s()\n", __FUNCTION__ );

     // return 0 to use the NXP hardware gfx accelerator
     return 0;
}

static VideoMode *
system_get_modes()
{
     D_DEBUG_AT( CETVFB_System, "%s()\n", __FUNCTION__ );

     return NULL;
}

static VideoMode *
system_get_current_mode()
{
     D_DEBUG_AT( CETVFB_System, "%s()\n", __FUNCTION__ );

     return NULL;
}

static DFBResult
system_thread_init()
{
     D_DEBUG_AT( CETVFB_System, "%s()\n", __FUNCTION__ );

     return DFB_OK;
}

static bool
system_input_filter( CoreInputDevice *device,
                     DFBInputEvent   *event )
{
     return false;
}

static unsigned long
system_video_memory_physical( unsigned int offset )
{
     return 0;
}

static void *
system_video_memory_virtual( unsigned int offset )
{
     return NULL;
}

static unsigned int
system_videoram_length()
{
     return 0;
}

static unsigned long
system_aux_memory_physical( unsigned int offset )
{
     return 0;
}

static void *
system_aux_memory_virtual( unsigned int offset )
{
     return NULL;
}

static unsigned int
system_auxram_length()
{
     return 0;
}

static void
system_get_busid( int *ret_bus, int *ret_dev, int *ret_func )
{
}

static void
system_get_deviceid( unsigned int *ret_vendor_id,
                     unsigned int *ret_device_id )
{
}

