/*
   (c) Copyright 2001-2007  The DirectFB Organization (directfb.org)
   (c) Copyright 2000-2004  Convergence (integrated media) GmbH

   All rights reserved.

   Written by Denis Oliver Kropp <dok@directfb.org>,
              Andreas Hundt <andi@fischlustig.de>,
              Sven Neumann <neo@directfb.org>,
              Ville Syrjälä <syrjala@sci.fi> and
              Claudio Ciccani <klan@users.sf.net>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <config.h>

//#define DIRECT_ENABLE_DEBUG

#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include <direct/debug.h>
#include <direct/hash.h>
#include <direct/mem.h>

#include <core/surface_pool.h>

#include "cetvfb.h"
#include "cetvfb_surfacepool.h"
#include "callbacks.h"

D_DEBUG_DOMAIN( CETVFB_TMML, "CETVFB/Tmml", "CETVFB System Tmml Surface Pool" );
D_DEBUG_DOMAIN( CETVFB_TMML_Cache, "CETVFB/Tmml/Cache", "CETVFB System Tmml Surface Pool Cache" );
D_DEBUG_DOMAIN( CETVFB_TMML_Lock, "CETVFB/Tmml/Lock", "CETVFB System Tmml Surface Pool Locking" );

/**********************************************************************************************************************/

typedef struct {
} TmmlPoolData;

typedef struct {
     DirectHash     *hash;
} TmmlPoolLocalData;

/**********************************************************************************************************************/

DFBResult
dfb_cetvfb_allocate_handler( cetvfbAllocationData *data )
{
     D_DEBUG_AT( CETVFB_TMML, "%s()\n", __FUNCTION__ );

     void *address;

     /* important input parameters: data->size, & data->address */

     D_ASSERT( dfb_cetvfb->shmAcquire != 0 );
     address = dfb_cetvfb->shmAcquire( data->resource_id, data->size );

     if (address == 0)
     {
          D_ERROR( "System/CETVFB: Error acquiring free memory of size '%d'!\n", data->size );
          return DFB_INIT;
     }

     data->address = address;

     D_DEBUG_AT( CETVFB_TMML, "  -> Allocate 'Acquire' Request 0x%p resid %lu size 0x%04x\n",
         data->address, data->resource_id, data->size );

     return DFB_OK;
}

DFBResult
dfb_cetvfb_deallocate_handler( cetvfbAllocationData *data )
{
     int rval;

     D_DEBUG_AT( CETVFB_TMML, "%s()\n", __FUNCTION__ );

     D_ASSERT( dfb_cetvfb->shmRelease != 0 );

     rval = dfb_cetvfb->shmRelease( data->resource_id, data->size, data->address );

     if (rval != Dfb_True)
     {
          D_ERROR( "System/CETVFB: Error releasing memory of size '%d'!\n", data->size );
          return DFB_INIT;
     }

     return DFB_OK;
}

/**********************************************************************************************************************/

static int
tmmlPoolDataSize()
{
     return sizeof(TmmlPoolData);
}

static int
tmmlPoolLocalDataSize()
{
     return sizeof(TmmlPoolLocalData);
}

static int
tmmlAllocationDataSize()
{
     return sizeof(cetvfbAllocationData);
}

static DFBResult
tmmlInitPool( CoreDFB                    *core,
              CoreSurfacePool            *pool,
              void                       *pool_data,
              void                       *pool_local,
              void                       *system_data,
              CoreSurfacePoolDescription *ret_desc )
{
     D_DEBUG_AT( CETVFB_TMML, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_ASSERT( ret_desc != NULL );

     ret_desc->caps              = CSPCAPS_PHYSICAL | CSPCAPS_VIRTUAL;
     ret_desc->access[CSAID_CPU] = CSAF_READ | CSAF_WRITE | CSAF_SHARED;
     ret_desc->access[CSAID_GPU] = CSAF_READ | CSAF_WRITE | CSAF_SHARED;
     ret_desc->types    = CSTF_LAYER | CSTF_WINDOW | CSTF_CURSOR | CSTF_FONT | CSTF_SHARED | CSTF_EXTERNAL;
     ret_desc->priority = CSPP_PREFERED;

     /* For hardware layers */
     ret_desc->access[CSAID_LAYER0] = CSAF_READ;
     ret_desc->access[CSAID_LAYER1] = CSAF_READ;
     ret_desc->access[CSAID_LAYER2] = CSAF_READ;
     ret_desc->access[CSAID_LAYER3] = CSAF_READ;
     ret_desc->access[CSAID_LAYER4] = CSAF_READ;
     ret_desc->access[CSAID_LAYER5] = CSAF_READ;
     ret_desc->access[CSAID_LAYER6] = CSAF_READ;
     ret_desc->access[CSAID_LAYER7] = CSAF_READ;

     snprintf( ret_desc->name, DFB_SURFACE_POOL_DESC_NAME_LENGTH, "CETVFB Resource Manager Pool" );

     return DFB_OK;
}

static DFBResult
tmmlDestroyPool( CoreSurfacePool *pool,
                 void            *pool_data,
                 void            *pool_local )
{
     D_DEBUG_AT( CETVFB_TMML, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );

     /* not implemented for now */

     return DFB_OK;
}

static DFBResult
tmmlAllocateBuffer( CoreSurfacePool       *pool,
                    void                  *pool_data,
                    void                  *pool_local,
                    CoreSurfaceBuffer     *buffer,
                    CoreSurfaceAllocation *allocation,
                    void                  *alloc_data )
{
     int                   ret;
     CoreSurface          *surface;
     cetvfbAllocationData *alloc = alloc_data;

     D_DEBUG_AT( CETVFB_TMML, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );

     surface = buffer->surface;
     D_MAGIC_ASSERT( surface, CoreSurface );

     dfb_surface_calc_buffer_size( surface, 8, 0, &alloc->pitch, &alloc->size );

     /* For layers, the surface->resource_id is a sequential number, starting at 0,
      * which is determined by the order of dfb_layers_register() calls in cetvfb.c.
      * Simple: the first call gets number 0.
      * However, to distinguish this from the 1-base WindowIDs in case of test code,
      * we add 2000 to get 2000,2001 and 2002 for layer 0,1 and 2.
      */
     alloc->resource_id = surface->resource_id;
     if( (surface->type & CSTF_LAYER) == CSTF_LAYER )
     {
         alloc->resource_id += 2000;
     }

     if (fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_TMML_ALLOCATE, alloc, &ret ))
          return DFB_FUSION;

     allocation->size   = alloc->size;
     allocation->offset = (unsigned long)alloc->address;

     return ret;
}

static DFBResult
tmmlDeallocateBuffer( CoreSurfacePool       *pool,
                      void                  *pool_data,
                      void                  *pool_local,
                      CoreSurfaceBuffer     *buffer,
                      CoreSurfaceAllocation *allocation,
                      void                  *alloc_data )
{
     int                   ret;
     cetvfbAllocationData *alloc = alloc_data;

     D_DEBUG_AT( CETVFB_TMML, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );

     if (fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_TMML_DEALLOCATE, alloc, &ret ))
          return DFB_FUSION;

     return DFB_OK;
}




/**************************************************************************************************/
#define IOCTL_IPC_CACHE_FLUSH          _IOR('S', 0x13, int)
#define IOCTL_IPC_CACHE_INVALIDATE     _IOR('S', 0x14, int)
#define IOCTL_IPC_CACHE_FLUSH_INV_ALL  _IOR('S', 0x15, int)

typedef struct {
        void* addr;
        unsigned int size;
} MemPages_t;

static int ShmemDriver = -1;

static void CacheFlush(void* addr, int size)
{
    int res=-1;
    MemPages_t pages = { addr, size };
   
    if ( ShmemDriver == -1 )
    {
        ShmemDriver = open("/dev/cachectl", O_RDWR );
        D_ASSERT( ShmemDriver != -1 );
    }
    
    if ( (ShmemDriver != -1) && (size != 0) )
    {
        res = ioctl( ShmemDriver, IOCTL_IPC_CACHE_FLUSH, &pages );
        D_ASSERT( res >= 0 );
    }   
}

static void CacheInvalidate(void* addr, int size)
{
    int res=-1;
    MemPages_t pages = { addr, size };
   
    if ( ShmemDriver == -1 )
    {
        ShmemDriver = open("/dev/cachectl", O_RDWR );
        D_ASSERT( ShmemDriver != -1 );
    }

    
    if ( (ShmemDriver != -1) && (size != 0) )
    {
        res = ioctl( ShmemDriver, IOCTL_IPC_CACHE_INVALIDATE, &pages );
        D_ASSERT( res >= 0 );
    }   
}

static void CacheFlushInvAll(void* addr, int size)
{
    int res=-1;
    MemPages_t pages = { addr, size };
   
    if ( ShmemDriver == -1 )
    {
        ShmemDriver = open("/dev/cachectl", O_RDWR );
        D_ASSERT( ShmemDriver != -1 );
    }

    
    if ( (ShmemDriver != -1) && (size != 0) )
    {
        res = ioctl( ShmemDriver, IOCTL_IPC_CACHE_FLUSH_INV_ALL, &pages );
        D_ASSERT( res >= 0 );
    }   
}



static DFBResult
tmmlLock( CoreSurfacePool       *pool,
          void                  *pool_data,
          void                  *pool_local,
          CoreSurfaceAllocation *allocation,
          void                  *alloc_data,
          CoreSurfaceBufferLock *lock )
{
     cetvfbAllocationData *alloc = alloc_data;

     D_DEBUG_AT( CETVFB_TMML_Lock, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( allocation, CoreSurfaceAllocation );
     D_MAGIC_ASSERT( lock, CoreSurfaceBufferLock );

     lock->addr   =                alloc->address;
     lock->phys   = (unsigned long)alloc->address;
     lock->offset = (unsigned long)alloc->address;
     lock->pitch  =                alloc->pitch;
     lock->handle =                alloc->address; /* whatever... */

     D_DEBUG_AT( CETVFB_TMML_Lock, "  -> Lock the surface. Virt: 0x%p Phys: 0x%04x\n",
         lock->addr, (unsigned int)lock->phys );

     if (lock->accessor & CSAID_CPU) {
          /* Invalidate read cache if GPU has written before */
          if (allocation->accessed[CSAID_GPU] & CSAF_WRITE) {
               D_DEBUG_AT( CETVFB_TMML_Cache, "  -> Invalidating cache...\n" );
//               CacheInvalidate(alloc->address, alloc->size);
               CacheFlushInvAll(alloc->address, alloc->size);
               D_DEBUG_AT( CETVFB_TMML_Cache, "  <- done.\n" );
          }
     }

     if (lock->accessor & CSAID_GPU) {
          /* Flush write cache */
          // PR brg36mgr#108355: [550r1cr2]Screen displays artefacts
          // Don't check if the previous lock was CPU or not, but always flush the cache.
          // Apparently this is not always reliable otherwise. Maybe the app does not always lock the buffer correctly?
          // In case the GPU accesses the buffer consecutively without CPU, it does not harm to flush twice
          // because the buffer will not be in the cache anyway.
//          if (allocation->accessed[CSAID_CPU] & CSAF_WRITE) {
               D_DEBUG_AT( CETVFB_TMML_Cache, "  -> Flushing cache...\n" );
//               CacheFlush(alloc->address, alloc->size);
               CacheFlushInvAll(alloc->address, alloc->size);
               D_DEBUG_AT( CETVFB_TMML_Cache, "  <- done.\n" );
//          }
     }

     return DFB_OK;
}


static DFBResult
tmmlUnlock( CoreSurfacePool       *pool,
            void                  *pool_data,
            void                  *pool_local,
            CoreSurfaceAllocation *allocation,
            void                  *alloc_data,
            CoreSurfaceBufferLock *lock )
{
     D_DEBUG_AT( CETVFB_TMML_Lock, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( allocation, CoreSurfaceAllocation );
     D_MAGIC_ASSERT( lock, CoreSurfaceBufferLock );

     return DFB_OK;
}

const SurfacePoolFuncs cetvfbSurfacePoolFuncs = {
     PoolDataSize:       tmmlPoolDataSize,
     PoolLocalDataSize:  tmmlPoolLocalDataSize,
     AllocationDataSize: tmmlAllocationDataSize,
     InitPool:           tmmlInitPool,
     DestroyPool:        tmmlDestroyPool,

     AllocateBuffer:     tmmlAllocateBuffer,
     DeallocateBuffer:   tmmlDeallocateBuffer,

     Lock:               tmmlLock,
     Unlock:             tmmlUnlock
};

