/*
   (c) Copyright 2000-2002  convergence integrated media GmbH.
   (c) Copyright 2002-2004  convergence GmbH.

   All rights reserved.

   Written by Denis Oliver Kropp <dok@directfb.org>,
              Andreas Hundt <andi@fischlustig.de>,
              Sven Neumann <neo@directfb.org> and
              Ville Syrjl <syrjala@sci.fi>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <stdio.h>

#include <directfb.h>

#include <fusion/fusion.h>
#include <fusion/shmalloc.h>

#include <core/core.h>
#include <core/coredefs.h>
#include <core/coretypes.h>
#include <core/layers.h>
#include <core/palette.h>
#include <core/surface.h>
#include <core/surface_buffer.h>
#include <core/system.h>

#include <gfx/convert.h>

#include <misc/conf.h>
#include <misc/util.h>

#include <direct/memcpy.h>
#include <direct/messages.h>

#define MAXCLUTNR (256)
#define COLORKEYNUMBER (0)
// Set to 1 if UI window detection is required.
// When a UI window is detected, a message will be sent to the 5100
// to disable the NM processing in this area.
#define NATURAL_MOTION_WINDOW (1) // set to 0 for now
// Define the layer ID that the UI uses
// This layer will be scanned for UI windows if NATURAL_MOTION_WINDOW is on
#define UI_LAYER_ID (1)
/**********************************************************************************************************************/

#include "cetvfb.h"
#include "primary.h"
#include "callbacks.h"

#define NM_LOGGING 0

#define CETVFB_PRIMARY_SUPPORTED_OPTIONS   (DLOP_ALPHACHANNEL | DLOP_OPACITY | DLOP_SRC_COLORKEY)

// Determines if the LAYER is a UI layer or not
#if NATURAL_MOTION_WINDOW
// Assumption is in this case that the layer is on UI_LAYER_ID
// and has a format ARGB4444 or ARGB8888.
#define IS_UI_LAYER( CETVLD ) \
        (                                               \
            ( (CETVLD)->layerId == UI_LAYER_ID )        \
         && ( ((CETVLD)->format == Dfb_PixFmtArgb4444)  \
           || ((CETVLD)->format == Dfb_PixFmtArgb8888 ) \
            )                                           \
        )
#else
// No need for this definition!
#define IS_UI_LAYER (0)
#endif

D_DEBUG_DOMAIN( CETVFB_Primary, "CETVFB/Primary", "Core System CETVFB Primary Screen & Layer" );

typedef struct _tmColor_t 
{
    unsigned char redOrY;
    unsigned char greenOrU;
    unsigned char blueOrV;
} tmColor_t;


/**********************************************************************************************************************/

static DFBResult callbacks_registered = DFB_INIT;

static DFBResult dfb_cetvfb_set_region   ( CoreDFB *core, CETVFBSetRegionData    *data );
static DFBResult dfb_cetvfb_update_screen( CoreDFB *core, CETVFBUpdateScreenData *data );
static DFBResult dfb_cetvfb_set_palette  ( CoreDFB *core, CETVFBSetPaletteData   *data );
static DFBResult dfb_cetvfb_remove_region( CoreDFB *core, CETVFBLayerData        *data );
static DFBResult dfb_cetvfb_set_level    ( CoreDFB *core, CETVFBSetGetLevelData  *data );
static DFBResult dfb_cetvfb_get_level    ( CoreDFB *core, CETVFBSetGetLevelData  *data );

#if NATURAL_MOTION_WINDOW
// Keeps the current status (on/off) and position of the UI window
static CETVFBNatMotionWdwData m_data={0,0,0,0,0};
#endif

/**********************************************************************************************************************/
/*== CALLBACK FUNCTIONS ==*/

static void (*pCetvfbGetOutputProperties) ( unsigned long *pwidth, unsigned long *pheight, unsigned long *pfieldrate, int *pprogressive );
static unsigned long (*pCetvfbGetSuppNrKeys)    ( int gfxlayer );
static void (*pCetvfbSetKey)                   ( int gfxlayer, unsigned long keynr, unsigned long keytype, unsigned long lowercolor, unsigned long uppercolor);
static void (*pCetvfbEnableKey)                ( int gfxlayer, unsigned long keynr, int enable );
static int  (*pCetvfbSetBuffer)                ( int gfxlayer, unsigned long width, unsigned long height, unsigned long stride, unsigned long fmtclass, unsigned long fmttype, void *pbuf );
static void (*pCetvfbSetClut)                  ( int gfxlayer, unsigned long nrentries, unsigned long *clut );
static void (*pCetvfbGetMinWindowSize)         ( int gfxlayer, unsigned long *pwidth, unsigned long *pheight );
static void (*pCetvfbGetMaxLayerWindowSize)    ( int gfxlayer, unsigned long *pwidth, unsigned long *pheight );
static void (*pCetvfbSetLayerWindow)           ( int gfxlayer, int ul_x, int ul_y, int lr_x, int lr_y );
static void (*pCetvfbSetDstWindow)             ( int gfxlayer, int ul_x, int ul_y, int lr_x, int lr_y );
static void (*pCetvfbSetSrcWindow)             ( int gfxlayer, int ul_x, int ul_y, int lr_x, int lr_y );
static void (*pCetvfbSetScaleMode)             ( int gfxlayer, unsigned long mode );
static void (*pCetvfbHideLayer)                ( int gfxlayer, int /*bool*/ hide );
static void (*pCetvfbActivateNewWindowSettings)( int gfxlayer, int /*bool*/ sync, unsigned long millisec );
static void (*pCetvfbSetGraphicLayerMode)      ( int gfxlayer, unsigned long layermode );
static void (*pCetvfbAcceleratedCopy) ( unsigned long SrcAddr, unsigned long DstAddr, int ul_x, int ul_y, int lr_x, int lr_y, int dst_x, int dst_y );
static void (*pCetvfbGetSharedMemoryLocation) ( CETVFBMemArea* uncached, CETVFBMemArea* cached, CETVFBMemArea* bis );
static void* (*pCetvfbMemoryAcquire) ( unsigned long resource_id, int size );
static int   (*pCetvfbMemoryRelease) ( unsigned long resource_id, int size, void* address );
#if NATURAL_MOTION_WINDOW
static void   (*pCetvfbSetNatMotionWindow) ( int gfxlayer, int ul_x, int ul_y, int lr_x, int lr_y, int enable );
#endif
static int (*pCetvfbSetLevel)      ( int gfxlayer, int level );
static int (*pCetvfbGetLevel)      ( int gfxlayer, int *level );

/*==   ==*/

static void ConvertPalette2Clut(unsigned int num_entries, DFBColor *entries, unsigned long *pClut);

#if NATURAL_MOTION_WINDOW
static void FindNaturalMotionWindow(
                CoreDFB               *core,
                CETVFBLayerData       *cetvld,
                CoreSurfaceBufferLock *lock
            );
#endif


/**********************************************************************************************************************/

static unsigned long
dfb_to_pixrgbfmt( DFBSurfacePixelFormat format, unsigned long *ret_class )
{
    switch (format) 
    {
        case DSPF_ARGB:
            if (ret_class)
                *ret_class = Dfb_PixFmtClassRgb;
            return Dfb_PixFmtArgb8888;

        case DSPF_ARGB4444:
            if (ret_class)
                *ret_class = Dfb_PixFmtClassRgb;
            return Dfb_PixFmtArgb4444;

        case DSPF_RGB16:
            if (ret_class)
                *ret_class = Dfb_PixFmtClassRgb;
            return Dfb_PixFmtRgb565;

        case DSPF_LUT8:
            if (ret_class)
                *ret_class = Dfb_PixFmtClassClut;
            return Dfb_PixFmtArgbClut8bpp;

        case DSPF_UYVY:
            if (ret_class)
                *ret_class = Dfb_PixFmtClassYuv;
            return Dfb_PixFmtUyvy;
        case DSPF_I420:
        case DSPF_NV12:
            if (ret_class)
                *ret_class = Dfb_PixFmtClassYuv;
            return Dfb_PixFmtYuv420;

        default:
            D_BUG( "unexpected pixelformat '%s'", dfb_pixelformat_name( format ) );
    }

    return 0;
}

/*
 * This function will translate DFB colour to MGR colour.
 * MGR needs a 'min' and a 'max' colour for colour keying.
 * We simply say min=max for LUT8, so we only need 1 input colour, which should be the colour key index.
 */
static void
get_color_minmax( DFBSurfacePixelFormat  format,
                  const DFBColorKey     *color,
                  unsigned int          *ret_min,  /* MGR color Nat32 */
                  unsigned int          *ret_max ) /* MGR color Nat32 */
{
     switch (format) {
          case DSPF_ARGB:
/*             ret_min->r   = color->r;
               ret_min->g   = color->g;
               ret_min->b   = color->b;
               ret_max->r   = color->r;
               ret_max->g   = color->g;
               ret_max->b   = color->b;*/
            
			   *ret_min = (unsigned int)((color->r <<16)+(color->g <<8)+(color->b));
			   *ret_max = *ret_min;
               break;

          case DSPF_ARGB4444:
/*             ret_min->r   = color->r & ~0xF;
               ret_min->g   = color->g & ~0xF;
               ret_min->b   = color->b & ~0xF;
               ret_max->r   = color->r |  0xF;
               ret_max->g   = color->g |  0xF;
               ret_max->b   = color->b |  0xF;*/

			   *ret_min = (unsigned int)(((color->r & ~0xF)<<16)+((color->g & ~0xF) <<8)+(color->b & ~0xF));
			   *ret_max = (unsigned int)(((color->r |  0xF)<<16)+((color->g |  0xF) <<8)+(color->b |  0xF));
               break;

          case DSPF_RGB16:
/*             ret_min->r   = color->r & ~0x7;
               ret_min->g   = color->g & ~0x3;
               ret_min->b   = color->b & ~0x7;
               ret_max->r   = color->r |  0x7;
               ret_max->g   = color->g |  0x3;
               ret_max->b   = color->b |  0x7;*/

			   *ret_min = (unsigned int)(((color->r & ~0x7)<<16)+((color->g & ~0x3) <<8)+(color->b & ~0x7));
			   *ret_max = (unsigned int)(((color->r |  0x7)<<16)+((color->g |  0x3) <<8)+(color->b |  0x7));
               break;

          case DSPF_LUT8:
               //D_WARN( "color keying does not work on UPPER layer" );

               /* do at least something... */
			   *ret_min = (unsigned int)((color->r <<16)+(color->g <<8)+(color->b));
			   *ret_max = *ret_min;
               break;

          default:
               D_BUG( "unexpected pixelformat '%s'", dfb_pixelformat_name( format ) );
     }
}





static void
SetSurfaceBuffer( CETVFBLayerData       *cetvld,
                  CoreSurface           *surface,
                  CoreSurfaceBufferLock *lock )
{
     CoreSurfaceAllocation *allocation;
     int width;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     D_ASSERT( cetvld != NULL );
     D_ASSERT( surface != NULL );
     D_ASSERT( lock != NULL );

     allocation = lock->allocation;
     D_ASSERT( allocation != NULL );

     D_DEBUG_AT( CETVFB_Primary, "  -> mix_SetBuffer( %dx%d, %d, Rgb %lx, %u, %p, %p )\n",
                 surface->config.size.w, surface->config.size.h, lock->pitch, cetvld->format, allocation->size,
                 lock->addr, cetvld );

      // PR brg36mgr#69514: Vertical line visible in mheg
      // The reason is that the width was an odd number, which causes NXP to read outside of the buffer
      // because they round up to the next even number.
      // Solution: round the width down to an even number in case the width is odd and 
      //           is equal to (or greater then, but this should not happen) the surface width
      // PR brg36mgr#100719: [tv550r1][Blocking][CRASH]DA-build 10/02 : crash when starting on digital channel.
      // In TV550 because of compressed buffers, this same trick needs to be done on the Setbuffer
      // call.
      // Not yet a nice solution to keep it matched with the source buffer, though!
      width = surface->config.size.w;
      if ( (width & 1) != 0 )
      {
        width--;
      }

     (void)(*pCetvfbSetBuffer)( cetvld->layerId,            /* int   gfxlayer */
                                width,                      /* Nat32 width    */
                                surface->config.size.h,     /* Nat32 height   */
                                lock->pitch,                /* Nat32 stride   */
                                cetvld->format_class,       /* Nat32 fmtclass */
                                cetvld->format,             /* Nat32 fmttype  */
                                lock->addr );               /* void *pbuf     */

}


/**********************************************************************************************************************/

static DFBResult
primaryInitScreen( CoreScreen           *screen,
                   CoreGraphicsDevice   *device,
                   void                 *driver_data,
                   void                 *screen_data,
                   DFBScreenDescription *description )
{
     static unsigned long dfb_ScreenWidth  = 1920;//1366;
     static unsigned long dfb_ScreenHeight = 1080;//768;
     static unsigned long dfb_ScreenFreq   = 50;
     static int           dfb_ProgressScan = Dfb_True;


     /* Set the screen capabilities. */
     description->caps = DSCCAPS_NONE;

     /* Set the screen name. */
     snprintf( description->name,
               DFB_SCREEN_DESC_NAME_LENGTH, "CETVFB Primary Screen" );


     /* Get the screen properties */
    (*pCetvfbGetOutputProperties)( &dfb_ScreenWidth, &dfb_ScreenHeight, &dfb_ScreenFreq, &dfb_ProgressScan );  /* !!! progressScan is Bool type !!! */


     fusion_skirmish_prevail( &dfb_cetvfb->lock );

     dfb_cetvfb->displayData.IntermediateWidth  = dfb_ScreenWidth;
     dfb_cetvfb->displayData.IntermediateHeight = dfb_ScreenHeight;
	 dfb_cetvfb->displayData.FinalDisplayWidth  = dfb_ScreenWidth;
	 dfb_cetvfb->displayData.FinalDisplayHeight = dfb_ScreenHeight;

     fusion_skirmish_dismiss( &dfb_cetvfb->lock );

	 D_DEBUG_AT( CETVFB_Primary, "InitScreen returned FinalDisplayWidth=%ld and FinalDisplayHeight=%ld\n", dfb_cetvfb->displayData.FinalDisplayWidth, dfb_cetvfb->displayData.FinalDisplayHeight);
	 D_DEBUG_AT( CETVFB_Primary, "InitScreen returned IntermediateWidth=%ld and IntermediateHeight=%ld\n", dfb_cetvfb->displayData.IntermediateWidth, dfb_cetvfb->displayData.IntermediateHeight);

     return DFB_OK;
}

static DFBResult
primaryGetScreenSize( CoreScreen *screen,
                      void       *driver_data,
                      void       *screen_data,
                      int        *ret_width,
                      int        *ret_height )
{
     D_ASSERT( dfb_cetvfb != NULL );
     D_ASSERT( ret_width != NULL );
     D_ASSERT( ret_height != NULL );


     fusion_skirmish_prevail( &dfb_cetvfb->lock );

     *ret_width  = dfb_cetvfb->displayData.IntermediateWidth;
     *ret_height = dfb_cetvfb->displayData.IntermediateHeight;

     fusion_skirmish_dismiss( &dfb_cetvfb->lock );

     D_DEBUG_AT( CETVFB_Primary, "GetScreenSize returned Width=%d and Height=%d\n", *ret_width, *ret_height );

     return DFB_OK;
}

ScreenFuncs cetvfbPrimaryScreenFuncs = 
{
     InitScreen:    primaryInitScreen,
     GetScreenSize: primaryGetScreenSize
};

/**********************************************************************************************************************/

static int
primaryLayerDataSize()
{
     return sizeof(CETVFBLayerData);
}

static int
primaryRegionDataSize()
{
     return 0;
}

static DFBResult
primaryInitLayer( CoreLayer                  *layer,
                  void                       *driver_data,
                  void                       *layer_data,
                  DFBDisplayLayerDescription *description,
                  DFBDisplayLayerConfig      *config,
                  DFBColorAdjustment         *adjustment )
{
     CETVFBLayerData  *cetvld = layer_data;
     int *data_driver = driver_data;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     cetvld->layerId  = (int)data_driver;

     D_DEBUG_AT( CETVFB_Primary, "Initializing Layer %u\n", cetvld->layerId );

     cetvld->srcColorKeyEnabled = false;
     cetvld->srcColorKey.r      = 0;
     cetvld->srcColorKey.g      = 0;
     cetvld->srcColorKey.b      = 0;
     cetvld->srcColorKey.index  = 0;

     cetvld->set_buffer         = false; 
     cetvld->unhide_layer       = false;

	 /** set the layer to scaled mode **/
//	 (*pCetvfbSetGraphicLayerMode)(cetvld->layerId, 0x20 /* LayerModeScaled */);


     /** set capabilities and type **/
     description->caps = DLCAPS_SURFACE | DLCAPS_ALPHACHANNEL | DLCAPS_OPACITY | DLCAPS_LEVELS |
                         DLCAPS_SRC_COLORKEY | DLCAPS_SCREEN_POSITION | DLCAPS_SCREEN_SIZE;

     description->type = DLTF_GRAPHICS;


     /** set name **/
     snprintf( description->name, DFB_DISPLAY_LAYER_DESC_NAME_LENGTH,
		 "CETVFB %u Layer", cetvld->layerId );

     /* fill out the default configuration */
     config->flags       = DLCONF_WIDTH       | DLCONF_HEIGHT |
                           DLCONF_PIXELFORMAT | DLCONF_BUFFERMODE | DLCONF_OPTIONS;
     config->buffermode  = DLBM_FRONTONLY;
     config->options     = DLOP_ALPHACHANNEL;

/*     if (dfb_config->mode.width)
          config->width  = dfb_config->mode.width;
     else*/
          config->width  = (cetvld->layerId == 0) ? 852 : 640;

/*     if (dfb_config->mode.height)
          config->height = dfb_config->mode.height;
     else*/
          config->height = 480;

/*     if (dfb_config->mode.format != DSPF_UNKNOWN)
          config->pixelformat = dfb_config->mode.format;
     else if (dfb_config->mode.depth > 0)
          config->pixelformat = dfb_pixelformat_for_depth( dfb_config->mode.depth );
     else*/
          config->pixelformat = DSPF_ARGB4444;

     /* Query maximum layer window size */
	 (*pCetvfbGetMaxLayerWindowSize)( cetvld->layerId , 
									&cetvld->maxLayerWindowWidth, 
									&cetvld->maxLayerWindowHeight );

     D_DEBUG_AT( CETVFB_Primary, "primaryInitLayer -> CetvfbGetMaxLayerWindowSize returned MaxWidth=%ld and MaxHeight= %ld\n",cetvld->maxLayerWindowWidth, cetvld->maxLayerWindowHeight);

     /* Query minimum window size */

     (*pCetvfbGetMinWindowSize)( cetvld->layerId,
								&cetvld->minWindowWidth,
								&cetvld->minWindowHeight );

     D_DEBUG_AT( CETVFB_Primary, "primaryInitLayer -> CetvfbGetMinWindowSize returned minWidth=%ld and minHeight= %ld\n",cetvld->minWindowWidth, cetvld->minWindowHeight);

     D_MAGIC_SET( cetvld, CETVFBLayerData );

     return DFB_OK;
}

static DFBResult
primaryTestRegion( CoreLayer                  *layer,
                   void                       *driver_data,
                   void                       *layer_data,
                   CoreLayerRegionConfig      *config,
                   CoreLayerRegionConfigFlags *failed )
{
     CETVFBLayerData               *cetvld  = layer_data;
     CoreLayerRegionConfigFlags  fail = 0;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

     if (config->options & ~CETVFB_PRIMARY_SUPPORTED_OPTIONS)
          fail |= CLRCF_OPTIONS;

     D_DEBUG_AT( CETVFB_Primary, "  -> src %d, %d - %dx%d\n",
                 DFB_RECTANGLE_VALS( &config->source ) );

     D_DEBUG_AT( CETVFB_Primary, "  -> dst %d, %d - %dx%d\n",
                 DFB_RECTANGLE_VALS( &config->dest ) );


     switch (config->format) {
          case DSPF_ARGB:
          case DSPF_ARGB4444:
          case DSPF_RGB16:
          case DSPF_LUT8:
          case DSPF_UYVY:
          case DSPF_I420:
          case DSPF_NV12:
               break;

          default:
               fail |= CLRCF_FORMAT;
     }


     if (config->source.w < cetvld->minWindowWidth)
          fail |= CLRCF_SOURCE;

     if (config->source.h < cetvld->minWindowHeight)
          fail |= CLRCF_SOURCE;


     if (config->dest.w < cetvld->minWindowWidth) {
          D_DEBUG_AT( CETVFB_Primary, "  -> %d < %lu!\n", config->dest.w, cetvld->minWindowWidth );
          fail |= CLRCF_DEST;
     }

     if (config->dest.h < cetvld->minWindowHeight) {
          D_DEBUG_AT( CETVFB_Primary, "  -> %d < %lu!\n", config->dest.h, cetvld->minWindowHeight );
          fail |= CLRCF_DEST;
     }


     if (config->dest.w > cetvld->maxLayerWindowWidth) {
          D_DEBUG_AT( CETVFB_Primary, "  -> %d > %lu!\n", config->dest.w, cetvld->maxLayerWindowWidth );
          fail |= CLRCF_DEST;
     }

     if (config->dest.h > cetvld->maxLayerWindowHeight) {
          D_DEBUG_AT( CETVFB_Primary, "  -> %d > %lu!\n", config->dest.h, cetvld->maxLayerWindowHeight );
          fail |= CLRCF_DEST;
     }

     if (failed)
          *failed = fail;

     if (fail) {
          D_DEBUG_AT( CETVFB_Primary, "  -> fail 0x%08x\n", fail );
          return DFB_UNSUPPORTED;
     }
     return DFB_OK;
}

static DFBResult
primaryAddRegion( CoreLayer             *layer,
                  void                  *driver_data,
                  void                  *layer_data,
                  void                  *region_data,
                  CoreLayerRegionConfig *config )
{
     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     return DFB_OK;
}

static DFBResult
primarySetRegion( CoreLayer                  *layer,
                  void                       *driver_data,
                  void                       *layer_data,
                  void                       *region_data,
                  CoreLayerRegionConfig      *config,
                  CoreLayerRegionConfigFlags  updated,
                  CoreSurface                *surface,
                  CorePalette                *palette,
                  CoreSurfaceBufferLock      *lock )
{
     DFBResult         ret;
     CETVFBSetRegionData  data;
     CETVFBLayerData     *cetvld = layer_data;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

     data.config    = *config;
     data.updated   = updated;
     data.surface   = surface;
     data.palette   = palette;
     data.layerData = layer_data;

     if (palette && (updated & CLRCF_PALETTE) && config->format == DSPF_LUT8)
	 {
          CETVFBSetPaletteData pdata;

          pdata.layerData = layer_data;
          pdata.palette   = palette;

          dfb_cetvfb_set_palette( dfb_cetvfb_core, &pdata );
     }

     ret = dfb_cetvfb_set_region( dfb_cetvfb_core, &data );
     if (ret)
          return ret;

     if (surface) {
          cetvld->surface    = surface;
          cetvld->set_buffer = true;
     }

     return DFB_OK;
}

static DFBResult
primaryRemoveRegion( CoreLayer             *layer,
                     void                  *driver_data,
                     void                  *layer_data,
                     void                  *region_data )
{
     CETVFBLayerData *cetvld = layer_data;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

     return dfb_cetvfb_remove_region( dfb_cetvfb_core, cetvld );
}

static DFBResult
primaryFlipRegion( CoreLayer             *layer,
                   void                  *driver_data,
                   void                  *layer_data,
                   void                  *region_data,
                   CoreSurface           *surface,
                   DFBSurfaceFlipFlags    flags,
                   CoreSurfaceBufferLock *lock )
{
     CETVFBLayerData *cetvld = layer_data;
     CETVFBUpdateScreenData data;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

     dfb_surface_flip( surface, false );

     data.region    = (DFBRegion){ 0, 0, surface->config.size.w - 1, surface->config.size.h - 1 };
     data.layerData = layer_data;
     data.lock      = lock;

     cetvld->set_buffer = true;

     return dfb_cetvfb_update_screen( dfb_cetvfb_core, &data );
}

static DFBResult
primaryUpdateRegion( CoreLayer             *layer,
                     void                  *driver_data,
                     void                  *layer_data,
                     void                  *region_data,
                     CoreSurface           *surface,
                     const DFBRegion       *update,
                     CoreSurfaceBufferLock *lock )
{
     CETVFBUpdateScreenData  data;
     CETVFBLayerData        *cetvld = layer_data;
     DFBResult               result = DFB_OK;

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

     if (update) 
     {
          data.region = *update;
     }
     else
     {
          data.region = (DFBRegion){ 0, 0, surface->config.size.w - 1, surface->config.size.h - 1 };
     }
     data.layerData = layer_data;
     data.lock      = lock;

     D_DEBUG_AT( CETVFB_Primary, "%s( %d,%d - %dx%d )\n", __FUNCTION__, DFB_RECTANGLE_VALS_FROM_REGION(&data.region) );

     /* this function is called whenever the buffer content changes.
         When compression is used by NXP (from TV550 onwards), we need to explicitly do a set_buffer call,
         since NXP needs a trigger to start the compressor on this buffer each time it changes.
         For the rest, this compression is transparent for directfb and platform.
         PR brg36mgr#101072: Do not do this compression in case of YUV inputs because there is no graphics compression
         done in this case. The YUV buffers are shown on the video layer.
      */
    if (   (surface->config.format != DSPF_UYVY)
        && (surface->config.format != DSPF_I420)
        && (surface->config.format != DSPF_NV21)
       )
    {      
        cetvld->set_buffer = true;
    }

#if NATURAL_MOTION_WINDOW
    // Only scan for a UI window in case of the UI layer
    if ( IS_UI_LAYER( cetvld ) )
    {
        FindNaturalMotionWindow(dfb_cetvfb_core, cetvld,lock);
    }
#endif         

     if( cetvld->set_buffer == true )
     {
         result = dfb_cetvfb_update_screen( dfb_cetvfb_core, &data );
     }

     return result;
}

static DFBResult
primarySetLevel( CoreLayer             *layer,
                     void                  *driver_data,
                     void                  *layer_data,
                     int                  level )
{
     CETVFBSetGetLevelData   data;
     CETVFBLayerData *cetvld = layer_data;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

     data.layerData  = layer_data;
     data.level      = level;

     return dfb_cetvfb_set_level( dfb_cetvfb_core, &data );
}
static DFBResult
primaryGetLevel( CoreLayer             *layer,
                     void                  *driver_data,
                     void                  *layer_data,
                     int                  *retlevel )
{
     CETVFBSetGetLevelData   data;
     CETVFBLayerData *cetvld = layer_data;
     DFBResult dfbresult;
     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );
     data.layerData     = layer_data;
     //data.retlevel      = retlevel;
     dfbresult =dfb_cetvfb_get_level( dfb_cetvfb_core, &data );
     *retlevel = data.level;
     return dfbresult;
}


DisplayLayerFuncs cetvfbPrimaryLayerFuncs = 
{
     LayerDataSize:      primaryLayerDataSize,
     RegionDataSize:     primaryRegionDataSize,
     InitLayer:          primaryInitLayer,

     TestRegion:         primaryTestRegion,
     AddRegion:          primaryAddRegion,
     SetRegion:          primarySetRegion,
     RemoveRegion:       primaryRemoveRegion,
     FlipRegion:         primaryFlipRegion,
     UpdateRegion:       primaryUpdateRegion,
     SetLevel:           primarySetLevel,
     GetLevel:           primaryGetLevel
};

/**********************************************************************************************************************/

static DFBResult
set_region( CETVFBLayerData            *cetvld,
            CoreLayerRegionConfig      *config,
            CoreLayerRegionConfigFlags  updated,
            CoreSurface                *surface,
            CorePalette                *palette )
{
     int    ul_x = 0;
     int    ul_y = 0;
     int    lr_x = 0;
     int    lr_y = 0;

     D_DEBUG_AT( CETVFB_Primary, "%s( %p, %p )\n", __FUNCTION__, cetvld, config );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );
     D_ASSERT( dfb_cetvfb != NULL );

    /** check if pixel format has changed **/
     if (updated & CLRCF_FORMAT) 
     {
          D_ASSERT( config != NULL );

          D_DEBUG_AT( CETVFB_Primary, "  -> set_region : pixel format has changed!\n");
          D_DEBUG_AT( CETVFB_Primary, "  -> format %s\n", dfb_pixelformat_name(config->format) );

          cetvld->format = dfb_to_pixrgbfmt( config->format, &cetvld->format_class );
     }

    /** check if source or destination windows have changed **/
     if (updated & (CLRCF_SOURCE | CLRCF_DEST)) 
     {
          D_ASSERT( config != NULL );

          D_DEBUG_AT( CETVFB_Primary, "  -> set_region : src or dst windows have changed!\n" );

          D_DEBUG_AT( CETVFB_Primary, "  -> src %d, %d - %dx%d\n",
                      config->source.x, config->source.y, config->source.w, config->source.h );

          D_DEBUG_AT( CETVFB_Primary, "  -> dst %d, %d - %dx%d\n",
                      config->dest.x, config->dest.y, config->dest.w, config->dest.h );

          /* Set source window. */
          ul_x = config->source.x;
          ul_y = config->source.y;
          // PR brg36mgr#69514: Vertical line visible in mheg
          // The reason is that the width was an odd number, which causes NXP to read outside of the buffer
          // because they round up to the next even number.
          // Solution: round the width down to an even number in case the width is odd and 
          //           is equal to (or greater then, but this should not happen) the surface width
          lr_x = config->source.x + config->source.w;
          if (   ( (config->source.w & 1) != 0 )
              && ( config->source.w >= surface->config.size.w )
             )
          {
            lr_x = config->source.x + ( config->source.w - 1 );
          }
          lr_y = config->source.y + config->source.h;

          D_DEBUG_AT( CETVFB_Primary, "  -> mix_SetSrcWindow( %d, %d, %d, %d )\n",
                      ul_x, ul_y, lr_x, lr_y );

          (*pCetvfbSetSrcWindow)( cetvld->layerId, ul_x , ul_y, lr_x, lr_y);

          /* Set destination window. */
          ul_x = config->dest.x;
          ul_y = config->dest.y;
          lr_x = config->dest.x + config->dest.w;
          lr_y = config->dest.y + config->dest.h;

          D_DEBUG_AT( CETVFB_Primary, "  -> mix_SetDstWindow( %d, %d, %d, %d )\n",
                      ul_x, ul_y, lr_x, lr_y );

          (*pCetvfbSetDstWindow)( cetvld->layerId, ul_x , ul_y, lr_x, lr_y);

          /* Set layer window. */
          ul_x = 0;
          ul_y = 0;
          lr_x = dfb_cetvfb->displayData.IntermediateWidth; //dfb_ScreenWidth;// coming from the Output Properties
          lr_y = dfb_cetvfb->displayData.IntermediateHeight;//dfb_ScreenHeight;// coming from the Output Properties

          D_DEBUG_AT( CETVFB_Primary, "  -> mix_SetLayerWindow( %d, %d, %d, %d )\n",
                      ul_x, ul_y, lr_x, lr_y );

          (*pCetvfbSetLayerWindow)( cetvld->layerId, ul_x , ul_y, lr_x, lr_y );

          /* Activate settings. */
          D_DEBUG_AT( CETVFB_Primary, "  -> mix_ActivateNewWindowSettings( Dfb_True, 0 )\n" );

          (*pCetvfbActivateNewWindowSettings)( cetvld->layerId, Dfb_False, 0 );

     }


     /** Check if source color key has changed **/
     if ((updated & CLRCF_SRCKEY) && (config->options & DLOP_SRC_COLORKEY)) 
     {
          unsigned int color_min = 0;
          unsigned int color_max = 0;

          /* Get the min max colors for the color keying */
          get_color_minmax( config->format, &config->src_key, &color_min, &color_max );

          D_DEBUG_AT( CETVFB_Primary, "  -> set_region : src color key has changed!\n" );
          D_DEBUG_AT( CETVFB_Primary, "  -> CetvfbSetKey( lower=%d, upper=%d )\n", color_min, color_max );

          (*pCetvfbSetKey)( cetvld->layerId, COLORKEYNUMBER, Dfb_KeySrcColorKey /*Nat32 keytype*/, color_min /*lowercolor*/, color_max /*uppercolor*/);
          cetvld->srcColorKey = config->src_key;
     }


     /** check if the options have changed **/
     if (updated & CLRCF_OPTIONS) 
     {
          bool enabled;
         
          enabled = !!(config->options & DLOP_SRC_COLORKEY); /* "!!" is used to make it a boolean */     
          D_DEBUG_AT( CETVFB_Primary, "  -> set_region : Enable Key\n" );
          (*pCetvfbEnableKey)( cetvld->layerId, COLORKEYNUMBER, enabled ? Dfb_True : Dfb_False );
          cetvld->srcColorKeyEnabled = enabled;
     }

     /** check if a new buffer needs to be passed */
     if (surface && (updated & (CLRCF_SURFACE | CLRCF_SOURCE | CLRCF_DEST | CLRCF_OPTIONS | CLRCF_SRCKEY | CLRCF_PALETTE)))
          cetvld->set_buffer = true;

     D_DEBUG_AT( CETVFB_Primary, "  -> set_buffer: %s\n", cetvld->set_buffer ? "true" : "false" );


     /* Finally show the layer.
        We will delay this if we need to do a set_buffer!
        Problem is here that we FIRST need to set the layer and THEN unhide it, to avoid artefacts. */
     if( cetvld->set_buffer == true )
     {
         /* delayed unhide */
         cetvld->unhide_layer = true;
     }
     else
     {
         D_DEBUG_AT( CETVFB_Primary, "  -> mix_Hide( %d, Dfb_False )\n", cetvld->layerId );
         (*pCetvfbHideLayer)( cetvld->layerId, Dfb_False );
     }

     return DFB_OK;
}

static DFBResult
update_screen( CETVFBLayerData *cetvld,
               int x, int y, int w, int h,
               CoreSurfaceBufferLock *lock )
{
     CoreSurface   *surface;

//   D_DEBUG_AT( CETVFB_Primary, "%s( %p, %d, %d - %dx%d )\n", __FUNCTION__, cetvld, x, y, w, h );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );
     D_ASSERT( dfb_cetvfb != NULL );

     surface = cetvld->surface;
     D_ASSERT( surface != NULL );

     /** check if a new buffer is available for this layer, and exit **/

     if (cetvld->set_buffer) {
          SetSurfaceBuffer( cetvld, surface, lock );

          cetvld->set_buffer = false;

          if (cetvld->unhide_layer) {
               D_DEBUG_AT( CETVFB_Primary, "  -> mix_Hide( %d, Dfb_False ) (delayed)\n", cetvld->layerId );

               (*pCetvfbHideLayer)( cetvld->layerId, Dfb_False );
               cetvld->unhide_layer = false;
          }
     }

     return DFB_OK;
}

/**********************************************************************************************************************/

DFBResult
dfb_cetvfb_set_region_handler( CETVFBSetRegionData *data )
{
     DFBResult     ret;
     CETVFBLayerData *cetvld = data->layerData;

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

     fusion_skirmish_prevail( &dfb_cetvfb->lock );

     ret = set_region( cetvld, &data->config, data->updated, data->surface, data->palette );

     fusion_skirmish_dismiss( &dfb_cetvfb->lock );

     return ret;
}

DFBResult
dfb_cetvfb_update_screen_handler( CETVFBUpdateScreenData *data )
{
     DFBResult ret;

     D_ASSERT( data != NULL );

     D_MAGIC_ASSERT( data->layerData, CETVFBLayerData );

     fusion_skirmish_prevail( &dfb_cetvfb->lock );

     ret = update_screen( data->layerData, DFB_RECTANGLE_VALS_FROM_REGION( &data->region ), data->lock );

     fusion_skirmish_dismiss( &dfb_cetvfb->lock );

     return ret;
}

DFBResult
dfb_cetvfb_set_palette_handler( CETVFBSetPaletteData *data )
{
     CETVFBLayerData  *cetvld  = data->layerData;
     CorePalette      *palette = data->palette;
     unsigned long    Clut[MAXCLUTNR]   = {0};

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );


     fusion_skirmish_prevail( &dfb_cetvfb->lock );



     ConvertPalette2Clut(palette->num_entries, palette->entries, (unsigned long *)Clut);


     (*pCetvfbSetClut)( cetvld->layerId,		/*gfxlayer*/
                        palette->num_entries,	/*Nat32 nrentries*/
                        (unsigned long *)Clut   /*Nat32 *clut */  /* Note: this virtual address will be marshalled later by DVP*/
                      );

     /* format is LUT8, but palette is changed. 
      * since we set the colour key as RGB rather than index, we probably need to update! */
     if( cetvld->srcColorKeyEnabled == true )
     {
          int index = cetvld->srcColorKey.index;
          if( index < palette->num_entries )
          {
               unsigned int color_min = 0;
               unsigned int color_max = 0;
               DFBColor*    color     = palette->entries + index;

               cetvld->srcColorKey.r = color->r;
               cetvld->srcColorKey.g = color->g;
               cetvld->srcColorKey.b = color->b;

               /* Get the min max colors for the color keying */
               get_color_minmax( DSPF_LUT8, &(cetvld->srcColorKey), &color_min, &color_max );

               (*pCetvfbSetKey)( cetvld->layerId, COLORKEYNUMBER, Dfb_KeySrcColorKey /*Nat32 keytype*/, color_min /*lowercolor*/, color_max /*uppercolor*/);
          }
     }

     /* After setting the pallette we must do a setbuffer. 
        This is according to DVP specifications: in this way, they can synchronize all colour keying stuff.
        Therefore, we make set_buffer true. According to Denis, this function will be followed by an UpdateRegion.
      */
     cetvld->set_buffer = true;

     fusion_skirmish_dismiss( &dfb_cetvfb->lock );

     return DFB_OK;
}

DFBResult
dfb_cetvfb_remove_region_handler( CETVFBLayerData *cetvld )
{

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     fusion_skirmish_prevail( &dfb_cetvfb->lock );

     D_DEBUG_AT( CETVFB_Primary, "  -> mix_Hide( %d, Dfb_True )\n", cetvld->layerId );

     (*pCetvfbHideLayer)( cetvld->layerId, Dfb_True );
     
#if NATURAL_MOTION_WINDOW
     if ( IS_UI_LAYER( cetvld ) )
     {
        m_data.enable = 0;
        if (pCetvfbSetNatMotionWindow)
        {
            (*pCetvfbSetNatMotionWindow)( m_data.gfxlayer, m_data.ul_x, m_data.ul_y, m_data.lr_x, m_data.lr_y, m_data.enable);
        }
     }
#endif

     fusion_skirmish_dismiss( &dfb_cetvfb->lock );

     return DFB_OK;
}

DFBResult
dfb_cetvfb_set_level_handler( CETVFBSetGetLevelData *data )
{
     DFBResult ret = DFB_OK;
     CETVFBLayerData  *cetvld  = data->layerData;

     D_ASSERT( data != NULL );

     D_MAGIC_ASSERT( data->layerData, CETVFBLayerData );

     fusion_skirmish_prevail( &dfb_cetvfb->lock );

     if (pCetvfbSetLevel)
     {
        ret=(*pCetvfbSetLevel)( cetvld->layerId, data->level );
     }
     fusion_skirmish_dismiss( &dfb_cetvfb->lock );

     return ret;
}


DFBResult
dfb_cetvfb_get_level_handler( CETVFBSetGetLevelData *data )
{
     DFBResult ret = DFB_OK;
     CETVFBLayerData  *cetvld  = data->layerData;

     D_ASSERT( data != NULL );

     D_MAGIC_ASSERT( data->layerData, CETVFBLayerData );

     fusion_skirmish_prevail( &dfb_cetvfb->lock );

     if (pCetvfbGetLevel)
     {
        ret=(*pCetvfbGetLevel)( cetvld->layerId, &(data->level) );
     }
     fusion_skirmish_dismiss( &dfb_cetvfb->lock );

     return ret;
}

/**********************************************************************************************************************/

static DFBResult
dfb_cetvfb_set_nat_motion_window( CoreDFB *core, CETVFBNatMotionWdwData *data )
{
     int               ret;
     CETVFBNatMotionWdwData *tmp = NULL;

     D_ASSERT( core != NULL );
     D_ASSERT( data != NULL );
     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     if ( dfb_core_is_master( core ) )
     {
          return dfb_cetvfb_set_nat_motion_handler( data );
     }

     if ( !fusion_is_shared( dfb_cetvfb_world, data ) )
     {
          tmp = SHMALLOC( dfb_cetvfb->shmpool, sizeof(*tmp) );
          if (!tmp)
          {
               return DFB_NOSYSTEMMEMORY;
          }

          direct_memcpy( tmp, data, sizeof(*tmp) );
     }

     fusion_call_execute(
                    &dfb_cetvfb->call,
                    FCEF_NONE,
                    CETVFB_SET_NAT_MOTION_WDW,
                    tmp ? tmp : data,
                    &ret
                );

     if ( tmp )
     {
          SHFREE( dfb_cetvfb->shmpool, tmp );
     }

     return ret;
}

static DFBResult
dfb_cetvfb_set_region( CoreDFB *core, CETVFBSetRegionData *data )
{
     int               ret;
     CETVFBSetRegionData *tmp = NULL;

     D_ASSERT( core != NULL );
     D_ASSERT( data != NULL );
     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     if (dfb_core_is_master( core ))
          return dfb_cetvfb_set_region_handler( data );

     if (!fusion_is_shared( dfb_cetvfb_world, data )) {
          tmp = SHMALLOC( dfb_cetvfb->shmpool, sizeof(*tmp) );
          if (!tmp)
               return DFB_NOSYSTEMMEMORY;

          direct_memcpy( tmp, data, sizeof(*tmp) );
     }

     fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_SET_REGION,
                          tmp ? tmp : data, &ret );

     if (tmp)
          SHFREE( dfb_cetvfb->shmpool, tmp );

     return ret;
}

static DFBResult
dfb_cetvfb_update_screen( CoreDFB *core, CETVFBUpdateScreenData *data )
{
     int                  ret;
     CETVFBUpdateScreenData *tmp = NULL;

     D_ASSERT( data != NULL );
     D_MAGIC_ASSERT( data->layerData, CETVFBLayerData );
     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     if (dfb_core_is_master( core ))
          return dfb_cetvfb_update_screen_handler( data );

     if (!fusion_is_shared( dfb_cetvfb_world, data )) {
          tmp = SHMALLOC( dfb_cetvfb->shmpool, sizeof(*data) );
          if (!tmp)
               return DFB_NOSYSTEMMEMORY;

          direct_memcpy( tmp, data, sizeof(*data) );

          D_MAGIC_ASSERT( tmp->layerData, CETVFBLayerData );
     }

     /* FIXME: removed ONEWAY because of SHFREE */
     fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_UPDATE_SCREEN,
                          tmp ? tmp : data, &ret );

     if (tmp)
          SHFREE( dfb_cetvfb->shmpool, tmp );

     return DFB_OK;
}

static DFBResult
dfb_cetvfb_set_palette( CoreDFB *core, CETVFBSetPaletteData *data )
{
//   int                ret;
     CETVFBSetPaletteData *tmp = NULL;

     D_ASSERT( data != NULL );
     D_MAGIC_ASSERT( data->layerData, CETVFBLayerData );
     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     if (dfb_core_is_master( core ))
          return dfb_cetvfb_set_palette_handler( data );

     if (!fusion_is_shared( dfb_cetvfb_world, data )) {

          tmp = SHMALLOC( dfb_cetvfb->shmpool, sizeof(*data) );
          if (!tmp)
               return DFB_NOSYSTEMMEMORY;

          direct_memcpy( tmp, data, sizeof(*data) );

          D_MAGIC_ASSERT( tmp->layerData, CETVFBLayerData );
     }

     fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_SET_PALETTE,
                          tmp ? tmp : data, NULL );

     if (tmp)
          SHFREE( dfb_cetvfb->shmpool, tmp );

     return DFB_OK;
}

static void ConvertPalette2Clut(unsigned int num_entries, DFBColor *entries, unsigned long *pClut)
{
     int i=0;

     for ( i = 0; i < num_entries; i++ )
     {
          pClut[i] = (u32)((entries->a <<24)+(entries->r <<16)+(entries->g <<8)+(entries->b));
          // printf("clut%u: a=%u r=%u g=%u b=%u Nat32=%lu\n",i,entries->a,entries->r,entries->g,entries->b,pClut[i]);
          entries++;
     }

  /******************************************/
  /* typedef struct {                       */
  /*    u8          a;   || alpha channel   */
  /*    u8          r;   || red channel     */
  /*    u8          g;   || green channel   */
  /*    u8          b;   || blue channel    */
  /* } DFBColor;                            */
  /******************************************/

  /******************************************/
  /* typedef struct {                       */
  /*    u8         index;|| color index     */
  /*    u8         r;    || red channel     */
  /*    u8         g;    || green channel   */
  /*    u8         b;    || blue channel    */
  /* } DFBColorKey;                         */
  /******************************************/

     return;
}

static DFBResult
dfb_cetvfb_remove_region( CoreDFB *core, CETVFBLayerData *cetvld )
{
     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );
     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     if (dfb_core_is_master( core ))
          return dfb_cetvfb_remove_region_handler( cetvld );

     fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_REMOVE_REGION, cetvld, NULL );

     return DFB_OK;
}

static DFBResult
dfb_cetvfb_set_level( CoreDFB *core, CETVFBSetGetLevelData *data )
{
//   int                ret;
     CETVFBSetGetLevelData *tmp = NULL;

     D_ASSERT( data != NULL );
     D_MAGIC_ASSERT( data->layerData, CETVFBLayerData );
     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     if (dfb_core_is_master( core ))
          return dfb_cetvfb_set_level_handler( data );

     if (!fusion_is_shared( dfb_cetvfb_world, data )) {

          tmp = SHMALLOC( dfb_cetvfb->shmpool, sizeof(*data) );
          if (!tmp)
               return DFB_NOSYSTEMMEMORY;

          direct_memcpy( tmp, data, sizeof(*data) );

          D_MAGIC_ASSERT( tmp->layerData, CETVFBLayerData );
     }

     fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_SET_LEVEL,
                          tmp ? tmp : data, NULL );
     
     if (tmp)
          SHFREE( dfb_cetvfb->shmpool, tmp );

     return DFB_OK;
}

static DFBResult
dfb_cetvfb_get_level( CoreDFB *core, CETVFBSetGetLevelData *data )
{
//   int                ret;
     CETVFBSetGetLevelData *tmp = NULL;

     D_ASSERT( data != NULL );
     D_MAGIC_ASSERT( data->layerData, CETVFBLayerData );
     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     if (dfb_core_is_master( core ))
          return dfb_cetvfb_get_level_handler( data );

     if (!fusion_is_shared( dfb_cetvfb_world, data )) {

          tmp = SHMALLOC( dfb_cetvfb->shmpool, sizeof(*data) );
          if (!tmp)
               return DFB_NOSYSTEMMEMORY;

          direct_memcpy( tmp, data, sizeof(*data) );

          D_MAGIC_ASSERT( tmp->layerData, CETVFBLayerData );
     }

     fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_GET_LEVEL,
                          tmp ? tmp : data, NULL );

     if (tmp)
        data-> level = tmp->level;

     if (tmp)
          SHFREE( dfb_cetvfb->shmpool, tmp );

     return DFB_OK;
}
// CALLBACKS REGISTRATION

void Dfb_RegisterCetvCallBackTable( CETVFBCallBackTable cetvfbfct )
{
     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     // ptrFunc = ptrFunc
     pCetvfbGetOutputProperties          = cetvfbfct.GetOutputProperties;
     pCetvfbGetSuppNrKeys                = cetvfbfct.GetSuppNrKeys;
     pCetvfbSetKey                       = cetvfbfct.SetKey;
     pCetvfbEnableKey                    = cetvfbfct.EnableKey;
     pCetvfbSetBuffer                    = cetvfbfct.SetBuffer;
     pCetvfbSetClut                      = cetvfbfct.SetClut;
     pCetvfbGetMinWindowSize             = cetvfbfct.GetMinWindowSize;
     pCetvfbGetMaxLayerWindowSize        = cetvfbfct.GetMaxLayerWindowSize;
     pCetvfbSetLayerWindow               = cetvfbfct.SetLayerWindow;
     pCetvfbSetDstWindow                 = cetvfbfct.SetDstWindow;
     pCetvfbSetSrcWindow                 = cetvfbfct.SetSrcWindow;
     pCetvfbSetScaleMode                 = cetvfbfct.SetScaleMode;
     pCetvfbHideLayer                    = cetvfbfct.HideLayer;
     pCetvfbActivateNewWindowSettings    = cetvfbfct.ActivateNewWindowSettings;
     pCetvfbSetGraphicLayerMode          = cetvfbfct.SetGraphicLayerMode;
     pCetvfbAcceleratedCopy              = cetvfbfct.AcceleratedCopy;

     pCetvfbGetSharedMemoryLocation      = cetvfbfct.GetSharedMemoryLocation;
     pCetvfbMemoryAcquire                = cetvfbfct.MemoryAcquire;
     pCetvfbMemoryRelease                = cetvfbfct.MemoryRelease;

#if NATURAL_MOTION_WINDOW  
     pCetvfbSetNatMotionWindow           = cetvfbfct.SetNatMotionWindow;
#endif
     pCetvfbSetLevel                     = cetvfbfct.SetLevel;
     pCetvfbGetLevel                     = cetvfbfct.GetLevel;

     callbacks_registered = DFB_OK;
}

/* this is used 'internally' in the system module,
 * to pass the pointers to other components (such as cetvfb.c or cetvfb_surfacepool.c) */
CETVFBCallBackTable dfb_cetvfb_get_callbacks( )
{
     CETVFBCallBackTable cetvfbfct = {0};

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );
     D_ASSERT( callbacks_registered == DFB_OK );

     if( callbacks_registered == DFB_OK )
     {
          cetvfbfct.GetOutputProperties         = pCetvfbGetOutputProperties;
          cetvfbfct.GetSuppNrKeys               = pCetvfbGetSuppNrKeys;
          cetvfbfct.SetKey                      = pCetvfbSetKey;
          cetvfbfct.EnableKey                   = pCetvfbEnableKey;
          cetvfbfct.SetBuffer                   = pCetvfbSetBuffer;
          cetvfbfct.SetClut                     = pCetvfbSetClut;
          cetvfbfct.GetMinWindowSize            = pCetvfbGetMinWindowSize;
          cetvfbfct.GetMaxLayerWindowSize       = pCetvfbGetMaxLayerWindowSize;
          cetvfbfct.SetLayerWindow              = pCetvfbSetLayerWindow;
          cetvfbfct.SetDstWindow                = pCetvfbSetDstWindow;
          cetvfbfct.SetSrcWindow                = pCetvfbSetSrcWindow;
          cetvfbfct.SetScaleMode                = pCetvfbSetScaleMode;
          cetvfbfct.HideLayer                   = pCetvfbHideLayer;
          cetvfbfct.ActivateNewWindowSettings   = pCetvfbActivateNewWindowSettings;
          cetvfbfct.SetGraphicLayerMode         = pCetvfbSetGraphicLayerMode;
          cetvfbfct.AcceleratedCopy             = pCetvfbAcceleratedCopy;

          cetvfbfct.GetSharedMemoryLocation     = pCetvfbGetSharedMemoryLocation;
          cetvfbfct.MemoryAcquire               = pCetvfbMemoryAcquire;
          cetvfbfct.MemoryRelease               = pCetvfbMemoryRelease;

#if NATURAL_MOTION_WINDOW  
          cetvfbfct.SetNatMotionWindow          = pCetvfbSetNatMotionWindow;
#endif //NATURAL_MOTION_WINDOW  
          cetvfbfct.SetLevel                    = pCetvfbSetLevel;
          cetvfbfct.GetLevel                    = pCetvfbGetLevel;
     }

     return cetvfbfct;
}

// Add code of PED from the 1UX branch.
// Added support for ARGB8888.
#if NATURAL_MOTION_WINDOW

#define MIN_WIDTH	(32)   // should be module 2 bigger then minimum window
#define MIN_HEIGHT  (32)


#ifndef MIN
#define MIN(a,b) ((a) < (b) ? (a) : (b))
#endif

static int m_nmbtest = 0;
static int stride = 0;

#if 0
bool IS_OPAQUE( int test_x, int test_y, unsigned short* buffer)
{	
    m_nmbtest ++;
    printf("Test Pix %d %d val %x\n",test_x,test_y,buffer[test_y*stride+test_x]);
	return( (buffer[test_y*stride+test_x] & 0xf000)!=0 );
}
#else
// Implicitly uses cetvld->format!
#define IS_OPAQUE( xx, yy, buf)   \
                (       \
                    (   \
                        ( cetvld->format == Dfb_PixFmtArgb4444 ) \
                     && ( ( ((unsigned short*)buf)[(yy)*stride+(xx)] & 0xf000 ) != 0 ) \
                    )   \
                 || (   \
                        ( cetvld->format == Dfb_PixFmtArgb8888 ) \
                     && ( ( ((unsigned long*)buf)[(yy)*stride+(xx)] & 0xff000000 ) != 0 ) \
                    )   \
                )

#endif

#define DIV_UP_BY_2(a)  ( (a) > 1 ? ((a)+1)>>1 : 0 )

void FindNaturalMotionWindow(
                CoreDFB               *core,
                CETVFBLayerData       *cetvld,
                CoreSurfaceBufferLock *lock
            )
{
    static bool m_WindowOn = false;
    static int ul_x,ul_y,lr_x,lr_y; /* top left and bottom right corner of UI window*/
    CoreSurface           *surface= cetvld->surface;

    int test_x,test_y; /* to find UL point */
    int offset_x,offset_y;

    int resolution_x = surface->config.size.w; /* resolution of UI window */
    int resolution_y = surface->config.size.h; 
    unsigned char *buffer= lock->addr;

    int x,y;
    int left=0,right=resolution_x,upper=0,bottom=resolution_y;
    int offset=0;
    int count;

#define MAXHEIGHT (1080/2)
#define DETECTSIZE ((MAXHEIGHT/MIN_HEIGHT)+1)
    int detect_pos[ DETECTSIZE ] = {0};
    int detect_cnt[ DETECTSIZE ] = {0};
    int detect_left[DETECTSIZE]  = {0};
    int detect_right[DETECTSIZE] = {0};

    int y_index=0;

    int detect_max = 0;
    int detect_max_start = 0;
    int detect_max_end = 0;
    int detect_max_running = false;
    int detect_max_left = 0;
    int detect_max_right = 0;
    int enable = 0;


    int i;

    if(resolution_y > MAXHEIGHT)
    {
        return;
    }

    m_nmbtest = 0;
    
    // Stride is expressed in pixels, not in bytes!
    switch ( cetvld->format )
    {
    case Dfb_PixFmtArgb4444:
        stride = lock->pitch>>1;
        break;
    case Dfb_PixFmtArgb8888:
        stride = lock->pitch>>2;
        break;
    default: // Invalid format!!
        stride = lock->pitch>>2;
    }

    /* todo do NOT go out of resolution */

    /******* test same window as last time *******/
#if NM_LOGGING
    printf("FindNaturalMotionWindow called with w,h %d %d stride %d size %d addr %x\n\n",resolution_x,resolution_y,stride,lock->allocation->size,(unsigned int)lock->addr);
#endif
    if (m_WindowOn )
    {
     /* check identical as last time */
        if( IS_OPAQUE(ul_x,ul_y, buffer) )
        {
            //printf("UL is opaque in test same\n");
	        if(    ( ( ul_x == 0 ) || !IS_OPAQUE( ul_x-1,ul_y,   buffer) ) 
                && ( ( ul_y == 0 ) || !IS_OPAQUE( ul_x,  ul_y-1, buffer) )
              )
            {
                //printf("UL -1 is NOT opaque in test same\n");
	            if( IS_OPAQUE(lr_x,lr_y, buffer) )
                {
		            if(    ( ( lr_x == resolution_x-1 ) || !IS_OPAQUE( lr_x+1,lr_y,   buffer) )
                        && ( ( lr_y == resolution_y-1 ) || !IS_OPAQUE( lr_x,  lr_y+1, buffer) )
                      )
                    {
#if NM_LOGGING
                        printf( "%s %d Same Window: %d %d %d %d\n",__FUNCTION__,__LINE__,ul_x,ul_y,lr_x,lr_y);
#endif
                        return; /* same window as last time */
                    }
                }
            }
        }
        else
        {
            /* other window must be set */
        }
    }

    /******* check if there is something transparent in the UI *******/
    m_WindowOn = false;
    m_nmbtest=0;

    for( y = MIN_HEIGHT ; (y < resolution_y) ;y += MIN_HEIGHT)
    {
        int prev_detect=false;
        
        detect_pos[y_index]=y;
        for( x = MIN_WIDTH ; (x < resolution_x) ;x += MIN_WIDTH)
        {
            int detect = IS_OPAQUE(x,y, buffer);
            m_nmbtest ++;
            
            if( detect )
            {
                detect_cnt[y_index]++;
            }
            if( !prev_detect && detect )
            {
                detect_left[y_index] = x;
            }
            if( prev_detect && detect)
            {
                detect_right[y_index] = x ;
            }
            prev_detect = detect;
        }
        y_index++; 
    }

    detect_max = 0;
    detect_max_start = 0;
    detect_max_end = 0;
    detect_max_running = false;
    for(y=0;y<y_index;y++)
    {
        if(detect_max_running && (detect_cnt[y] < (detect_max-1)))
        {
            detect_max_running = false;
        }
        if(detect_max_running)
        {
            detect_max_end      = (detect_pos[y+1] > 0) ? detect_pos[y+1]   : resolution_y;
        }
        if(detect_cnt[y] > (detect_max+1))
        {
            detect_max = detect_cnt[y];
            detect_max_start    = (y-1) >= 0 ? detect_pos[y-1] : 0;
            detect_max_left     = detect_left[y]- MIN_WIDTH > 0 ? detect_left[y]- MIN_WIDTH : 0;
            detect_max_right    = detect_right[y] + MIN_WIDTH <  resolution_x ? (detect_right[y] + MIN_WIDTH) : resolution_x;
            detect_max_running = true;
        }
    }


    left   = detect_max_left;
    right  = detect_max_right;
    upper  = detect_max_start;
    bottom = detect_max_end;

    #if NM_LOGGING
    printf("NM Window (raw)left:%d right:%d upper:%d bottom:%d\n",left,right,upper,bottom);
    #endif


    if((left >= right) || (upper >= bottom))
    {

        m_data.enable = 0;
        dfb_cetvfb_set_nat_motion_window(core, &m_data);

        m_data.enable = 0;
        m_data.ul_x   = 0;
        m_data.ul_y   = 0;
        m_data.lr_x   = 0;
        m_data.lr_y   = 0;

        return;
    }


    y = upper + MIN_HEIGHT;
    offset = MIN_WIDTH/2;
    for(x=left,count =100;count;count--)
    {
        int x1= (x+offset) > resolution_x ? resolution_x : x+offset;
        m_nmbtest ++;
        if( !IS_OPAQUE(x1,y, buffer) )
        {
            x = x1;
            left = x;
        }
        offset = (offset > 1) ? (offset / 2) : 1;
        if ((offset == 1) && (count > 2))
        {
            count = 2;
        }
    }

    y = bottom - MIN_HEIGHT;
    offset = MIN_WIDTH/2;
    for(x=right,count =100;count;count--)
    {
        int x1 = (x-offset) > 0 ? x-offset : 0;
        m_nmbtest ++;
        if( !IS_OPAQUE(x1,y, buffer) )
        {
            x = x1;
            right = x;
        }
        offset = (offset > 1) ? (offset / 2) : 1;
        if ((offset == 1) && (count > 2))
        {
            count = 2;
        }
    }

    x = left + 1;
    offset = MIN_HEIGHT/2;
    for(y=upper,count =100;count;count--)
    {
        int y1= (y+offset) > resolution_y ? resolution_y : y+offset;
        m_nmbtest ++;
        if( !IS_OPAQUE(x,y1, buffer) )
        {
            y = y1;
            upper = y;
        }
        offset = (offset > 1) ? (offset / 2) : 1;
        if ((offset == 1) && (count > 2))
        {
            count = 2;
        }
    }

    x = right - 1;
    offset = (bottom > MIN_HEIGHT/2 ) ? MIN_HEIGHT/2 : bottom/2;
    for(y=bottom,count =100;count;count--)
    {
        int y1=y-offset > 0 ? y-offset : 0;
        m_nmbtest ++;
        if( !IS_OPAQUE(x,y1, buffer) )
        {
            y = y1;
            bottom = y;
        }
        offset = (offset > 1) ? (offset / 2) : 1;
        if ((offset == 1) && (count > 2))
        {
            count = 2;
        }
    }
    /******* Set the pip window ************/
    #if NM_LOGGING
    printf("NM Window (fine)left:%d right:%d upper:%d bottom:%d\n",left,right,upper,bottom);
    #endif

    m_data.ul_x     = left;
    m_data.ul_y     = upper;
    m_data.lr_x     = right;
    m_data.lr_y     = bottom;
    m_data.gfxlayer = cetvld->layerId;

    // send to ceplffb
    m_data.enable   = 1;
    dfb_cetvfb_set_nat_motion_window(core, &m_data);
    m_WindowOn = true;

}

/* following only executed in master */
DFBResult 
dfb_cetvfb_set_nat_motion_handler( CETVFBNatMotionWdwData *data )
{
    if ( pCetvfbSetNatMotionWindow )
    {

        (*pCetvfbSetNatMotionWindow)(
                            data->gfxlayer,
                            data-> ul_x,
                            data-> ul_y, 
                            data-> lr_x,
                            data-> lr_y,
                            data->enable);
    }
    else
        printf("&&&&&&&&&&&& THIS SHOULD NOT BE CALLED &&&&&\n");
    return DFB_OK;
}

#endif // NATURAL_MOTION_WINDOW
