/*
   (c) Copyright 2000-2002  convergence integrated media GmbH.
   (c) Copyright 2002-2004  convergence GmbH.

   All rights reserved.

   Written by Denis Oliver Kropp <dok@directfb.org>,
              Andreas Hundt <andi@fischlustig.de>,
              Sven Neumann <neo@directfb.org> and
              Ville Syrjl <syrjala@sci.fi>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <fcntl.h>
#include <sys/mman.h>

#include <directfb.h>

#include <fusion/arena.h>
#include <fusion/shmalloc.h>

#include <core/core.h>
#include <core/coredefs.h>
#include <core/coretypes.h>
#include <core/layers.h>
#include <core/palette.h>
#include <core/surface.h>
#include <core/system.h>

#include <gfx/convert.h>

#include <misc/conf.h>

#include <direct/messages.h>


#include "jag.h"
#include "tmml_surface_pool.h"
#include "primary.h"
#include "video.h"

#include <core/core_system.h>


DFB_CORE_SYSTEM( jag )

D_DEBUG_DOMAIN( JAG_System, "JAG/System", "Core System JAG" );

/**********************************************************************************************************************/

DFBJAG             *dfb_jag        = NULL;
CoreDFB            *dfb_jag_core   = NULL;
FusionWorld        *dfb_jag_world  = NULL;

tmIVmix       *dfb_pIVmix1      = NULL;
tmIVmix       *dfb_pIVmix2      = NULL;
tmIPlfInstGfx *dfb_pIPlfInstGfx = NULL;

/**********************************************************************************************************************/

__attribute__((weak))
Void
tmDbg_AssertError(
    String  expr,   //  I: The actual expression that failed in string format
    String  fil,    //  I: The file name of the file that contained the assert
    UInt32  lin,    //  I: The line number where the assert occurred
    String  message //  I: An extra message to be output (may be Null)
    )
{
}

__attribute__((weak))
pChar
tmDbg_StrAssertFormat(
    pChar format,           // I: printf like formatting string
    ...                     // I: parameters
)
{
     return NULL;
}

__attribute__((weak))
tmErrorCode_t STDCALL
tmCom_SimpleQueryInterface (
    ptmThif thif,      // I: thif pointer
    tmRefIid_t  riid,  // I: interface ID of interface to be returned
    void**      ppv    // O: interface pointer (with ID 'iid')
    )
{
     return TM_ERR_BAD_INSTANCE;
}

__attribute__((weak))
UInt32 STDCALL
tmCom_SimpleAddRef (
    ptmThif thif      // I: thif pointer
    )
{
    return 0;
}

__attribute__((weak))
UInt32 STDCALL
tmCom_SimpleRelease (
    ptmThif thif      // I: thif pointer
    )
{
    return 0;
}

__attribute__((weak))
tmErrorCode_t
tmmlRetailMalloc (
    tmmlMmspHandle_t  handle,     /* I: handle to memspace to allocate from */
    UInt32            nrOfBytes,  /* I: number of bytes to allocate         */
    pVoid             *ppAddress, /* O: address returned upon success       */
    tmmlMallocFlags_t flags       /* I: memory allocation flags             */
    )
{
     return TM_ERR_BAD_INSTANCE;
}

__attribute__((weak))
tmErrorCode_t
tmmlCreate (
    ptmmlMmspHandle_t     pHandle,   /* O: pointer to memspace handle returned */
                                     /*    upon success                        */
    UInt32                nrOfBytes, /* I: size of the memspace                */
    tmmlMmspCreateFlags_t flags      /* I: memspace creation flags             */
    )
{
     return TM_ERR_BAD_INSTANCE;
}

__attribute__((weak))
const tmGuid_t
TMIID_tmIVmixColorKey;

__attribute__((weak))
const tmGuid_t
TMIID_tmIVmixGfxLayer;

__attribute__((weak))
const tmGuid_t
TMIID_tmIVmixVidLayer;

__attribute__((weak))
const tmGuid_t
TMIID_tmIPlfInstGfxNtf;

/**********************************************************************************************************************/

void
DirectFBPreInit( tmIVmix       *pIVmix1,
                 tmIPlfInstGfx *pIPlfInstGfx )
{
     dfb_pIVmix1      = pIVmix1;
     dfb_pIPlfInstGfx = pIPlfInstGfx;
}

void
DirectFBPreInit2nd( tmIVmix *pIVmix2 )
{
     dfb_pIVmix2 = pIVmix2;
}

/**********************************************************************************************************************/

static DFBResult
MapPhysToVirt( void          *virtualAddress,
               unsigned long  physicalAddress,
               int            nrOfBytes )
{
     static int  fd = -1;
     void       *ptr;

     if (fd < 0) {
          fd = open( "/dev/mem", O_RDWR );
          if (fd < 0) {
               D_PERROR( "System/JAG: Couldn't open /dev/mem!\n" );
               return DFB_INIT;
          }
     }

     ptr = mmap( virtualAddress, nrOfBytes, PROT_READ | PROT_WRITE, MAP_FIXED | MAP_SHARED, fd, physicalAddress );
     if (ptr == MAP_FAILED) {
          D_PERROR( "System/JAG: Couldn't map /dev/mem (offset 0x%08lx, size 0x%x, virtual %p!\n",
                    physicalAddress, nrOfBytes, virtualAddress );
          return DFB_INIT;
     }

     return DFB_OK;
}

/**********************************************************************************************************************/

static DFBResult
JAG_Init( bool master )
{
     unsigned int rval;

     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );

     setpgid( 0, 0 );

#if 1
     if (master) {
          D_ASSERT( dfb_pIVmix1 != NULL );
          D_ASSERT( dfb_pIPlfInstGfx != NULL );

          /* Create memory space */
          rval = tmmlCreate( &dfb_jag->memHandle, JAG_SYSTEM_GFXMEM_SIZE + 0x1000, tmmlMmspMpCached );
          if (rval) {
               D_ERROR( "System/JAG: Error '%u' creating memspace!\n", rval );
               return DFB_INIT;
          }

          D_DEBUG_AT( JAG_System, "  -> memspace handle 0x%08x\n", (unsigned int) dfb_jag->memHandle );


          /* Get physical address of graphics memory */
     /*     rval = tmmlMemHandleToPhys( dfb_jag->memHandle, &dfb_jag->memPhys );
          if (rval) {
               D_ERROR( "System/JAG: Error '%u' getting the physical address of the memspace!\n", rval );
               return DFB_INIT;
          }*/

          /* Allocate whole memspace area. */
          rval = tmmlMalloc( dfb_jag->memHandle, JAG_SYSTEM_GFXMEM_SIZE, &dfb_jag->memVirt, tmmlMallocCacheAligned );
          if (rval) {
               D_ERROR( "System/JAG: Error '%u' allocating memory within memspace!\n", rval );
               return DFB_INIT;
          }

          dfb_jag->memPhys = (tmmlPhysAddr_t) dfb_jag->memVirt;
     }
     else {
          if (MapPhysToVirt( dfb_jag->memVirt, dfb_jag->memPhys, JAG_SYSTEM_GFXMEM_SIZE ))
               return DFB_INIT;
     }
#else
     dfb_jag->memPhys = JAG_SYSTEM_GFXMEM_ADDR;

     /* Map the physical address to virtual address of graphics memory */
     rval = MapPhysToVirt( &dfb_jag->memVirt, dfb_jag->memPhys, JAG_SYSTEM_GFXMEM_SIZE );
     if (rval) {
          D_ERROR( "System/JAG: Error '%u' mapping physical to virtual!\n", rval );
          return rval;
     }
#endif


     D_DEBUG_AT( JAG_System, "  -> physical 0x%08x\n", (unsigned int) dfb_jag->memPhys );
     D_DEBUG_AT( JAG_System, "  -> virtual 0x%08x\n", (unsigned int) dfb_jag->memVirt );


//     D_ASSERT( dfb_jag->memPhys != 0 );
     D_ASSERT( dfb_jag->memVirt != NULL );


     return DFB_OK;
}

static void
JAG_Exit()
{
    /* TODO: destroy memspace */
}

static FusionCallHandlerResult
dfb_jag_call_handler( int           caller,
                      int           call_arg,
                      void         *call_ptr,
                      void         *ctx,
                      unsigned int  serial,
                      int          *ret_val )
{
     switch (call_arg) {
          case JAG_SET_REGION:
               *ret_val = dfb_jag_set_region_handler( call_ptr );
               break;

          case JAG_UPDATE_SCREEN:
               *ret_val = dfb_jag_update_screen_handler( call_ptr );
               break;

          case JAG_SET_PALETTE:
               *ret_val = dfb_jag_set_palette_handler( call_ptr );
               break;

          case JAG_VIDEO_SET_REGION:
               *ret_val = dfb_jag_video_set_region_handler( call_ptr );
               break;

          case JAG_REMOVE_REGION:
               *ret_val = dfb_jag_remove_region_handler( call_ptr );
               break;

          case JAG_TMML_ALLOCATE:
               *ret_val = dfb_jag_tmml_allocate_handler( call_ptr );
               break;

          case JAG_TMML_DEALLOCATE:
               *ret_val = dfb_jag_tmml_deallocate_handler( call_ptr );
               break;

          default:
               D_BUG( "unknown call" );
               *ret_val = DFB_BUG;
               break;
     }

     return FCHR_RETURN;
}

/**********************************************************************************************************************/

static void
system_get_info( CoreSystemInfo *info )
{
     info->type = CORE_ANY;
     info->caps = CSCAPS_ACCELERATION;

     snprintf( info->name, DFB_CORE_SYSTEM_INFO_NAME_LENGTH, "JAG" );
}

static DFBResult
system_initialize( CoreDFB *core, void **data )
{
     DFBResult            ret;
     CoreScreen          *screen;
     FusionWorld         *world;
     FusionSHMPoolShared *pool;

     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );

     D_ASSERT( dfb_jag == NULL );

     world = dfb_core_world( core );
     pool  = dfb_core_shmpool( core );

     dfb_jag = (DFBJAG*) SHCALLOC( pool, 1, sizeof(DFBJAG) );
     if (!dfb_jag) {
          D_ERROR( "DirectFB/JAG: Couldn't allocate shared memory!\n" );
          return DFB_NOSYSTEMMEMORY;
     }

     dfb_jag->shmpool = pool;
     dfb_jag_core     = core;
     dfb_jag_world    = world;

     ret = JAG_Init( true );
     if (ret) {
          SHFREE( pool, dfb_jag );
          return ret;
     }

     fusion_skirmish_init( &dfb_jag->lock, "JAG System", world );

     fusion_call_init( &dfb_jag->call, dfb_jag_call_handler, NULL, world );

     screen = dfb_screens_register( NULL, NULL, &jagPrimaryScreenFuncs );

     dfb_layers_register( screen, (void*) 0, &jagPrimaryLayerFuncs );

     if (!dfb_pIVmix2)
          dfb_pIVmix2 = dfb_pIVmix1;

     dfb_layers_register( screen, (void*) 1, &jagPrimaryLayerFuncs );

     pthread_cond_init( &dfb_jag->mbsCond, NULL );
     pthread_mutex_init( &dfb_jag->mbsLock, NULL );

     fusion_arena_add_shared_field( dfb_core_arena( core ), "JAG", dfb_jag );

     dfb_surface_pool_initialize( core, &tmmlSurfacePoolFuncs, &dfb_jag->tmml_surface_pool );

     *data = dfb_jag;

     return DFB_OK;
}

static DFBResult
system_join( CoreDFB *core, void **data )
{
     void       *ret;
     CoreScreen *screen;

     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );

     D_ASSERT( dfb_jag == NULL );

     fusion_arena_get_shared_field( dfb_core_arena( core ), "JAG", &ret );

     dfb_jag       = ret;
     dfb_jag_core  = core;
     dfb_jag_world = dfb_core_world( core );

     if (JAG_Init( false ))
          return DFB_INIT;

     screen = dfb_screens_register( NULL, NULL, &jagPrimaryScreenFuncs );

     dfb_layers_register( screen, (void*) 0, &jagPrimaryLayerFuncs );

     dfb_layers_register( screen, (void*) 1, &jagPrimaryLayerFuncs );

     dfb_surface_pool_join( core, dfb_jag->tmml_surface_pool, &tmmlSurfacePoolFuncs );

     *data = dfb_jag;

     return DFB_OK;
}

static DFBResult
system_shutdown( bool emergency )
{
     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );

     D_ASSERT( dfb_jag != NULL );

     fusion_call_destroy( &dfb_jag->call );

     fusion_skirmish_prevail( &dfb_jag->lock );

     JAG_Exit();

     dfb_surface_pool_destroy( dfb_jag->tmml_surface_pool );

     fusion_skirmish_destroy( &dfb_jag->lock );

     pthread_cond_destroy( &dfb_jag->mbsCond );
     pthread_mutex_destroy( &dfb_jag->mbsLock );

     SHFREE( dfb_jag->shmpool, dfb_jag );

     dfb_jag       = NULL;
     dfb_jag_core  = NULL;
     dfb_jag_world = NULL;

     return DFB_OK;
}

static DFBResult
system_leave( bool emergency )
{
     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );

     D_ASSERT( dfb_jag != NULL );

     dfb_jag       = NULL;
     dfb_jag_core  = NULL;
     dfb_jag_world = NULL;

     return DFB_OK;
}

static DFBResult
system_suspend()
{
     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );

     return DFB_UNIMPLEMENTED;
}

static DFBResult
system_resume()
{
     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );

     return DFB_UNIMPLEMENTED;
}

static volatile void *
system_map_mmio( unsigned int    offset,
                 int             length )
{
     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );

     return NULL;
}

static void
system_unmap_mmio( volatile void  *addr,
                   int             length )
{
     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );
}

static int
system_get_accelerator()
{
     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );

     return 8550;
}

static VideoMode *
system_get_modes()
{
     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );

     return NULL;
}

static VideoMode *
system_get_current_mode()
{
     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );

     return NULL;
}

static DFBResult
system_thread_init()
{
     D_DEBUG_AT( JAG_System, "%s()\n", __FUNCTION__ );

     return DFB_OK;
}

static bool
system_input_filter( CoreInputDevice *device,
                     DFBInputEvent   *event )
{
     return false;
}

static unsigned long
system_video_memory_physical( unsigned int offset )
{
     return dfb_jag->memPhys + offset;
}

static void *
system_video_memory_virtual( unsigned int offset )
{
     return dfb_jag->memVirt + offset;
}

static unsigned int
system_videoram_length()
{
     return JAG_SYSTEM_GFXMEM_SIZE;
}

static unsigned long
system_aux_memory_physical( unsigned int offset )
{
     return 0;
}

static void *
system_aux_memory_virtual( unsigned int offset )
{
     return NULL;
}

static unsigned int
system_auxram_length()
{
     return 0;
}

static void
system_get_busid( int *ret_bus, int *ret_dev, int *ret_func )
{
}

static void
system_get_deviceid( unsigned int *ret_vendor_id,
                     unsigned int *ret_device_id )
{
}

