/*
   (c) Copyright 2000-2002  convergence integrated media GmbH.
   (c) Copyright 2002-2004  convergence GmbH.

   All rights reserved.

   Written by Denis Oliver Kropp <dok@directfb.org>,
              Andreas Hundt <andi@fischlustig.de>,
              Sven Neumann <neo@directfb.org> and
              Ville Syrjl <syrjala@sci.fi>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#define DIRECT_FORCE_DEBUG

#include <config.h>

#include <stdio.h>

#include <directfb.h>

#include <fusion/fusion.h>
#include <fusion/shmalloc.h>

#include <core/core.h>
#include <core/coredefs.h>
#include <core/coretypes.h>
#include <core/layers.h>
#include <core/palette.h>
#include <core/surface.h>
#include <core/surface_buffer.h>
#include <core/system.h>

#include <gfx/convert.h>

#include <misc/conf.h>
#include <misc/util.h>

#include <direct/memcpy.h>
#include <direct/messages.h>

#include <tmCom.h>        // convenience macros/functions for defining classes

#include <tmIPlfInstGfxNtf.h>

// interface name to number mapping used by tmCom_GetIntf()
#define tmIPlfInstGfxNtf_IntfNr 0

//-----------------------------------------------------------------------------
// Internal Prototypes:
//-----------------------------------------------------------------------------
//

// tmIPlfInstGfxNtf
static Void STDCALL tmIPlfInstGfxNtfOnSubscriptionChanged (
    ptmThif thif,                       //  I: object or interface pointer
    UInt32 cookie,                      //  I:
    tmPlfInstGfx_Notifications_t notifs //  I:
    );

static Void STDCALL tmIPlfInstGfxNtfOnScaledBlitCompleted (
    ptmThif thif, //  I: object or interface pointer
    UInt32 cookie //  I:
    );


//-----------------------------------------------------------------------------
// Global data:
//-----------------------------------------------------------------------------
//
// interface tables
static TMCOM_ITABLE(tmIPlfInstGfxNtf) =
{
    TMCOM_ITABLE_DECL(TMIID_tmIPlfInstGfxNtf),
    {
    TMCOM_SIMPLE_ITABLE_TMIUNKNOWN(0),
    TMCOM_CFRONT_INIT_OFFSET_FIELDS(0)     tmIPlfInstGfxNtfOnSubscriptionChanged,
    TMCOM_CFRONT_INIT_OFFSET_FIELDS(0)     tmIPlfInstGfxNtfOnScaledBlitCompleted
    }
};

// the singleton object
static tmIPlfInstGfxNtf gtmIPlfInstGfxNtf_Singleton = { &(tmIPlfInstGfxNtf_Itbl.vt) };

/**********************************************************************************************************************/

#include "jag.h"
#include "primary.h"


#define JAG_PRIMARY_SUPPORTED_OPTIONS   (DLOP_ALPHACHANNEL | DLOP_OPACITY | DLOP_SRC_COLORKEY)


D_DEBUG_DOMAIN( JAG_Primary, "JAG/Primary", "Core System JAG Primary Screen & Layer" );

/**********************************************************************************************************************/

static DFBResult dfb_jag_set_region   ( CoreDFB *core, JAGSetRegionData    *data );
static DFBResult dfb_jag_update_screen( CoreDFB *core, JAGUpdateScreenData *data );
static DFBResult dfb_jag_set_palette  ( CoreDFB *core, JAGSetPaletteData   *data );
static DFBResult dfb_jag_remove_region( CoreDFB *core, JAGLayerData        *data );

/**********************************************************************************************************************/

static tmPixRgbFmt_t
dfb_to_pixrgbfmt( DFBSurfacePixelFormat format, tmPixFmtCls_t *ret_class )
{
    switch (format) {
        case DSPF_ARGB:
            if (ret_class)
                *ret_class = tmPixFmtCls_Rgb;
            return tmPixRgbFmt_Argb8888;

        case DSPF_ARGB4444:
            if (ret_class)
                *ret_class = tmPixFmtCls_Rgb;
            return tmPixRgbFmt_Argb4444;

        case DSPF_RGB16:
            if (ret_class)
                *ret_class = tmPixFmtCls_Rgb;
            return tmPixRgbFmt_Rgb565;

        case DSPF_LUT8:
            if (ret_class)
                *ret_class = tmPixFmtCls_Clut;
            return tmPixClutFmt_ArgbClut8bpp;

        default:
            D_BUG( "unexpected pixelformat '%s'", dfb_pixelformat_name( format ) );
    }

    return 0;
}

static void
get_color_minmax( DFBSurfacePixelFormat  format,
                  const DFBColorKey     *color,
                  tmColor_t             *ret_min,
                  tmColor_t             *ret_max )
{
     switch (format) {
          case DSPF_ARGB:
               ret_min->redOrY   = color->r;
               ret_min->greenOrU = color->g;
               ret_min->blueOrV  = color->b;
               ret_max->redOrY   = color->r;
               ret_max->greenOrU = color->g;
               ret_max->blueOrV  = color->b;
               break;

          case DSPF_ARGB4444:
               ret_min->redOrY   = color->r & ~0xF;
               ret_min->greenOrU = color->g & ~0xF;
               ret_min->blueOrV  = color->b & ~0xF;
               ret_max->redOrY   = color->r |  0xF;
               ret_max->greenOrU = color->g |  0xF;
               ret_max->blueOrV  = color->b |  0xF;
               break;

          case DSPF_RGB16:
               ret_min->redOrY   = color->r & ~0x7;
               ret_min->greenOrU = color->g & ~0x3;
               ret_min->blueOrV  = color->b & ~0x7;
               ret_max->redOrY   = color->r |  0x7;
               ret_max->greenOrU = color->g |  0x3;
               ret_max->blueOrV  = color->b |  0x7;
               break;

          case DSPF_LUT8:
               D_WARN( "color keying for LUT8 not supported" );
               break;

          default:
               D_BUG( "unexpected pixelformat '%s'", dfb_pixelformat_name( format ) );
     }
}

static void
SetSurfaceBuffer( JAGLayerData          *jld,
                  CoreSurface           *surface,
                  CoreSurfaceBufferLock *lock )
{
     tmErrorCode_t          rval;
     CoreSurfaceAllocation *allocation;

     D_DEBUG_AT( JAG_Primary, "%s()\n", __FUNCTION__ );

     D_ASSERT( jld != NULL );
     D_ASSERT( surface != NULL );
     D_ASSERT( lock != NULL );

     allocation = lock->allocation;
     D_ASSERT( allocation != NULL );

     direct_log_printf( NULL, "  -> tmIVmixGfxLayer_SetBuffer( %dx%d, %d, Rgb %x, %d, %p, %p )...\n",
                        surface->config.size.w, surface->config.size.h, lock->pitch, jld->format, allocation->size,
                        lock->addr, jld );

     rval = tmIVmixGfxLayer_SetBuffer( jld->pIGfxLayer,
                                       surface->config.size.w,
                                       surface->config.size.h,
                                       lock->pitch,
                                       jld->format_class,
                                       jld->format,
                                       allocation->size,
                                       lock->addr,
                                       (UInt32) jld );
     D_ASSERT( rval == TM_OK );

     direct_log_printf( NULL, "  -> tmIVmixGfxLayer_SetBuffer( %dx%d, %d, Rgb %x, %d, %p, %p ) done.\n",
                        surface->config.size.w, surface->config.size.h, lock->pitch, jld->format, allocation->size,
                        lock->addr, jld );
}

static void
SetScaleBuffer( JAGLayerData *jld )
{
     tmErrorCode_t rval;

     D_DEBUG_AT( JAG_Primary, "%s()\n", __FUNCTION__ );

     D_ASSERT( jld != NULL );

     direct_log_printf( NULL, "  -> tmIVmixGfxLayer_SetBuffer( %dx%d, %d, %s %x, %d, %lu, %p )...\n",
                 jld->scale_width, jld->scale_height, jld->scale_pitch,
                 jld->scale_format_class == tmPixFmtCls_Rgb ? "Rgb" : "Clut",
                 jld->scale_format, jld->scale_size,
                 dfb_jag->memPhys /* Virt */ + (jld->first ?
                                                dfb_jag->scale_buffer :
                                                dfb_jag->scale_buffer2), jld );


     rval = tmIVmixGfxLayer_SetBuffer( jld->pIGfxLayer,
                                       jld->scale_width,
                                       jld->scale_height,
                                       jld->scale_pitch,
                                       jld->scale_format_class,
                                       jld->scale_format,
                                       jld->scale_size,
                                       (void*) dfb_jag->memVirt + (jld->first ?
                                                                   dfb_jag->scale_buffer :
                                                                   dfb_jag->scale_buffer2),
                                       (UInt32) jld );
     D_ASSERT( rval == TM_OK );

     direct_log_printf( NULL, "  -> tmIVmixGfxLayer_SetBuffer( %dx%d, %d, %s %x, %d, %lu, %p ) done.\n",
                 jld->scale_width, jld->scale_height, jld->scale_pitch,
                 jld->scale_format_class == tmPixFmtCls_Rgb ? "Rgb" : "Clut",
                 jld->scale_format, jld->scale_size,
                 dfb_jag->memPhys /* Virt */ + (jld->first ?
                                                dfb_jag->scale_buffer :
                                                dfb_jag->scale_buffer2), jld );
}

/**********************************************************************************************************************/

static DFBResult
primaryInitScreen( CoreScreen           *screen,
                   CoreGraphicsDevice   *device,
                   void                 *driver_data,
                   void                 *screen_data,
                   DFBScreenDescription *description )
{
     /* Set the screen capabilities. */
     description->caps = DSCCAPS_NONE;

     /* Set the screen name. */
     snprintf( description->name,
               DFB_SCREEN_DESC_NAME_LENGTH, "JAG Primary Screen" );

     return DFB_OK;
}

static DFBResult
primaryGetScreenSize( CoreScreen *screen,
                      void       *driver_data,
                      void       *screen_data,
                      int        *ret_width,
                      int        *ret_height )
{
//     tmErrorCode_t       rval;
     UInt32              width = 1366;
     UInt32              height = 768;
//     tmVmix_FieldRate_t  fieldRate;
//     tmVmix_ScanType_t   scanType;

     D_ASSERT( dfb_jag != NULL );
     D_ASSERT( ret_width != NULL );
     D_ASSERT( ret_height != NULL );

     //rval = tmIVmix_GetOutputProperties( dfb_pIVmix1, &width, &height, &fieldRate, &scanType );
     //D_ASSERT( rval == TM_OK );

     *ret_width  = width;
     *ret_height = height;

     return DFB_OK;
}

ScreenFuncs jagPrimaryScreenFuncs = {
     InitScreen:    primaryInitScreen,
     GetScreenSize: primaryGetScreenSize
};

/**********************************************************************************************************************/

static int
primaryLayerDataSize()
{
     return sizeof(JAGLayerData);
}

static int
primaryRegionDataSize()
{
     return 0;
}

static DFBResult
primaryInitLayer( CoreLayer                  *layer,
                  void                       *driver_data,
                  void                       *layer_data,
                  DFBDisplayLayerDescription *description,
                  DFBDisplayLayerConfig      *config,
                  DFBColorAdjustment         *adjustment )
{
     tmErrorCode_t  rval;
     JAGLayerData  *jld = layer_data;

     D_DEBUG_AT( JAG_Primary, "%s()\n", __FUNCTION__ );

     jld->first  = !driver_data;
     jld->pIVmix = jld->first ? dfb_pIVmix1 : dfb_pIVmix2;

     /* set capabilities and type */
     description->caps = DLCAPS_SURFACE | DLCAPS_ALPHACHANNEL | DLCAPS_OPACITY |
                         DLCAPS_SCREEN_POSITION | DLCAPS_SCREEN_SIZE;
     description->type = DLTF_GRAPHICS;

     if (jld->first) {
         rval = tmIPlfInstGfx_Subscribe( dfb_pIPlfInstGfx,
                                         &gtmIPlfInstGfxNtf_Singleton,
                                         0, tmPlfInstGfx_AllNotifications );
         D_ASSERT( rval == TM_OK );
     }

     /* set name */
     snprintf( description->name, DFB_DISPLAY_LAYER_DESC_NAME_LENGTH,
               "JAG %s Layer", jld->first ? "Primary" : "Secondary" );

     /* fill out the default configuration */
     config->flags       = DLCONF_WIDTH       | DLCONF_HEIGHT |
                           DLCONF_PIXELFORMAT | DLCONF_BUFFERMODE | DLCONF_OPTIONS;
     config->buffermode  = DLBM_FRONTONLY;
     config->options     = DLOP_ALPHACHANNEL;

     if (dfb_config->layers[!jld->first].config.width)
          config->width  = dfb_config->layers[!jld->first].config.width;
     else
          config->width  = jld->first ? 852 : 640;

     if (dfb_config->layers[!jld->first].config.height)
          config->height = dfb_config->layers[!jld->first].config.height;
     else
          config->height = 480;

     if (dfb_config->layers[!jld->first].config.pixelformat != DSPF_UNKNOWN)
          config->pixelformat = dfb_config->layers[!jld->first].config.pixelformat;
     else
          config->pixelformat = DSPF_ARGB4444;

     /* Get interface to graphics layer */
     if (jld->first)
         rval = tmIVmix_GetLayer( dfb_pIVmix1, 3, &jld->pILayer );
     else
         rval = tmIVmix_GetLayer( dfb_pIVmix2,  /* test mode? */
                                  dfb_pIVmix1 == dfb_pIVmix2 ? 2 : 1, &jld->pILayer );
     D_ASSERT( rval == TM_OK );

     /* Get graphics layer interface */
     rval = tmIVmixGfxLayer_QueryInterface( jld->pILayer, &TMIID_tmIVmixGfxLayer, &jld->pIGfxLayer);
     if (rval) {
          D_ERROR( "JAG/Primary: Could not query interface tmIVmixGfxLayer! (%lu)\n", rval );
          return DFB_INIT;
     }

     /* Get color key interface */
     rval = tmIVmixColorKey_QueryInterface( jld->pILayer, &TMIID_tmIVmixColorKey, &jld->pIColorKey);
     if (rval) {
          D_ERROR( "JAG/Primary: Could not query interface tmIVmixColorKey! (%lu)\n", rval );
          return DFB_INIT;
     }

     /* Set scale mode */
     rval = tmIVmixLayer_SetScaleMode( jld->pILayer, tmVmix_Horizontal );
     D_ASSERT( rval == TM_OK );

     /* Query maximum layer window size */
     tmIVmixLayer_GetMaxLayerWindowSize( jld->pILayer,
                                         &jld->maxLayerWindowWidth,
                                         &jld->maxLayerWindowHeight );

     /* Query minimum window size */
     tmIVmixLayer_GetMinWindowSize( jld->pILayer,
                                    &jld->minWindowWidth,
                                    &jld->minWindowHeight );

     D_MAGIC_SET( jld, JAGLayerData );

     return DFB_OK;
}

static DFBResult
primaryTestRegion( CoreLayer                  *layer,
                   void                       *driver_data,
                   void                       *layer_data,
                   CoreLayerRegionConfig      *config,
                   CoreLayerRegionConfigFlags *failed )
{
     JAGLayerData               *jld  = layer_data;
     CoreLayerRegionConfigFlags  fail = 0;

     D_DEBUG_AT( JAG_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( jld, JAGLayerData );

     if (config->options & ~JAG_PRIMARY_SUPPORTED_OPTIONS)
          fail |= CLRCF_OPTIONS;

     if (getenv("JAG_NOSCALE")) {
          config->dest.w = config->width;
          config->dest.h = config->height;
     }

     D_DEBUG_AT( JAG_Primary, "  -> src %d, %d - %dx%d\n",
                 DFB_RECTANGLE_VALS( &config->source ) );

     D_DEBUG_AT( JAG_Primary, "  -> dst %d, %d - %dx%d\n",
                 DFB_RECTANGLE_VALS( &config->dest ) );


     switch (config->format) {
          case DSPF_ARGB:
          case DSPF_ARGB4444:
          case DSPF_RGB16:
          case DSPF_LUT8:
               break;

          default:
               fail |= CLRCF_FORMAT;
     }


     if (config->source.w < jld->minWindowWidth)
          fail |= CLRCF_SOURCE;

     if (config->source.h < jld->minWindowHeight)
          fail |= CLRCF_SOURCE;


     if (config->dest.w < jld->minWindowWidth) {
          D_DEBUG_AT( JAG_Primary, "  -> %d < %lu!\n", config->dest.w, jld->minWindowWidth );
          fail |= CLRCF_DEST;
     }

     if (config->dest.h < jld->minWindowHeight) {
          D_DEBUG_AT( JAG_Primary, "  -> %d < %lu!\n", config->dest.h, jld->minWindowHeight );
          fail |= CLRCF_DEST;
     }


     if (config->dest.w > jld->maxLayerWindowWidth) {
          D_DEBUG_AT( JAG_Primary, "  -> %d > %lu!\n", config->dest.w, jld->maxLayerWindowWidth );
          fail |= CLRCF_DEST;
     }

     if (config->dest.h > jld->maxLayerWindowHeight) {
          D_DEBUG_AT( JAG_Primary, "  -> %d > %lu!\n", config->dest.h, jld->maxLayerWindowHeight );
          fail |= CLRCF_DEST;
     }


     /* Maximum scale buffer size is 852x768 for 32 bit */
     if (DFB_BYTES_PER_PIXEL(config->format) > 2 &&
         config->dest.h != config->source.h && (config->source.w > 852 || config->dest.h > 768))
     {
          D_DEBUG_AT( JAG_Primary, "  -> %dx%d -> %dx%d failed\n",
                      config->source.w, config->source.h, config->dest.w, config->dest.h );

          fail |= CLRCF_SOURCE | CLRCF_DEST;
     }


     if (failed)
          *failed = fail;

     if (fail) {
          D_DEBUG_AT( JAG_Primary, "  -> fail 0x%08x\n", fail );
          return DFB_UNSUPPORTED;
     }

     return DFB_OK;
}

static DFBResult
primaryAddRegion( CoreLayer             *layer,
                  void                  *driver_data,
                  void                  *layer_data,
                  void                  *region_data,
                  CoreLayerRegionConfig *config )
{
     D_DEBUG_AT( JAG_Primary, "%s()\n", __FUNCTION__ );

     return DFB_OK;
}

static DFBResult
primarySetRegion( CoreLayer                  *layer,
                  void                       *driver_data,
                  void                       *layer_data,
                  void                       *region_data,
                  CoreLayerRegionConfig      *config,
                  CoreLayerRegionConfigFlags  updated,
                  CoreSurface                *surface,
                  CorePalette                *palette,
                  CoreSurfaceBufferLock      *lock )
{
     DFBResult         ret;
     JAGSetRegionData  data;
     JAGLayerData     *jld = layer_data;

     D_DEBUG_AT( JAG_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( jld, JAGLayerData );

     data.config    = *config;
     data.updated   = updated;
     data.surface   = surface;
     data.palette   = palette;
     data.layerData = layer_data;

     ret = dfb_jag_set_region( dfb_jag_core, &data );
     if (ret)
          return ret;

     if (surface) {
          jld->surface    = surface;
          jld->set_buffer = true;
     }

//     direct_log_printf(NULL,"%s() surface %p\n", __FUNCTION__, surface );

     if (palette && (updated & CLRCF_PALETTE) && config->format == DSPF_LUT8) {
          JAGSetPaletteData pdata;

          pdata.layerData = layer_data;
          pdata.palette   = palette;

          dfb_jag_set_palette( dfb_jag_core, &pdata );
     }

     return DFB_OK;
}

static DFBResult
primaryRemoveRegion( CoreLayer             *layer,
                     void                  *driver_data,
                     void                  *layer_data,
                     void                  *region_data )
{
     JAGLayerData *jld = layer_data;

     D_DEBUG_AT( JAG_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( jld, JAGLayerData );

     return dfb_jag_remove_region( dfb_jag_core, jld );
}

static DFBResult
primaryFlipRegion( CoreLayer             *layer,
                   void                  *driver_data,
                   void                  *layer_data,
                   void                  *region_data,
                   CoreSurface           *surface,
                   DFBSurfaceFlipFlags    flags,
                   CoreSurfaceBufferLock *lock )
{
     JAGLayerData *jld = layer_data;
     JAGUpdateScreenData data;

     D_DEBUG_AT( JAG_Primary, "%s()\n", __FUNCTION__ );
//     direct_log_printf(NULL,"%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( jld, JAGLayerData );

     dfb_surface_flip( surface, false );

     data.region    = (DFBRegion){ 0, 0, surface->config.size.w - 1, surface->config.size.h - 1 };
     data.layerData = layer_data;
     data.lock      = lock;

     if (!jld->scaling)
          jld->set_buffer = true;

     return dfb_jag_update_screen( dfb_jag_core, &data );
}

static DFBResult
primaryUpdateRegion( CoreLayer             *layer,
                     void                  *driver_data,
                     void                  *layer_data,
                     void                  *region_data,
                     CoreSurface           *surface,
                     const DFBRegion       *update,
                     CoreSurfaceBufferLock *lock )
{
     JAGUpdateScreenData  data;
     JAGLayerData        *jld = layer_data;

     D_DEBUG_AT( JAG_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( jld, JAGLayerData );

//     if (!jld->scaling)
//          return DFB_OK;

     if (update) {
          /* Temporary HACK to optimize Opera! */
          if (getenv( "OPERA_DIR" )) {
               if (update->x1 == 100 && update->y1 == 100 && update->x2 == 101 && update->y2 == 101)
                    return DFB_OK;
          }

          data.region = *update;
     }
     else
          data.region = (DFBRegion){ 0, 0, jld->surface->config.size.w - 1, jld->surface->config.size.h - 1 };

     data.layerData = layer_data;
     data.lock      = lock;

     if (!jld->scaling)
          jld->set_buffer = true;

     return dfb_jag_update_screen( dfb_jag_core, &data );
}

DisplayLayerFuncs jagPrimaryLayerFuncs = {
     LayerDataSize:      primaryLayerDataSize,
     RegionDataSize:     primaryRegionDataSize,
     InitLayer:          primaryInitLayer,

     TestRegion:         primaryTestRegion,
     AddRegion:          primaryAddRegion,
     SetRegion:          primarySetRegion,
     RemoveRegion:       primaryRemoveRegion,
     FlipRegion:         primaryFlipRegion,
     UpdateRegion:       primaryUpdateRegion
};

/**********************************************************************************************************************/

static DFBResult
set_region( JAGLayerData               *jld,
            CoreLayerRegionConfig      *config,
            CoreLayerRegionConfigFlags  updated,
            CoreSurface                *surface,
            CorePalette                *palette )
{
     tmErrorCode_t       rval;
     tmRect_t            rect;
     UInt32              width;
     UInt32              height;
     tmVmix_FieldRate_t  fieldRate;
     tmVmix_ScanType_t   scanType;

     D_DEBUG_AT( JAG_Primary, "%s( %p, %p )\n", __FUNCTION__, jld, config );

     D_MAGIC_ASSERT( jld, JAGLayerData );
     D_ASSERT( dfb_jag != NULL );
     D_ASSERT( jld->pIVmix != NULL );

     rval = tmIVmix_GetOutputProperties( jld->pIVmix, &width, &height, &fieldRate, &scanType );
     D_ASSERT( rval == TM_OK );

     if (updated & CLRCF_FORMAT) {
          D_ASSERT( config != NULL );

          D_DEBUG_AT( JAG_Primary, "  -> format %s\n", dfb_pixelformat_name(config->format) );

          jld->format = dfb_to_pixrgbfmt( config->format, &jld->format_class );
     }

     if ((updated & CLRCF_SRCKEY) && (config->options & DLOP_SRC_COLORKEY)) {
          tmColor_t color_min, color_max;

          get_color_minmax( config->format, &config->src_key, &color_min, &color_max );

          D_DEBUG_AT( JAG_Primary, "  -> tmIVmixColorKey_SetKey( %d, %d, %d - %d, %d, %d )\n",
                      color_min.redOrY, color_min.greenOrU, color_min.blueOrV,
                      color_max.redOrY, color_max.greenOrU, color_max.blueOrV );

          rval = tmIVmixColorKey_SetKey( jld->pIColorKey, 0, tmVmix_SrcColorKey, color_min, color_max );
          if (rval) {
               D_ERROR( "JAG/Primary: Could not set color key! (%lu)\n", rval );
               return DFB_INIT;
          }
     }

     if (updated & CLRCF_OPTIONS) {
          D_DEBUG_AT( JAG_Primary, "  -> tmIVmixColorKey_EnableKey( %s )\n",
                      (config->options & DLOP_SRC_COLORKEY) ? "True" : "False" );

          rval = tmIVmixColorKey_EnableKey( jld->pIColorKey, 0, !!(config->options & DLOP_SRC_COLORKEY) );
          if (rval) {
               D_ERROR( "JAG/Primary: Could not %sable color key! (%lu)\n",
                        (config->options & DLOP_SRC_COLORKEY) ? "en" : "dis", rval );
               return DFB_INIT;
          }
     }

     if (updated & (CLRCF_SOURCE | CLRCF_DEST)) {
          D_ASSERT( config != NULL );

          D_DEBUG_AT( JAG_Primary, "  -> src %d, %d - %dx%d\n",
                      config->source.x, config->source.y, config->source.w, config->source.h );

          D_DEBUG_AT( JAG_Primary, "  -> dst %d, %d - %dx%d\n",
                      config->dest.x, config->dest.y, config->dest.w, config->dest.h );


          jld->scaling = (config->dest.h != config->source.h);// || (config->format == DSPF_LUT8);

          D_DEBUG_AT( JAG_Primary, "  -> %sscaling\n", jld->scaling ? "" : "not " );

          if (jld->scaling) {
               jld->scale_format = dfb_to_pixrgbfmt( (config->format != DSPF_LUT8) ?
                                                     config->format : DSPF_RGB16,
                                                     &jld->scale_format_class );
               jld->scale_width  = config->source.w;
               jld->scale_height = config->dest.h;
               jld->scale_pitch  = (DFB_BYTES_PER_LINE( config->format, jld->scale_width ) + 31) & ~31;
               jld->scale_size   = DFB_PLANE_MULTIPLY( config->format, jld->scale_pitch * jld->scale_height );

               jld->scale_rect.ul.x = config->source.x;
               jld->scale_rect.ul.y = config->source.y;
               jld->scale_rect.lr.x = config->source.x + config->source.w;
               jld->scale_rect.lr.y = config->source.y + config->source.h;

               /* Set source window. */
               rect.ul.x = 0;
               rect.ul.y = 0;
               rect.lr.x = jld->scale_width;
               rect.lr.y = jld->scale_height;
          }
          else {
               /* Set source window. */
               rect.ul.x = config->source.x;
               rect.ul.y = config->source.y;
               rect.lr.x = config->source.x + config->source.w;
               rect.lr.y = config->source.y + config->source.h;
          }

          D_DEBUG_AT( JAG_Primary, "  -> tmIVmixLayer_SetSrcWindow( %ld, %ld, %ld, %ld )\n",
                      rect.ul.x, rect.ul.y, rect.lr.x, rect.lr.y );

          rval = tmIVmixLayer_SetSrcWindow( jld->pILayer, rect );
          D_ASSERT( rval == TM_OK );


          /* Set destination window. */
          rect.ul.x = config->dest.x;
          rect.ul.y = config->dest.y;
          rect.lr.x = config->dest.x + config->dest.w;
          rect.lr.y = config->dest.y + config->dest.h;

          D_DEBUG_AT( JAG_Primary, "  -> tmIVmixLayer_SetDstWindow( %ld, %ld, %ld, %ld )\n",
                      rect.ul.x, rect.ul.y, rect.lr.x, rect.lr.y );

          rval = tmIVmixLayer_SetDstWindow( jld->pILayer, rect );
          D_ASSERT( rval == TM_OK );


          /* Set layer window. */
          rect.ul.x = 0;
          rect.ul.y = 0;
          rect.lr.x = width;
          rect.lr.y = height;

          D_DEBUG_AT( JAG_Primary, "  -> tmIVmixLayer_SetLayerWindow( %ld, %ld, %ld, %ld )\n",
                      rect.ul.x, rect.ul.y, rect.lr.x, rect.lr.y );

          rval = tmIVmixLayer_SetLayerWindow( jld->pILayer, rect );
          D_ASSERT( rval == TM_OK );


          /* Activate settings. */
          D_DEBUG_AT( JAG_Primary, "  -> tmIVmixLayer_ActivateNewWindowSettings( True, 0 )\n" );

          rval = tmIVmixLayer_ActivateNewWindowSettings( jld->pILayer, True, 0 );
          D_ASSERT( rval == TM_OK );
     }

     if (surface && (updated & (CLRCF_SURFACE | CLRCF_SOURCE | CLRCF_DEST | CLRCF_OPTIONS | CLRCF_SRCKEY)))
          jld->set_buffer = true;

     D_DEBUG_AT( JAG_Primary, "  -> set_buffer: %s\n", jld->set_buffer ? "true" : "false" );

     /* Finally show the layer. */
     D_DEBUG_AT( JAG_Primary, "  -> tmIVmixLayer_Hide( False )\n" );

     rval = tmIVmixLayer_Hide( jld->pILayer, False );
     D_ASSERT( rval == TM_OK );

     return DFB_OK;
}

static DFBResult
update_screen( JAGLayerData *jld, int x, int y, int w, int h,
               CoreSurfaceBufferLock *lock )
{
     tmErrorCode_t  rval;
     CoreSurface   *surface;
     tmRect_t       rect;

     D_DEBUG_AT( JAG_Primary, "%s( %p, %d, %d - %dx%d )\n", __FUNCTION__, jld, x, y, w, h );
//     direct_log_printf( NULL, "%s( %p, %d, %d - %dx%d )\n", __FUNCTION__, jld, x, y, w, h );

     D_MAGIC_ASSERT( jld, JAGLayerData );
     D_ASSERT( dfb_jag != NULL );

     surface = jld->surface;
     D_ASSERT( surface != NULL );

     if (!jld->scaling) {
          if (jld->set_buffer) {
               SetSurfaceBuffer( jld, surface, lock );

               jld->set_buffer = false;
          }

          return DFB_OK;
     }

     if (x < 0 || y < 0 || x > surface->config.size.w - 1 || y > surface->config.size.h - 1)
          return DFB_INVARG;

//     direct_log_printf(NULL,"surface %p  (%dx%d), jld %p\n",surface, surface->config.size.w, surface->config.size.h, jld);

     if (x) {
          x--;
          w++;
     }

     if (y) {
          y--;
          h++;
     }



     if (w < 16)
          w = 16;

     if (h < 16)
          h = 16;

     if (x + w > surface->config.size.w) {
          x = surface->config.size.w - w;
     }

     if (y + h > surface->config.size.h) {
          y = surface->config.size.h - h;
     }

     w++;
     h++;

     if (x + w > surface->config.size.w) {
          w = surface->config.size.w - x;
     }

     if (y + h > surface->config.size.h) {
          h = surface->config.size.h - y;
     }

     rect.ul.x = x;
     rect.ul.y = y;
     rect.lr.x = x + w;
     rect.lr.y = y + h;

     pthread_mutex_lock( &dfb_jag->mbsLock );

     if (dfb_jag->mbsBusy) {
          D_DEBUG_AT( JAG_Primary, "  -> waiting for idle MBS...\n" );

          while (dfb_jag->mbsBusy)
               pthread_cond_wait( &dfb_jag->mbsCond, &dfb_jag->mbsLock );

          D_DEBUG_AT( JAG_Primary, "  -> finished waiting for MBS.\n" );
     }

     dfb_jag->mbsBusy = true;

     if (surface->config.format == DSPF_LUT8) {
          CorePalette *palette = surface->palette;

          D_ASSERT( palette != NULL );

          D_DEBUG_AT( JAG_Primary, "  -> tmIPlfInstGfx_SetClut( %p, %d, %p )\n",
                      dfb_pIPlfInstGfx, palette->num_entries, (void*)palette->entries );

          rval = tmIPlfInstGfx_SetClut( dfb_pIPlfInstGfx, palette->num_entries, (void*)palette->entries );
          if (rval)
               D_ERROR( "JAG/Primary: tmIPlfInstGfx_SetClut() returned error code %lu!\n", rval );
     }

     do {
         D_DEBUG_AT( JAG_Primary, "  -> tmIPlfInstGfx_ScaledBlit()...\n" );

         rval = tmIPlfInstGfx_ScaledBlit( dfb_pIPlfInstGfx,
                                          surface->config.size.w,
                                          surface->config.size.h,
                                          rect,
                                          jld->format_class,
                                          jld->format,
                                          lock->pitch,
                                          ( lock->pitch * surface->config.size.h ),
                                          lock->addr,
                                          jld->scale_width,
                                          jld->scale_height,
                                          jld->scale_format_class,
                                          jld->scale_format,
                                          jld->scale_pitch,
                                          ( jld->scale_pitch * jld->scale_height ),
                                          (void*) dfb_jag->memVirt + (jld->first ?
                                                                      dfb_jag->scale_buffer :
                                                                      dfb_jag->scale_buffer2) );

         D_DEBUG_AT( JAG_Primary, "  -> tmIPlfInstGfx_ScaledBlit() returned %lu.\n", rval );

         if (rval == TM_ERR_BUSY) {
              //direct_log_printf( NULL, "BUSY ERROR\n" );
         }
     } while (rval == TM_ERR_BUSY);

     pthread_mutex_unlock( &dfb_jag->mbsLock );

     if (rval) {
          D_ERROR( "JAG/Primary: tmIPlfInstGfx_ScaledBlit() returned error code %lu!\n", rval );
          return DFB_FAILURE;
     }

     if (jld->set_buffer) {
          SetScaleBuffer( jld );

          jld->set_buffer = false;
     }

     return DFB_OK;
}

/**********************************************************************************************************************/

DFBResult
dfb_jag_set_region_handler( JAGSetRegionData *data )
{
     DFBResult     ret;
     JAGLayerData *jld = data->layerData;

     D_MAGIC_ASSERT( jld, JAGLayerData );

     fusion_skirmish_prevail( &dfb_jag->lock );

     ret = set_region( jld, &data->config, data->updated, data->surface, data->palette );

     fusion_skirmish_dismiss( &dfb_jag->lock );

     return ret;
}

DFBResult
dfb_jag_update_screen_handler( JAGUpdateScreenData *data )
{
     DFBResult ret;

     D_ASSERT( data != NULL );

     D_MAGIC_ASSERT( data->layerData, JAGLayerData );

     fusion_skirmish_prevail( &dfb_jag->lock );

     ret = update_screen( data->layerData, DFB_RECTANGLE_VALS_FROM_REGION( &data->region ), data->lock );

     fusion_skirmish_dismiss( &dfb_jag->lock );

     return ret;
}

DFBResult
dfb_jag_set_palette_handler( JAGSetPaletteData *data )
{
     tmErrorCode_t  rval;
     JAGLayerData  *jld     = data->layerData;
     CorePalette   *palette = data->palette;

     D_DEBUG_AT( JAG_Primary, "%s()\n", __FUNCTION__ );

     fusion_skirmish_prevail( &dfb_jag->lock );

     rval = tmIVmixGfxLayer_SetClut( jld->pIGfxLayer, palette->num_entries, (void*)palette->entries );

     fusion_skirmish_dismiss( &dfb_jag->lock );

     return rval ? DFB_FAILURE : DFB_OK;
}

DFBResult
dfb_jag_remove_region_handler( JAGLayerData *jld )
{
     tmErrorCode_t rval;

     D_DEBUG_AT( JAG_Primary, "%s()\n", __FUNCTION__ );

     fusion_skirmish_prevail( &dfb_jag->lock );

     rval = tmIVmixLayer_Hide( jld->pILayer, True );

     fusion_skirmish_dismiss( &dfb_jag->lock );

     return rval ? DFB_FAILURE : DFB_OK;
}

/**********************************************************************************************************************/

static DFBResult
dfb_jag_set_region( CoreDFB *core, JAGSetRegionData *data )
{
     int               ret;
     JAGSetRegionData *tmp = NULL;

     D_ASSERT( core != NULL );
     D_ASSERT( data != NULL );

     if (dfb_core_is_master( core ))
          return dfb_jag_set_region_handler( data );

     if (!fusion_is_shared( dfb_jag_world, data )) {
          tmp = SHMALLOC( dfb_jag->shmpool, sizeof(*tmp) );
          if (!tmp)
               return DFB_NOSYSTEMMEMORY;

          direct_memcpy( tmp, data, sizeof(*tmp) );
     }

     fusion_call_execute( &dfb_jag->call, FCEF_NONE, JAG_SET_REGION,
                          tmp ? tmp : data, &ret );

     if (tmp)
          SHFREE( dfb_jag->shmpool, tmp );

     return ret;
}

static DFBResult
dfb_jag_update_screen( CoreDFB *core, JAGUpdateScreenData *data )
{
     int                  ret;
     JAGUpdateScreenData *tmp = NULL;

     D_ASSERT( data != NULL );
     D_MAGIC_ASSERT( data->layerData, JAGLayerData );

     if (dfb_core_is_master( core ))
          return dfb_jag_update_screen_handler( data );

     if (!fusion_is_shared( dfb_jag_world, data )) {
          tmp = SHMALLOC( dfb_jag->shmpool, sizeof(*data) );
          if (!tmp)
               return DFB_NOSYSTEMMEMORY;

          direct_memcpy( tmp, data, sizeof(*data) );

          D_MAGIC_ASSERT( tmp->layerData, JAGLayerData );
     }

     /* FIXME: removed ONEWAY because of SHFREE */
     fusion_call_execute( &dfb_jag->call, FCEF_NONE, JAG_UPDATE_SCREEN,
                          tmp ? tmp : data, &ret );

     if (tmp)
          SHFREE( dfb_jag->shmpool, tmp );

     return DFB_OK;
}

static DFBResult
dfb_jag_set_palette( CoreDFB *core, JAGSetPaletteData *data )
{
     int                ret;
     JAGSetPaletteData *tmp = NULL;

     D_ASSERT( data != NULL );
     D_MAGIC_ASSERT( data->layerData, JAGLayerData );

     if (dfb_core_is_master( core ))
          return dfb_jag_set_palette_handler( data );

     if (!fusion_is_shared( dfb_jag_world, data )) {
          tmp = SHMALLOC( dfb_jag->shmpool, sizeof(*data) );
          if (!tmp)
               return DFB_NOSYSTEMMEMORY;

          direct_memcpy( tmp, data, sizeof(*data) );

          D_MAGIC_ASSERT( tmp->layerData, JAGLayerData );
     }

     /* FIXME: removed ONEWAY because of SHFREE */
     fusion_call_execute( &dfb_jag->call, FCEF_NONE, JAG_SET_PALETTE,
                          tmp ? tmp : data, &ret );

     if (tmp)
          SHFREE( dfb_jag->shmpool, tmp );

     return DFB_OK;
}

static DFBResult
dfb_jag_remove_region( CoreDFB *core, JAGLayerData *jld )
{
     D_MAGIC_ASSERT( jld, JAGLayerData );

     if (dfb_core_is_master( core ))
          return dfb_jag_remove_region_handler( jld );

     fusion_call_execute( &dfb_jag->call, FCEF_NONE, JAG_REMOVE_REGION, jld, NULL );

     return DFB_OK;
}



//-----------------------------------------------------------------------------
// Internal functions:
//-----------------------------------------------------------------------------
//

//-----------------------------------------------------------------------------
// FUNCTION:    tmIPlfInstGfxNtfOnSubscriptionChanged
//
// DESCRIPTION:
//
// RETURN:      Nothing
//
// NOTES:
//-----------------------------------------------------------------------------
//
static Void STDCALL
tmIPlfInstGfxNtfOnSubscriptionChanged (
    ptmThif thif,                       //  I: object or interface pointer
    UInt32 cookie,                      //  I:
    tmPlfInstGfx_Notifications_t notifs //  I:
    )
{
} // tmIPlfInstGfxNtfOnSubscriptionChanged ()

//-----------------------------------------------------------------------------
// FUNCTION:    tmIPlfInstGfxNtfOnScaledBlitCompleted
//
// DESCRIPTION:
//
// RETURN:      Nothing
//
// NOTES:
//-----------------------------------------------------------------------------
//
static Void STDCALL
tmIPlfInstGfxNtfOnScaledBlitCompleted (
    ptmThif thif, //  I: object or interface pointer
    UInt32 cookie //  I:
    )
{
    pthread_mutex_lock( &dfb_jag->mbsLock );

    D_ASSUME( dfb_jag->mbsBusy );

//    direct_log_printf( NULL, "         completed!\n" );
    dfb_jag->mbsBusy = false;

    pthread_cond_signal( &dfb_jag->mbsCond );

    pthread_mutex_unlock( &dfb_jag->mbsLock );
} // tmIPlfInstGfxNtfOnScaledBlitCompleted ()

