/*
   (c) Copyright 2000-2002  convergence integrated media GmbH.
   (c) Copyright 2002-2004  convergence GmbH.

   All rights reserved.

   Written by Denis Oliver Kropp <dok@directfb.org>,
              Andreas Hundt <andi@fischlustig.de>,
              Sven Neumann <neo@directfb.org> and
              Ville Syrjl <syrjala@sci.fi>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <stdio.h>

#include <directfb.h>

#include <fusion/fusion.h>
#include <fusion/shmalloc.h>

#include <core/core.h>
#include <core/coredefs.h>
#include <core/coretypes.h>
#include <core/layers.h>
#include <core/palette.h>
#include <core/system.h>

#include <gfx/convert.h>

#include <misc/conf.h>
#include <misc/util.h>

#include <direct/memcpy.h>
#include <direct/messages.h>


#include "jag.h"
#include "video.h"


#define JAG_VIDEO_SUPPORTED_OPTIONS   (DLOP_NONE)


D_DEBUG_DOMAIN( JAG_Video, "JAG/Video", "Core System JAG Video Layer" );

/**********************************************************************************************************************/

static DFBResult dfb_jag_video_set_region( CoreDFB *core, JAGVideoSetRegionData *data );

/**********************************************************************************************************************/

static int
videoLayerDataSize()
{
     return sizeof(JAGVideoLayerData);
}

static int
videoRegionDataSize()
{
     return 0;
}

static DFBResult
videoInitLayer( CoreLayer                  *layer,
                void                       *driver_data,
                void                       *layer_data,
                DFBDisplayLayerDescription *description,
                DFBDisplayLayerConfig      *config,
                DFBColorAdjustment         *adjustment )
{
     tmErrorCode_t      rval;
     JAGVideoLayerData *jvld = layer_data;

     D_DEBUG_AT( JAG_Video, "%s()\n", __FUNCTION__ );

     /* set capabilities and type */
     description->caps = DLCAPS_SCREEN_POSITION | DLCAPS_SCREEN_SIZE;
     description->type = DLTF_VIDEO;

     /* set name */
     snprintf( description->name,
               DFB_DISPLAY_LAYER_DESC_NAME_LENGTH, "JAG Video Layer" );

     /* fill out the default configuration */
     config->flags   = DLCONF_OPTIONS;
     config->options = DLOP_NONE;

     /* Get interface to video layer */
     rval = tmIVmix_GetLayer( dfb_pIVmix1, 0, &jvld->pILayer );
     DBG_ASSERT2( rval == TM_OK, ("Error = 0x%x", rval) );

     /* Get video layer interface */
     rval = tmIVmixVidLayer_QueryInterface( jvld->pILayer, &TMIID_tmIVmixVidLayer, &jvld->pIVidLayer);
     DBG_ASSERT2( rval == TM_OK, ("Error = 0x%x", rval) );

     /* Query maximum layer window size */
     tmIVmixLayer_GetMaxLayerWindowSize( jvld->pILayer,
                                         &jvld->maxLayerWindowWidth,
                                         &jvld->maxLayerWindowHeight );

     /* Query minimum window size */
     tmIVmixLayer_GetMinWindowSize( jvld->pILayer,
                                    &jvld->minWindowWidth,
                                    &jvld->minWindowHeight );

     D_DEBUG_AT( JAG_Video, "  -> min %lux%lu\n", jvld->minWindowWidth, jvld->minWindowHeight );
     D_DEBUG_AT( JAG_Video, "  -> max %lux%lu\n", jvld->maxLayerWindowWidth, jvld->maxLayerWindowHeight );

     dfb_jag->videoLayerData = jvld;

     return DFB_OK;
}

static DFBResult
videoTestRegion( CoreLayer                  *layer,
                 void                       *driver_data,
                 void                       *layer_data,
                 CoreLayerRegionConfig      *config,
                 CoreLayerRegionConfigFlags *failed )
{
     JAGVideoLayerData          *jvld = layer_data;
     CoreLayerRegionConfigFlags  fail = 0;

     D_DEBUG_AT( JAG_Video, "%s()\n", __FUNCTION__ );

     if (config->options & ~JAG_VIDEO_SUPPORTED_OPTIONS)
          fail |= CLRCF_OPTIONS;

     D_DEBUG_AT( JAG_Video, "  -> dst %d, %d - %dx%d\n",
                 DFB_RECTANGLE_VALS( &config->dest ) );


     if (config->dest.w < jvld->minWindowWidth) {
          D_DEBUG_AT( JAG_Video, "  -> %d < %lu!\n", config->dest.w, jvld->minWindowWidth );
          fail |= CLRCF_DEST;
     }

     if (config->dest.h < jvld->minWindowHeight) {
          D_DEBUG_AT( JAG_Video, "  -> %d < %lu!\n", config->dest.h, jvld->minWindowHeight );
          fail |= CLRCF_DEST;
     }


     if (config->dest.w > jvld->maxLayerWindowWidth) {
          D_DEBUG_AT( JAG_Video, "  -> %d > %lu!\n", config->dest.w, jvld->maxLayerWindowWidth );
          fail |= CLRCF_DEST;
     }

     if (config->dest.h > jvld->maxLayerWindowHeight) {
          D_DEBUG_AT( JAG_Video, "  -> %d > %lu!\n", config->dest.h, jvld->maxLayerWindowHeight );
          fail |= CLRCF_DEST;
     }


     if (failed)
          *failed = fail;

     if (fail) {
          D_DEBUG_AT( JAG_Video, "  -> fail 0x%08x\n", fail );
          return DFB_UNSUPPORTED;
     }

     return DFB_OK;
}

static DFBResult
videoAddRegion( CoreLayer             *layer,
                void                  *driver_data,
                void                  *layer_data,
                void                  *region_data,
                CoreLayerRegionConfig *config )
{
     D_DEBUG_AT( JAG_Video, "%s()\n", __FUNCTION__ );

     return DFB_OK;
}

static DFBResult
videoSetRegion( CoreLayer                  *layer,
                void                       *driver_data,
                void                       *layer_data,
                void                       *region_data,
                CoreLayerRegionConfig      *config,
                CoreLayerRegionConfigFlags  updated,
                CoreSurface                *surface,
                CorePalette                *palette,
                CoreSurfaceBufferLock      *lock )
{
     JAGVideoSetRegionData data;

     D_DEBUG_AT( JAG_Video, "%s()\n", __FUNCTION__ );

     data.config  = *config;
     data.updated = updated;

     return dfb_jag_video_set_region( dfb_jag_core, &data );
}

static DFBResult
videoRemoveRegion( CoreLayer             *layer,
                   void                  *driver_data,
                   void                  *layer_data,
                   void                  *region_data )
{
     D_DEBUG_AT( JAG_Video, "%s()\n", __FUNCTION__ );

     return DFB_OK;
}

DisplayLayerFuncs jagVideoLayerFuncs = {
     LayerDataSize:      videoLayerDataSize,
     RegionDataSize:     videoRegionDataSize,
     InitLayer:          videoInitLayer,

     TestRegion:         videoTestRegion,
     AddRegion:          videoAddRegion,
     SetRegion:          videoSetRegion,
     RemoveRegion:       videoRemoveRegion
};

/**********************************************************************************************************************/

static DFBResult
set_region( JAGVideoLayerData          *jvld,
            CoreLayerRegionConfig      *config,
            CoreLayerRegionConfigFlags  updated )
{
     tmErrorCode_t       rval;
     tmRect_t            rect;
     UInt32              width;
     UInt32              height;
     tmVmix_FieldRate_t  fieldRate;
     tmVmix_ScanType_t   scanType;

     D_DEBUG_AT( JAG_Video, "%s( %p, %p )\n", __FUNCTION__, jvld, config );

     D_ASSERT( jvld != NULL );
     D_ASSERT( config != NULL );
     D_ASSERT( dfb_jag != NULL );
     D_ASSERT( dfb_pIVmix1 != NULL );

     /* Hide the layer? */
     if (!config->opacity) {
          D_DEBUG_AT( JAG_Video, "  -> tmIVmixLayer_Hide( True )\n" );

          rval = tmIVmixLayer_Hide( jvld->pILayer, True );
          DBG_ASSERT2( rval == TM_OK, ("Error = 0x%x", rval) );
     }

     rval = tmIVmix_GetOutputProperties( dfb_pIVmix1, &width, &height, &fieldRate, &scanType );
     DBG_ASSERT2( rval == TM_OK, ("Error = 0x%x", rval) );

     if (updated & CLRCF_DEST) {
          D_DEBUG_AT( JAG_Video, "  -> dst %d, %d - %dx%d\n",
                      config->dest.x, config->dest.y, config->dest.w, config->dest.h );


          /* Set destination window. */
          rect.ul.x = config->dest.x;
          rect.ul.y = config->dest.y;
          rect.lr.x = config->dest.x + config->dest.w;
          rect.lr.y = config->dest.y + config->dest.h;

          D_DEBUG_AT( JAG_Video, "  -> tmIVmixLayer_SetDstWindow( %ld, %ld, %ld, %ld )\n",
                      rect.ul.x, rect.ul.y, rect.lr.x, rect.lr.y );

          rval = tmIVmixLayer_SetDstWindow( jvld->pILayer, rect );
          DBG_ASSERT2( rval == TM_OK, ("Error = 0x%x", rval) );


          /* Set layer window. */
          rect.ul.x = 0;
          rect.ul.y = 0;
          rect.lr.x = width;
          rect.lr.y = height;

          D_DEBUG_AT( JAG_Video, "  -> tmIVmixLayer_SetLayerWindow( %ld, %ld, %ld, %ld )\n",
                      rect.ul.x, rect.ul.y, rect.lr.x, rect.lr.y );

          rval = tmIVmixLayer_SetLayerWindow( jvld->pILayer, rect );
          DBG_ASSERT2( rval == TM_OK, ("Error = 0x%x", rval) );


          /* Activate settings. */
          D_DEBUG_AT( JAG_Video, "  -> tmIVmixLayer_ActivateNewWindowSettings( True, 0 )\n" );

          rval = tmIVmixLayer_ActivateNewWindowSettings( jvld->pILayer, True, 0 );
          DBG_ASSERT2( rval == TM_OK, ("Error = 0x%x", rval) );
     }

     /* Show the layer? */
     if (config->opacity) {
          D_DEBUG_AT( JAG_Video, "  -> tmIVmixLayer_Hide( False )\n" );

          rval = tmIVmixLayer_Hide( jvld->pILayer, False );
          DBG_ASSERT2( rval == TM_OK, ("Error = 0x%x", rval) );
     }

     return DFB_OK;
}

/**********************************************************************************************************************/

DFBResult
dfb_jag_video_set_region_handler( JAGVideoSetRegionData *data )
{
     DFBResult          ret;
     JAGVideoLayerData *jvld = dfb_jag->videoLayerData;

     fusion_skirmish_prevail( &dfb_jag->lock );

     ret = set_region( jvld, &data->config, data->updated );

     fusion_skirmish_dismiss( &dfb_jag->lock );

     return ret;
}

static DFBResult
dfb_jag_video_set_region( CoreDFB *core, JAGVideoSetRegionData *data )
{
     int                    ret;
     JAGVideoSetRegionData *tmp = NULL;

     D_ASSERT( core != NULL );
     D_ASSERT( data != NULL );

     if (dfb_core_is_master( core ))
          return dfb_jag_video_set_region_handler( data );

     if (!fusion_is_shared( dfb_jag_world, data )) {
          tmp = SHMALLOC( dfb_jag->shmpool, sizeof(*tmp) );
          if (!tmp)
               return DFB_NOSYSTEMMEMORY;

          direct_memcpy( tmp, data, sizeof(*tmp) );
     }

     fusion_call_execute( &dfb_jag->call, FCEF_NONE, JAG_VIDEO_SET_REGION,
                          tmp ? tmp : data, &ret );

     if (tmp)
          SHFREE( dfb_jag->shmpool, tmp );

     return ret;
}

