/*
 *  Copyright (C) 2010 Koninklijke Philips Electronics N.V.
 *  Pieter Van Loocke
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 */


#include <asm/io.h>
#include <asm/page.h>
#include <asm/sigcontext.h>
#include <asm/signal.h>
#include <asm/ucontext.h>
#include <linux/errno.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <linux/hash.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/list.h>
#include <linux/marker.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/pagemap.h>
#include <linux/pid.h>
#include <linux/proc_fs.h>
#include <linux/sched.h>
#include <linux/seq_file.h>
#include <linux/spinlock.h> 
#include <linux/string.h>
#include <linux/syscalls.h>
#include <linux/vmalloc.h>

MODULE_LICENSE("GPL");

/* configuration */
#define BIGBUF ( 0 )

/*
 * Noclear data
 */

#define MAGICTRUE 0x20100906
#define LOGBUFSIZE 4096
#define STACKBUFSIZE 200000
#if BIGBUF
    #define TRACEBUFSIZE ( 131072 )
    #define PRINTBUFSIZE ( 100000 )
#else
    #define TRACEBUFSIZE ( 32768 )
    #define PRINTBUFSIZE ( 30000 )
#endif /* BIGBUF */
#define TASKBUCKETORDER ( 10 )
#define TASKBUCKETLIM ( 1 << TASKBUCKETORDER )   /* search by hash */
#define TASKLIM ( 2000 )
#define TASKNAMELIM ( 20 )
#define PUMPLIM ( 3000 )
#define PUMPNAMELIM ( 40 )

typedef struct
{
    unsigned cw;
    unsigned dw;
} TraceRecord;

typedef enum
{
    TaskAddNative,
    TaskAddTpRtk,
    TaskAddMgr,
    TaskAddEngine,
    TaskAddPump
} TaskType;

typedef struct
{
    struct hlist_node head;
    TaskType type;
    char name[ TASKNAMELIM ];
    int customname;
    int sampleref;
    int childrefs;
    int ppid;
    int pid;
    int tgid;
    int mgrid;
    void* entry;
    int priority;
    void* unused1;
    void* unused2;
    struct hlist_head pumplist;
} TaskEntry; /* 80 bytes */

typedef struct
{
    struct hlist_node head;
    char name[ PUMPNAMELIM ];
    int pid;
    int pen;
    int pump;
    void* entry;
    int replacing;
} PumpEntry; /* 68 bytes */

struct crash_data
{
    int logbufvalid;
    char logbuf[ LOGBUFSIZE ];

    int stackbufvalid;
    char stackbuf[ STACKBUFSIZE ];
    char* stackcur;

    int tracebufvalid;
    TraceRecord tracebuf[ TRACEBUFSIZE ];
    TraceRecord* tracecur;
    int tracemode;
    int tracefull;

    char printbuf[ PRINTBUFSIZE ];
    char* printcur;
    int printfull;

    TaskEntry TaskMemPool[ TASKLIM ];
    struct hlist_head TaskList[ TASKBUCKETLIM ];
    struct hlist_head TaskFreeList;
    int TaskCount;

    PumpEntry PumpMemPool[ PUMPLIM ];
    struct hlist_head PumpFreeList;
    int PumpCount;
};

static struct crash_data kernel_crash_dump;


/*
 * Kernel trace
 */

/* portability */
#define TIMESTAMP PNX8XXX_MMIO_BASE_VIRT+0x14C0C0 /* master timer 0..31 @ 27 MHz*/

#define FALSE 0
#define TRUE  1

typedef unsigned long Nat32;

/* ISR safe critical section */
static spinlock_t trcbuf_lock = SPIN_LOCK_UNLOCKED;


/* frozen stored with cookie to survive reboot */
#define MODEFIFO 1
#define MODESTACK 2

/* trace buffer admin */
#define TRACEFROZEN ( kernel_crash_dump.tracebufvalid )
#define TRACEBUF ( kernel_crash_dump.tracebuf )
#define TRACECUR ( kernel_crash_dump.tracecur )
#define TRACEMODE ( kernel_crash_dump.tracemode )
#define TRACEFULL ( kernel_crash_dump.tracefull )
static int samplecounter = 0;

/* tracing options */
static int stracing = FALSE;
static int irqtracing = FALSE;
static int softirqtracing = FALSE;
static int signaltracing = TRUE;

/* print buffer admin */
#define PRINTSTRLIM ( 100 )

#define PRINTBUF ( kernel_crash_dump.printbuf )
#define PRINTCUR ( kernel_crash_dump.printcur )
#define PRINTFULL ( kernel_crash_dump.printfull )

/* task data lock */
static spinlock_t taskdata_lock = SPIN_LOCK_UNLOCKED;

/* task data */
typedef struct
{
    TaskType type;
    int pid;
    union
    {
        struct
        {
            int tid;
            int mgrid;
            char name[ TASKNAMELIM ];
            void* entry;
            int priority;
        } task;
        
        struct
        {
            int pump;
            int pen;
            char name[ PUMPNAMELIM ];
            void* entry;
            int replacing;
        } pump;
    };
} TaskAddData;

typedef struct
{
    int pid;
} TaskDelData;


#define TASKMEMPOOL ( kernel_crash_dump.TaskMemPool )
#define TASKLIST ( kernel_crash_dump.TaskList )
#define TASKFREELIST ( kernel_crash_dump.TaskFreeList )
#define TASKCOUNT ( kernel_crash_dump.TaskCount )

#define PUMPMEMPOOL ( kernel_crash_dump.PumpMemPool )
#define PUMPFREELIST ( kernel_crash_dump.PumpFreeList )
#define PUMPCOUNT ( kernel_crash_dump.PumpCount )

static int PeriodicTaskCleanupStartComplete = FALSE;
DECLARE_COMPLETION( PeriodicTaskCleanupExit );


/*
 * Writing samples in the buffer
 */

static inline void stamp( Nat32 cw, Nat32 dw )
{
    if( TRACEFROZEN != MAGICTRUE )
    {
        /* store sample */
        TRACECUR->cw = cw;
        TRACECUR->dw = dw;
    
        /* buffer wrapping */
        if( ++TRACECUR >= TRACEBUF + TRACEBUFSIZE ) /* end of buf */
        {
            if( TRACEMODE == MODEFIFO )
            {
                TRACECUR = TRACEBUF;     /* overwrite first samples in fifo mode */
                TRACEFULL = TRUE;
            }
            else  /* mode == MODESTACK */
            {
                TRACEFROZEN = MAGICTRUE;
                TRACEFULL = TRUE;
            }
        }
    }
}

#define CUSTOM_SAMPLE ( 13 )

static void probe_arch_mips_kernel_trcext_trace_print_irq( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned data = 0;

    data = va_arg( *args, typeof( data ) );

    stamp( CUSTOM_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), data );
}


static void probe_arch_mips_kernel_trcext_trace_print( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    unsigned data = 0;

    data = va_arg( *args, typeof( data ) );

    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( CUSTOM_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), data );
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}


/*
 * Downloading trace data
 */

static ssize_t local_traceread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    char* start = ( char* )TRACEBUF;
    char* lim = ( char* )( TRACEBUF + TRACEBUFSIZE );
    char* head = ( char* )( TRACEFULL ? TRACECUR : TRACEBUF );
    Nat32 size = ( TRACEFULL ? TRACEBUFSIZE : TRACECUR - TRACEBUF ) * sizeof( TraceRecord );
    size_t smallcount = count;

    char* addr;
    Nat32 len;

    if( TRACEFROZEN != MAGICTRUE )
    {
        return -EBUSY;
    }
    
    if( *ppos >= size ) 
    {
        return 0;
    }
    
    if( *ppos + smallcount > size )
    {
        smallcount = size - *ppos;
    }

    addr = head + *ppos;
    if( addr >= lim ) addr -= size;
    
    len = smallcount;
    if( addr + len > lim ) len = lim - addr;
    
    if( copy_to_user( buf, addr, len ) )
    {
        return -EFAULT;
    };
    
    buf += len;
    
    if( smallcount - len > 0 )
    {
        addr = start;
        len = smallcount - len;
        
        if( copy_to_user( buf, addr, len ) )
        {
            return -EFAULT;
        }
    }
    
    *ppos += smallcount;
    return smallcount;
}

static ssize_t local_tasksread( struct file* file, char __user* buf, size_t count, loff_t* ppos )
{
    /* todo: optimize repeated reads by making some locals static (=tricky) */
    size_t totalcount = 0;
    char __user* bufpos = buf;
    
    int tasknr = ( int )( *ppos ) / sizeof( TaskEntry );
    int offset = ( int )( *ppos ) % sizeof( TaskEntry );
    size_t smallcount = sizeof( TaskEntry ) - offset;
    int bucket = 0;
    int taskcount = 0;
    struct hlist_node* list = NULL;
    TaskEntry* t = NULL;
    
    if( TRACEFROZEN != MAGICTRUE )
    {
        return -EBUSY;
    }

    if( smallcount > count )
    {
        smallcount = count;
    }

    /* iterate to right task */
    for( bucket = 0; bucket < TASKBUCKETLIM; ++bucket )
    {
        hlist_for_each_entry( t, list, &TASKLIST[ bucket ], head )
        {
            if( count == 0 ) goto OuterBreak;
            if( taskcount >= tasknr )
            {
                /* check last known name of the process */
                if( !t->customname )
                {
                    struct task_struct* task = find_task_by_pid_ns( t->pid, &init_pid_ns );
                    if( task )
                    {
                        strncpy( t->name, task->comm, TASKNAMELIM );
                        t->name[ TASKNAMELIM-1 ] = '\0';
                    }
                }
                
                /* copy data to user */
                if( copy_to_user( bufpos, ( char* )t + offset, smallcount ) )
                {
                    return -EFAULT;
                }
                offset = 0;
                count -= smallcount;
                totalcount += smallcount;
                bufpos += smallcount;
                smallcount = sizeof( TaskEntry ) < count ? sizeof( TaskEntry ) : count;
            }
            else
            {
                /* skip task */
            }
            
            taskcount += 1;
        }
    }
OuterBreak:
    
    *ppos += totalcount;
    return totalcount;  
}


static ssize_t local_pumpsread( struct file* file, char __user* buf, size_t count, loff_t* ppos )
{
    /* todo: optimize repeated reads by making some locals static (=tricky) */
    size_t totalcount = 0;
    char __user* bufpos = buf;
    
    int pumpnr = ( int )( *ppos ) / sizeof( PumpEntry );
    int offset = ( int )( *ppos ) % sizeof( PumpEntry );
    size_t smallcount = sizeof( PumpEntry ) - offset;
    int bucket = 0;
    int pumpcount = 0;
    struct hlist_node* tasklist = NULL;
    struct hlist_node* pumplist = NULL;
    TaskEntry* t = NULL;
    PumpEntry* p = NULL;
    
    if( TRACEFROZEN != MAGICTRUE )
    {
        return -EBUSY;
    }

    if( smallcount > count )
    {
        smallcount = count;
    }

    /* iterate to right task */
    for( bucket = 0; bucket < TASKBUCKETLIM; ++bucket )
    {
        hlist_for_each_entry( t, tasklist, &TASKLIST[ bucket ], head )
        {
            hlist_for_each_entry( p, pumplist,  &t->pumplist, head )
            {
                if( count == 0 ) goto OuterBreak;
                if( pumpcount >= pumpnr )
                {
                    if( copy_to_user( bufpos, ( char* )p + offset, smallcount ) )
                    {
                        return -EFAULT;
                    }
                    offset = 0;
                    count -= smallcount;
                    totalcount += smallcount;
                    bufpos += smallcount;
                    smallcount = sizeof( PumpEntry ) < count ? sizeof( PumpEntry ) : count;
                }
                else
                {
                    /* skip pump */
                }
                
                pumpcount += 1;
            }
        }
    }
OuterBreak:
    
    *ppos += totalcount;
    return totalcount;  
}


static ssize_t local_printsread( struct file* file, char __user* buf, size_t count, loff_t* ppos )
{
    char* head = PRINTFULL ? PRINTCUR : PRINTBUF;
    size_t smallcount = count;

    char* addr;
    Nat32 len;

    if( TRACEFROZEN != MAGICTRUE )
    {
        return -EBUSY;
    }
    
    if( *ppos >= PRINTBUFSIZE ) 
    {
        return 0;
    }
    
    if( *ppos + smallcount > PRINTBUFSIZE )
    {
        smallcount = PRINTBUFSIZE - *ppos;
    }

    addr = head + *ppos;
    if( addr >= PRINTBUF + PRINTBUFSIZE ) addr -= PRINTBUFSIZE;
    
    len = smallcount;
    if( addr + len > PRINTBUF + PRINTBUFSIZE ) len = PRINTBUF + PRINTBUFSIZE - addr;
    
    if( copy_to_user( buf, addr, len ) )
    {
        return -EFAULT;
    };
    
    buf += len;
    
    if( smallcount - len > 0 )
    {
        addr = PRINTBUF;
        len = smallcount - len;
        
        if( copy_to_user( buf, addr, len ) )
        {
            return -EFAULT;
        }
    }
    
    *ppos += smallcount;
    return smallcount;
}


static ssize_t trcext_traceread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    static int subfile = 0;
    static loff_t tasksoff = 0;
    static loff_t pumpsoff = 0;
    static loff_t printsoff = 0;
    ssize_t result = 0;
    
    if( TRACEFROZEN != MAGICTRUE )
    {
        return -EBUSY;
    }

    if( *ppos == 0 )
    {
        subfile = 0;
        tasksoff = 0;
        pumpsoff = 0;
        printsoff = 0;
    }
    
    switch( subfile )
    {
    case 0: /* trace */
        result = local_traceread( file, buf, count, ppos );
        if( result != 0 ) return result;
        tasksoff = *ppos;
        subfile = 1;
        /* no break -> fall through if result == 0 */

    case 1: /* tasks */
        *ppos -= tasksoff;
        result = local_tasksread( file, buf, count, ppos );
        *ppos += tasksoff;
        if( result != 0 ) return result;
        pumpsoff = *ppos;
        subfile = 2;
        /* no break -> fall through if result == 0 */

    case 2: /* pumps */
        *ppos -= pumpsoff;
        result = local_pumpsread( file, buf, count, ppos );
        *ppos += pumpsoff;
        if( result != 0 ) return result;
        printsoff = *ppos;
        subfile = 3;
        /* no break -> fall through if result == 0 */

    case 3: /* prints */
        *ppos -= printsoff;
        result = local_printsread( file, buf, count, ppos );
        *ppos += printsoff;
        if( result != 0 ) return result;
        subfile = 4;
        /* no break -> fall through if result == 0 */

    case 4: /* index */
        result = sizeof( tasksoff ) + sizeof( pumpsoff ) + sizeof( printsoff );
        if( result > count )
        {
            return -EINVAL;  /* can't handle too small user buf */
        }
        
        if( copy_to_user( buf, ( char* )&tasksoff, sizeof( tasksoff ) ) )
        {
            return -EFAULT;
        }
        if( copy_to_user( buf + sizeof( tasksoff ), ( char* )&pumpsoff, sizeof( pumpsoff ) ) )
        {
            return -EFAULT;
        }
        if( copy_to_user( buf + sizeof( tasksoff ) + sizeof( pumpsoff ), ( char* )&printsoff, sizeof( printsoff ) ) )
        {
            return -EFAULT;
        }
        subfile = 5;
        *ppos += result;
        return result;
        break;

    case 5: /* done */
        return 0;
        break;
        
    default:
        subfile = 0;
        return -EINVAL;
        break;
    }
}


static ssize_t trcext_countread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    unsigned long irqflags = 0;
    Nat32 samplecount;
    char kerbuf[ 20 ];
    int size = 0;
    
    spin_lock_irqsave( &trcbuf_lock, irqflags );
    if( TRACEFULL )
    {
        samplecount = TRACEBUFSIZE;
    }
    else
    {
        samplecount = TRACECUR - TRACEBUF;
    }
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
    
    size = sprintf( kerbuf, "%lu\n", samplecount );
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( count > size - *ppos )
    {
        count = size - *ppos;
    }
    
    if( copy_to_user( buf, kerbuf + *ppos, count ) )
    {
        return -EFAULT;
    }
    
    *ppos += count;
    return count;
}

static ssize_t trcext_moderead( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    char* kerbuf;
    int size;

    if( TRACEMODE == MODEFIFO )
    {    
        kerbuf = "fifo\n";
        size = strlen( kerbuf ) + 1;
    }
    else
    {
        kerbuf = "stack\n";
        size = strlen( kerbuf ) + 1;
    }
     
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( count > size - *ppos )
    {
        count = size - *ppos;
    }
    
    if( copy_to_user( buf, kerbuf + *ppos, count ) )
    {
        return -EFAULT;
    }

    *ppos += count;
    return count;
}

/* this implicitly starts a new trace with the given mode */
static ssize_t trcext_modewrite( struct file * file, const char * buf, size_t count, loff_t *ppos )
{
    unsigned long irqflags = 0;
    size_t smallcount = count;
    char kerbuf[ 6 ];
    int size = 6;
    int new_mode = 0;
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( smallcount > size - *ppos )
    {
        smallcount = size - *ppos;
    }
    
    if( copy_from_user( kerbuf + *ppos, buf, smallcount ) )
    {
        return -EFAULT;
    }
    kerbuf[ size-1 ] = '\0';

    if( !strncmp( kerbuf, "fifo", 4 ) )
    {
        new_mode = MODEFIFO;
    }
    else if( !strncmp( kerbuf, "stack", 5 ) )
    {
        new_mode = MODESTACK;
    }
    else
    {
        return -EINVAL;
    }
    
    spin_lock_irqsave( &trcbuf_lock, irqflags );
    {
        int bucket = 0;
        for( bucket = 0; bucket < TASKBUCKETLIM; ++bucket )
        {
            struct hlist_node* list = NULL;
            TaskEntry* t = NULL;
            
            hlist_for_each_entry( t, list, &TASKLIST[ bucket ], head )
            {
                t->sampleref = 0;
            }
        }

        TRACEMODE = new_mode;
        TRACECUR = TRACEBUF;
        TRACEFULL = FALSE;
        TRACEFROZEN = FALSE;
        samplecounter = 0;
    }
    
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
    
    *ppos += count;
    return count;
}

static ssize_t trcext_frozenread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    char kerbuf[ 20 ];
    int size = 0;
    
    if( TRACEFROZEN == MAGICTRUE )
    {
        size = sprintf( kerbuf, "1\n" );
    }
    else
    {
        size = sprintf( kerbuf, "0\n" );
    }
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( count > size - *ppos )
    {
        count = size - *ppos;
    }
    
    if( copy_to_user( buf, kerbuf + *ppos, count ) )
    {
        return -EFAULT;
    }
    
    *ppos += count;
    return count;
}

/* when unfreezing, a new trace will implicitly be started in the same mode */
static ssize_t trcext_frozenwrite( struct file * file, const char * buf, size_t count, loff_t *ppos )
{
    unsigned long irqflags = 0;
    size_t smallcount = count;
    char kerbuf[ 2 ];
    int size = 2;
    int new_frozen = FALSE;
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( smallcount > size - *ppos )
    {
        smallcount = size - *ppos;
    }
    
    if( copy_from_user( kerbuf + *ppos, buf, smallcount ) )
    {
        return -EFAULT;
    }
    kerbuf[ size-1 ] = '\0';

    if( !strncmp( kerbuf, "1", 1 ) )
    {
        new_frozen = MAGICTRUE;
    }
    else
    {
        return -EINVAL;
    }
    
    spin_lock_irqsave( &trcbuf_lock, irqflags );
    if( TRACEFROZEN != MAGICTRUE && new_frozen == MAGICTRUE )
    {
        TRACEFROZEN = MAGICTRUE;
    }
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
    
    *ppos += count;
    return count;
}



/*
   Tracking process names
   
   A process name is available in the 16 bytes comm field of the 3 struct.
   To provide process names for processes in the kernel trace, we must copy
   this name, since processes may be deleted while references to it are still
   in the trace buffer.
 */

/* caller must lock task data structures */
static inline TaskEntry* FindTask( int pid )
{
    TaskEntry* retval = NULL;
    int bucket = hash_long( pid, TASKBUCKETORDER );
    struct hlist_node* list = NULL;
    TaskEntry* t = NULL;

    hlist_for_each_entry( t, list, &TASKLIST[ bucket ], head )
    {
        if( t->pid == pid )
        {
            retval = t;
            break;
        }
    }
    
    return retval;
}

/* caller must lock task data structures */
static inline TaskEntry* FindOrCreateTask( int pid )
{
    struct hlist_node* tasklist = NULL;
    TaskEntry* taskinfo = FindTask( pid );
    if( !taskinfo )
    {
        /* task not found => allocate */
        int bucket = hash_long( pid, TASKBUCKETORDER );
        hlist_for_each_entry( taskinfo, tasklist, &TASKFREELIST, head ){ hlist_del( &taskinfo->head ); break; } /* simply get first element */
        if( taskinfo )
        {
            struct task_struct* task = ( pid == 0 ? &init_task : find_task_by_pid_ns( pid, &init_pid_ns ) );
            if( task )
            {
                taskinfo->type = TaskAddNative;
                strncpy( taskinfo->name, task->comm, TASKNAMELIM );
                taskinfo->name[ TASKNAMELIM-1 ] = '\0';
                taskinfo->childrefs = 0;
                taskinfo->sampleref = 0;
                taskinfo->customname = 0;
                taskinfo->ppid = task->parent == NULL ? -1 : task->parent->pid;
                taskinfo->pid = task->pid;
                taskinfo->tgid = task->tgid;
                taskinfo->mgrid = 0;
                taskinfo->entry = NULL;
                taskinfo->priority = task->prio;
                INIT_HLIST_HEAD( &taskinfo->pumplist );
                hlist_add_head( &taskinfo->head, &TASKLIST[ bucket ] );
                TASKCOUNT += 1;
                
                if( taskinfo->pid != -1 )
                {
                    /* increase parent childref count if applicable */
                    TaskEntry* parent = FindTask( taskinfo->ppid );
                    if( parent )
                    {
                        parent->childrefs += 1;
                    }
                }
            
                if( taskinfo->tgid != taskinfo->pid )
                {
                    /* increase parent childref count if applicable */
                    TaskEntry* group = FindTask( taskinfo->tgid );
                    if( group )
                    {
                        group->childrefs += 1;
                    }
                }
            }
            else
            {
                /* linux couldn't find task via task_find_by_pid */
            }
        }
        else
        {
            /* couldn't allocate memory :-o */
            taskinfo = NULL;
        }
    }
    else
    {
        /* task found */
    }

    return taskinfo;
}


/* caller must lock task data structures */
static int TryDeleteTask( int pid )
{
    /* delete considerations
       - even if task is killed by Linux, it may still be in the trace
       - applications may dynamically quit and start again; we can't keep
         all task info
       - user task info is only registered once; if we delete it and the task
         later shows again in the trace, we can't recover the extra info
     */
    int deleted = 0;
    TaskEntry* t = FindTask( pid );

    if( t )
    {   
        if( samplecounter - t->sampleref > TRACEBUFSIZE  /* no samples in trace */
         && find_task_by_pid_ns( pid, &init_pid_ns ) == NULL /* task doesn't exist in Linux */
         && t->childrefs <= 0 /* task with no children referred in trace */
          )
        {
            struct hlist_node* pumplist = NULL;
            struct hlist_node* temp = NULL;
            PumpEntry* p = NULL;
            
            deleted = 1;

            /* free memory for all pumps on the task */
            hlist_for_each_entry_safe( p, pumplist, temp, &t->pumplist, head )
            {
                hlist_del( &p->head );
                hlist_add_head( &p->head, &PUMPFREELIST );
                PUMPCOUNT -= 1;
            }
            hlist_del( &t->head );

            /* try delete parent */
            if( t->pid != -1 )
            {
                TaskEntry* parent = FindTask( t->pid );
                if( parent )
                {
                    parent->childrefs -= 1;
                    TryDeleteTask( parent->pid );
                }
            } 
            
            /* try delete group owner */
            if( t->tgid != t->pid )
            {
                TaskEntry* group = FindTask( t->tgid );
                if( group )
                {
                    group->childrefs -= 1;
                    TryDeleteTask( group->tgid );
                }
            }

            /* finally, free task memory */
            hlist_add_head( &t->head, &TASKFREELIST );
            TASKCOUNT -= 1;
        }
        else
        {
            /* criteria to delete task not met */
        }
    }
    else
    {
        /* task to delete not found */
    }
    
    return deleted;
}

/* This kernel thread periodically check which tasks have been killed. It would
   be easier if we could get a notify but there's no way to do this without
   patching the kernel. */
static int PeriodicTaskCleanup( void* param )
{
    int bucket = 0;
    int active = TRUE;
    
    daemonize( "ktrcext" );
    
    while( active )
    {
        struct hlist_node* tasklist = NULL;
        struct hlist_node* temp = NULL;
        TaskEntry* t = NULL;

        /* check one bucket at a time */
        spin_lock( &taskdata_lock );
        if( !PeriodicTaskCleanupStartComplete )
        {
            hlist_for_each_entry_safe( t, tasklist, temp, &TASKLIST[ bucket ], head )
            {
                if( TryDeleteTask( t->pid ) )
                {
                    /* delete only one task per loop*/
                    break;
                }
            }
        }
        else
        {
            active = FALSE;
        }
        spin_unlock( &taskdata_lock );

        bucket += 1;
        bucket %= TASKBUCKETLIM;
        
        set_current_state( TASK_INTERRUPTIBLE );
        schedule_timeout( HZ / 4 );
    }
    
    complete( &PeriodicTaskCleanupExit );
    return 0;
}

static ssize_t trcext_taskaddwrite( struct file * file, const char * buf, size_t count, loff_t *ppos )
{
    if( TRACEFROZEN != MAGICTRUE )
    {
        TaskAddData taskdata;
        
        if( *ppos != 0 || count != sizeof( taskdata ) )
        {
            return -EINVAL;
        }
        
        if( copy_from_user( &taskdata, buf, count ) )
        {
            return -EFAULT;
        }
    
        spin_lock( &taskdata_lock );
        {
            switch( taskdata.type )
            {
                case TaskAddTpRtk:
                case TaskAddMgr:
                case TaskAddEngine:
                {
                    TaskEntry* t = FindOrCreateTask( taskdata.task.tid );
                    if( t )
                    {
                        t->type = taskdata.type;
                        strlcpy( t->name, taskdata.task.name, TASKNAMELIM );
                        t->customname = 1;
                        t->mgrid = taskdata.task.mgrid;
                        if( taskdata.type != TaskAddEngine )
                        {
                            t->entry = taskdata.task.entry;
                        }
                    }
                    else
                    {
                        /* task not found nor created (out of memory?) */ 
                    }
                    break;
                }
            
                case TaskAddPump:
                {
                    TaskEntry* t = FindTask( taskdata.pid );
                    if( t )
                    {
                        struct hlist_node* pumplist = NULL;
                        PumpEntry* p = NULL;
                        hlist_for_each_entry( p, pumplist, &PUMPFREELIST, head ){ hlist_del( &p->head ); break; }
                        if( p )
                        {
                            int i = 0;
                            for( i = 0; i < PUMPNAMELIM - 1; ++i ) p->name[ i ] = taskdata.pump.name[ i ];
                            p->name[ PUMPNAMELIM - 1 ] = '\0';
                            p->pid = taskdata.pid;
                            p->pen = taskdata.pump.pen;
                            p->pump = taskdata.pump.pump;
                            p->entry = taskdata.pump.entry;
                            p->replacing = taskdata.pump.replacing;

                            hlist_add_head( &p->head, &t->pumplist );
                            PUMPCOUNT += 1;
                        }
                        else
                        {
                            /* couldn't alloc from pump memory */
                        }
                    }
                    else
                    {
                        /* creating pump for process that is not in our list
                           we could fix this with a FindOrCreateTask but if we
                           end up here, something is really wrong anyway */
                    }
                    break;
                }
            
                default:
                {
                    printk( KERN_ERR "illegal data written to proc/taskadd\n" );
                    break;
                }
            }
        }
        spin_unlock( &taskdata_lock );
    }
    else
    {
        /* todo: should we add tasks if trace is frozen ? 
           - pro: collect task info that may be useful when trace gets unfrozen
           - contra: when trace frozen, tasks are not cleaned up
           - contra: when tasks are added while frozen, there may be race conditions when reading tasklist and pumplist
         */
    }
    
    return count;
}


static ssize_t trcext_taskdelwrite(struct file * file, const char * buf, size_t count, loff_t *ppos)
{
    TaskDelData taskdata;
    
    if( *ppos != 0 || count != sizeof( taskdata ) )
    {
        return -EINVAL;
    }
    
    if( copy_from_user( &taskdata, buf, count ) )
    {
        return -EFAULT;
    }
    
    *ppos += count;
    return count;
}



static ssize_t trcext_samplewrite(struct file * file, const char * buf, size_t count, loff_t *ppos)
{
    unsigned long irqflags = 0;
    size_t smallcount = count;
    TraceRecord sample[ 4 ];  /* max four samples */
    int size = sizeof( sample );
    int s;
    
    if( count > size )
    {
        smallcount = size;
    }
    
    if( copy_from_user( &sample, buf, smallcount ) )
    {
        return -EFAULT;
    }
    
    spin_lock_irqsave( &trcbuf_lock, irqflags );
    sample[ 0 ].cw = ( sample[ 0 ].cw & 0x0000000FUL )
                   | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL );
    for( s = 0; s < smallcount / sizeof( TraceRecord ); ++s )
    {
        stamp( sample[ s ].cw, sample[ s ].dw );
    }
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
    
    *ppos += smallcount;
    return smallcount;
}

static ssize_t trcext_printwrite(struct file * file, const char * buf, size_t count, loff_t *ppos)
{
    unsigned long irqflags = 0;
    size_t smallcount = count < PRINTSTRLIM ? count : PRINTSTRLIM;
    static char kerbuf[ PRINTSTRLIM ]; /* max string length */
    static unsigned printcount = 0;
    int i = 0;
    
    if( TRACEFROZEN != MAGICTRUE )
    {
        if( copy_from_user( kerbuf, buf, smallcount ) )
        {
            return -EFAULT;
        }
        kerbuf[ PRINTSTRLIM-1 ] = '\0';
        
        spin_lock_irqsave( &trcbuf_lock, irqflags );
        if( smallcount != 0 && kerbuf[ 0 ] != '\n' )
        {
            smallcount += 1;  /* we add \0 if not present */
        
            for( i = 0; i < smallcount; ++i )
            {
                if( i == smallcount - 1 ) kerbuf[ i ] = '\0';
                if( kerbuf[ i ] == '\n' ) kerbuf[ i ] = '\0';
        
                *PRINTCUR = kerbuf[ i ];
                PRINTCUR += 1;
                if( PRINTCUR >= PRINTBUF + PRINTBUFSIZE )
                {
                    PRINTCUR = PRINTBUF;
                    PRINTFULL = TRUE;
                }
        
                if( kerbuf[ i ] == '\0' ) break;
            }
            stamp( 3 | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), printcount++ );
        }
        else
        {
            /* printing empty line -> ignore else we get \0\0 in our PRINTBUF */
        }
        spin_unlock_irqrestore( &trcbuf_lock, irqflags );
    }
    else
    {
        /* frozen => ignore print */
    }
    
    return count;
}

#define KERNEL_EVENT_SAMPLE ( 14 )

static void probe_kernel_syscall_entry( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    long scno = 0;
    struct pt_regs* regs = NULL;

    scno = va_arg( *args, typeof( scno ) );
    regs = va_arg( *args, typeof( regs ) );

    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( KERNEL_EVENT_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), 0x00000000 + scno );
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}

static void probe_kernel_syscall_exit( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( KERNEL_EVENT_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), 0x10000000 );
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}

static ssize_t trcext_stracingread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    char kerbuf[ 20 ];
    int size = 0;
    
    if( stracing )
    {
        size = sprintf( kerbuf, "1\n" );
    }
    else
    {
        size = sprintf( kerbuf, "0\n" );
    }
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( count > size - *ppos )
    {
        count = size - *ppos;
    }
    
    if( copy_to_user( buf, kerbuf + *ppos, count ) )
    {
        return -EFAULT;
    }
    
    *ppos += count;
    return count;
}

static ssize_t trcext_stracingwrite( struct file * file, const char * buf, size_t count, loff_t *ppos )
{
    size_t smallcount = count;
    char kerbuf[ 2 ];
    int size = 2;
    int new_stracing = FALSE;
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( smallcount > size - *ppos )
    {
        smallcount = size - *ppos;
    }
    
    if( copy_from_user( kerbuf + *ppos, buf, smallcount ) )
    {
        return -EFAULT;
    }
    kerbuf[ size-1 ] = '\0';

    if( !strncmp( kerbuf, "0", 1 ) )
    {
        new_stracing = FALSE;
    }
    else if( !strncmp( kerbuf, "1", 1 ) )
    {
        new_stracing = TRUE;
    }
    else
    {
        return -EINVAL;
    }
    
    if( !stracing && new_stracing )
    {
        /* install probes for tracing syscalls */
        marker_probe_register( "kernel_syscall_entry", "%ld struct pt_regs %p", probe_kernel_syscall_entry, NULL );
        marker_probe_register( "kernel_syscall_exit", MARK_NOARGS, probe_kernel_syscall_exit, NULL );
        stracing = TRUE;
    }
    else 
    {
        /* remove probes for tracing syscalls */
        marker_probe_unregister( "kernel_syscall_exit", probe_kernel_syscall_exit, NULL );
        marker_probe_unregister( "kernel_syscall_entry", probe_kernel_syscall_entry, NULL );
        stracing = FALSE;
    }

    *ppos += count;
    return count;
}

static void probe_kernel_irq_entry( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    unsigned irqnr = 0;
    unsigned ku = 0;

    irqnr = va_arg( *args, typeof( irqnr ) );
    ku = va_arg( *args, typeof( ku ) );

    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( KERNEL_EVENT_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), 0x20000000 + irqnr );
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}

static void probe_kernel_irq_exit( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( KERNEL_EVENT_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), 0x30000000 );
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}

static ssize_t trcext_irqtracingread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    char kerbuf[ 20 ];
    int size = 0;
    
    if( irqtracing )
    {
        size = sprintf( kerbuf, "1\n" );
    }
    else
    {
        size = sprintf( kerbuf, "0\n" );
    }
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( count > size - *ppos )
    {
        count = size - *ppos;
    }
    
    if( copy_to_user( buf, kerbuf + *ppos, count ) )
    {
        return -EFAULT;
    }
    
    *ppos += count;
    return count;
}

static ssize_t trcext_irqtracingwrite( struct file * file, const char * buf, size_t count, loff_t *ppos )
{
    size_t smallcount = count;
    char kerbuf[ 2 ];
    int size = 2;
    int new_irqtracing = FALSE;
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( smallcount > size - *ppos )
    {
        smallcount = size - *ppos;
    }
    
    if( copy_from_user( kerbuf + *ppos, buf, smallcount ) )
    {
        return -EFAULT;
    }
    kerbuf[ size-1 ] = '\0';

    if( !strncmp( kerbuf, "0", 1 ) )
    {
        new_irqtracing = FALSE;
    }
    else if( !strncmp( kerbuf, "1", 1 ) )
    {
        new_irqtracing = TRUE;
    }
    else
    {
        return -EINVAL;
    }
    
    if( !irqtracing && new_irqtracing )
    {
        /* install probes for tracing irqs */
        marker_probe_register( "kernel_irq_entry", "%u %u", probe_kernel_irq_entry, NULL );
        marker_probe_register( "kernel_irq_exit", MARK_NOARGS, probe_kernel_irq_exit, NULL );
        irqtracing = TRUE;
    }
    else
    {
        /* remove probes for tracing irqs */
        marker_probe_unregister( "kernel_irq_exit", probe_kernel_irq_exit, NULL );
        marker_probe_unregister( "kernel_irq_entry", probe_kernel_irq_entry, NULL );
        irqtracing = FALSE;
    }
    
    *ppos += count;
    return count;
}

static void probe_kernel_softirq_entry( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    unsigned long irqnr = 0;

    irqnr = va_arg( *args, typeof( irqnr ) );

    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( KERNEL_EVENT_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), 0x40000000 + irqnr );
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}

static void probe_kernel_softirq_exit( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    unsigned long irqnr = 0;

    irqnr = va_arg( *args, typeof( irqnr ) );

    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( KERNEL_EVENT_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), 0x50000000 + irqnr );
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}

static void probe_kernel_tasklet_high_entry( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    void* func = 0;
    unsigned long data = 0;

    func = va_arg( *args, typeof( func ) );
    data = va_arg( *args, typeof( data ) );

    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( KERNEL_EVENT_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), 0x60000000  + ( ( unsigned )func & 0x0FFFFFFF ) );
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}

static void probe_kernel_tasklet_high_exit( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    void* func = 0;
    unsigned long data = 0;

    func = va_arg( *args, typeof( func ) );
    data = va_arg( *args, typeof( data ) );

    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( KERNEL_EVENT_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), 0x70000000  + ( ( unsigned )func & 0x0FFFFFFF ) );
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}

static void probe_kernel_tasklet_low_entry( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    void* func = 0;
    unsigned long data = 0;

    func = va_arg( *args, typeof( func ) );
    data = va_arg( *args, typeof( data ) );

    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( KERNEL_EVENT_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), 0x80000000  + ( ( unsigned )func & 0x0FFFFFFF ) );
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}

static void probe_kernel_tasklet_low_exit( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    void* func = 0;
    unsigned long data = 0;

    func = va_arg( *args, typeof( func ) );
    data = va_arg( *args, typeof( data ) );

    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( KERNEL_EVENT_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), 0x90000000  + ( ( unsigned )func & 0x0FFFFFFF ) );
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}

static ssize_t trcext_softirqtracingread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    char kerbuf[ 20 ];
    int size = 0;
    
    if( softirqtracing )
    {
        size = sprintf( kerbuf, "1\n" );
    }
    else
    {
        size = sprintf( kerbuf, "0\n" );
    }
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( count > size - *ppos )
    {
        count = size - *ppos;
    }
    
    if( copy_to_user( buf, kerbuf + *ppos, count ) )
    {
        return -EFAULT;
    }
    
    *ppos += count;
    return count;
}

static ssize_t trcext_softirqtracingwrite( struct file * file, const char * buf, size_t count, loff_t *ppos )
{
    size_t smallcount = count;
    char kerbuf[ 2 ];
    int size = 2;
    int new_softirqtracing = FALSE;
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( smallcount > size - *ppos )
    {
        smallcount = size - *ppos;
    }
    
    if( copy_from_user( kerbuf + *ppos, buf, smallcount ) )
    {
        return -EFAULT;
    }
    kerbuf[ size-1 ] = '\0';

    if( !strncmp( kerbuf, "0", 1 ) )
    {
        new_softirqtracing = FALSE;
    }
    else if( !strncmp( kerbuf, "1", 1 ) )
    {
        new_softirqtracing = TRUE;
    }
    else
    {
        return -EINVAL;
    }
    
    if( !softirqtracing && new_softirqtracing )
    {
        /* install probes for tracing softirqs and tasklets */
        marker_probe_register( "kernel_softirq_entry", "%lu", probe_kernel_softirq_entry, NULL );
        marker_probe_register( "kernel_softirq_exit", "%lu", probe_kernel_softirq_exit, NULL );
        marker_probe_register( "kernel_tasklet_high_entry", "%p %lu", probe_kernel_tasklet_high_entry, NULL );
        marker_probe_register( "kernel_tasklet_high_exit", "%p %lu", probe_kernel_tasklet_high_exit, NULL );
        marker_probe_register( "kernel_tasklet_low_entry", "%p %lu", probe_kernel_tasklet_low_entry, NULL );
        marker_probe_register( "kernel_tasklet_low_exit", "%p %lu", probe_kernel_tasklet_low_exit, NULL );
        softirqtracing = TRUE;
    }
    else
    {
        /* remove probes for tracing softirqs and tasklets */
        marker_probe_unregister( "kernel_tasklet_low_exit", probe_kernel_tasklet_low_exit, NULL );
        marker_probe_unregister( "kernel_tasklet_low_entry", probe_kernel_tasklet_low_entry, NULL );
        marker_probe_unregister( "kernel_tasklet_high_exit", probe_kernel_tasklet_high_exit, NULL );
        marker_probe_unregister( "kernel_tasklet_high_entry", probe_kernel_tasklet_high_entry, NULL );
        marker_probe_unregister( "kernel_softirq_exit", probe_kernel_softirq_exit, NULL );
        marker_probe_unregister( "kernel_softirq_entry", probe_kernel_softirq_entry, NULL );
        softirqtracing = FALSE;
    }
    
    *ppos += count;
    return count;
}

static void probe_kernel_signal( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    struct task_struct* task = NULL;
    int sig = 0;

    task = va_arg( *args, typeof( task ) );
    sig = va_arg( *args, typeof( sig ) );

    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( KERNEL_EVENT_SAMPLE | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), 0xA0000000 | ( sig & 0x000000FFUL ) << 16 | task->pid );
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}

static ssize_t trcext_signaltracingread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    char kerbuf[ 20 ];
    int size = 0;
    
    if( signaltracing )
    {
        size = sprintf( kerbuf, "1\n" );
    }
    else
    {
        size = sprintf( kerbuf, "0\n" );
    }
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( count > size - *ppos )
    {
        count = size - *ppos;
    }
    
    if( copy_to_user( buf, kerbuf + *ppos, count ) )
    {
        return -EFAULT;
    }
    
    *ppos += count;
    return count;
}

static ssize_t trcext_signaltracingwrite( struct file * file, const char * buf, size_t count, loff_t *ppos )
{
    size_t smallcount = count;
    char kerbuf[ 2 ];
    int size = 2;
    int new_signaltracing = FALSE;
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( smallcount > size - *ppos )
    {
        smallcount = size - *ppos;
    }
    
    if( copy_from_user( kerbuf + *ppos, buf, smallcount ) )
    {
        return -EFAULT;
    }
    kerbuf[ size-1 ] = '\0';

    if( !strncmp( kerbuf, "0", 1 ) )
    {
        new_signaltracing = FALSE;
    }
    else if( !strncmp( kerbuf, "1", 1 ) )
    {
        new_signaltracing = TRUE;
    }
    else
    {
        return -EINVAL;
    }
    
    if( !signaltracing && new_signaltracing )
    {
        /* install probes for tracing signals */
        marker_probe_register( "kernel_signal", "%p %d", probe_kernel_signal, NULL );
        signaltracing = TRUE;
    }
    else
    {
        /* remove probes for tracing signals */
        marker_probe_unregister( "kernel_signal", probe_kernel_signal, NULL );
        signaltracing = FALSE;
    }
    
    *ppos += count;
    return count;
}


#define TASK_SWITCH_SAMPLE ( 12 )

static void probe_kernel_sched_schedule( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    unsigned long irqflags = 0;
    struct task_struct* prev = NULL;
    struct task_struct* cur = NULL;
    TaskEntry* t = NULL;

    prev = va_arg( *args, typeof( prev ) );
    cur = va_arg( *args, typeof( cur ) );

    spin_lock( &taskdata_lock );
    t = FindOrCreateTask( cur->pid );
        /* we need the OrCreate variant since trcext is loaded dynamically and may have missed the fork */
    if( t )
    {
        t->sampleref = samplecounter++;
    }
    spin_unlock( &taskdata_lock );

    spin_lock_irqsave( &trcbuf_lock, irqflags );
    stamp( TASK_SWITCH_SAMPLE | task_thread_info(cur)->cpu << 3 | ( ( *( ( unsigned long* )( TIMESTAMP ) ) << 1 ) & 0xFFFFFFF0UL ), ( Nat32 )cur->pid );
    trace_mark(trcext_sched_schedule, "%d %d %ld", prev->pid , cur->pid, cur->state);
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );
}

#if 0
static void probe_kernel_sched_wakeup_new_task( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    struct task_struct* task = NULL;
    long state = 0;

    task = va_arg( *args, typeof( task ) );
    state = va_arg( *args, typeof( state ) );
    
    printk( "kernel_sched_wakeup_new_task(%d, %ld, %s)\n", task->pid, state, task->comm );
}
#endif

static void probe_kernel_thread_create( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    long pid = 0;
    void* fn = NULL;
    TaskEntry* t = NULL;

    pid = va_arg( *args, typeof( pid ) );
    fn = va_arg( *args, typeof( fn ) );

    spin_lock( &taskdata_lock );
    t = FindOrCreateTask( ( int )pid );
    if( !t )
    {
        /* error, no task created (no memory?) */
    }
    spin_unlock( &taskdata_lock );

#if 0        
    printk( "kernel_thread_create(%ld, 0x%08x)\n", pid, ( unsigned )fn );
#endif
}

static void probe_kernel_process_fork( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    struct task_struct* orig = NULL;
    struct task_struct* copy = NULL;
    TaskEntry* t = NULL;

    orig = va_arg( *args, typeof( orig ) );
    copy = va_arg( *args, typeof( copy ) );

    spin_lock( &taskdata_lock );
    t = FindOrCreateTask( ( int )copy->pid );
    if( !t )
    {
        /* error, no task created (no memory?) */
    }
    spin_unlock( &taskdata_lock );

#if 0        
    printk( "kernel_process_fork(%d, %d, %d, %s->%s)\n", orig->pid, copy->pid, copy->tgid, orig->comm, copy->comm );
#endif
}

static void probe_kernel_process_exit( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    TaskEntry* t = NULL;
    struct task_struct* task = NULL;

    task = va_arg( *args, typeof( task ) );

    spin_lock( &taskdata_lock );
    t = FindTask( task->pid );
    if( t )
    {
        if( !t->customname )
        {
            /* last chance to copy the latest name of the task */
            strncpy( t->name, task->comm, TASKNAMELIM );
            t->name[ TASKNAMELIM-1 ] = '\0';
        }
    }
    spin_unlock( &taskdata_lock );
    
#if 0
    printk( "kernel_process_exit(%d)\n", pid);
#endif
}

static void probe_kernel_process_free( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    struct task_struct* task = NULL;

    task = va_arg( *args, typeof( task ) );

    spin_lock( &taskdata_lock );
    /* at this point linux already removed the task_struct */
    TryDeleteTask( task->pid );
    spin_unlock( &taskdata_lock );

#if 0    
    printk( "kernel_process_free(%d)\n", task->pid );
#endif
}

/*
 * Stack dump
 */

//#define PRINTDEBUG0( fmt, args...) do { printk( "TRC " fmt, ## args ); } while( 0 )
//#define PRINTDEBUG1( fmt, args...) do { printk( "TRC " fmt, ## args ); } while( 0 )
//#define PRINTDEBUG2( fmt, args...) do { printk( "TRC " fmt, ## args ); } while( 0 )

#ifndef PRINTDEBUG0
# define PRINTDEBUG0( fmt, args...) do {  } while( 0 )
#endif

#ifndef PRINTDEBUG1
# define PRINTDEBUG1( fmt, args...) do {  } while( 0 )
#endif

#ifndef PRINTDEBUG2
# define PRINTDEBUG2( fmt, args...) do {  } while( 0 )
#endif

#define MAX_SCAN_LEN 5000
#define MAX_BRANCH_LEN 100
#define MAX_LEVEL 50

#define SCAN_END_ERET     55
#define SCAN_END_SIGFRAME 36
#define SCAN_END_MEM      79
#define SCAN_END_LEVEL0   80

#define MISSING_RA        -1

#define GENERAL_OPCODE_MASK 0xFC000000
#define SPECIAL_OPCODE_MASK 0x0000003F
#define RT_REG_POS          16
#define RT_REG_MASK         ( 0x1F << RT_REG_POS )
#define RS_REG_POS          21
#define RS_REG_MASK         ( 0x1F << RS_REG_POS )
#define RD_REG_POS          11
#define RD_REG_MASK         ( 0x1F << RD_REG_POS )
#define BASE_REG_POS        21
#define BASE_REG_MASK       0x03E00000

// not all opcodes map the registers to the same location !
// RS2/RD2 is for ORI
#define RS2_REG_POS          21
#define RS2_REG_MASK       ( 0x1f << RS2_REG_POS )
#define RD2_REG_POS          16
#define RD2_REG_MASK       ( 0x1f << RD2_REG_POS )


#define BEQ_INSTR   0x10000000
#define	J_INSTR     0x08000000
#define	JR_INSTR    0x00000008
#define ADDIU_INSTR 0x24000000
#define ORI_INSTR   0x34000000
#define LW_INSTR    0x8C000000
#define SW_INSTR    0xAC000000
#define ERET_INSTR  0x42000018          
#define ADDU_INSTR  0x00000021
#define OR_INSTR    0x00000025
/* move d,s => addu d,s,$zero */
#define SYSCALL_INSTR 0x0000000c

#define RA_REG  31
#define SP_REG  29
#define S8_REG  30
#define T0_REG   8
#define T1_REG   9
#define ZERO_REG 0

#define INSTRMATCH( instr, pattern, mask) ( ( instr & mask ) == pattern )

#define STACKBUF ( kernel_crash_dump.stackbuf )
#define STACKCUR ( kernel_crash_dump.stackcur )
#define STACKBUFLEFT ( STACKBUF + STACKBUFSIZE - STACKCUR )

#define PIDSELECTALL 99
#define PIDSELECTLIM 10
static int pidselectlist[ PIDSELECTLIM ];
static int pidselectcount = 0;
static int freezetraceonstackread = FALSE;



/* begin copy-paste reuse from ptrace, removed flush_icache_page (which isn't nice, we should really compile this module in the kernel) */

static int local_access_process_vm(struct task_struct *tsk, unsigned long addr, void *buf, int len, int write)
{
    struct mm_struct *mm;
    struct vm_area_struct *vma;
    void *old_buf = buf;

    mm = get_task_mm(tsk);
    if (!mm)
        return 0;

    down_read(&mm->mmap_sem);
    /* ignore errors, just check how much was sucessfully transfered */
    while (len) {
        int bytes, ret, offset;
        void *maddr;
    	struct page *page = NULL;

        ret = get_user_pages(tsk, mm, addr, 1,
                write, 1, &page, &vma);
        if (ret <= 0)
            break;

        bytes = len;
        offset = addr & (PAGE_SIZE-1);
        if (bytes > PAGE_SIZE-offset)
            bytes = PAGE_SIZE-offset;

        maddr = kmap(page);
        if (write) {
#if 0  /* doesn't compile, not needed anyway */
            copy_to_user_page(vma, page, addr,
                      maddr + offset, buf, bytes);
            set_page_dirty_lock(page);
#endif
        } else {
            copy_from_user_page(vma, page, addr,
                        buf, maddr + offset, bytes);
        }
        kunmap(page);
        page_cache_release(page);
        len -= bytes;
        buf += bytes;
        addr += bytes;
    }
    up_read(&mm->mmap_sem);
    mmput(mm);
    
    return buf - old_buf;
}

/* end copy-paste reuse from ptrace */

static struct task_struct* UserProcess = NULL;

static inline int ReadMem( unsigned address, int stack, unsigned* data )
{
    int success = 0;
    unsigned long requestpage = ( unsigned )address & PAGE_MASK;
    static struct task_struct* lastprocess = NULL;
    static char* stackpagedata[ PAGE_SIZE ] __attribute__ ((aligned (PAGE_SIZE)));
    static char* codepagedata[ PAGE_SIZE ] __attribute__ ((aligned (PAGE_SIZE)));
    static unsigned long loadedstackpage = 0;
    static unsigned long loadedcodepage = 0;
    
    if( UserProcess == NULL )
    {
        if( 0x80000000 < address && address < 0xc1000000 && address % 4 == 0 )
        {
            *data = *( ( unsigned* )address );
            success = 1;
        }
        else
        {
            success = 0;
        }
    }
    else
    {
        if( stack )
        {
            if( requestpage != loadedstackpage || UserProcess != lastprocess )
            {
                int copied = 0;
                copied = local_access_process_vm( UserProcess, requestpage, stackpagedata, PAGE_SIZE, 0 );
                
                if( copied )
                {
                    loadedstackpage = requestpage;
                    lastprocess = UserProcess;
                    success = 1;
                }
                else
                {
                    loadedstackpage = 0;
                    lastprocess = NULL;
                    success = 0;
                }
            }
            else
            {
                success = 1;
            }
            
            if( success )
            {
                *data = *( ( unsigned* )( ( ( char* )stackpagedata ) + ( address & ~PAGE_MASK ) ) );
            }
        }
        else
        {
            if( requestpage != loadedcodepage || UserProcess != lastprocess )
            {
                int copied = local_access_process_vm( UserProcess, requestpage, codepagedata, PAGE_SIZE, 0 );
                
                if( copied )
                {
                    loadedcodepage = requestpage;
                    lastprocess = UserProcess;
                    success = 1;
                }
                else
                {
                    loadedcodepage = 0;
                    lastprocess = NULL;
                    success = 0;
                }
            }
            else
            {
                success = 1;
            }
            
            if( success )
            {
                *data = *( ( unsigned* )( ( ( char* )codepagedata ) + ( address & ~PAGE_MASK ) ) );
            }
        }
    }
    
    return success;
}

static void GetExceptionFrame( struct task_struct* task, unsigned* programcounter, unsigned* stackpointer, unsigned* returnaddress, unsigned* framepointer, unsigned* cause, unsigned* badvaddr )
{
    struct pt_regs* exceptframe  = task_pt_regs(task);
    
    *programcounter = ( unsigned )exceptframe->cp0_epc;
    *stackpointer   = ( unsigned )exceptframe->regs[ 29 ];
    *returnaddress  = ( unsigned )exceptframe->regs[ 31 ];
    *cause          = ( unsigned )exceptframe->cp0_cause;
    *badvaddr       = ( unsigned )exceptframe->cp0_badvaddr;

    // *framepointer   = exceptframe->regs[ 30 ]; // NDC: the framepointer is not stored in the exceptframe (unless PTRACE is active !)
    if( *framepointer != exceptframe->regs[30] )
    {
        PRINTDEBUG2("exceptionframe RA=0x%08lx, active RA=0x%08x\n", exceptframe->regs[30], *framepointer );
    }
}

typedef struct
{
    u32 arg[ 4 ];
    u32 retval[ 2 ];
    struct siginfo info;
    struct ucontext context;
} SignalFrame;

static void GetSignalFrame( unsigned framebase, unsigned* programcounter, unsigned* stackpointer, unsigned* returnaddress, unsigned* framepointer, int* signo, int* sigcode, unsigned (* regs )[ 32 ] )
{
    SignalFrame* sigframe = ( SignalFrame* )framebase;
    int i = 0;
    
    ReadMem( ( unsigned )&( sigframe->context.uc_mcontext.sc_pc ),         1, programcounter );
    ReadMem( ( unsigned )&( sigframe->context.uc_mcontext.sc_regs[ 29 ] ), 1, stackpointer   );
    ReadMem( ( unsigned )&( sigframe->context.uc_mcontext.sc_regs[ 31 ] ), 1, returnaddress  );
    ReadMem( ( unsigned )&( sigframe->info.si_signo ),                     1, signo          );
    ReadMem( ( unsigned )&( sigframe->info.si_code ),                      1, sigcode        );
    for( i = 0; i < 32; ++i )
    {
        ReadMem( ( unsigned )&( sigframe->context.uc_mcontext.sc_regs[ i ] ), 1, &( ( *regs )[ i ] ) ); 
    }
    *framepointer = ( *regs )[30];
}

/* in the case of optimization __noreturn__ we will hit a case
 * where JR RA is reached without a corresponding  lw ra, XXX(sp) 
 *
 * this is because we are in a path where the compiler has optimized this read out.
 * the solution is to look up to the corresponding sw ra, XXX(sp) instruction
 * having found this, we find XXX and read RA from the stack
 *
 * input:  
 *    level            : stack level (used only for debug prints)
 *    programcount     : PC to start scanning from
 *    stackpointer     : read RA relative to SP
 *
 * output:
 *    *pCaller         : returned RA value
 * 
 * returns:            
 *    1 in case we got a value for RA
 *    0 if no RA was found
 */
static int FixupNoreturn( int level
                        , unsigned  programcounter
                        , unsigned  stackpointer
                        , unsigned* pCaller
                        )
{
    unsigned instr        = 0;
    unsigned scanpc       = programcounter;
    int      scanlen      = 0;


    for( scanlen = 0; scanlen < MAX_SCAN_LEN; scanlen++, scanpc -= 4 )
    {
        if( !ReadMem( scanpc, 0, &instr ) )
        {
            return 0;
        }

        if( INSTRMATCH( instr, ( JR_INSTR | RA_REG << RS_REG_POS ), ( GENERAL_OPCODE_MASK | SPECIAL_OPCODE_MASK | RS_REG_MASK ) ) )
        {
            PRINTDEBUG2( "fix %d: jr ra found (previous function ?) => EXIT\n", scanlen );
            return 0;
        }
        
        if( INSTRMATCH( instr, ( SW_INSTR | RA_REG << RT_REG_POS | SP_REG << BASE_REG_POS ), ( GENERAL_OPCODE_MASK | RT_REG_MASK | BASE_REG_MASK ) ) )
        {
            int offset = ( short )( instr & 0xFFFF );
            PRINTDEBUG2( "fix %d: sw ra,%d(sp) found\n", scanlen, offset );
            if( !ReadMem( stackpointer + offset, 1, pCaller ) )
            {
                *pCaller = 0;
                return 0;
            }
            PRINTDEBUG2( "        read stack -> caller = 0x%08x\n", *pCaller );
            return 1;
        }
    }

    PRINTDEBUG2( "fix %d: nothing found => EXIT\n", scanlen );
    return 0;
}

struct RegionLimit {
  unsigned first;
  unsigned last;
};

// pLimit holds the region we scanned so far
// pc is checked if it increases this region on either side
static void UpdateRegion( unsigned addr, struct RegionLimit * pLimit )
{
  if( addr < pLimit->first)
  {
    pLimit->first = addr;
  }

  if( addr > pLimit->last )
  {
    pLimit->last = addr;
  }
}

// return 1 if addr is inside pLimit
static int  IsInsideRegion( unsigned addr, struct RegionLimit* pLimit )
{
  if( addr < pLimit->first)
  {
    return 0;
  }
  if( addr > pLimit->last )
  {
    return 0;
  }
  return 1;
}

static void InitRegion( unsigned addr, struct RegionLimit* pLimit )
{
  pLimit->first = pLimit->last = addr;
}

 
/* search forward from PC to end of function */
static void StackSearchForward( int          level
                              , unsigned   programcounter
                              , unsigned   stackpointer
                              , unsigned   returnaddress
                              , int        followBranch 
                              , unsigned*  pCaller
                              , int*       pFramesize
                              , int*       pSpecial
                              , int*       pFuncEnd
                              , unsigned*  pFramepointer
                              )
{
    unsigned instr        = 0;
    unsigned scanpc       = programcounter;
    int      scanlen      = 0;
    unsigned branchpc     = 0;
    int      branchlen    = MAX_BRANCH_LEN;
    unsigned reg_t0       = 0;
    unsigned reg_t1       = 0;
    struct RegionLimit region;

    InitRegion( programcounter, &region );
        
    for( scanlen = 0; scanlen < MAX_SCAN_LEN; scanlen++, scanpc += 4 )
    {
        if( !ReadMem( scanpc, 0, &instr ) )
        {
            *pCaller    = 0;
            *pFramesize = 0;
            *pSpecial   = SCAN_END_MEM;
            break;
        }
        PRINTDEBUG2( "  scan fwd  [%03d] at 0x%08x instr=0x%08x\n", scanlen, scanpc, instr );
        
        UpdateRegion( scanpc, &region ); 

        if( branchlen < MAX_BRANCH_LEN )
        {
            ++branchlen;
            if( branchlen == MAX_BRANCH_LEN )
            {
                PRINTDEBUG2( "fwd %d: return from branch\n", scanlen );
                scanpc = branchpc;  /* stop following branch */
            }
        }
        else
        {
            // NDC 14/3/2011 - the conditional branch, we only follow in the followBranch iteration (but take care this line does not trigger on unconditional branches)
            if( !INSTRMATCH( instr, ( BEQ_INSTR | 0 << RS_REG_POS | 0 << RT_REG_POS ), ( GENERAL_OPCODE_MASK | RS_REG_MASK | RT_REG_MASK ) ) &&
                 INSTRMATCH( instr, BEQ_INSTR , GENERAL_OPCODE_MASK ) )
            {
                if( followBranch )
                {
                    int offset = ( short )( instr & 0xFFFF );
                    unsigned new_pc = scanpc + 4 * (offset -1 );

                    if( IsInsideRegion( new_pc, &region ))
                    {
                        PRINTDEBUG2( "fwd %d: skip already visited conditional branch with offset %d to 0x%08x\n", scanlen, offset, new_pc );
                    }
                    else
                    {
                        branchlen  = 0;
                        branchpc   = scanpc;
                        scanpc     = new_pc;
                        UpdateRegion( scanpc, &region ); 
                        PRINTDEBUG2( "fwd %d: follow conditional branch with offset %d to 0x%08x\n", scanlen, offset, scanpc );
                    } 
                    continue;
                }
            }
        }

        // NDC 22/3/11 - always follow non-conditional jump and unconditional branch
        if(  INSTRMATCH( instr, ( BEQ_INSTR | 0 << RS_REG_POS | 0 << RT_REG_POS ), ( GENERAL_OPCODE_MASK | RS_REG_MASK | RT_REG_MASK ) ) ) 
        {
            int      offset = ( short )( instr & 0xFFFF );
            unsigned new_pc = scanpc + 4 * (offset -1 );

            if( IsInsideRegion( new_pc, &region ))
            {
                PRINTDEBUG2( "fwd %d: skip already visited unconditional branch with offset %d to 0x%08x\n", scanlen, offset, new_pc );
            }
            else
            {
                scanpc = new_pc;
                UpdateRegion( scanpc, &region ); 
                PRINTDEBUG2( "fwd %d: follow unconditional branch with offset %d to 0x%08x\n", scanlen, offset, scanpc );
            }
        }
        else if( INSTRMATCH( instr, J_INSTR, GENERAL_OPCODE_MASK ) )
        {
            unsigned new_pc = ( ( scanpc + 4 ) & 0xFC000000 )
                              | ( ( instr & 0x03FFFFFF ) << 2 );

            if( IsInsideRegion( new_pc, &region ))
            {
                PRINTDEBUG2( "fwd %d: skip already visited jump to 0x%08x\n", scanlen, new_pc );
            }
            else
            {
                PRINTDEBUG2( "fwd %d: follow jump to 0x%08x\n", scanlen, new_pc );;
                UpdateRegion( scanpc, &region ); 
                scanpc    = new_pc;
            }
        }
        else if( INSTRMATCH( instr, ( JR_INSTR | RA_REG << RS_REG_POS ), ( GENERAL_OPCODE_MASK | SPECIAL_OPCODE_MASK | RS_REG_MASK ) ) )
        {
            PRINTDEBUG2( "fwd %d: jr ra found (wait for possible sp change in delay slot)\n", scanlen );
            *pFuncEnd = scanpc + 4; // code ends one line further
            if( ( level == 0 ) && ( *pCaller == 0 ) )
            {
                *pSpecial = SCAN_END_LEVEL0;  // NDC I hope this solves the ioctl issue
                *pCaller = returnaddress;
            }
            else if( *pCaller == 0 )
            {
                PRINTDEBUG2("        JR RA missing RA data\n");
                
                if( FixupNoreturn( level, programcounter, stackpointer, pCaller ) )
                {
                    PRINTDEBUG2("        fixup RA: 0x%08x\n", *pCaller);
                }
            }
            scanlen = MAX_SCAN_LEN - 2;  /* allow reading delay slot */
        }
        else if( *pFramepointer
               &&( ( INSTRMATCH( instr, ( ADDU_INSTR | SP_REG << RD_REG_POS | S8_REG << RS_REG_POS ), ( GENERAL_OPCODE_MASK | SPECIAL_OPCODE_MASK | RD_REG_MASK | RS_REG_MASK ) ) )
                 ||( INSTRMATCH( instr, (   OR_INSTR | SP_REG << RD_REG_POS | S8_REG << RS_REG_POS ), ( GENERAL_OPCODE_MASK | SPECIAL_OPCODE_MASK | RD_REG_MASK | RS_REG_MASK ) ) ) ) )
        {
            int      offset;
            unsigned newStackpointer;

	    if( INSTRMATCH( instr, ZERO_REG << RT_REG_POS, RT_REG_MASK ))
	    {
		offset = 0;
	    }
	    else if( INSTRMATCH( instr, T0_REG << RT_REG_POS, RT_REG_MASK ))
            {
		offset = reg_t0;
            }
	    else if( INSTRMATCH( instr, T1_REG << RT_REG_POS, RT_REG_MASK ))
            {
		offset = reg_t1;
            }
            else
            {
	        PRINTDEBUG2("fwd %d: move sp, s8, XX found with unrecognised RT 0x%08x\n", scanlen, instr );
		offset = 0;  // we have to do something
            }
            newStackpointer = *pFramepointer + offset;
            PRINTDEBUG2( "fwd %d: add sp,s8,rt found. sp: 0x%08x --> 0x%08x\n", scanlen, stackpointer, newStackpointer );
            *pFramesize  = newStackpointer - stackpointer;
            stackpointer = newStackpointer;
        }
        else if( INSTRMATCH( instr, ( ORI_INSTR | ZERO_REG << RS2_REG_POS | T0_REG << RD2_REG_POS ), ( GENERAL_OPCODE_MASK | RS2_REG_MASK | RD2_REG_MASK ) ) )
        {
            reg_t0 =  instr & 0xFFFF;
            PRINTDEBUG2( "fwd %d: li t0 %d found\n", scanlen, reg_t0 );
        }
        else if( INSTRMATCH( instr, ( ORI_INSTR | ZERO_REG << RS2_REG_POS | T1_REG << RD2_REG_POS ), ( GENERAL_OPCODE_MASK | RS2_REG_MASK | RD2_REG_MASK ) ) )
        {
            reg_t1 =  instr & 0xFFFF;
            PRINTDEBUG2( "fwd %d: li t1 %d found\n", scanlen, reg_t1 );
        }
        else if( INSTRMATCH( instr, ( ADDIU_INSTR | SP_REG << RT_REG_POS ), ( GENERAL_OPCODE_MASK | RT_REG_MASK ) ) )
        {
            int offset =  ( short )( instr & 0xFFFF );
            
            PRINTDEBUG2( "fwd %d: addiu sp %d found\n", scanlen, offset );
                                
            if ( offset < 0 )
            {
                PRINTDEBUG2( "        offset < 0 in SP mode -> begin of next function ?\n" );
                break;
            }
            else if( scanlen > 0 )
            {
                PRINTDEBUG2( "        framesize change: %d --> %d\n", *pFramesize, *pFramesize + offset );
                *pFramesize += offset;
                if( ( ( *pCaller == 0 ) && ( level == 0 ) )
                   || ( *pCaller != 0 ) )
                {
                    PRINTDEBUG2( "        offset > 0 found -> exit\n" );
                    break;
                }
                else
                {
                    PRINTDEBUG2( "        offset > 0 -> continue\n" );
                }
            }
        }
        else if( INSTRMATCH( instr, ( LW_INSTR | RA_REG << RT_REG_POS | SP_REG << BASE_REG_POS ), ( GENERAL_OPCODE_MASK | RT_REG_MASK | BASE_REG_MASK ) ) )
        {
            int offset =  ( short )( instr & 0xFFFF );
            
            PRINTDEBUG2( "fwd %d: lw ra,%d(sp) found\n", scanlen, offset );
            if( !ReadMem( stackpointer + offset, 1, pCaller ) )
            {
                *pCaller    = 0;
                *pFramesize = 0;
                *pSpecial   = SCAN_END_MEM;
                break;
            }
            PRINTDEBUG2( "        read stack -> Call: 0x%08x\n", *pCaller );
        }
        else if( INSTRMATCH( instr, ( LW_INSTR | S8_REG << RT_REG_POS | SP_REG << BASE_REG_POS ), ( GENERAL_OPCODE_MASK | RT_REG_MASK | BASE_REG_MASK ) ) )
        {
            int offset =  ( short )( instr & 0xFFFF );
            
            PRINTDEBUG2( "fwd %d: lw s8,%d(sp) found\n", scanlen, offset );
            if( !ReadMem( stackpointer + offset, 1, pFramepointer ) )
            {
                *pCaller    = 0;
                *pFramesize = 0;
                *pSpecial   = SCAN_END_MEM;
                break;
            }
            PRINTDEBUG2( "        read stack -> Framepointer: 0x%08x\n", *pFramepointer );
        }
        else if( INSTRMATCH( instr, ERET_INSTR, 0xFFFFFFFF ) )
        {
            PRINTDEBUG2( "fwd %d: eret found (exit)\n", scanlen );
            *pCaller    = 0;
            *pFramesize = 0;
            *pSpecial   = SCAN_END_ERET;
            break;
        }
        else if( INSTRMATCH( instr, SYSCALL_INSTR, 0xffffffff ) )
        {
            PRINTDEBUG2( "fwd %d: syscall found (bail out)\n", scanlen );
            *pCaller    = 0;
            *pFramesize = 0;
            *pSpecial   = 0;
            break;
        }
    }
    PRINTDEBUG2("FWD %02d => Call: 0x%08x, frame: %d, special: %d\n"
                , level
                , *pCaller
                , *pFramesize
                , *pSpecial );
}


/* search backward from PC to start of function */
static void StackSearchBackward( int       level
                                , unsigned  programcounter
                                , unsigned  stackpointer
                                , unsigned  returnaddress
                                , unsigned* pCaller
                                , int*      pFramesize
                                , int*      pSpecial
                                , int*      pFuncStart
                                , unsigned* pFramepointer
                                )
{
    unsigned instr        = 0;
    unsigned scanpc       = programcounter;
    int      scanlen      = 0;

    
    for( scanlen = 0; scanlen < MAX_SCAN_LEN; scanlen++, scanpc -= 4 )
    {
        if( !ReadMem( scanpc, 0, &instr ) )
        {
            *pCaller    = 0;
            *pFramesize = 0;
            *pSpecial   = SCAN_END_MEM;
            break;
        }
        PRINTDEBUG2( "  scan back [%03d] at 0x%08x instr=0x%08x\n", scanlen, scanpc, instr );

        if( INSTRMATCH( instr, ( JR_INSTR | RA_REG << RS_REG_POS ), ( GENERAL_OPCODE_MASK | SPECIAL_OPCODE_MASK | RS_REG_MASK ) ) )
        {
            PRINTDEBUG2( "bck %d: jr ra found (previous function ?) => EXIT\n", scanlen );
            *pFuncStart = scanpc + 8; // code ends one line further

            if( level == 0 )
            {
                *pCaller = returnaddress;
            }
            break;
        }
        else
        {
            // we only want to look at the stack stuff if we can trust it
            // if we know the stack is S8 based, don't bother trying to interpret the SP relative stuff (yet)
            if( INSTRMATCH( instr, ( ADDIU_INSTR | SP_REG << RT_REG_POS ), ( GENERAL_OPCODE_MASK | RT_REG_MASK ) ) )
            {
                int offset =  ( short )( instr & 0xFFFF );
                
                PRINTDEBUG2( "bck %d: addiu sp %d found\n", scanlen, offset );

                if( offset > 0 )
                {
                    PRINTDEBUG2( "        offset > 0 ( end of previous function ?) -> exit\n" );
                    if( *pCaller == 0 )
                    {
                        *pCaller = returnaddress;
                    }
                    break;
                }
                else
                {
                    /* "sw ra" should have been seen when reaching here         */
                    /* if not, we assume it was not stored yet thus still valid */
                    PRINTDEBUG2( "        offset < 0 -> adjust framesize\n" );

                    *pFramesize -= offset;
                    if( ( level == 0 ) && ( *pCaller == 0 ) )
                    {
                        *pCaller = returnaddress;
                    }
                    if( *pCaller != 0 )
                    {
                        break;
                    }
                }
            }
            else if( INSTRMATCH( instr, ( SW_INSTR | RA_REG << RT_REG_POS | SP_REG << BASE_REG_POS ), ( GENERAL_OPCODE_MASK | RT_REG_MASK | BASE_REG_MASK ) ) )
            {
                int offset = ( short )( instr & 0xFFFF );
                PRINTDEBUG2( "bck %d: sw ra,%d(sp) found\n", scanlen, offset );

                /* get ra from stack + continue searching for framesize */
                if( !ReadMem( stackpointer + offset, 1, pCaller ) )
                {
                    *pCaller    = 0;
                    *pFramesize = 0;
                    *pSpecial   = SCAN_END_MEM;
                    break;
                }
                PRINTDEBUG2( "        read stack -> caller = 0x%08x\n", *pCaller );
            }
            else if( INSTRMATCH( instr, ( SW_INSTR | S8_REG << RT_REG_POS | SP_REG << BASE_REG_POS ), ( GENERAL_OPCODE_MASK | RT_REG_MASK | BASE_REG_MASK ) ) )
            {
                int offset = ( short )( instr & 0xFFFF );
                PRINTDEBUG2( "bck %d: sw s8,%d(sp) found\n", scanlen, offset );

                /* get ra from stack + continue searching for framesize */
                if( !ReadMem( stackpointer + offset, 1, pFramepointer ) )
                {
                    *pCaller    = 0;
                    *pFramesize = 0;
                    *pSpecial   = SCAN_END_MEM;
                    break;
                }
                PRINTDEBUG2( "        read stack -> FP: 0x%08x\n", *pFramepointer );
            }
        }
    }
    PRINTDEBUG2("BCK %02d => Call:0x%08x, frame: %d, special: %d\n"
                , level
                , *pCaller
                , *pFramesize
                , *pSpecial );
}


/* search forward from start of function untill PC */
static void StackSearchTop2PC( int          level
                                , unsigned   programcounter
                                , unsigned   stackpointer
                                , unsigned   funcStart
                                , unsigned   funcEnd
                                , unsigned*  pCaller
                                , int*       pFramesize
                                , int*       pSpecial
                                )
{
    unsigned instr        = 0;
    unsigned scanpc       = funcStart;
    int      scanlen      = 0;
    int      scanoffset   = 0;                        // offset on SP untill scan point
    int      ra_offset    = 0;                        // offset to find RA stored on stack (relative to SP at start of function)
#define BRANCH_WIDTH   10    
    unsigned branchpc[BRANCH_WIDTH];                  // save PC when taking a branch
    unsigned branchoffset[BRANCH_WIDTH];              // save offset when taking branch
    int      branchlen[BRANCH_WIDTH];
    int      branch = -1;  // indicates there is no active branch
    
    if( !(   ( funcStart      < programcounter ) 
          && ( programcounter < funcEnd        ) ) )
    {
        PRINTDEBUG2( "top impossible input: S:0x%08x, PC:0x%08x, E:0x%08x\n", funcStart, programcounter, funcEnd );
        return;
    }

    for( scanlen = 0; scanlen < MAX_SCAN_LEN; scanlen++, scanpc += 4 )
    {
        if( !ReadMem( scanpc, 0, &instr ) )
        {
            *pCaller    = 0;
            *pFramesize = 0;
            *pSpecial   = SCAN_END_MEM;
            break;
        }
        PRINTDEBUG2( "  scan top2pc [%03d] at 0x%08x instr=0x%08x\n", scanlen, scanpc, instr );

        // if we are in a branch, handle the max depth counter
        if( branch >= 0 )
        {
            ++branchlen[branch];
            if( branchlen[branch] >= MAX_BRANCH_LEN )
            {
                PRINTDEBUG2( "top %d: return from branch %d\n", scanlen, branch );
                scanpc     = branchpc[branch];         /* stop following branch */
                scanoffset = branchoffset[branch];
                --branch;
                continue;
            }
        }
        
        if( INSTRMATCH( instr, ( BEQ_INSTR | 0 << RS_REG_POS | 0 << RT_REG_POS ), ( GENERAL_OPCODE_MASK | RS_REG_MASK | RT_REG_MASK ) ) )
        {
            int offset = ( short )( instr & 0xFFFF );
            
            if( offset > 0 ) // only follow forward branches, otherwise we fall into endless iterations
            {
                branch++;
                if( branch >= BRANCH_WIDTH )
                {
                    PRINTDEBUG2( "top %d: too many branches in parallel\n", scanlen );
                    break;
                }                     
                branchlen[branch]    = 0;
                branchpc[branch]     = scanpc;
                branchoffset[branch] = scanoffset;
                scanpc              += 4 * ( offset - 1 );
                PRINTDEBUG2( "top %d: follow unconditional branch %d with offset %d to 0x%08x\n", scanlen, branch, offset, scanpc );
            }
            continue;      // todo: handle branch delay slot
        }
        
        if( INSTRMATCH( instr, J_INSTR, GENERAL_OPCODE_MASK ) )
        {
            unsigned target = ( ( scanpc + 4 ) & 0xFC000000 )
                            | ( ( instr & 0x03FFFFFF ) << 2 );

            if( target > scanpc )  // only follow forward jumps, otherwise we fall into endless iterations
            {
                branch++;
                if( branch >= BRANCH_WIDTH )
                {
                    PRINTDEBUG2( "top %d: too many branches in parallel\n", scanlen );
                    break;
                }                     
                
                PRINTDEBUG2( "top %d: follow jump %d to 0x%08x\n", scanlen, branch, target );;
                branchlen[branch]    = 0;
                branchpc[branch]     = scanpc;
                branchoffset[branch] = scanoffset;
                scanpc               = target;
            }
            continue;       // todo: handle branch delay slot ?
        }

        if( INSTRMATCH( instr, ( ADDIU_INSTR | SP_REG << RT_REG_POS ), ( GENERAL_OPCODE_MASK | RT_REG_MASK ) ) )
        {
            int offset =  ( short )( instr & 0xFFFF );
            
            scanoffset += offset;
            
            PRINTDEBUG2( "top %d: addiu sp %d found -> offset = %d\n", scanlen, offset, scanoffset );
        }
        else if( INSTRMATCH( instr, ( SW_INSTR | RA_REG << RT_REG_POS | SP_REG << BASE_REG_POS ), ( GENERAL_OPCODE_MASK | RT_REG_MASK | BASE_REG_MASK ) ) )
        {
            ra_offset    = ( short )( instr & 0xFFFF );

            PRINTDEBUG2( "bck %d: sw ra found, ra_offset= %d + %d = %d\n", scanlen, ra_offset, scanoffset, ra_offset + scanoffset );
            ra_offset += scanoffset;
        }
        else if( scanpc == programcounter )
        {
            PRINTDEBUG2( "top %d: PC reached\n", scanlen );
            
            *pFramesize = -scanoffset; // offset should be negative at this moment
            
            if( !ReadMem( stackpointer - scanoffset + ra_offset, 1, pCaller ) )
            {
                *pCaller    = 0;
                *pFramesize = 0;
                *pSpecial   = SCAN_END_MEM;
                break;
            }
            PRINTDEBUG2( "        read stack at offset %d (0x%08x) -> caller = 0x%08x\n"
                        , ra_offset - scanoffset
                        , stackpointer + ra_offset - scanoffset
                        , *pCaller );
        }
    }
    PRINTDEBUG2("TOP %02d => Call: 0x%08x, frame: %d, special: %d\n"
                , level
                , *pCaller
                , *pFramesize
                , *pSpecial );
}


static void GetStackFrame( int       level
                         , unsigned  programcounter
                         , unsigned  stackpointer
                         , unsigned  returnaddress
                         , unsigned* pCaller
                         , int*      pFramesize
                         , int*      pSpecial
                         , unsigned* pFramepointer
                         )
{
    unsigned instr           = 0;
    unsigned funcStart       = 0;
    unsigned funcEnd         = 0;
    unsigned oldFramepointer = *pFramepointer;
    
    *pFramesize = 0;
    *pCaller    = 0;
    *pSpecial   = 0;
    
    if( stackpointer + 16 == programcounter + 8 )
    {
        PRINTDEBUG2( "signal frame => EXIT\n" );
        *pCaller    = 0;
        *pFramesize = 0;
        *pSpecial   = SCAN_END_SIGFRAME;
        return;
    }
    
    if( level == 0 )
    {
        ReadMem( programcounter, 0, &instr );
        if( INSTRMATCH( instr, ( ADDIU_INSTR | SP_REG << RT_REG_POS ), ( GENERAL_OPCODE_MASK | RT_REG_MASK ) ) )
        {
            *pCaller = returnaddress;
        }
    }
        
    /* 1 - search forward - ignore branches */
    StackSearchForward( level, programcounter, stackpointer, returnaddress, FALSE, pCaller, pFramesize, pSpecial, &funcEnd, pFramepointer );
    
    /* 2 - search forward - take branches */
    if(  ( *pSpecial == 0                    )
      && ( *pCaller == 0 || *pFramesize == 0 ) )
    {      
        *pFramepointer = oldFramepointer;
        StackSearchForward( level, programcounter, stackpointer, returnaddress, TRUE, pCaller, pFramesize, pSpecial, &funcEnd, pFramepointer );
    }

    /* search backward */
    if(  ( *pSpecial == 0                    ) 
      && ( *pCaller == 0 || *pFramesize == 0 ) )
    {
        *pFramepointer = oldFramepointer;
        StackSearchBackward(level, programcounter, stackpointer, returnaddress, pCaller, pFramesize, pSpecial, &funcStart, pFramepointer );
    }

    /* handle suspicious stuff */
    /* NB: in original code from NDC, this if-condition included an extra IsStackFrameS8 test, which was
       never true so StackSearchTop2PC never executed.  By removing the extra test, the search will now
       execute if the above searches fail. */
    
    if(  ( *pSpecial == 0                    ) 
      && ( *pCaller == 0 || *pFramesize == 0 ) 
      && ( funcStart != 0                    )
      && ( funcEnd   != 0                    ) )
    {
        *pFramepointer = oldFramepointer;
        StackSearchTop2PC(level, programcounter, stackpointer, funcStart, funcEnd, pCaller, pFramesize, pSpecial );
    }

    /* fixup caller */
    if( *pCaller != 0 )
    {
        *pCaller -= 8;
    }

    /* fixup hack to circumvent level0 with framesize 0 */
    if ( *pSpecial == SCAN_END_LEVEL0 )
    {
        *pSpecial = 0;
    }
    
    PRINTDEBUG2("--- %02d => Call: 0x%08x, frame: %d, special: %d\n"
                , level
                , *pCaller
                , *pFramesize
                , *pSpecial );
}


#ifdef LINKED_IN_KERNEL_IMAGE
extern struct seq_operations modules_op;
#endif

static void MakeStackTrace( void )
{
    unsigned long irqflags = 0;
    struct task_struct* process = NULL;
    struct task_struct* task = NULL;
        
    read_lock( &tasklist_lock );

    /* file format version */
    STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "version = 20100715\n\n" );

    /* kernel command line */
#ifdef LINKED_IN_KERNEL_IMAGE
    STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "command line = %s\n\n", saved_command_line );
#else
    if( !in_interrupt() )
    {
        struct file* file = NULL;
        char buffer[ 20 ];
        int len = 0;
        mm_segment_t fs;
        
        fs = get_fs();
        set_fs( KERNEL_DS );

        file = filp_open( "/proc/cmdline", O_RDONLY, 0 );
        if( file >= 0 )
        {
            STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "cmdline = " );
            while( ( len = vfs_read( file, buffer, 19, &file->f_pos ) ) > 0 )
            {
                buffer[ len ] = '\0';
                STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "%s", buffer );
            }
            STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "\n" );
            fput( file );
        }

        set_fs(fs);
    }
    else
    {
        STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "command line = <can't retrieve in interrupt context>\n" );
    }    
#endif
    
    /* ko-mapping */
    {
#ifdef LINKED_IN_KERNEL_IMAGE
        loff_t off = 0;
        void* entry = NULL;

        STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "modules =" );
        for( entry = modules_op.start( NULL, &off ); entry != NULL; modules_op.next( NULL, entry, &off ) )
        {
            struct module* mod = list_entry( entry, struct module, list );
            
            STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, " 0x%08x->%s", ( unsigned )mod->module_core, mod->name );
        }
        modules_op.stop( NULL, entry );
        STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "\n\n" );
#else 
        /* as a ko mod we can't access modules_op so we have do this trick */
        struct module* mod = NULL;
        
        STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "modules = 0x%08x->%s", ( unsigned )THIS_MODULE->module_core, THIS_MODULE->name );

        list_for_each_entry( mod, &THIS_MODULE->list, list )
        {
            if( 0xc0000000 < ( unsigned )mod->module_core && ( unsigned )mod->module_core < 0xe0000000 )
                /* this clumsy test proves mod is not the list_head */
            {
                STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, " 0x%08x->%s", ( unsigned )mod->module_core, mod->name );
            }
        }
        STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "\n\n" );
#endif
    }    

    /* call stacks */
    spin_lock_irqsave( &trcbuf_lock, irqflags );
    if( freezetraceonstackread )
    {
        if( TRACEFROZEN != MAGICTRUE )
        {
            TRACEFROZEN = MAGICTRUE;
        }
        freezetraceonstackread = FALSE;
    }
    spin_unlock_irqrestore( &trcbuf_lock, irqflags ); /* don't spinlock during stacktrace, else scheduler callback may deadlock on other locks */
    do_each_thread( process, task )
    {
        unsigned stackpointer   = ( unsigned )task->thread.reg29;
        unsigned programcounter = ( unsigned )task->thread.reg31;
        unsigned returnaddress  = 0;
        unsigned framepointer   = 0;
        int i = 0;
        
        if( pidselectcount != PIDSELECTALL )
        {
            /* find pid in pidselectlist */
            for( i = 0; i < pidselectcount; ++i )
            {
                if( pidselectlist[ i ] == task->pid || pidselectlist[ i ] == task->tgid ) break;
            }
        }
        
        if( i < pidselectcount ) /* always succeeds if pidselectcount == PIDSELECTALL */
        {
            int kthread = task->mm ? 0 : 1;
            TaskEntry* trcinfo = NULL;
            unsigned caller = 0;
            int framesize = 0;                      
            int special = 0;
            int level = 0;


            /* general process info */
            spin_lock( &taskdata_lock );
            trcinfo = FindTask( task->pid );

            PRINTDEBUG1( "pid: %d, tgid: %d, pc: 0x%08x, sp: 0x%08x\n"
                        , task->pid, task->tgid
                        , programcounter
                        , stackpointer
                        );

            STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "process %d (%s%s%s, %s)\n", task->pid, kthread ? "[" : "", task->comm, kthread ? "]" : "", trcinfo ? trcinfo->name : "<no taskname>" );
            STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "pid = %d, tgid = %d, parent = %d, real parent = %d\n", task->pid, task->tgid, task->parent ? task->parent->pid : 0, task->real_parent ? task->real_parent->pid : 0 );
            STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "pc = 0x%08x, baduaddr = 0x%08x, badvaddr = 0x%08x\n", programcounter, ( unsigned )task->thread.cp0_baduaddr, ( unsigned )task->thread.cp0_badvaddr );
            STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "state = %ld, flags = 0x%08x\n", task->state, ( unsigned )task->flags );
            STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "policy = %u, prio = %d, static prio = %d, normal prio = %d, rt prio = %u\n", task->policy, task->prio, task->static_prio, task->normal_prio, task->rt_priority );
            spin_unlock( &taskdata_lock );

            /* memory context for selected processes */
            if( task->tgid == task->pid && !kthread )
            {
                if( !in_interrupt() )
                {
                    struct mm_struct* mm = get_task_mm( task );
                    if( mm )
                    {
                        struct vm_area_struct* vma = NULL;
                        
                        down_read( &mm->mmap_sem );
                        
                        STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "mmap =" );
                        
                        for( vma = mm->mmap; vma != NULL; vma = vma->vm_next )
                        {
                            if( ( vma->vm_flags & VM_READ )
                             && ( vma->vm_flags & VM_EXEC )
                             && !(vma->vm_flags & VM_WRITE ) )
                            {
                                struct file* f = vma->vm_file;
                                
                                if( f )
                                {
                                    struct dentry* d = f->f_dentry;
                                    
                                    if( d )
                                    {
                                        STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, " 0x%08x->%s", ( unsigned )vma->vm_start, d->d_name.name );
                                    }
                                }
                            }
                        }
                        
                        STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "\n" );
                        
                        up_read( &mm->mmap_sem );
                        mmput( mm );
                    }
                    else
                    {
                        /* no mm, this isn't possible since we've used the mm-test to discriminate a kernel thread */
                    }
                }
                else
                {
                    STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "mmap = <can't retrieve in interrupt context>\n" );
                }
            }
            
            /* unwind kernel stack */
            STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "kernel stack =" );
            
            do
            {
                GetStackFrame( level, programcounter, stackpointer, 0, &caller, &framesize, &special, &framepointer );
                PRINTDEBUG1( "== [%03d] %02d === cal: 0x%08x, frame: %d, special: %d\n", task->pid, level, caller, framesize, special );
                PRINTDEBUG2( "=======================================================\n"); 
                
                STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, " 0x%08x", caller );
                programcounter = caller;
                stackpointer  += framesize;
                level         += 1;
            } while( caller != 0 && special == 0 && level < MAX_LEVEL );

            STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "\n" );
            
            if( !kthread )
            {
                /* decode exception frame */
                if( 1 /* special == SCAN_END_ERET || special == SCAN_END_MEM */ )
                {
                    unsigned cause    = 0;
                    unsigned badvaddr = 0;
    
                    GetExceptionFrame( task, &programcounter, &stackpointer, &returnaddress, &framepointer, &cause, &badvaddr );
                    PRINTDEBUG1( "Exception frame: PC: 0x%08x, SP: 0x%08x, S8: 0x%08x, RA: 0x%08x\n"
                              , programcounter
                              , stackpointer
                              , framepointer
                              , returnaddress
                              );

                    STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "trap pc = 0x%08x, ra = 0x%08x, badva = 0x%08x\n", programcounter, returnaddress, badvaddr );
    
                    if( !in_interrupt() )
                    {
                        /* unwind user stack */
                        STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "user stack = 0x%08x", programcounter );
        
                        UserProcess = process;
                        caller      = 0;
                        framesize   = 0;
                        special     = 0;
                        level       = 0;
                        do
                        {
                            PRINTDEBUG2( "=======================================================\n"); 
    
                            GetStackFrame( level, programcounter, stackpointer, returnaddress, &caller, &framesize, &special, &framepointer );
                            PRINTDEBUG1( "== [%03d] %02d === cal: 0x%08x, frame: %d, special: %d\n", task->pid, level, caller, framesize, special );
    
                            if( special == SCAN_END_SIGFRAME )
                            {
                                int signo = 0;
                                int sigcode = 0;
                                unsigned regs[ 32 ];
                                int i = 0;
                                
                                GetSignalFrame( stackpointer, &programcounter, &stackpointer, &returnaddress, &framepointer, &signo, &sigcode, &regs );
                                PRINTDEBUG1( "SIGNAL FRAME pc: 0x%08x, sp: 0x%08x, fp: 0x%08x, ra: 0x%08x\n"
                                            , programcounter
                                            , stackpointer
                                            , framepointer
                                            , returnaddress
                                            );
    
                                STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "\nsignal no = %d, code = %d, pc = 0x%08x, ra = 0x%08x, regs =", signo, sigcode, programcounter, returnaddress );
                                for( i = 0; i < 32; i += 8 )
                                {
                                    STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, " 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x 0x%08x", regs[ i + 0 ], regs[ i + 1 ], regs[ i + 2 ], regs[ i + 3 ], regs[ i + 4 ], regs[ i + 5 ], regs[ i + 6 ], regs[ i + 7 ] );
                                }
                                STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "\nuser stack = 0x%08x", programcounter );
                                
                                caller    = programcounter; /* continue scanning */
                                framesize = 1;              /* continue scanning */
                                special   = 0;              /* continue scanning */
                                level     = 0;
                            }
    
                            else if( special == 0 )
                            {
                                static int recurse_count = 0;
    
                                PRINTDEBUG1( "[%02d] stack frame - caller: 0x%08x, framesize: %d, framepointer: 0x%08x\n"
                                            , level
                                            , caller
                                            , framesize
                                            , framepointer
                                            );
    
                                STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, " 0x%08x", caller );
                                
                                /* this test stops tracing at recursive functions; still we need better tests to stop backtracing */
                                if( programcounter == caller ) 
                                {
                                    if( recurse_count++ > 3 )
                                    {
    				                    level = MAX_LEVEL;  /* break out */
                                    }
                                }
                                else
                                {
                                    recurse_count = 0;
                                }
    
                                programcounter = caller;
                                stackpointer  += framesize;
                                level         += 1;
                            }
                        }
                        while( caller != 0 && special == 0 && level < MAX_LEVEL );
                        UserProcess = NULL;
        
                        STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "\n" );
                    }
                    else
                    {
                        STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "user stack = <can't retrieve in interrupt context>\n" );
                    }    
                }
            }
            STACKCUR += snprintf( STACKCUR, STACKBUFLEFT, "\n" );
        }
    } while_each_thread( process, task );

    read_unlock( &tasklist_lock );

}

static ssize_t trcext_stackwrite( struct file * file, const char * buf, size_t count, loff_t *ppos )
{
    size_t smallcount = count;
    char kerbuf[ 40 ];
    char* kerbufscan = kerbuf;
    int size = 40;
    int pid = 0;
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( smallcount > size - *ppos )
    {
        smallcount = size - *ppos;
    }
    
    if( copy_from_user( kerbuf + *ppos, buf, smallcount ) )
    {
        return -EFAULT;
    }
    kerbuf[ size-1 ] = '\0';

    if( ! strncmp( "allfreeze", kerbuf, 9 ) )
    {
        pidselectcount = PIDSELECTALL;
        freezetraceonstackread = TRUE;
    }
    else if( ! strncmp( "all", kerbuf, 3 ) )
    {
        pidselectcount = PIDSELECTALL;
    }
    else
    {
        pidselectcount = 0;
        while( sscanf( kerbufscan, "%d", &pid ) == 1 )
        {
            pidselectlist[ pidselectcount++ ] = pid;
            strsep( &kerbufscan, "," );
            if( kerbufscan == NULL ) break;
        }
    }

    *ppos += count;
    return count;
}


static ssize_t trcext_stackread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    static int size = 0;
    size_t smallcount = count ;

    /* make stack dump if file read from beginning */
    if( *ppos == 0 )
    {
        STACKCUR = STACKBUF;
        kernel_crash_dump.stackbufvalid = 0;
        MakeStackTrace();
        size = STACKCUR - STACKBUF;
    }

    /* return generated stack */
    if( *ppos >= size )
    {
        size = 0;
        return 0;
    }

    if( smallcount > size - *ppos )
    {
        smallcount = size - *ppos;
    }
    
    
    if( copy_to_user( buf, STACKBUF + *ppos, smallcount ) )
    {
        return -EFAULT;
    }
   
    *ppos += smallcount;
    return smallcount;
}

static void probe_arch_kernel_signal_force_sigsegv( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    struct task_struct* task = NULL;
    int sig = 0;

    task = va_arg( *args, typeof( task ) );
    sig = va_arg( *args, typeof( sig ) );

    printk( KERN_EMERG "PVL: force_sigsegv pid = %d, signal = %d; try backtrace\n", task->pid, sig );

    STACKCUR = STACKBUF;
    pidselectlist[ 0 ] = task->tgid;
    pidselectcount = 1;
    kernel_crash_dump.stackbufvalid = 0;
    MakeStackTrace();
    {
        char* line = NULL;
        char* stackrun = STACKBUF;
        while( ( line = strsep( &stackrun, "\n" ) ) )
        {
            printk( KERN_EMERG "%s\n", line );
        }
    }
}

static void probe_kernel_signal_force_blocked( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    struct task_struct* task = NULL;
    int sig = 0;

    task = va_arg( *args, typeof( task ) );
    sig = va_arg( *args, typeof( sig ) );

    printk( KERN_EMERG "PVL: force_sig_info blocked pid = %d, signal = %d; try backtrace\n", task->pid, sig );

    STACKCUR = STACKBUF;
    pidselectlist[ 0 ] = task->tgid;
    pidselectcount = 1;
    kernel_crash_dump.stackbufvalid = 0;
    MakeStackTrace();
    {
        char* line = NULL;
        char* stackrun = STACKBUF;
        while( ( line = strsep( &stackrun, "\n" ) ) )
        {
            printk( KERN_EMERG "%s\n", line );
        }
    }
}


/*
 * Save kernel trace and stack through reboot
 */
 
static void probe_arch_mips_kernel_trcext_prepare_debugdump( void* probe_private, void* call_private, const char* fmt, va_list* args )
{
    STACKCUR = STACKBUF;
    pidselectcount = PIDSELECTALL;
    kernel_crash_dump.stackbufvalid = 0;
    freezetraceonstackread = TRUE;
    MakeStackTrace();
    kernel_crash_dump.stackbufvalid = MAGICTRUE;
}

static ssize_t trcext_savedwrite( struct file * file, const char * buf, size_t count, loff_t *ppos )
{
    unsigned long irqflags = 0;

    spin_lock_irqsave( &trcbuf_lock, irqflags );
    if( TRACEFROZEN != MAGICTRUE )
    {
        TRACECUR = TRACEBUF;
        TRACEMODE = MODEFIFO;
        TRACEFULL = FALSE;

        PRINTCUR = PRINTBUF;
        PRINTFULL = FALSE;

        {
            int bucket = 0;
            for( bucket = 0; bucket < TASKBUCKETLIM; ++bucket )
            {
                INIT_HLIST_HEAD( &TASKLIST[ bucket ] );
            }
        }
        {
            TaskEntry* e = NULL;
            INIT_HLIST_HEAD( &TASKFREELIST );
            for( e = TASKMEMPOOL; e < TASKMEMPOOL + TASKLIM; ++e )
            {
                hlist_add_head( &e->head, &TASKFREELIST );
            }
        }
        TASKCOUNT = 0;
    
        {
            PumpEntry* e = NULL;
            INIT_HLIST_HEAD( &PUMPFREELIST );
            for( e = PUMPMEMPOOL; e < PUMPMEMPOOL + PUMPLIM; ++e )
            {
                hlist_add_head( &e->head, &PUMPFREELIST );
            }
        }
        PUMPCOUNT = 0;
    }
    spin_unlock_irqrestore( &trcbuf_lock, irqflags );

    kernel_crash_dump.logbufvalid = 0;
    kernel_crash_dump.stackbufvalid = 0;

    *ppos += count;
    return count;
}

static ssize_t trcext_savedread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    char kerbuf[ 20 ];
    int size = 0;
    
    if( kernel_crash_dump.logbufvalid == MAGICTRUE )
    {
        size = sprintf( kerbuf, "1\n" );
    }
    else
    {
        size = sprintf( kerbuf, "0\n" );
    }
    
    if( *ppos >= size )
    {
        return 0;
    }
    
    if( count > size - *ppos )
    {
        count = size - *ppos;
    }
    
    if( copy_to_user( buf, kerbuf + *ppos, count ) )
    {
        return -EFAULT;
    }
    
    *ppos += count;
    return count;
}

static ssize_t trcext_savedtraceread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    return trcext_traceread( file, buf, count, ppos );
}

static ssize_t trcext_savedstackread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    static int size = 0;
    size_t smallcount = count;


    if( kernel_crash_dump.stackbufvalid != MAGICTRUE )
    {
        return -EINVAL;
    }
    
    if( *ppos == 0 )
    {
        size = STACKCUR - STACKBUF;
    }

    if( *ppos >= size ) 
    {
        return 0;
    }
    
    if( *ppos + smallcount > size )
    {
        smallcount = size - *ppos;
    }

    
    if( copy_to_user( buf, STACKBUF + *ppos, smallcount ) )
    {
        return -EFAULT;
    }
    
    buf += smallcount;
    *ppos += smallcount;
    return smallcount;
}

static ssize_t trcext_savedprintkread( struct file * file, char __user * buf, size_t count, loff_t *ppos )
{
    size_t smallcount = count;

    if( kernel_crash_dump.logbufvalid != MAGICTRUE )
    {
        return -EINVAL;
    }
    
    if( *ppos >= LOGBUFSIZE ) 
    {
        return 0;
    }
    
    if( *ppos + smallcount > LOGBUFSIZE )
    {
        smallcount = LOGBUFSIZE - *ppos;
    }

    
    if( copy_to_user( buf, kernel_crash_dump.logbuf + *ppos, smallcount ) )
    {
        return -EFAULT;
    }
    
    buf += smallcount;
    *ppos += smallcount;
    return smallcount;
}

/*
 * Init
 */

static int create_proc_entries( void )
{
    int err = 0;

    {
        struct proc_dir_entry* trcext = NULL;
        trcext = create_proc_entry( "trcext", S_IFDIR | S_IRUSR | S_IXUSR, NULL );
        if( !trcext )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext->owner = THIS_MODULE;
    }
    
    {
        struct proc_dir_entry* trcext_trace = NULL;
        static struct file_operations trcext_traceops =
        {
            .read   = trcext_traceread,
        };
     
        trcext_trace = create_proc_entry( "trcext/trace", S_IRUSR, NULL );
        if( !trcext_trace )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }

        trcext_trace->owner = THIS_MODULE;
        trcext_trace->proc_fops = &trcext_traceops;
    }
    
    {
        struct proc_dir_entry* trcext_count = NULL;
        static struct file_operations trcext_countops =
        {
            .read   = trcext_countread,
        };

        trcext_count = create_proc_entry( "trcext/count", S_IRUSR, NULL );
        if( !trcext_count )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_count->owner = THIS_MODULE;
        trcext_count->proc_fops = &trcext_countops;
    }
    
    {
        struct proc_dir_entry* trcext_mode = NULL;
        static struct file_operations trcext_modeops =
        {
            .read   = trcext_moderead,
            .write  = trcext_modewrite,
        };

        trcext_mode = create_proc_entry( "trcext/mode", S_IRUSR | S_IWUSR, NULL );
        if( !trcext_mode )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_mode->owner = THIS_MODULE;
        trcext_mode->proc_fops = &trcext_modeops;
    }
    
    {
        struct proc_dir_entry* trcext_frozen = NULL;
        static struct file_operations trcext_frozenops =
        {
            .read   = trcext_frozenread,
            .write  = trcext_frozenwrite,
        };

        trcext_frozen = create_proc_entry( "trcext/frozen", S_IRUSR | S_IWUSR, NULL );
        if( !trcext_frozen )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_frozen->owner = THIS_MODULE;
        trcext_frozen->proc_fops = &trcext_frozenops;
    }
    
    {
        struct proc_dir_entry* trcext_taskadd = NULL;
        static struct file_operations trcext_taskaddops =
        {
            .write  = trcext_taskaddwrite,
        };

        trcext_taskadd = create_proc_entry( "trcext/taskadd", S_IWUSR, NULL );
        if( !trcext_taskadd )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_taskadd->owner = THIS_MODULE;
        trcext_taskadd->proc_fops = &trcext_taskaddops;
    }

    {
        struct proc_dir_entry* trcext_taskdel = NULL;
        static struct file_operations trcext_taskdelops =
        {
            .write  = trcext_taskdelwrite,
        };

        trcext_taskdel = create_proc_entry( "trcext/taskdel", S_IWUSR, NULL );
        if( !trcext_taskdel )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_taskdel->owner = THIS_MODULE;
        trcext_taskdel->proc_fops = &trcext_taskdelops;
    }

    {
        struct proc_dir_entry* trcext_sample = NULL;
        static struct file_operations trcext_sampleops =
        {
            .write  = trcext_samplewrite,
        };

        trcext_sample = create_proc_entry( "trcext/sample", S_IWUSR, NULL );
        if( !trcext_sample )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_sample->owner = THIS_MODULE;
        trcext_sample->proc_fops = &trcext_sampleops;
    }
    
    {
        struct proc_dir_entry* trcext_print = NULL;
        static struct file_operations trcext_printops =
        {
            .write  = trcext_printwrite,
        };

        trcext_print = create_proc_entry( "trcext/print", S_IWUSR, NULL );
        if( !trcext_print )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_print->owner = THIS_MODULE;
        trcext_print->proc_fops = &trcext_printops;
    }

    {
        struct proc_dir_entry* trcext_stracing = NULL;
        static struct file_operations trcext_stracingops =
        {
            .read   = trcext_stracingread,
            .write  = trcext_stracingwrite,
        };

        trcext_stracing = create_proc_entry( "trcext/stracing", S_IRUSR | S_IWUSR, NULL );
        if( !trcext_stracing )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_stracing->owner = THIS_MODULE;
        trcext_stracing->proc_fops = &trcext_stracingops;
    }
    
    {
        struct proc_dir_entry* trcext_irqtracing = NULL;
        static struct file_operations trcext_irqtracingops =
        {
            .read   = trcext_irqtracingread,
            .write  = trcext_irqtracingwrite,
        };

        trcext_irqtracing = create_proc_entry( "trcext/irqtracing", S_IRUSR | S_IWUSR, NULL );
        if( !trcext_irqtracing )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_irqtracing->owner = THIS_MODULE;
        trcext_irqtracing->proc_fops = &trcext_irqtracingops;
    }

    {
        struct proc_dir_entry* trcext_softirqtracing = NULL;
        static struct file_operations trcext_softirqtracingops =
        {
            .read   = trcext_softirqtracingread,
            .write  = trcext_softirqtracingwrite,
        };

        trcext_softirqtracing = create_proc_entry( "trcext/softirqtracing", S_IRUSR | S_IWUSR, NULL );
        if( !trcext_softirqtracing )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_softirqtracing->owner = THIS_MODULE;
        trcext_softirqtracing->proc_fops = &trcext_softirqtracingops;
    }

    {
        struct proc_dir_entry* trcext_signaltracing = NULL;
        static struct file_operations trcext_signaltracingops =
        {
            .read   = trcext_signaltracingread,
            .write  = trcext_signaltracingwrite,
        };

        trcext_signaltracing = create_proc_entry( "trcext/signaltracing", S_IRUSR | S_IWUSR, NULL );
        if( !trcext_signaltracing )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_signaltracing->owner = THIS_MODULE;
        trcext_signaltracing->proc_fops = &trcext_signaltracingops;
    }

    {
        struct proc_dir_entry* trcext_stack = NULL;
        static struct file_operations trcext_stackops =
        {
            .write  = trcext_stackwrite,
            .read   = trcext_stackread,
        };

        trcext_stack = create_proc_entry( "trcext/stack", S_IRUSR | S_IWUSR, NULL );
        if( !trcext_stack )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_stack->owner = THIS_MODULE;
        trcext_stack->proc_fops = &trcext_stackops;
    }
    {
        struct proc_dir_entry* trcext_saved = NULL;
        static struct file_operations trcext_savedops =
        {
            .write  = trcext_savedwrite,
            .read   = trcext_savedread,
        };

        trcext_saved = create_proc_entry( "trcext/saved", S_IRUSR | S_IWUSR, NULL );
        if( !trcext_saved )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_saved->owner = THIS_MODULE;
        trcext_saved->proc_fops = &trcext_savedops;
    }

    {
        struct proc_dir_entry* trcext_savedtrace = NULL;
        static struct file_operations trcext_savedtraceops =
        {
            .read   = trcext_savedtraceread,
        };

        trcext_savedtrace = create_proc_entry( "trcext/savedtrace", S_IRUSR, NULL );
        if( !trcext_savedtrace )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_savedtrace->owner = THIS_MODULE;
        trcext_savedtrace->proc_fops = &trcext_savedtraceops;
    }

    {
        struct proc_dir_entry* trcext_savedstack = NULL;
        static struct file_operations trcext_savedstackops =
        {
            .read   = trcext_savedstackread,
        };

        trcext_savedstack = create_proc_entry( "trcext/savedstack", S_IRUSR, NULL );
        if( !trcext_savedstack )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_savedstack->owner = THIS_MODULE;
        trcext_savedstack->proc_fops = &trcext_savedstackops;
    }

    {
        struct proc_dir_entry* trcext_savedprintk = NULL;
        static struct file_operations trcext_savedprintkops =
        {
            .read   = trcext_savedprintkread,
        };

        trcext_savedprintk = create_proc_entry( "trcext/savedprintk", S_IRUSR, NULL );
        if( !trcext_savedprintk )
        {
            err = -ENOMEM;
            goto exit_errmem;
        }
        trcext_savedprintk->owner = THIS_MODULE;
        trcext_savedprintk->proc_fops = &trcext_savedprintkops;
    }

exit_errmem:
    return err;
}

static void remove_proc_entries( void )
{
    remove_proc_entry( "trcext/savedprintk", NULL );
    remove_proc_entry( "trcext/savedstack", NULL );
    remove_proc_entry( "trcext/savedtrace", NULL );
    remove_proc_entry( "trcext/saved", NULL );
    remove_proc_entry( "trcext/stack", NULL );
    remove_proc_entry( "trcext/signaltracing", NULL );
    remove_proc_entry( "trcext/softirqtracing", NULL );
    remove_proc_entry( "trcext/irqtracing", NULL );
    remove_proc_entry( "trcext/stracing", NULL );
    remove_proc_entry( "trcext/print", NULL );
    remove_proc_entry( "trcext/sample", NULL );
    remove_proc_entry( "trcext/taskdel", NULL );
    remove_proc_entry( "trcext/taskadd", NULL );
    remove_proc_entry( "trcext/frozen", NULL );
    remove_proc_entry( "trcext/mode", NULL );
    remove_proc_entry( "trcext/count", NULL );
    remove_proc_entry( "trcext/pumps", NULL );
    remove_proc_entry( "trcext/tasks", NULL );
    remove_proc_entry( "trcext/trace", NULL );
    remove_proc_entry( "trcext", NULL );
}

static int trcctl_init( void )
{
    int err = 0;
    
#if BIGBUF
#warning "BIGBUF option enabled"
    printk( KERN_WARNING "trcext.ko is compiled with increased trace buffer size\n" );
#endif

    /* initialize trace buffer */    
    if( TRACEFROZEN != MAGICTRUE )
    {
        TRACECUR = TRACEBUF;
        TRACEMODE = MODEFIFO;
        TRACEFULL = FALSE;

        PRINTCUR = PRINTBUF;
        PRINTFULL = FALSE;

        {
            int bucket = 0;
            for( bucket = 0; bucket < TASKBUCKETLIM; ++bucket )
            {
                INIT_HLIST_HEAD( &TASKLIST[ bucket ] );
            }
        }
        {
            TaskEntry* e = NULL;
            INIT_HLIST_HEAD( &TASKFREELIST );
            for( e = TASKMEMPOOL; e < TASKMEMPOOL + TASKLIM; ++e )
            {
                hlist_add_head( &e->head, &TASKFREELIST );
            }
        }
        TASKCOUNT = 0;
    
        {
            PumpEntry* e = NULL;
            INIT_HLIST_HEAD( &PUMPFREELIST );
            for( e = PUMPMEMPOOL; e < PUMPMEMPOOL + PUMPLIM; ++e )
            {
                hlist_add_head( &e->head, &PUMPFREELIST );
            }
        }
        PUMPCOUNT = 0;
    }

    /* create proc entries */
    if( (err = create_proc_entries()) != 0 )
    {
        goto exit_errprocentries;
    }    
   

    /* start task cleanup loop */
    {
        kernel_thread( &PeriodicTaskCleanup, NULL, CLONE_KERNEL );
    }

    /* add hook for undeliverable signals*/
    {
        marker_probe_register( "arch_kernel_signal_force_sigsegv", "%p %d", probe_arch_kernel_signal_force_sigsegv, NULL );
        marker_probe_register( "kernel_signal_force_blocked", "%p %d", probe_kernel_signal_force_blocked, NULL );
    }   


    /* add task hook */
    {
    	marker_probe_register( "kernel_process_free", "%p", probe_kernel_process_free, NULL );
    	marker_probe_register( "kernel_process_exit", "%p", probe_kernel_process_exit, NULL );
        marker_probe_register( "kernel_process_fork", "%p %p", probe_kernel_process_fork, NULL );
        marker_probe_register( "kernel_thread_create", "%ld %p", probe_kernel_thread_create, NULL );
        /* marker_probe_register( "kernel_sched_wakeup_new_task", "%p %ld", probe_kernel_sched_wakeup_new_task, NULL ); */
        marker_probe_register( "kernel_sched_schedule", "%p %p", probe_kernel_sched_schedule, NULL );
    }

    /* add signal hook */
    if( signaltracing )
    {
        /* install probes for tracing signals */
        marker_probe_register( "kernel_signal", "%p %d", probe_kernel_signal, NULL );
    }

    /* add trace print probes */
    {
        marker_probe_register( "arch_mips_kernel_trcext_trace_print_irq", "%d", probe_arch_mips_kernel_trcext_trace_print_irq, NULL );
        marker_probe_register( "arch_mips_kernel_trcext_trace_print", "%d", probe_arch_mips_kernel_trcext_trace_print, NULL );
    }

    /* add probe for kernel crash dump */
    marker_probe_register( "arch_mips_kernel_trcext_prepare_debugdump", MARK_NOARGS, probe_arch_mips_kernel_trcext_prepare_debugdump, NULL );

    return 0;

exit_errprocentries:
    remove_proc_entries();
    return err;
}

static void trcctl_exit( void )
{
    /* remove proc entries */
    remove_proc_entries();
    
    /* deregister probes */
    marker_probe_unregister( "arch_mips_kernel_trcext_prepare_debugdump", probe_arch_mips_kernel_trcext_prepare_debugdump, NULL );
    marker_probe_unregister( "arch_mips_kernel_trcext_trace_print", probe_arch_mips_kernel_trcext_trace_print, NULL );
    marker_probe_unregister( "arch_mips_kernel_trcext_trace_print_irq", probe_arch_mips_kernel_trcext_trace_print_irq, NULL );
    marker_probe_unregister( "arch_kernel_signal_force_sigsegv", probe_arch_kernel_signal_force_sigsegv, NULL );
    marker_probe_unregister( "kernel_sched_schedule", probe_kernel_sched_schedule, NULL );
    /* marker_probe_unregister( "kernel_sched_wakeup_new_task", probe_kernel_sched_wakeup_new_task, NULL ); */
    marker_probe_unregister( "kernel_thread_create", probe_kernel_thread_create, NULL );
    marker_probe_unregister( "kernel_process_fork", probe_kernel_process_fork, NULL );
    marker_probe_unregister( "kernel_process_exit", probe_kernel_process_exit, NULL );
    marker_probe_unregister( "kernel_process_free", probe_kernel_process_free, NULL );

    /* stop periodic task cleanup */
    PeriodicTaskCleanupStartComplete = TRUE;  /* cleanup task will exit by itself when it sees this */
    wait_for_completion( &PeriodicTaskCleanupExit );
}

module_init( trcctl_init );
module_exit( trcctl_exit );
