/*-
 * Copyright (c) 2009  Atheros Communications Inc
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/version.h>
#include <linux/usb.h>
#include <linux/netdevice.h>
#include <linux/skbuff.h>

#include "hif.h"
#include "hif_usb_internal.h"


#ifdef HIF_USB_ENABLE_STREAM_MODE
#if  ZM_USB_TX_STREAM_MODE == 1

int aggcnt = 5;//ZM_MAX_TX_AGGREGATE_NUM;
int aggtimer = 100;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
module_param(aggcnt, int, S_IRUGO);
module_param(aggtimer, int, S_IRUGO);
#else
MODULE_PARM(aggcnt, "i");
MODULE_PARM(aggtimer, "i");
#endif
#endif
#endif


volatile int surprise = 0;
static void *HIFInit(struct usb_interface *interface);
static hif_status_t HIFFirmwareDownload(hif_handle_t hHIF, const uint32_t* fw, uint32_t len, 
                    uint32_t offset,uint32_t testoffset);

drvreg_callbacks_t DrvRegCallbacks;
extern const uint32_t zcFwImage[];
extern const uint32_t zcFwImageSize;
extern const uint32_t zcFwImageK2[];
extern const uint32_t zcFwImageK2Size;
ath_usb_unload g_ath_usb_unload = NULL;

int 
notifyDeviceInsertedHandler(hif_handle_t hHIF)
{
    daemonize("notifyDeviceInsertedHandler");
    DrvRegCallbacks.deviceInsertedHandler(hHIF);
    return 0;
}


static hif_status_t
HIFFirmwareDownload(hif_handle_t hHIF, const uint32_t* fw, uint32_t len, 
                    uint32_t offset,uint32_t uTextOfst)
{
    hif_status_t ret = HIF_OK;
    uint32_t uCodeOfst = offset;
    uint8_t *image, *ptr;
    uint32_t result;
    
    image = (uint8_t*) fw;
    ptr = image;
  
    while (len > 0) 
    {
        uint32_t translen = (len > 4096) ? 4096 : len;
    /**
     * Avoiding the conversion to original buffer,
     * otherwise next hotplug might have inverted firmware
     */
#if 0
        uint32_t ii;
        uint32_t *pfw = (uint32_t *)image;

        /* Convert firmware image to correct byte ordering */
        for (ii = 0; ii < (translen >> 2); ii++)
        {
            *pfw = cpu_to_le32(*pfw);
             pfw++;
        }
#endif

        result = zfLnxUsbSubmitControl(hHIF, FIRMWARE_DOWNLOAD, 
                                      (uint16_t) (uCodeOfst >> 8), 
                                      0, image, translen);

        if (result != 0)
        {
            printk("FIRMWARE_DOWNLOAD failed\n");
            ret = HIF_ERROR;
            goto exit;
        }

        len -= translen;
        image += translen;
        uCodeOfst += translen; // in Word (16 bit)

        result = 0;
    }

    /* If download firmware success, issue a command to firmware */
    if (ret == 0)
    {   
        result = zfLnxUsbSubmitControl(hHIF, FIRMWARE_DOWNLOAD_COMP, 
                                       (uint16_t) (uTextOfst >> 8), 
                                       0, NULL, 0);

        if (result != 0)
        {
            printk("FIRMWARE_DOWNLOAD_COMP failed\n");
            ret = HIF_ERROR;
            goto exit;
        }
    }

exit:

    return ret;

}
#define VENDOR_ATHR             0x0CF3  //Atheros
#define PRODUCT_AR7010          0x7010  
#define PRODUCT_AR7011          0x7011  
#define PRODUCT_AR9271          0x9271  
#define PRODUCT_TV550 0x209e


void 
HIF_USBDeviceInserted(struct usb_interface *interface, ath_usb_unload unload,short product_id)
{
    hif_handle_t hHIF = NULL;
    unsigned int  *FwImage = NULL;
    unsigned int FwImageSize = 0;
    unsigned int offset = 0 ;
    unsigned int  pid = ( cpu_to_le16( product_id ) & 0xffff );

    //pid =  (((product_id & 0xff) << 8)  | ((product_id & 0xff00) >> 8) ) & 0x0000ffff  ;
    printk("GVC  pid %x *************\n",pid);
    g_ath_usb_unload = unload;
    switch(pid)
    {
        case PRODUCT_AR7010 :
        case PRODUCT_AR7011 :
                             printk("Magpie Detected PID %x \n",pid); 
                             FwImage     = (uint *)zcFwImage;
                             FwImageSize = zcFwImageSize;
                             offset = ZM_FIRMWARE_TEXT_ADDR_MAGPIE ; 
                             break ;
        case PRODUCT_AR9271 :
        case PRODUCT_TV550:
                             printk("K2 Detected PID %x \n",pid); 
                             FwImage     = (uint *)zcFwImageK2;
                             FwImageSize = zcFwImageK2Size;
                             offset = ZM_FIRMWARE_TEXT_ADDR_K2 ; 

                             break ;
                             
        default :
                             printk("Unknown  Prodcut ID %d  Asserting \n",pid);
                             break;

    }

	surprise = 0;
    hHIF = HIFInit(interface);

     

#ifdef USB_DISCONNECT_RECONNECT
    static int first_time =1;
    if (first_time)
    {
	first_time = 0;
    /* FW download */
    if (HIFFirmwareDownload(hHIF, (uint32_t *)FwImage,
                            (uint32_t)FwImageSize,
                            ZM_FIRMWARE_WLAN_ADDR,offset) != HIF_OK) {
        printk("[%s] FW download failed.\n", __FUNCTION__);
        //hif_usb_assert(0);
    } else 
        printk("[%s] FW download successful.\n", __FUNCTION__);
    } else {
        first_time = 1;
        /* Inform HTC */
        if (hHIF != NULL && DrvRegCallbacks.deviceInsertedHandler) {
            //htcDrvRegCallbacks.deviceInsertedHandler(hHIF);
            kernel_thread(notifyDeviceInsertedHandler, hHIF, 
                    CLONE_FS|CLONE_FILES);
        }
    }
#else /* USB_DISCONNECT_RECONNECT */

#if 1
    /* FW download */
    if (HIFFirmwareDownload(hHIF, (uint32_t *)FwImage, 
                            (uint32_t)FwImageSize, 
                            ZM_FIRMWARE_WLAN_ADDR,offset) != HIF_OK) {

        printk("[%s] FW download failed.\n", __FUNCTION__);
    //    hif_usb_assert(0);
    }
    else 
        printk("[%s] FW download successful.\n", __FUNCTION__);
    
#endif

    /* Inform HTC */
    if (hHIF != NULL && DrvRegCallbacks.deviceInsertedHandler) {
        //htcDrvRegCallbacks.deviceInsertedHandler(hHIF);
        kernel_thread(notifyDeviceInsertedHandler, hHIF, CLONE_FS|CLONE_FILES);
    }
#endif /* USB_DISCONNECT_RECONNECT */
    //return 0;
}

static void send_usb_reboot_cmd(HIF_DEVICE_USB *macp)
{
    printk("Send reboot command to target HIF USB...\n");
    //if ( PipeID == USB_REG_OUT_PIPE ) 
    {
        uint32_t rebootCmd = 0xffffffff;
        uint8_t *data = (uint8_t *)&rebootCmd;
        uint32_t len = 4;
        UsbCmdUrbContext *ctx;
        int ret;
        
        ctx = zfLnxAllocCmdUrbCtx(macp);
        ctx->buf = NULL;

#if 0
        ret = zfLnxUsbSubmitBulkUrb(ctx->RegOutUrb, macp->udev,
                                    USB_REG_OUT_PIPE, USB_DIR_OUT, data,
                                    adf_nbuf_len(sendBuf), 
                                    zfLnxUsbRegOut_callback, ctx);
#else
         ret = zfLnxUsbSubmitIntUrb(ctx->RegOutUrb, macp->udev,
                USB_REG_OUT_PIPE, USB_DIR_OUT, data,
                len, zfLnxUsbRegOut_callback, ctx, 1);
#endif        
        //adf_os_assert( ret == 0 );                                    
        udelay(10000);                   
    }

}

int 
notifyDeviceSurprisedRemovedHandler(void *handle)
{
    daemonize("notifyDeviceSurprisedRemovedHandler");
    DrvRegCallbacks.deviceRemovedHandler(handle, 1);
    return 0;
}

void 
HIF_USBDeviceDetached(struct usb_interface *interface, uint8_t surpriseRemoved)
{
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
    HIF_DEVICE_USB *macp = (HIF_DEVICE_USB *) usb_get_intfdata(interface);
#else
    HIF_DEVICE_USB *macp = (HIF_DEVICE_USB *)ptr;
#endif   

    printk("HIF_USBDeviceDetached\n");

    macp->surpriseRemoved = surpriseRemoved;
    surprise = surpriseRemoved;

    if ( surpriseRemoved ) {
        kernel_thread(notifyDeviceSurprisedRemovedHandler, macp->handle, 
                CLONE_FS|CLONE_FILES);
        goto free_rez;
    }

#if ZM_USB_TX_STREAM_MODE == 1
    /* del timer */
    del_timer(&macp->tm_txagg);
#endif

    del_timer(&macp->tm_rxbuf_alloc);

    /* Inform HTC */
    if (DrvRegCallbacks.deviceRemovedHandler) {
        DrvRegCallbacks.deviceRemovedHandler(macp->handle, 
                surpriseRemoved);
    }

    if ( surpriseRemoved == 0 )
        send_usb_reboot_cmd(macp);  

free_rez:
    zfLnxUnlinkAllUrbs(macp);
    zfLnxFreeAllUrbs(macp);
    zfLnxFreeUsbTxBuffer(macp, &macp->TxPipe);
#ifndef DISABLE_USB_MPHP    
    zfLnxFreeUsbTxBuffer(macp, &macp->HPTxPipe);
    zfLnxFreeUsbTxBuffer(macp, &macp->MPTxPipe);        
#endif 
    HIFShutDown(macp);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
    usb_set_intfdata(interface, NULL);
#endif
}

hif_status_t 
HIF_register(drvreg_callbacks_t *callbacks)
{
    DrvRegCallbacks.deviceInsertedHandler = callbacks->deviceInsertedHandler;
    DrvRegCallbacks.deviceRemovedHandler = callbacks->deviceRemovedHandler;

    return HIF_OK;
}

hif_status_t 
HIF_deregister()
{
    g_ath_usb_unload();
    return HIF_OK;
}
#ifdef ADF_OS_MEM_DEBUG
void __adf_os_free_debug (void * );
#endif
hif_status_t 
HIFSend(hif_handle_t hHIF, uint8_t PipeID, struct sk_buff *hdr_buf, 
        struct sk_buff *buf)
{
    hif_status_t status = HIF_OK;
    HIF_DEVICE_USB *macp = (HIF_DEVICE_USB *)hHIF;
    struct sk_buff *sendBuf;

    if ( macp->surpriseRemoved ) 
        return HIF_ERROR;
    

    /* If necessary, link hdr_buf & buf */
    if (hdr_buf != NULL) {
        if(!pskb_expand_head(hdr_buf, 0, buf->len, GFP_ATOMIC))
            memcpy(skb_put(hdr_buf, buf->len), buf->data, buf->len);

        sendBuf = hdr_buf;
    } else
        sendBuf = buf;
   

    if ( PipeID == USB_REG_OUT_PIPE ) {
        uint8_t *data = NULL;
        uint32_t len;
        UsbCmdUrbContext *ctx;
        //int ret;
        
        ctx = zfLnxAllocCmdUrbCtx(macp);
        if ( ctx == NULL ) {
            status = -1;
            goto done;
        }

        //macp->regUsbWriteBuf = sendBuf;
        ctx->buf = sendBuf;
        data     = sendBuf->data;
        len      = sendBuf->len;

	if(surprise)
		return -1;
        status = zfLnxUsbSubmitIntUrb(ctx->RegOutUrb, macp->udev,
                USB_REG_OUT_PIPE, USB_DIR_OUT, data,
                sendBuf->len, zfLnxUsbRegOut_callback, ctx, 1);
    } else if ( PipeID == USB_WLAN_TX_PIPE )
        status = zfLnxUsbOut(macp, sendBuf, &macp->TxPipe);
   
#ifndef DISABLE_USB_MPHP
    else if ( PipeID == USB_WLAN_HP_TX_PIPE )
        status = zfLnxUsbOut(macp, sendBuf, &macp->HPTxPipe);
    else if ( PipeID == USB_WLAN_MP_TX_PIPE )
        status = zfLnxUsbOut(macp, sendBuf, &macp->MPTxPipe);
#endif
    else {
        printk("Unknown pipe %d\n", PipeID);
        dev_kfree_skb_any(sendBuf);
#ifdef ADF_OS_MEM_DEBUG
            __adf_os_free_debug ((void *) sendBuf);
#endif
     }
    if(status == -1){
	    dev_kfree_skb_any(sendBuf);
#ifdef ADF_OS_MEM_DEBUG
            __adf_os_free_debug ((void *) sendBuf);
#endif
	    return -1;
    }

done:
    return ( status ? HIF_ERROR : HIF_OK);
}

uint16_t 
HIFGetFreeQueueNumber(hif_handle_t hHIF, uint8_t PipeID)
{
    HIF_DEVICE_USB *macp = (HIF_DEVICE_USB *)hHIF;

    if ( PipeID == USB_WLAN_TX_PIPE )
    {
        return zfLnxGetFreeTxBuffer(macp, &macp->TxPipe);
    }
#ifndef DISABLE_USB_MPHP
    else if ( PipeID == USB_WLAN_HP_TX_PIPE )
    {
        return zfLnxGetFreeTxBuffer(macp, &macp->HPTxPipe);
    }
    else if ( PipeID == USB_WLAN_MP_TX_PIPE )
    {
        return zfLnxGetFreeTxBuffer(macp, &macp->MPTxPipe);
    }
#endif
    else
    {
        return 1; 
    }
}

void 
HIFPostInit(hif_handle_t hHIF, void *handle, completion_callbacks_t *callbacks)
{
    HIF_DEVICE_USB *hif_dev = (HIF_DEVICE_USB *) hHIF;

    printk("HIFPostInit %p\n", handle);

    hif_dev->handle = handle;
    hif_dev->Callbacks.Context = callbacks->Context;
    hif_dev->Callbacks.rxCompletionHandler = callbacks->rxCompletionHandler;
    hif_dev->Callbacks.txCompletionHandler = callbacks->txCompletionHandler;    
}

static hif_handle_t
HIFInit(struct usb_interface *interface)
{
    //uint8_t i;
    HIF_DEVICE_USB *hif_dev;
    struct usb_device *dev = interface_to_usbdev(interface);

    /* allocate memory for HIF_DEVICE */
    hif_dev = (HIF_DEVICE_USB *)kzalloc(sizeof(HIF_DEVICE_USB), GFP_ATOMIC);
    if (!hif_dev)
        return NULL;
    
    hif_dev->udev = dev;
    /* set up the endpoint information */
    /* check out the endpoints */
    hif_dev->interface = interface;

    printk("HIFInit #1\n");
    if (!zfLnxAllocAllUrbs(hif_dev))
        goto fail2;
   

    spin_lock_init(&(hif_dev->cs_lock));

    printk("HIFInit #2\n");
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0))
    usb_set_intfdata(interface, hif_dev);
#endif

    return hif_dev;

fail2:
    kfree(hif_dev);
    return NULL;
}

void 
HIFStart(hif_handle_t hHIF)
{
    HIF_DEVICE_USB *hif_dev = (HIF_DEVICE_USB *) hHIF;

    /* Initialize USB TxQ */
    zfLnxInitUsbTxQ(hif_dev);

    /* Initialize USB RxQ */
    zfLnxInitUsbRxQ(hif_dev);

    zfLnxSubmitRegInUrb(hif_dev);
}

/* undo what was done in HIFInit() */
void 
HIFShutDown(hif_handle_t hHIF)
{
    HIF_DEVICE_USB *hif_dev = (HIF_DEVICE_USB *)hHIF;

    /* XXX:need to handle packets queued in the HIF */

    /* free memory for hif device */
    kfree(hif_dev);

}

void 
HIFGetDefaultPipe(hif_handle_t hHIF, uint8_t *ULPipe, uint8_t *DLPipe)
{
    *ULPipe = HIF_USB_PIPE_COMMAND;
    *DLPipe = HIF_USB_PIPE_INTERRUPT;
}

uint8_t 
HIFGetULPipeNum(void)
{
    return HIF_USB_MAX_TXPIPES;
}

uint8_t 
HIFGetDLPipeNum(void)
{
    return HIF_USB_MAX_RXPIPES;
}

int init_usb_hif(void)
{
    printk("USBHIF Version 1.xx Loaded...\n");
    return 0;
}

void exit_usb_hif(void)
{
    printk("USB HIF UnLoaded...\n");
}

EXPORT_SYMBOL(HIF_register);
EXPORT_SYMBOL(HIF_deregister);
EXPORT_SYMBOL(HIFShutDown);
EXPORT_SYMBOL(HIFSend);
EXPORT_SYMBOL(HIFStart);
EXPORT_SYMBOL(HIFPostInit);
EXPORT_SYMBOL(HIF_USBDeviceInserted);
EXPORT_SYMBOL(HIF_USBDeviceDetached);
EXPORT_SYMBOL(HIFGetDefaultPipe);
EXPORT_SYMBOL(HIFGetULPipeNum);
EXPORT_SYMBOL(HIFGetDLPipeNum);
EXPORT_SYMBOL(HIFGetFreeQueueNumber);

module_init(init_usb_hif);
module_exit(exit_usb_hif);

MODULE_LICENSE("GPL");
