/*
 * (C) Copyright 2004
 * Pete Popov, Embedded Alley Solutions, Inc
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */
#include <common.h>
#include <command.h>
#include <asm/io.h>
#include <asm/mipsregs.h>
#include <asm/addrspace.h>
#include <nxp_uart_ip0107.h>
#include <nxp_global1_ip0126.h>
#include <netdev.h>
#include <tm328x.h>

#ifdef CONFIG_NAND_LEGACY
#include <linux/mtd/nand_ids.h>
#include <nxp_pcixio_ipa051.h>
#endif

#define SHOW_INFO

#ifdef SHOW_INFO
    #define info(format, arg...) printf("INFO: " format "\n", ## arg)
#else
    #define info(format, arg...) do {} while(0)
#endif /* SHOW_INFO */

#if defined(CONFIG_PCI)
#  include <pci.h>
#endif /* CONFIG_PCI */

#if defined(CONFIG_XIO)
#if defined(CONFIG_NAND_LEGACY)
#  include <linux/mtd/nand_legacy.h>
#else
#  include <linux/mtd/nand.h>
#endif
#endif /* CONFIG_XIO */

#if defined(CONFIG_PCI) || defined(CONFIG_XIO)
#  include <nxp_pcixio_ipa051.h>
#endif /* CONFIG_PCI || CONFIG_XIO */

#ifdef CONFIG_STANDBY_SLAVE_ADDR
#  include <i2c.h>
#endif /* CONFIG_STANDBY_SLAVE_ADDR */

DECLARE_GLOBAL_DATA_PTR;

phys_size_t initdram(int board_type)
{
	unsigned long	memsize;
	unsigned long 	*mips_rst_vector;

	/*
	 * SDRAM has been initialized at power-up with parameters
	 * from I2C NVM by the boot module.
	 */

    mips_rst_vector = (unsigned long)IP0126_MIPS_RESET_VEC;
    memsize = ((*mips_rst_vector & IP0126_MIPS_RESET_VEC_MTL_MASK) >> IP0126_MIPS_RESET_VEC_MTL_SHIFT);
    memsize *= (16*1024*1024);

    return get_ram_size((volatile long *)CONFIG_SYS_SDRAM_BASE, memsize);
}

#if defined(CONFIG_PCI)
static struct pci_controller hose;

extern void init_nxp_ipa051_pci (struct pci_controller *);

int board_early_init_f(void)
{
        /* Unlock the setup register */
        writel( IPA051_UNLOCK_REGISTER__UNLOCK_SETUP,
                IPA051_UNLOCK_REGISTER + IPA051 );

        /* Configure the setup register (locks automatically) */
        writel( IPA051_PCI_SETUP__EN_TA |
                IPA051_PCI_SETUP__EN_PCI2MMI |
                IPA051_PCI_SETUP__EN_XIO |
                IPA051_PCI_SETUP__BASE18_SIZ(IPA051_PCI_SETUP__BASExx_SIZ_128M) |
                IPA051_PCI_SETUP__EN_BASE18 |
                IPA051_PCI_SETUP__EN_BASE14 |
                IPA051_PCI_SETUP__BASE10_PREFETCHABLE |
                IPA051_PCI_SETUP__BASE10_SIZ(IPA051_PCI_SETUP__BASExx_SIZ_16M) |
                IPA051_PCI_SETUP__EN_CONFIG_MANAG |
                IPA051_PCI_SETUP__EN_PCI_ARB,
                IPA051_PCI_SETUP + IPA051 );
        
	return 0;
}

void pci_init_board (void)
{
	int pci_mem_code;

	/* Calc the PCI mem size code */
	if ((gd->ram_size >> 20) > 128)
		pci_mem_code = IPA051_PCI_SETUP__BASExx_SIZ_256M;
	else if ((gd->ram_size >> 20) >= 128)
		pci_mem_code = IPA051_PCI_SETUP__BASExx_SIZ_128M;
	else if ((gd->ram_size>>20) >= 64)
		pci_mem_code = IPA051_PCI_SETUP__BASExx_SIZ_64M;
	else if ((gd->ram_size>>20) >= 32)
		pci_mem_code = IPA051_PCI_SETUP__BASExx_SIZ_32M;
	else
		pci_mem_code = IPA051_PCI_SETUP__BASExx_SIZ_16M;

	/* Configure BASE1 for PCI Memory aperture */
	writel(PCIMEM_BASE,               IPA051 + IPA051_PCI_BASE1_LO);
	writel(PCIMEM_BASE + PCIMEM_SIZE, IPA051 + IPA051_PCI_BASE1_HI);

	/* Configure BASE2 PCI IO aperture */
	writel(PCIIO_BASE,              IPA051 + IPA051_PCI_BASE2_LO);
	writel(PCIIO_BASE + PCIIO_SIZE, IPA051 + IPA051_PCI_BASE2_HI);

	/* Let BASE2 forward PCI2DTL transanctions as memory transations with unchanged address */
	writel(0x00000000, IPA051 + IPA051_PCI_IO);

	/*
	 * Configure SDRAM aperture on PCI bus to the configured
	 * virtual SDRAM address to enable us to send RAM addresses
	 * to PCI devices, like USB OHCI
	 */
	writel( (readl(IPA051 + IPA051_BASE10) & 0x001fffff) |
		IPA051_BASE10__BASE_ADDRESS_10(SDRAM_BASE),
		IPA051_BASE10 + IPA051);

	/*
	 * These two bars are set by default or the boot code.
	 * However, it's safer to set them here so we're not boot
	 * code dependent.
	 */
	writel(MMIO_BASE, IPA051 + IPA051_BASE14);	/* MMIO */
	writel(XIO_BASE,  IPA051 + IPA051_BASE18);	/* XIO      */

	/* Unlock the setup register */
	writel( IPA051_UNLOCK_REGISTER__UNLOCK_SETUP,
		IPA051_UNLOCK_REGISTER + IPA051 );

	/* Configure the setup register (locks automatically) */
	writel( IPA051_PCI_SETUP__EN_TA |
		IPA051_PCI_SETUP__EN_PCI2MMI |
		IPA051_PCI_SETUP__EN_XIO |
		IPA051_PCI_SETUP__BASE18_SIZ(IPA051_PCI_SETUP__BASExx_SIZ_128M) |
		IPA051_PCI_SETUP__EN_BASE18 |
		IPA051_PCI_SETUP__EN_BASE14 |
		IPA051_PCI_SETUP__BASE10_PREFETCHABLE |
		IPA051_PCI_SETUP__BASE10_SIZ(pci_mem_code) |
		IPA051_PCI_SETUP__EN_CONFIG_MANAG |
		IPA051_PCI_SETUP__EN_PCI_ARB,
		IPA051_PCI_SETUP + IPA051 );

	/* PCI_CONTROL */
	writel( IPA051_PCI_CONTROL__DECOUPLE_TARG |
		IPA051_PCI_CONTROL__EN_SERR_SEEN |
		IPA051_PCI_CONTROL__EN_BASE10_SPEC_RD |
		IPA051_PCI_CONTROL__DISABLE_SUBWORD2_14 |
		IPA051_PCI_CONTROL__EN_RETRY_TIMER,
		IPA051_PCI_CONTROL + IPA051 );

	/* SLV_TUNING */
	writel( IPA051_SLV_TUNING__SLV_MEMRD_FETCH(15),
		IPA051_SLV_TUNING + IPA051 );

	init_nxp_ipa051_pci(&hose);
}
#endif

int checkboard (void)
{
	/* No board tests implemented yet */

	return 0;
}

#if defined(CONFIG_CMD_NAND) && defined(CONFIG_XIO) && defined(CONFIG_NAND_LEGACY)
extern struct nand_chip nand_dev_desc[CFG_MAX_NAND_DEVICE];
extern unsigned long nand_probe(unsigned long physadr);

void legacy_nand_init (void)
{
	int		count ;
	unsigned long	totalNand = 0 ;
	unsigned long	xioProfile[CFG_MAX_NAND_DEVICE] =
	{
		readl(IPA051 + IPA051_XIO_SEL0_PROF),
		readl(IPA051 + IPA051_XIO_SEL1_PROF),
		readl(IPA051 + IPA051_XIO_SEL2_PROF),
		readl(IPA051 + IPA051_XIO_SEL3_PROF),
		readl(IPA051 + IPA051_XIO_SEL4_PROF)
	} ;

	for (count = 0; count < CFG_MAX_NAND_DEVICE; count ++)
	{
		/* Check if profile 'count' is enabled and configured for NAND FLASH */
		if (  (xioProfile[count] & IPA051_XIO_SELx_PROF__ENAB) &&
		     ((xioProfile[count] & IPA051_XIO_SELx_PROF__TYPE) == IPA051_XIO_SELx_PROF__TYPE_NAND) )
		{
			unsigned long address = xioProfile[count] ;

			/* Calculate the offset from the profile value */
			address = ((address & IPA051_XIO_SELx_PROF__OFFSET_EXT) >> 23) |
				  ((address & IPA051_XIO_SELx_PROF__OFFSET) >> 5 ) ;
			address *= 0x00800000UL ;

			/* Add the address off XIO aperture */
			address += readl(IPA051 + IPA051_BASE18) & IPA051_BASExx__ADDRESS ;

			/* Install the device into the NAND FLASH subsystem */
			nand_probe(address);
		}
	}

	/* Add up the sizes of all know devices */
	for (count = 0; count < CFG_MAX_NAND_DEVICE; count++)
		if (nand_dev_desc[count].ChipID != NAND_ChipID_UNKNOWN)
			totalNand += nand_dev_desc[count].totlen ;

	if (totalNand > 0)
		print_size(totalNand, " total\n");
	else
		printf ("No NAND found\n");
}
#endif /* CONFIG_XIO */

/*
 * Let the standby uP powercycle the board, only if it is there.  Delay for
 * 30ms to give the standby uP some time to do it's work.  Complain if the
 * I2C message could not be sent and restore the previous selected I2C bus.
 */
#ifdef CONFIG_STANDBY_SLAVE_ADDR
void pnx85xx_sby_reset(void)
{
	int	old_bus;
	uchar	i2cMsgBuf[] = { 0x01, 0x1c, 0x04, 0x04, 0x00, 0x00, 0x01, 0x01 };

	old_bus = i2c_get_bus_num();
	i2c_set_bus_num(CONFIG_STANDBY_SLAVE_BUS);

	if (i2c_write(CONFIG_STANDBY_SLAVE_ADDR, 0, 0, i2cMsgBuf, sizeof(i2cMsgBuf)))
		printf("Sby:   force cold boot FAILED!\n");
	udelay(30000);

	i2c_set_bus_num(old_bus);
}
#endif /* CONFIG_STANDBY_SLAVE_ADDR */

#ifdef CONFIG_MISC_INIT_R
/*
 * Miscelaneous platform dependent initialisations after monitor
 * has been relocated into ram
 */

int misc_init_r (void)
{
#ifdef CONFIG_STANDBY_SLAVE_ADDR
	int	i;
	int	old_bus ;
	uchar	i2cMsgBuf[6];
	int	msglen = sizeof(i2cMsgBuf);

	/* Save current bus number */
	old_bus = i2c_get_bus_num();

	/* Select stand-by processor's bus */
	i2c_set_bus_num( CONFIG_STANDBY_SLAVE_BUS );

	/* Synthesize 'bootbatch done'-signal packet */
	i2cMsgBuf[0] = 0x01;
	i2cMsgBuf[1] = 0xe9;
	i2cMsgBuf[2] = 0x03;
	i2cMsgBuf[3] = 0x02;
	i2cMsgBuf[4] = 0x1e;
	i2cMsgBuf[5] = 0x00;

	printf("Sby:   ");
	if (i2c_write(CONFIG_STANDBY_SLAVE_ADDR, 0, 0, i2cMsgBuf, msglen)) {
		printf("'bootbatch done'-signal FAILED!\n");
		return 1;
	}
	printf("'bootbatch done'-signal ok.\n");
	
	for (i = 0; i < 65535; i++)
		; /* nil */

	/* Synthesize 'disable hearbeat'-signal packet */
	i2cMsgBuf[0] = 0x1;
	i2cMsgBuf[1] = 0xf3;
	i2cMsgBuf[2] = 0x03;
	i2cMsgBuf[3] = 0x01;
	i2cMsgBuf[4] = 0;
	i2cMsgBuf[5] = 0;

	printf("Sby:   ");
	if (i2c_write(CONFIG_STANDBY_SLAVE_ADDR, 0, 0, i2cMsgBuf, msglen)) {
		printf("'disable hearbeat'-signal FAILED!\n");
		return 1;
	}
	printf("'disable hearbeat'-signal ok.\n");

	/* Restore previous bus number */
	i2c_set_bus_num(old_bus);
#endif	/*  CONFIG_STANDBY_SLAVE_ADDR */

	return (0);
}
#endif /* CONFIG_MISC_INIT_R */


/*
 * dev_addr	I2C address of EEPROM device to enable.
 * state	-1: deliver current state
 *		 0: disable write
 *		 1: enable write
 * return	-1: wrong device address
 *		 0: dis-/en- able done
 *		0/1: current state if state was -1.
 */
#if defined(CFG_EEPROM_WREN) && defined(CONFIG_TV522_32_SAMSUNG)
int eeprom_write_enable(unsigned dev_addr, int state)
{
	int     old_bus;
	uchar   i2cMsgBuf[6];

	if (dev_addr != CFG_I2C_EEPROM_ADDR)
		return -1;

	/*
	 * Save the currently selected I2C bus number and select the one
	 * the Stby uP is on.
	 */
	old_bus = i2c_get_bus_num();
	i2c_set_bus_num(CONFIG_STANDBY_SLAVE_BUS);

	switch (state) {
	default:
		return -1;

	case -1:
		return -1;

	case 0:
	case 1:
		i2cMsgBuf[0] = 0x01;
		i2cMsgBuf[1] = 0x04;
		i2cMsgBuf[2] = 0x00;
		i2cMsgBuf[3] = 0x02;
		i2cMsgBuf[4] = 0x27;
		i2cMsgBuf[5] = state ? 0x1 : 0x0;
		break;
	}

	if (i2c_write(CONFIG_STANDBY_SLAVE_ADDR, 0, 0, i2cMsgBuf, sizeof(i2cMsgBuf)))
		return -1;

	i2c_set_bus_num(old_bus);

	return 0;
}
#endif	/* CFG_EEPROM_WREN && CONFIG_TV522_32_SAMSUNG */

int board_eth_init(bd_t *bis)
{
	int c = 0;

#if defined(CONFIG_CMD_USB) && defined(CONFIG_CMD_NET)	
	extern int usb_eth_init(bd_t *bis);
    c += usb_eth_init(bis);
#endif

#ifdef CONFIG_CMD_PCI
	c += pci_eth_init(bis);
#endif
	
	return c;
}

// Start TM
#define	TM32_CTL_RUN(tm) \
	writel( TM32_CTL__ACTION__MASK(TM32_CTL__ACTION__START) | TM32_CTL__JMP_TGT_ENABLE, \
			MMIO_BASE + OFFS_TM##tm + OFFS_TM32_CTL );
#define	TM32_CTL_HALT(tm) \
	writel( TM32_CTL__ACTION__MASK(TM32_CTL__ACTION__RESET), (MMIO_BASE + OFFS_TM##tm + OFFS_TM32_CTL));
#define	TM32_START_ADDR_SET(tm,addr)  writel(addr, MMIO_BASE + OFFS_TM##tm + OFFS_TM32_START_ADDR);
#define	TM32_DRAM_HI_SET(tm,addr)     writel(addr, MMIO_BASE + OFFS_TM##tm + OFFS_TM32_DRAM_HI);
#define	TM32_DRAM_LO_SET(tm,addr)     writel(addr, MMIO_BASE + OFFS_TM##tm + OFFS_TM32_DRAM_LO);
#define	TM32_DRAM_CLIMIT_SET(tm,addr) writel(addr, MMIO_BASE + OFFS_TM##tm + OFFS_TM32_DRAM_CLIMIT);

inline unsigned int get_mmio_cpu(	int	cpu) {
	switch (cpu) {
	case 1:
		return (OFFS_TM1);
	case 2:
		return (OFFS_TM2);
	case 3:
		return (OFFS_TM3);
	default:
		printf(	"ERROR: (%s) invalid cpu number: %d\n", __FUNCTION__, cpu);
		return (0);
	}
}

int dsp_stop(int cpu) {
	/* stop the specified cpu by clearing the 'run' bit in the
	 * tm32_ctl register.. */

	unsigned int	mmio_offs_cpu;
	unsigned int	tm32_ctl;

	/* get mmio offset for cpu */
	mmio_offs_cpu = get_mmio_cpu(cpu);

	if (mmio_offs_cpu == 0)
		return (~0);

	/* read current tm32_ctl register */
	tm32_ctl = readl(MMIO_BASE + mmio_offs_cpu + OFFS_TM32_CTL);

	/* mask all action bits */
	tm32_ctl &= ~(TM32_CTL__ACTION__MASK(-1));

	/* set the appropriate action bits */
	tm32_ctl |= TM32_CTL__ACTION__MASK(TM32_CTL__ACTION__RESET);
	mydebug("tm32_ctl=0x%08x\n", tm32_ctl);

	/* read/modify/write the tm32_ctl register */
	writel(	tm32_ctl, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_CTL);

	return (0);
}

int dsp_start( int cpu, unsigned int	start_addr) {
	unsigned int	mmio_offs_cpu;
	unsigned int	tm32_ctl;

	/* get mmio offset for cpu */
	mmio_offs_cpu = get_mmio_cpu(cpu);

	if (mmio_offs_cpu == 0)
		return (~0);

	if (gd->ram_size == (phys_size_t)0x08000000) {
		// 128 MB
		myinfo("128mb ddr\n");
		writel(0x00000000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_DRAM_LO);
		writel(0x08000000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_DRAM_HI);
		writel(0x07FA0000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_DRAM_CLIMIT);
		writel(0x1C000000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_APERT1_LO);
		writel(0x20000000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_APERT1_HI);
	} else if (gd->ram_size == (phys_size_t)0x10000000) {
		// 256 MB
		myinfo("256mb ddr\n");
		writel(0x00000000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_DRAM_LO);
		writel(0x10000000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_DRAM_HI);
		writel(0x0FFA0000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_DRAM_CLIMIT);
		writel(0x1C000000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_APERT1_LO);
		writel(0x20000000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_APERT1_HI);
	} else {
		//YW: not implemented yet; use 256MB layout
		printf("unsupported DDR\n");
		writel(0x00000000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_DRAM_LO);
		writel(0x08000000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_DRAM_HI);
		writel(0x07FA0000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_DRAM_CLIMIT);
		writel(0x1C000000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_APERT1_LO);
		writel(0x20000000, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_APERT1_HI);
	}	

	// TM_Misc
	writel(2, 0x1be63708);
	
	// Start addr
	writel(start_addr, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_START_ADDR);

	// START	
	tm32_ctl  = readl(MMIO_BASE + mmio_offs_cpu + OFFS_TM32_CTL);
	tm32_ctl &= ~(TM32_CTL__ACTION__MASK(-1));
	tm32_ctl |= TM32_CTL__ACTION__MASK(TM32_CTL__ACTION__START);

	mydebug("tm32_ctl=0x%08x\n", tm32_ctl);
	writel(	tm32_ctl, MMIO_BASE + mmio_offs_cpu + OFFS_TM32_CTL);

	/* start cpu */
	return (0);
}

