/*
 * (C) Copyright 2006
 * NXP Semiconductors,
 * Robert Delien robert.delien@nxp.com, Hans Zuidam hans.zuidam@nxp.com
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 *
 */

#include <common.h>

#ifdef CONFIG_NXP_I2C

#include <asm/io.h>
#include <i2c.h>

#ifdef CONFIG_NXP_I2C_IP0105
#  include "nxp_i2c_ip0105.h"
#endif /* CONFIG_NXP_I2C_IP0105 */
#ifdef CONFIG_NXP_I2C_IP3203
#  include "nxp_i2c_ip3203.h"
#endif /* CONFIG_NXP_I2C_IP3203 */

#undef NXP_I2C_DEBUG

#ifdef NXP_I2C_DEBUG
	#warning NXP I2C Debug enabled
	#define NXP_I2C_DBG(f, ...)	printf(f, ## __VA_ARGS__ );
#else
	#warning NXP I2C Debug disabled
	#define NXP_I2C_DBG(f, ...)
#endif

/*-----------------------------------------------------------------------
 * Definitions
 */

#ifdef CONFIG_TV55X
#define MAX_I2C_BUSSES 4
#else
#define MAX_I2C_BUSSES 3
#endif
typedef struct
{
	unsigned long	address ;
	int		(*init)		(unsigned long, int, int) ;
	int		(*probe)	(unsigned long, uchar) ;
	int		(*read)		(unsigned long, uchar, uint, int, uchar*, int) ;
	int		(*write)	(unsigned long, uchar, uint, int, uchar*, int) ;
	uchar		(*reg_read)	(unsigned long, uchar, uchar) ;
	void		(*reg_write)	(unsigned long, uchar, uchar, uchar) ;
	int		(*set_speed)	(unsigned long, unsigned int) ;
	unsigned int	(*get_speed)	(unsigned long) ;
} i2cbus ;


/*-----------------------------------------------------------------------
 * Local functions
 */
DECLARE_GLOBAL_DATA_PTR;

#if defined(CONFIG_I2C_MULTI_BUS)
/* Initialize the bus pointer to whatever one the EEPROM is on.
 * Default is bus 0.  This is necessary because the DDR initialization
 * runs from ROM, and we can't switch buses because we can't modify
 * the global variables.  Taken from cpu/ppc4xx/i2c.c
 */
#ifdef CFG_EEPROM_I2C_BUS
static unsigned int curr_bus __attribute__ ((section ("data"))) = CFG_EEPROM_I2C_BUS;
static unsigned int last_bus __attribute__ ((section ("data"))) = CFG_EEPROM_I2C_BUS;
#else
static unsigned int curr_bus __attribute__ ((section ("data"))) = 0;
static unsigned int last_bus __attribute__ ((section ("data"))) = 0;
#endif
#endif /* CONFIG_I2C_MULTI_BUS */

/* WARNING: We're ignoring the warning about globals because we have RAM
 * from the moment we've started */
static i2cbus	i2c_buses[MAX_I2C_BUSSES] ;

/*=====================================================================*/
/*                         Public Functions                            */
/*=====================================================================*/

/*-----------------------------------------------------------------------
 * Initialization
 */
void i2c_init (int speed, int slaveaddr)
{
	/*
	 * WARNING: Do NOT save speed in a static variable: if the
	 * I2C routines are called before RAM is initialized (to read
	 * the DIMM SPD, for instance), RAM won't be usable and your
	 * system will crash.
	 */
	unsigned long	addr;
#ifdef	CFG_MMIO_WARN_INV_MODID_RANGES
	unsigned long	corrupt_addr_start = 0x00000000;
	unsigned long	corrupt_addr_size = 0;
#endif	/* CFG_MMIO_WARN_INV_MODID_RANGES */
#ifdef	CFG_MMIO_WARN_INV_MODID_COUNT
	int		num_invalid_module_ids = 0;
#endif	/* CFG_MMIO_WARN_INV_MODID_COUNT */
	int		new_bus = 0;
	unsigned long	module_id = 0 ;
        unsigned int	mod_id = 0;
	unsigned int	mod_rev = 0;
	unsigned int	mod_size = 0;
        
	addr = (MMIO_BASE) ;

#ifdef CONFIG_NXP_I2C_IP3203
	/* The I2C blocks are powered down in hotboot handler.
	 * Access to I2C registers will not work, if the I2C block is powered down.
	 * Hence to read the module IDs, power up the I2C blocks 
	 * */ 	
	unsigned long	powerdown_reg = 0 ;
        writel(powerdown_reg, addr + 0x45FF4);
        writel(powerdown_reg, addr + 0x46FF4);
        writel(powerdown_reg, addr + 0x48FF4);
#ifdef CONFIG_TV55X
        writel(powerdown_reg, addr + 0x49FF4);
#endif
#endif        
        
        while ( (new_bus < MAX_I2C_BUSSES) &&
		(addr < (MMIO_BASE + MMIO_SIZE)) ) {
		
		module_id	= readl(addr + 0x0FFC);
                mod_id = 0;
		mod_rev = 0;
		mod_size = 0;

		NXP_I2C_DBG("Probe %08x = %08x", addr, module_id);

		/* these 2 scenarios should not occur.. */
		if (module_id == 0x00000000) {
			/* empty mmio block encountered, skip it! */
			addr += 0x1000;

			NXP_I2C_DBG(" - no IP found\n");
			continue;
		}
		
		if (module_id == 0xffffffff) {
#ifdef	CFG_MMIO_WARN_INV_MODID_RANGES
			/* `corrupt' mmio block encountered, administer it */
			if (corrupt_addr_start == 0x00000000)
				corrupt_addr_start = addr;
			/* keep track of `corrupted' mmio space */
			corrupt_addr_size += 0x1000;
#endif	/* CFG_MMIO_WARN_INV_MODID_RANGES */
#ifdef	CFG_MMIO_WARN_INV_MODID_COUNT
			++num_invalid_module_ids;
#endif	/* CFG_MMIO_WARN_INV_MODID_COUNT */
			/* and skip it! */
			addr += 0x1000;

			NXP_I2C_DBG(" - no IP found\n");

			continue;
		}

#ifdef	CFG_MMIO_WARN_INV_MODID_RANGES
		/* if we get here, we are at a valid mmio module in mmio space
		 * if we encountered some `corrupt' mmio space, warn user about this */
		if (
			(corrupt_addr_start != 0x00000000) &&
			(corrupt_addr_size != 0)) {
			printf("WARNING: 0x%lx bytes of corrupt MMIO space encountered at 0x%p\n",
				corrupt_addr_size,
				(void*)corrupt_addr_start);
			/* reset the `corrupted' area administration */
			corrupt_addr_start = 0x00000000;
			corrupt_addr_size = 0;
		}
#endif	/* CFG_MMIO_WARN_INV_MODID_RANGES */

		mod_id = (module_id & 0xffff0000) >> 16;
		mod_rev = (module_id & 0x0000ff00) >> 8;
		mod_size = (module_id & 0x000000ff) >> 0;
		mod_size += 1;
		mod_size *= 0x1000;
		
		NXP_I2C_DBG("- found IP\n");

		switch (mod_id) {
                case 0x1067:
			/*
 			 * Change skipsize for MIPSBROM IP size aperture is not correct
			 */
      			mod_size = 0x8000;
			break;
#ifdef CONFIG_NXP_I2C_IP0105
		case 0x0105:
			i2c_buses[new_bus].address	= addr ;
			i2c_buses[new_bus].init		= &nxp_i2c_ip0105_init ;
			i2c_buses[new_bus].probe	= &nxp_i2c_ip0105_probe ;
			i2c_buses[new_bus].read		= &nxp_i2c_ip0105_read ;
			i2c_buses[new_bus].write	= &nxp_i2c_ip0105_write ;
			i2c_buses[new_bus].reg_read	= &nxp_i2c_ip0105_reg_read ;
			i2c_buses[new_bus].reg_write	= &nxp_i2c_ip0105_reg_write ;
			i2c_buses[new_bus].set_speed	= &nxp_i2c_ip0105_set_speed ;
			i2c_buses[new_bus].get_speed	= &nxp_i2c_ip0105_get_speed ;
			break ;
#endif /* CONFIG_NXP_I2C_IP0105 */
#ifdef CONFIG_NXP_I2C_IP3203
		case 0x3203:
			i2c_buses[new_bus].address	= addr ;
			i2c_buses[new_bus].init		= &nxp_i2c_ip3203_init ;
			i2c_buses[new_bus].probe	= &nxp_i2c_ip3203_probe ;
			i2c_buses[new_bus].read		= &nxp_i2c_ip3203_read ;
			i2c_buses[new_bus].write	= &nxp_i2c_ip3203_write ;
			i2c_buses[new_bus].reg_read	= &nxp_i2c_ip3203_reg_read ;
			i2c_buses[new_bus].reg_write	= &nxp_i2c_ip3203_reg_write ;
			i2c_buses[new_bus].set_speed	= &nxp_i2c_ip3203_set_speed ;
			i2c_buses[new_bus].get_speed	= &nxp_i2c_ip3203_get_speed ;
			break ;
#endif /* CONFIG_NXP_I2C_IP3203 */
		default:
			i2c_buses[new_bus].address	= 0 ;
			break ;
		}

		if (i2c_buses[new_bus].address) {
			if( i2c_buses[new_bus].init( i2c_buses[new_bus].address, speed, slaveaddr) == 0 ) {
				last_bus = new_bus ;
				new_bus++ ;
			}
		}

		/* Increment address pointer to the next IP block */
#ifdef CONFIG_TV55X_I2C_PATCH 
		addr += 0x1000;
#else
		addr += mod_size ;
#endif

	}
	if (last_bus != -1)
		curr_bus = 0 ;
		
#ifdef	CFG_MMIO_WARN_INV_MODID_COUNT
	if (num_invalid_module_ids != 0)
		printf("WARNING: encountered %d invalid module ids during MMIO scan\n",
			num_invalid_module_ids);
#endif	/* CFG_MMIO_WARN_INV_MODID_COUNT */
}

/*-----------------------------------------------------------------------
 * Probe to see if a chip is present.  Also good for checking for the
 * completion of EEPROM writes since the chip stops responding until
 * the write completes (typically 10mSec).
 */
int i2c_probe(uchar addr)
{
	if (i2c_buses[curr_bus].address)
		return( i2c_buses[curr_bus].probe( i2c_buses[curr_bus].address, addr ) ) ;
	else
		return (-1);
}

/*-----------------------------------------------------------------------
 * Read bytes
 */
int  i2c_read(uchar chip, uint addr, int alen, uchar *buffer, int len)
{
	if (i2c_buses[curr_bus].address)
		return( i2c_buses[curr_bus].read( i2c_buses[curr_bus].address, chip, addr, alen, buffer, len ) ) ;
	else
		return (-1);
}

/*-----------------------------------------------------------------------
 * Write bytes
 */
int  i2c_write(uchar chip, uint addr, int alen, uchar *buffer, int len)
{
	if (i2c_buses[curr_bus].address)
		return( i2c_buses[curr_bus].write( i2c_buses[curr_bus].address, chip, addr, alen, buffer, len ) ) ;
	else
		return (-1);
}

#if 0
/*-----------------------------------------------------------------------
 * Read a register
 */
uchar i2c_reg_read(uchar i2c_addr, uchar reg)
{
	if (i2c_buses[curr_bus].address)
		return( i2c_buses[curr_bus].reg_read( i2c_buses[curr_bus].address, i2c_addr, reg ) ) ;
	else
		return (-1);
}

/*-----------------------------------------------------------------------
 * Write a register
 */
void i2c_reg_write(uchar i2c_addr, uchar reg, uchar val)
{
	if (i2c_buses[curr_bus].address)
		i2c_buses[curr_bus].reg_write( i2c_buses[curr_bus].address, i2c_addr, reg, val ) ;
}
#endif

int i2c_set_bus_num(unsigned int bus)
{
	if (bus > last_bus)
		return(-1);

	curr_bus = bus ;

	return(0);
}

unsigned int i2c_get_bus_num(void)
{
	return(curr_bus);
}

int i2c_set_bus_speed(unsigned int speed)
{
	if (i2c_buses[curr_bus].address)
		return( i2c_buses[curr_bus].set_speed( i2c_buses[curr_bus].address, speed ) ) ;
	else
		return (-1);
}

unsigned int i2c_get_bus_speed(void)
{
	if (i2c_buses[curr_bus].address)
		return( i2c_buses[curr_bus].get_speed( i2c_buses[curr_bus].address ) ) ;
	else
		return (-1);
}

#endif	/* CONFIG_NXP_I2C */
