/*
 * Device driver for NAND flash connected IP_2070 EFM.
 *
 * Copyright (C) 2008 NXP B.V.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Revision history
 * Version      Date          Remarks		
 * 0.0.1	 		20081211		Draft-Initial version
 */

#include <common.h>
#include <malloc.h>
#include <watchdog.h>
#include <linux/err.h>
#include <linux/mtd/compat.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/nand_ecc.h>

#include <asm/io.h>
#include <asm/errno.h>

#ifdef CONFIG_JFFS2_NAND
#include <jffs2/jffs2.h>
#endif

/* NAND specific files */
#include "nx_nand_ip2070.h"
#include <linux/nx_dmac_ip1902.h>

extern void flush_dcache_range(ulong start_addr, ulong stop);
extern void invalidate_dcache_range(ulong start_addr, ulong stop);

#ifdef CONFIG_MTD_NX_NAND_HWECC
/**
*	OOB structure
*/
/* For LPF */
static struct nand_ecclayout nx_nand_oob_128 = {
	.eccbytes = 96,
	.eccpos = {
		   4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15,
		   20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
			 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
			 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63,
			 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79,
			 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95,
			 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111,
			 116, 117, 118, 119, 120, 121, 122, 123, 124, 125,126, 127
			 },
			
	.oobfree = {
		{.offset = 2,
		 .length = 2},
		{.offset = 16,
		 .length = 4},
		{.offset = 32,
		 .length = 4},
		{.offset = 48,
		 .length = 4},
		{.offset = 64,
		 .length = 4},
		{.offset = 80,
		 .length = 4},
		{.offset = 96,
		 .length = 4},
		{.offset = 112,
		 .length = 4},
	}
};

static struct nand_ecclayout nx_nand_oob_64 = {
	.eccbytes = 48,
	.eccpos = {
		   4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15,
		   20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
			 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47,
			 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63},
	.oobfree = {
		{.offset = 2,
		 .length = 2},
		{.offset = 16,
		 .length = 4},
		{.offset = 32,
		 .length = 4},
		{.offset = 48,
		 .length = 4},
	}
};

/* For SPF */
static struct nand_ecclayout nx_nand_oob_16 = {
	.eccbytes = 16,
	.eccpos = {
				4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
	.oobfree = {
		{.offset = 0,
		 .length = 4},
		}
};

/**
*	Flash based BBT information
*/
static uint8_t nx_bbt_pattern[] = {'N', 'X', 'P' };
static uint8_t nx_mirror_pattern[] = {'P', 'X', 'N' };

static struct nand_bbt_descr nx_bbt_main = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	0,
	.len = 3,
	.veroffs = 3,
	.maxblocks = 4,
	.pattern = nx_bbt_pattern
};

static struct nand_bbt_descr nx_bbt_mirror = {
	.options = NAND_BBT_LASTBLOCK | NAND_BBT_CREATE | NAND_BBT_WRITE
		| NAND_BBT_2BIT | NAND_BBT_VERSION | NAND_BBT_PERCHIP,
	.offs =	0,
	.len = 3,
	.veroffs = 3,
	.maxblocks = 4,
	.pattern = nx_mirror_pattern
};

#endif

/**
* NAND control structure
*/
struct nx_nand_ctrl *nx_nc = NULL;

#ifdef CONFIG_NX_MEMCPY
extern void nx_memcpy( void* pDest, void* pSrc, unsigned int length);  
#endif

/*------------------------------------------------------------------------------------
* Internal functions 
--------------------------------------------------------------------------------------*/
#ifdef CONFIG_NAND_PERF_MEAS
#define	TSU_COUNTER_REG		(volatile unsigned int *)(MMIO_BASE + 0x0014C0C0)
#endif

/**
* nx_nand_dmac_init - Configure the DMAC scatter gather list
* @nc: NAND control structure
* @read: Read command
* @req: Request structure
* @stgt: Scatter gather array
*
* Initialise the DMAC scatter gather list
*/
static inline void nx_nand_dmac_init(struct nx_nand_ctrl *nc, 
			uint32_t cmd, nx_dmac_tfr_t *req, nx_dmac_stgt_t *stgt)
{
	int i;
#ifdef CONFIG_NAND_PERF_MEAS
	//volatile uint32_t sc1, sc2, sc3, sc4, sc5, sc6, sc7, sc8;
#endif

#ifdef CONFIG_NAND_PERF_MEAS
	//sc1 = readl(TSU_COUNTER_REG);
#endif
	if(cmd) {
			
#ifdef CONFIG_NAND_PERF_MEAS
		//sc2 = readl(TSU_COUNTER_REG);
#endif
		/* Read Main Area */
		for(i=0; i < nc->num_blks; i++) {
			stgt[i].src_addr = NX_NAND_AHB_BUF;     
	  	stgt[i].dst_addr = CPHYSADDR(nc->dmabuf + (i * NX_NAND_BLK_SIZE));
  		stgt[i].tfr_size = NX_NAND_BLK_SIZE >> 2;    
			stgt[i].flowctl = nx_dmac_per2mem_dma;         
 			stgt[i].src_per = 0;		          
  		stgt[i].dst_per = 0;
			stgt[i].src_ahb = 1;		          /* Source AHB master 1 */
			stgt[i].dst_ahb = 0;              /* Dest AHB master 0 */
			stgt[i].src_inc = 1;
			stgt[i].dst_inc = 1;
			stgt[i].src_brst = nx_dmac_128;
			stgt[i].dst_brst = nx_dmac_128;
			stgt[i].src_width = nx_dmac_width_32;
			stgt[i].dst_width = nx_dmac_width_32;
		}

#ifdef CONFIG_NAND_PERF_MEAS
	//sc3 = readl(TSU_COUNTER_REG);
#endif
	
		/* Read OOB area */
		stgt[i].src_addr = NX_NAND_AHB_BUF + NX_NAND_BLK_SIZE;     
  	stgt[i].dst_addr = CPHYSADDR(nc->dmabuf + (i * NX_NAND_BLK_SIZE));
  	stgt[i].tfr_size = nc->mtd->oobsize >> 2;    
		stgt[i].flowctl = nx_dmac_per2mem_dma;         
 		stgt[i].src_ahb = 1;		          /* Source AHB master 1 */
  	stgt[i].dst_ahb = 0;              /* Dest AHB master 0 */
		stgt[i].src_per = 0;		          
 		stgt[i].dst_per = 0;
 		stgt[i].src_inc = 1;
 		stgt[i].dst_inc = 1;
   	stgt[i].src_width = nx_dmac_width_32;
    stgt[i].dst_width = nx_dmac_width_32;
		if(nc->mtd->oobsize == 128) {
 			stgt[i].src_brst = nx_dmac_32;
 			stgt[i].dst_brst = nx_dmac_32;
		}
		else if(nc->mtd->oobsize == 64) {
 			stgt[i].src_brst = nx_dmac_16;
 			stgt[i].dst_brst = nx_dmac_16;
		}
		else {
 			stgt[i].src_brst = nx_dmac_4;
 			stgt[i].dst_brst = nx_dmac_4;
		}
#ifdef CONFIG_NAND_PERF_MEAS
		//sc4 = readl(TSU_COUNTER_REG);
#endif
	}
	else { 
		
		/* Write OOB area */
		i=0;	
		stgt[i].dst_addr = NX_NAND_AHB_BUF + NX_NAND_BLK_SIZE;     
  	stgt[i].src_addr = CPHYSADDR(nc->dmabuf + (nc->num_blks * NX_NAND_BLK_SIZE));
  	stgt[i].tfr_size = nc->mtd->oobsize >> 2;    
		stgt[i].flowctl = nx_dmac_mem2per_dma;         
 		stgt[i].src_ahb = 0;		          /* Source AHB master 0 */
 		stgt[i].dst_ahb = 1;              /* Dest AHB master 1 */
		stgt[i].src_per = 0;		          
 		stgt[i].dst_per = 0;
 		stgt[i].src_inc = 1;
 		stgt[i].dst_inc = 1;
    stgt[i].src_width = nx_dmac_width_32;
    stgt[i].dst_width = nx_dmac_width_32;
		if(nc->mtd->oobsize == 128) {
 			stgt[i].src_brst = nx_dmac_32;
 			stgt[i].dst_brst = nx_dmac_32;
		}
		else if(nc->mtd->oobsize == 64) {
 			stgt[i].src_brst = nx_dmac_16;
 			stgt[i].dst_brst = nx_dmac_16;
		}
		else {
 			stgt[i].src_brst = nx_dmac_4;
 			stgt[i].dst_brst = nx_dmac_4;
		}
			
		/* Write Main area */
		for(i=1; i< (nc->num_blks+1); i++) {
  		stgt[i].src_addr = CPHYSADDR(nc->dmabuf + ( (i-1) * NX_NAND_BLK_SIZE));
			stgt[i].dst_addr = NX_NAND_AHB_BUF;     
  		stgt[i].tfr_size = NX_NAND_BLK_SIZE >> 2;    
			stgt[i].flowctl = nx_dmac_mem2per_dma;         
 			stgt[i].src_per = 0;		         
  		stgt[i].dst_per = 0;
 			stgt[i].src_ahb = 0;		          /* Source AHB master 0 */
  		stgt[i].dst_ahb = 1;              /* Dest AHB master 1 */
  		stgt[i].src_inc = 1;
  		stgt[i].dst_inc = 1;
  		stgt[i].src_brst = nx_dmac_128;
  		stgt[i].dst_brst = nx_dmac_128;
    	stgt[i].src_width = nx_dmac_width_32;
    	stgt[i].dst_width = nx_dmac_width_32;
		}
	}
 
	req->num_reqs = nc->num_blks + 1;
	req->req = &stgt[0];
#ifdef CONFIG_NAND_PERF_MEAS
	//sc5 = readl(TSU_COUNTER_REG);
	//printk(KERN_INFO "sc1: %d sc2: %d sc3: %d sc4: %d sc5: %d \r\n", sc1, sc2, sc3, sc4, sc5);
#endif
}

/**
* nx_nand_cmd_addr - Send cmd & address cycles to chip
* @nc: NAND control structure
* @cmd: Command to be send
* @data: command or data
* @last: Last cycle
*
* Send command & address cycles to chip for small page chips
*/
static inline void nx_nand_cmd_addr(struct nx_nand_ctrl *nc, uint32_t cmd,
				     uint32_t data, int last)
{
	uint32_t cmd_addr = 0;
	
	/* Send command & address to chip */
	cmd_addr = (0 << (nc->slotid + CMD_FIFO_CE_POS));
	cmd_addr |= ((cmd & CMD_FIFO_CYCLE_MSK) << CMD_FIFO_CYCLE_POS);
	cmd_addr |= ((last & CMD_FIFO_LAST_MSK) << CMD_FIFO_LAST_POS);
	cmd_addr |= (data & CMD_FIFO_IO_MSK);
	writel(cmd_addr, (nx_nc->ctrl_base + CMD_ADDR_FIFO_OFFSET));	

	return;
}

/**
* nx_nand_erase - Erase the block
* @nc: NAND control structure
*
* Erase the block
*/
static inline void nx_nand_erase(struct nx_nand_ctrl *nc)
{
	struct mtd_info *mtd=nc->mtd;		
	struct nand_chip *chip = mtd->priv;
	uint32_t intr;
	volatile uint32_t int_status;
	int addr;

	/* Enable READY interupt */
	intr = (1 << (NX_NAND_INT_READY_START + nc->slotid));
	writel(intr, (nx_nc->ctrl_base + INT_SET_ENABLE_OFFSET));
		
	/* Clear interrupt */
	writel(intr, (nx_nc->ctrl_base + INT_CLR_STATUS_OFFSET));
		
	/* Send address ERASE1 command */
	nx_nand_cmd_addr(nc, 1, NAND_CMD_ERASE1, 0);
		
	/* Send address cycles */
	addr = nc->cur_page & NX_NAND_SP_ADDR_MASK;
	nx_nand_cmd_addr(nc, 0, addr, 0);
	
	addr = (nc->cur_page >> 8) & NX_NAND_SP_ADDR_MASK;
	nx_nand_cmd_addr(nc, 0, addr, 0);
	
	if(nc->lb_chip) {
		/* LPF */	
		if (chip->chipsize > (1 << 27)) {
			addr = (nc->cur_page >> 16) & NX_NAND_SP_ADDR_MASK;
			nx_nand_cmd_addr(nc, 0, addr, 0);
		}
	}
	else {
		/* SPF */	
		if (chip->chipsize > (32 << 20)) {
			addr = (nc->cur_page >> 16) & NX_NAND_SP_ADDR_MASK;
			nx_nand_cmd_addr(nc, 0, addr, 0);
		}
	}
	
	/* Send ERASE2 command */
	nx_nand_cmd_addr(nc, 1, NAND_CMD_ERASE2, 1);
		
	/* Wait for ready interrupt */
	set_timeout(PNX8XXX_TIMEOUT);
	int_status = readl(nx_nc->ctrl_base + INT_STATUS_OFFSET);		
	while (!(int_status & (1 << (NX_NAND_INT_READY_START + nc->slotid)))) {
		if(did_timeout()) {
			printk("TIMEOUT:Erase \r\n");
			return;
		}
		int_status = readl(nx_nc->ctrl_base + INT_STATUS_OFFSET);		
	}

	/* Clear interrupt */
	writel(intr, (nx_nc->ctrl_base + INT_CLR_STATUS_OFFSET));
		
	/* Disable interrupt */
	writel(intr, (nx_nc->ctrl_base + INT_CLR_ENABLE_OFFSET));

	return;
}

/**
* nx_nand_read - Read page into driver buffer
* @nc: NAND control structure
*
* Read a page into driver buffer
*/
static inline void nx_nand_read(struct nx_nand_ctrl *nc)
{
	struct mtd_info *mtd=nc->mtd;		
	struct nand_chip *chip = mtd->priv;
	uint32_t intr, page_cfg=0, int_ecc;
	volatile uint32_t int_status;
	int status, addr, chanid, i=0;
	nx_dmac_tfr_t	req;
	nx_dmac_stgt_t	stgt[5];
	int do_cached_read = NAND_CANCACHEDREAD(chip) ? 1 : 0;
#ifdef CONFIG_NAND_PERF_MEAS
	//volatile uint32_t s1, s2, s3, s4, s5, s6, s7, s8;
#endif

#ifdef CONFIG_NAND_PERF_MEAS
	//s1 = readl(TSU_COUNTER_REG);
#endif

	/* Init DMAC Scatter gather list */
	nx_nand_dmac_init(nc, 1, &req, stgt);

#ifdef CONFIG_NAND_PERF_MEAS
	//s2 = readl(TSU_COUNTER_REG);
#endif
 
	/* Set DMAC Flow control */
	writel(DMA_CTRL_PER_2_MEM, (nc->ctrl_base + DMA_CTRL_OFFSET));
		
	/* Enable the SEQ READ PAGE DONE interrupt */
	intr = NX_NAND_INT_SEQ_READ; 
	writel(intr, (nc->ctrl_base + INT_SET_ENABLE_OFFSET));
		
	/* Clear interrupt */
	writel(intr, (nc->ctrl_base + INT_CLR_STATUS_OFFSET));
	
	/* Start DMAC */
	chanid = nx_dmac_tfr(&req);
	if(chanid < 0) {
		printk( "nx_nand: NAND_READ0 DMAC config \r\n");	
		return;
	}
	
	if(nc->cur_cmd == NAND_CMD_READ0) {

		/* Page operation */
#ifdef CONFIG_NUMONYX_70NM_CACHEREAD_PATCH
		page_cfg = PAGE_RW_OOB_MSK | PAGE_RW_PAGE_READ_MSK;
#else		
		if(!do_cached_read) {	
			page_cfg = PAGE_RW_OOB_MSK | PAGE_RW_PAGE_READ_MSK;
		}
#endif
		if(nc->aes) {
			page_cfg |= PAGE_RW_AES_MSK;
		}
		if(chip->ecc.mode == NAND_ECC_HW) {
			page_cfg |= PAGE_RW_ECC_MSK;
		}
		writel(page_cfg, (nc->ctrl_base + PAGE_RW_OFFSET));
	
		/* Send READ0 command */
		nx_nand_cmd_addr(nc, 1, NAND_CMD_READ0, 0);

		/* Send address cycles */
		addr = nc->cur_col & NX_NAND_SP_ADDR_MASK;
		nx_nand_cmd_addr(nc, 0, addr, 0);

		if(nc->lb_chip) {
			addr = (nc->cur_col >> 8) & NX_NAND_SP_ADDR_MASK;
			nx_nand_cmd_addr(nc, 0, addr, 0);
		}	
		
		addr = nc->cur_page & NX_NAND_SP_ADDR_MASK;
		nx_nand_cmd_addr(nc, 0, addr, 0);
	
		if(nc->lb_chip) {
			/* LPF */	
			addr = (nc->cur_page >> 8) & NX_NAND_SP_ADDR_MASK;
			nx_nand_cmd_addr(nc, 0, addr, 0);
		
			if (chip->chipsize > (1 << 27)) {
				addr = (nc->cur_page >> 16) & NX_NAND_SP_ADDR_MASK;
				nx_nand_cmd_addr(nc, 0, addr, 0);
			}
	
			/* Send READSTART command */
#ifdef CONFIG_NUMONYX_70NM_CACHEREAD_PATCH
			if(do_cached_read) {
				nx_nand_cmd_addr(nc, 1, NAND_CMD_CACHEREAD, 1);
			}
			else {
				nx_nand_cmd_addr(nc, 1, NAND_CMD_READSTART, 1);
			}
#else
			nx_nand_cmd_addr(nc, 1, NAND_CMD_READSTART, 1);
#endif			

#ifndef CONFIG_NUMONYX_70NM_CACHEREAD_PATCH
			if(do_cached_read) {	
				/* Page operation */		
				page_cfg |= PAGE_RW_OOB_MSK | PAGE_RW_PAGE_READ_MSK;
				writel(page_cfg, (nc->ctrl_base + PAGE_RW_OFFSET));

				/* Send CACHED READ cmd */
				nx_nand_cmd_addr(nc, 1, NAND_CMD_CACHEREAD, 1);
			}
#endif		
		}
		else {
			/* SPF */	
			if (chip->chipsize > (32 << 20)) {
				/* Chip size > 64MB */	
				addr = (nc->cur_page >> 8) & NX_NAND_SP_ADDR_MASK;
				nx_nand_cmd_addr(nc, 0, addr, 0);

				addr = (nc->cur_page >> 16) & NX_NAND_SP_ADDR_MASK;
				nx_nand_cmd_addr(nc, 0, addr, 1);
			}
			else {
				/* Chip size < 64MB */	
				addr = (nc->cur_page >> 8) & NX_NAND_SP_ADDR_MASK;
				nx_nand_cmd_addr(nc, 0, addr, 1);
			}
		}

	}
	else {
		/* Page operation */		
		page_cfg = PAGE_RW_OOB_MSK | PAGE_RW_PAGE_READ_MSK;
		if(nc->aes) {
			page_cfg |= PAGE_RW_AES_MSK;
		}	

		if(chip->ecc.mode == NAND_ECC_HW) {
			page_cfg |= PAGE_RW_ECC_MSK;
		}
		writel(page_cfg, (nc->ctrl_base + PAGE_RW_OFFSET));

		/* Send CACHED READ cmd */
#ifdef CONFIG_NUMONYX_70NM_CACHEREAD_PATCH
		/* For Numonyx 78nm Flash chip, command is not required.
		 * But due to IP_2017 limitation, we need to send dummy command */
		nx_nand_cmd_addr(nc, 1, 0x77, 1);
#else		
		nx_nand_cmd_addr(nc, 1, NAND_CMD_CACHEREAD, 1);
#endif		
	}

	/* Wait for SEQ_READ interrupt */
	set_timeout(PNX8XXX_TIMEOUT);
	int_status = readl(nx_nc->ctrl_base + INT_STATUS_OFFSET);		
	while (!(int_status & NX_NAND_INT_SEQ_READ))
	{
		/* Check Timeout */
		if(did_timeout()) {
			printk("TIMEOUT: Read\r\n");	
			return;
		}	
		int_status = readl(nx_nc->ctrl_base + INT_STATUS_OFFSET);		
	}
	
	/* Check ECC status */
	if(chip->ecc.mode == NAND_ECC_HW) {
		/* Get HW ECC status */
		int_ecc = int_status & (NX_NAND_INT_DEC_UNCOR |
                   NX_NAND_INT_DEC_0_ERR |
                   NX_NAND_INT_DEC_1_ERR |
                   NX_NAND_INT_DEC_2_ERR |
                   NX_NAND_INT_DEC_3_ERR |
                   NX_NAND_INT_DEC_4_ERR |
                   NX_NAND_INT_DEC_5_ERR);
 	
		if(int_ecc) {
		 	nc->ecc_status[i] = int_ecc;
			writel(int_ecc, (nx_nc->ctrl_base + INT_CLR_STATUS_OFFSET));
			i++;
		}
	}

	/* Clear interrupt */
	writel(int_status, (nx_nc->ctrl_base + INT_CLR_STATUS_OFFSET));

#ifdef CONFIG_NAND_PERF_MEAS
	//s6 = readl(TSU_COUNTER_REG);
#endif
	
	/* Complete DMAC transfer */
	status = nx_dmac_tfr_comp(chanid);
	if(status) {
		printk(KERN_ERR "nx_nand: NAND_READ0 DMAC complete\r\n");	
		return;
	}

#ifdef CONFIG_NAND_PERF_MEAS
	//s7 = readl(TSU_COUNTER_REG);
#endif

	/* Disable interrupt */
	writel(intr, (nx_nc->ctrl_base + INT_CLR_ENABLE_OFFSET));

	/* Disable DMAC Flow control */
	writel(0, (nx_nc->ctrl_base + DMA_CTRL_OFFSET));

#ifdef CONFIG_NAND_PERF_MEAS
	//s8 = readl(TSU_COUNTER_REG);
	//printk(KERN_INFO "s1: %d s2: %d s3: %d s4: %d s5: %d s6: %d s7: %d s8: %d\r\n", s1, s2, s3, s4, s5, s6, s7, s8);
#endif

	return;
}


/*------------------------------------------------------------------------------------
* NAND chip specific functions 
--------------------------------------------------------------------------------------*/
#ifdef CONFIG_MTD_NX_NAND_HWECC
/**
* nx_nand_calculate_ecc - HW ECC calculate
* @mtd: MTD information structure
* @dat: Databuffer
* @ecc_code: ECC code buffer
*
* Dummy function for HW ECC calculation
*/
static int nx_nand_calculate_ecc(struct mtd_info *mtd, const uint8_t *dat,
							uint8_t *ecc_code)
{
	return 0;
}

/**
* nx_nand_correct_data - HW ECC correct
* @mtd: MTD information structure
* @dat: Databuffer
* @read_ecc: Read ECC code buffer
* @calc_ecc: Calculated ECC buffer
*
* Dummy function for HW ECC calculation
*/
static int nx_nand_correct_data(struct mtd_info *mtd, uint8_t *dat,
		             uint8_t *read_ecc, uint8_t *calc_ecc)
{
	return 0;
}

/**
* nx_nand_hwctl - HW ECC control function
* @mtd: MTD information structure
* @mode: Mode
*
* Dummy function for HW ECC calculation
*/
static void nx_nand_hwctl(struct mtd_info *mtd, int mode)
{
	return;
}

#endif

/**
* nx_nand_select_chip - Enable or Disable chip
* @mtd: MTD information structure
* @chipnr: Chip number
*
* Enable the chip if it chipnr >= 0, else disable the chip
*/
static void nx_nand_select_chip(struct mtd_info *mtd, int chipnr)
{
	struct nand_chip *chip= mtd->priv;
	struct nx_nand_ctrl *nc = chip->priv;

	if(chipnr == -1)
		return;
	
	/* Store the value in nand control structure 
	 * Chip enable/disable done in command function */
	nc->slotid = chipnr;
}

/**
* nx_nand_dev_ready - Check device ready
* @mtd: MTD information structure
*
* Return true if the device is ready, false otherwise
*/
static int nx_nand_dev_ready(struct mtd_info *mtd)
{
	struct nand_chip *chip= mtd->priv;
	struct nx_nand_ctrl *nc = chip->priv;
	int status, ready = 0;
	
	/* Get status from controller */
	status = readl(nc->ctrl_base + NAND_STATUS_OFFSET);
	if(status & (NAND_STATUS_RB_DEV0 << nc->slotid)) {
		ready = 1;
	} 

	/* Return R/B status */
	return ready;
}

/**
* nx_nand_read_byte - Read a byte from chip
* @mtd: MTD information structure
*
* Read a byte from the nand chip
*/
static uint8_t nx_nand_read_byte(struct mtd_info *mtd)
{
	struct nand_chip *chip= mtd->priv;
	struct nx_nand_ctrl *nc = chip->priv;
	int data;

	/* Read data */
	data = readl(nc->ctrl_base + SINGLE_READ_OFFSET);
	return (uint8_t) cpu_to_le16(data);
}

/**
* nx_nand_read_byte16 - Read a byte from 16bit chip
* @mtd: MTD information structure
*
* Read a byte from the 16bit nand chip
*/
static uint8_t nx_nand_read_byte16(struct mtd_info *mtd)
{
	struct nand_chip *chip= mtd->priv;
	struct nx_nand_ctrl *nc = chip->priv;
	uint16_t data;

	/* Read data */
	data = readl(nc->ctrl_base + SINGLE_READ_OFFSET);
	return (uint8_t) cpu_to_le16(data);
}

/**
* nx_nand_read_buf - Read data from chip
* @mtd: MTD information structure
* @buf: Data buffer
* @len: Transfer size
*
* Read specified number of bytes from the driver buffer
*/
static void nx_nand_read_buf(struct mtd_info *mtd, uint8_t *buf, int len)
{
	struct nand_chip *chip= mtd->priv;
	struct nx_nand_ctrl *nc = chip->priv;
#ifdef CONFIG_NAND_PERF_MEAS
	//uint32_t rb1, rb2, rb3;
#endif
	
	/* Copy from driver buffer */
#ifdef CONFIG_NAND_PERF_MEAS
	//rb1 = readl(TSU_COUNTER_REG);
#endif

#ifdef CONFIG_NX_MEMCPY
	nx_memcpy(buf, nc->dmabuf + nc->offset, len);
#else	
	memcpy(buf, nc->dmabuf + nc->offset, len);
#endif	

#ifdef CONFIG_NAND_PERF_MEAS
	//rb2 = readl(TSU_COUNTER_REG);
#endif
	nc->offset += len;
#ifdef CONFIG_NAND_PERF_MEAS
	//rb3 = readl(TSU_COUNTER_REG);
	//printk(KERN_INFO "rb1: %d rb2: %d rb3: %d \r\n", rb1, rb2, rb3);
#endif
}

/**
* nx_nand_read_page_raw - Read 1 page data from chip
* @mtd: MTD information structure
* @chip: Chip information structure
* @buf: Data buffer
*
* Read a full page + oob into the buffer
*/
static int nx_nand_read_page_raw(struct mtd_info *mtd, struct nand_chip *chip,
				  uint8_t *buf)
{
	struct nx_nand_ctrl *nc = chip->priv;

#ifdef CONFIG_NAND_PERF_MEAS
	//uint32_t c1, c2, c3, c4;
#endif

	/* Copy to user buffer */
#ifdef CONFIG_NAND_PERF_MEAS
	//c1 = readl(TSU_COUNTER_REG);
#endif

	invalidate_dcache_range(nc->dmabuf, (nc->dmabuf + mtd->writesize + mtd->oobsize));

#ifdef CONFIG_NAND_PERF_MEAS
	//c2 = readl(TSU_COUNTER_REG);
#endif

	chip->read_buf(mtd, buf, mtd->writesize);

#ifdef CONFIG_NAND_PERF_MEAS
	//c3 = readl(TSU_COUNTER_REG);
#endif

	chip->read_buf(mtd, chip->oob_poi, mtd->oobsize);

#ifdef CONFIG_NAND_PERF_MEAS
	//c4 = readl(TSU_COUNTER_REG);
	//printk(KERN_INFO "c1: %d c2: %d c3 %d c4 %d \r\n", c1, c2, c3, c4);
#endif

	return 0;
}

#ifdef CONFIG_MTD_NX_NAND_HWECC

#define NUM_BITFLIPS 			(4)

/**
* get_flip_count - Get the number of bitflips
* @buf: Buffer pointer
* @size: size of the buffer (in bytes)
*
* Get number of bitflips (in case of empty page) in the buffer
*/
static uint32_t get_flip_count(uint32_t *buf, uint32_t size)
{
	int j;
	uint32_t	bit_flips=0;
	uint32_t	*buf_ptr = (uint32_t *) buf;

	/* Check if data bitflips */
	for(j=0; j < (size >> 2); j++) {
		uint32_t dat = ~buf_ptr[j];
		if (likely(!dat)) continue;
		else {
			while (dat){
				bit_flips++;
				dat &= (dat -1);
			}
		}
	}
	return bit_flips;
}

/**
* nx_nand_read_page - Read 1 page data from chip with HWECC
* @mtd: MTD information structure
* @chip: Chip information structure
* @buf: Data buffer
*
* Read a full page + oob into the buffer
*/
static int nx_nand_read_page(struct mtd_info *mtd, struct nand_chip *chip,
				  uint8_t *buf)
{
	int stat=0,i, j;	
	struct nx_nand_ctrl *nc = chip->priv;
	int eccsteps = chip->ecc.steps;
	int no_all_ffs=0;
	int word_data, *ptr;
	
#ifdef CONFIG_NAND_PERF_MEAS
	//volatile uint32_t rp1, rp2, rp3, rp4, rp5, rp6, rp7, rp8;
#endif

#ifdef CONFIG_NAND_PERF_MEAS
	//rp1 = readl(TSU_COUNTER_REG);
#endif

	/* Read page data */
	nx_nand_read_page_raw(mtd, chip, buf);

#ifdef CONFIG_NAND_PERF_MEAS
	//rp2 = readl(TSU_COUNTER_REG);
#endif

	/* Check ECC status */
	for (i = 0 ; i<eccsteps; i++) {
		if((nc->ecc_status[i] == NX_NAND_INT_DEC_UNCOR) || 
		  (nc->ecc_status[i] & NX_NAND_INT_DEC_5_ERR)) {

#ifdef CONFIG_NAND_PERF_MEAS
			//rp4 = readl(TSU_COUNTER_REG);
#endif
			/* The IP_2017 does not have empty page detector.
			 * The ECC will fail when empty pages are read
			 * If ECC failed, check if page is empty (all 0xFFs).
			 * If empty page, ignore ECC errors.
			 * Check if empty page has bit flips.
			 * If number of bitflips is <= to NUM_BITFLIPS,
			 * correct bitflips & retun.
			 * Else return with ECC failure without correction
			 * */
			uint32_t bit_flips, bit_flip_limit;
			bit_flips = get_flip_count((uint32_t *)buf, mtd->writesize) +
								get_flip_count ((uint32_t *)chip->oob_poi, mtd->oobsize);
			bit_flip_limit = (nc->ecc_status[i] == NX_NAND_INT_DEC_UNCOR) ?
					 NUM_BITFLIPS : (45 + NUM_BITFLIPS);
			if(bit_flips <= bit_flip_limit) {
				if(bit_flips) {
					/* Correct data if bitflips are present in teh page */
					memset(buf, 0xFF, mtd->writesize);
					memset(chip->oob_poi, 0xFF, mtd->oobsize);
					/* We don't know how many actual bitflips were there
					 * before ECC correction; we assume 1 for the stats */
					printk(KERN_INFO "step %d:ECC 1 bit corrected \r\n", i);
					mtd->ecc_stats.corrected++;
				}
			} else {
				if (nc->ecc_status[i] == NX_NAND_INT_DEC_UNCOR) {
				/* If not empty page, update ECC statistics */
				printk(KERN_INFO "ECC failed step %d  \r\n", i);
				mtd->ecc_stats.failed++;
				} else {
					printk(KERN_INFO "step %d:ECC 5 bits corrected \r\n", i);
					mtd->ecc_stats.corrected += 5;
				}
			}

#ifdef CONFIG_NAND_PERF_MEAS
			//rp5 = readl(TSU_COUNTER_REG);
#endif
		}
		else {
#ifdef CONFIG_NAND_PERF_MEAS
			//rp6 = readl(TSU_COUNTER_REG);
#endif
			/* Update ECC stats */
			if(nc->ecc_status[i] != NX_NAND_INT_DEC_0_ERR) {
				for(j=0; j < 4; j++) {	/* NX_NAND_INT_DEC_5_ERR is handled above */
					if(nc->ecc_status[i] & (NX_NAND_INT_DEC_1_ERR << j)) {
						printk(KERN_INFO "step %d:ECC %d bits corrected \r\n", i, j+1);
						stat = j+1;	
					}
				}
				mtd->ecc_stats.corrected += stat;
			}
#ifdef CONFIG_NAND_PERF_MEAS
			//rp7 = readl(TSU_COUNTER_REG);
#endif
		}
	}
#ifdef CONFIG_NAND_PERF_MEAS
	//rp3 = readl(TSU_COUNTER_REG);
	//printk(KERN_INFO "rp1: %d rp2: %d rp3: %d rp4: %d rp5: %d \r\n", rp1, rp2, rp3, rp4, rp5);
#endif
	
	return 0;
}
#endif

/**
* nx_nand_read_oob - Read OOB data
* @mtd: MTD information structure
* @chip: Chip information structure
* @page: Page address
* @sndcmd: Send command flag
*
* Read OOB data into the buffer
*/
static int nx_nand_read_oob(struct mtd_info *mtd, struct nand_chip *chip,
				  int page, int sndcmd)
{
	struct nx_nand_ctrl *nc = chip->priv;
	uint8_t *buf = chip->oob_poi;
  	int length = mtd->oobsize;
	int column, addr, i;
	uint32_t	data;
	uint32_t page_cfg;

	if(mtd->flags & MTD_USE_DEV_OOB_LAYOUT) {
		/* Use device OOB layout */	
		/* No page operation */
		printk("************Reading OOB without ECC!!!\r\n");
		page_cfg = PAGE_RW_OOB_MSK;
		writel(page_cfg, (nc->ctrl_base + PAGE_RW_OFFSET)); //NOECC!!
	
		/* Send READOOB command */	
		if(sndcmd) {
			chip->cmdfunc(mtd, NAND_CMD_READOOB, 0, page);
		}
	
		/* Check if CE DON't care is supported */
		column = nc->cur_col;
		i = 0;
		while(length) {
			/* Send Address & cmd cycles */
			if(sndcmd) {
				/* Send READ0 command */	
				nx_nand_cmd_addr(nc, 1, nc->cur_cmd, 0);
	
				/* Send address cycles */	
				addr = column & NX_NAND_SP_ADDR_MASK;
				nx_nand_cmd_addr(nc, 0, addr, 0);

				if(nc->lb_chip) {
					addr = (column >> 8) & NX_NAND_SP_ADDR_MASK;
					nx_nand_cmd_addr(nc, 0, addr, 0);
				}
		
				addr = nc->cur_page & NX_NAND_SP_ADDR_MASK;
				nx_nand_cmd_addr(nc, 0, addr, 0);
			
				addr = (nc->cur_page >> 8) & NX_NAND_SP_ADDR_MASK;
				nx_nand_cmd_addr(nc, 0, addr, 0);
					
				if(nc->lb_chip) {
					/* if > 2Gb, extra address cycle */
					if (chip->chipsize > (1 << 27)) {
						addr = (nc->cur_page >> 16) & NX_NAND_SP_ADDR_MASK;
						nx_nand_cmd_addr(nc, 0, addr, 0);
					}
				
					/* Send READSTART command */
					nx_nand_cmd_addr(nc, 1, NAND_CMD_READSTART, 1);
				}
				else {
					if (chip->chipsize > (32 << 20)) {
						addr = (nc->cur_page >> 8) & NX_NAND_SP_ADDR_MASK;
						nx_nand_cmd_addr(nc, 0, addr, 0);
					
						addr = (nc->cur_page >> 16) & NX_NAND_SP_ADDR_MASK;
						nx_nand_cmd_addr(nc, 0, addr, 1);
					}
					else {
						addr = (nc->cur_page >> 8) & NX_NAND_SP_ADDR_MASK;
						nx_nand_cmd_addr(nc, 0, addr, 1);
					}
				}
				sndcmd = 0;
					
				/* Wait for completion */
				udelay(chip->chip_delay);
			}

			/* Read data from chip */
			data = readl(nx_nc->ctrl_base + SINGLE_READ_OFFSET);
			buf[i] = (uint8_t) data; 
		
			length--;
			i++;
		
			if(!nx_nc->cedontcare) {
				sndcmd = 1;	
				column++;
			}
		}
	}
	else {
		/* Use IP_2017 OOB layout - (512 Bytes data + 16 bytes OOB data) */
		/* Send READOOB command */	
		if(sndcmd) {
			chip->cmdfunc(mtd, NAND_CMD_READ0, 0, page);
			sndcmd = 0;
		}

		/* Copy into user buffer */
		nc->offset = mtd->writesize;	
		invalidate_dcache_range(nc->dmabuf, (nc->dmabuf + mtd->writesize + mtd->oobsize));
		chip->read_buf(mtd, chip->oob_poi, mtd->oobsize);
	
		/* Stop cached read */
		if (NAND_CANCACHEDREAD(chip)) {
			chip->cmdfunc(mtd, NAND_CMD_EXIT_CACHEREAD, -1, -1);
			nand_wait_ready(mtd);
		}
	}

	return 0;
}

/**
* nx_nand_write_buf - Write data into chip
* @mtd: MTD information structure
* @buf: Data buffer
* @len: Transfer size
*
* Write specified number of bytes into the nand chip
*/
static void nx_nand_write_buf(struct mtd_info *mtd, const uint8_t *buf, 
		int len)
{
	struct nand_chip *chip= mtd->priv;
	struct nx_nand_ctrl *nc = chip->priv;

	/* Copy data to driver buffer */
#ifdef CONFIG_NX_MEMCPY
	nx_memcpy(nc->dmabuf + nc->offset, buf, len);
#else	
	memcpy(nc->dmabuf + nc->offset, buf, len);
#endif	
	nc->offset += len;
}

/**
* nx_nand_write_page_raw - Write 1 page data into chip
* @mtd: MTD information structure
* @chip: Chip information structure
* @buf: Data buffer
*
* Write a full page + oob into the buffer
*/
static void nx_nand_write_page_raw(struct mtd_info *mtd, struct nand_chip *chip,
				  const uint8_t *buf)
{
	struct nx_nand_ctrl *nc = chip->priv;
	unsigned long len = mtd->writesize + mtd->oobsize;
	uint32_t	intr=0;
	volatile 	uint32_t int_status;
	uint16_t	addr;
	int chanid, status;
	uint32_t page_cfg;
	nx_dmac_tfr_t				req;
	nx_dmac_stgt_t			stgt[5];

	/* Copy data into buffer */
	chip->write_buf(mtd, buf, mtd->writesize);
	chip->write_buf(mtd, chip->oob_poi, mtd->oobsize);
	flush_dcache_range(nc->dmabuf, nc->dmabuf + len);

	/* Scatter gather list for DMAC */
	nx_nand_dmac_init(nc, 0, &req, stgt);
	
	/* Configure Flow control */
	writel(DMA_CTRL_MEM_2_PER, (nx_nc->ctrl_base + DMA_CTRL_OFFSET));

	/* Start DMA */
	chanid = nx_dmac_tfr(&req);
	if(chanid < 0) {
		printk(KERN_ERR "nx_nand: NAND_SEQIN DMAC config \r\n");	
		return;
	}

	/* Page operation */
	page_cfg = PAGE_RW_OOB_MSK | PAGE_RW_PAGE_WRITE_MSK;
	if(nc->aes) {
		page_cfg |= PAGE_RW_AES_MSK;
	}

	if(chip->ecc.mode == NAND_ECC_HW) {
		page_cfg |= PAGE_RW_ECC_MSK;
	}
	writel(page_cfg, (nx_nc->ctrl_base + PAGE_RW_OFFSET));
	
	/* Clear interrupt */
	writel(NX_NAND_INT_ALL, (nx_nc->ctrl_base + INT_CLR_STATUS_OFFSET));
	
	/* Enable SEQ WRIEt & READY interrupts */
	intr = NX_NAND_INT_SEQ_WRITE | (1 << (NX_NAND_INT_READY_START + nc->slotid));
	writel(intr, (nx_nc->ctrl_base + INT_SET_ENABLE_OFFSET));

	/* Send SEQIN command */
	nx_nand_cmd_addr(nc, 1, NAND_CMD_SEQIN, 0);

	/* Send address commands */	
	addr = nc->cur_col & NX_NAND_SP_ADDR_MASK;
	nx_nand_cmd_addr(nc, 0, addr, 0);
		
	if(nc->lb_chip) {
		addr = (nc->cur_col >> 8) & NX_NAND_SP_ADDR_MASK;
		nx_nand_cmd_addr(nc, 0, addr, 0);
	}
		
	addr = nc->cur_page & NX_NAND_SP_ADDR_MASK;
	nx_nand_cmd_addr(nc, 0, addr, 0);
			
	addr = (nc->cur_page >> 8) & NX_NAND_SP_ADDR_MASK;
	nx_nand_cmd_addr(nc, 0, addr, 0);
	
	if(nc->lb_chip) {
		/* if > 2Gb, extra address cycle */
		if (chip->chipsize > (1 << 27)) {
			addr = (nc->cur_page >> 16) & NX_NAND_SP_ADDR_MASK;
			nx_nand_cmd_addr(nc, 0, addr, 0);
		}
	}
	else {
		/* if > 64MB, extra adddress cycle */
		if (chip->chipsize > (32 << 20)) {
			addr = (nc->cur_page >> 16) & NX_NAND_SP_ADDR_MASK;
			nx_nand_cmd_addr(nc, 0, addr, 0);
		}
	}
	
	/* Send PAGEPROG - post write command */
	nx_nand_cmd_addr(nc, 2, NAND_CMD_PAGEPROG, 1);

	/* Complete DMAC transfer */
	status = nx_dmac_tfr_comp(chanid);
	if(status) {
		printk(KERN_ERR "nx_nand: write_page_raw \r\n");	
		return;
	}
	
	/* Wait till READY interrupt */
	set_timeout(PNX8XXX_TIMEOUT);
	int_status = readl(nx_nc->ctrl_base + INT_STATUS_OFFSET);		
	while (!(1 << (NX_NAND_INT_READY_START + nc->slotid))) {
		/* Timeout */
		if(did_timeout()) {
			printk("TIMEOUT: Write \r\n");
			return;
		}
		
		/* Get Page done */		
		if(int_status & NX_NAND_INT_SEQ_WRITE) {
			writel(NX_NAND_INT_SEQ_WRITE, (nx_nc->ctrl_base + INT_CLR_STATUS_OFFSET));	
		}
		int_status = readl(nx_nc->ctrl_base + INT_STATUS_OFFSET);		
	}

	/* Clear interrupt */
	writel(int_status, (nx_nc->ctrl_base + INT_CLR_STATUS_OFFSET));

	/* Disable interrupt */
	writel(intr, (nx_nc->ctrl_base + INT_CLR_ENABLE_OFFSET));

	/* Disable Flow control */
	writel(0, (nx_nc->ctrl_base + DMA_CTRL_OFFSET));
	
	return;
}

#ifdef CONFIG_MTD_NX_NAND_HWECC
/**
* nx_nand_write_page - Write 1 page data into chip when HW ECC enabled
* @mtd: MTD information structure
* @chip: Chip information structure
* @buf: Data buffer
*
* Write a full page + oob into the buffer
*/
static void nx_nand_write_page(struct mtd_info *mtd, struct nand_chip *chip,
				  const uint8_t *buf)
{
	/* write page */	
	nx_nand_write_page_raw(mtd, chip, buf);

	return;
}
#endif

static uint8_t temp_buf[NAND_MAX_PAGESIZE+NAND_MAX_OOBSIZE];
#define OOB_BYTES_PER_BLK (16)

/**
* nx_nand_write_oob - Write OOB data
* @mtd: MTD information structure
* @chip: Chip information structure
* @page: Page address
*
* Write OOB data into the chip
*/
static int nx_nand_write_oob(struct mtd_info *mtd, struct nand_chip *chip,
				  int page)
{
	struct nx_nand_ctrl *nc = chip->priv;
	int status = 0, sndcmd = 1;
  	const uint8_t *buf = chip->oob_poi;
  	int length;
	int column, addr, i;
	uint16_t	data;
	uint32_t page_cfg;
	int	ecc_old, aes_old;
	uint8_t	*oob_poi_orig, *temp1;
		
	/* Store ECC,AES values & Disbale */ 	
	ecc_old = chip->ecc.mode;
	aes_old = nc->aes;
	chip->ecc.mode = NAND_ECC_NONE;
	nc->aes = 0;
	
	/* Send SEQIN command */
	chip->cmdfunc(mtd, NAND_CMD_SEQIN, 0, page);
			
	/* Write dummy_buf for page write */
	memset(temp_buf, 0xff, (mtd->writesize+mtd->oobsize));
		
	if(mtd->flags & MTD_USE_DEV_OOB_LAYOUT)
	{
		/* Call write page raw */
		nx_nand_write_page_raw(mtd, chip, temp_buf);
	}
	else {
		/* Use IP_2017 OOB layout - write page */
		/* Save original OOB pointer */ 	
		oob_poi_orig = chip->oob_poi; 
	
		/* copy OOB to temparory buffer as per IP_2017 layout*/	
		temp1 = temp_buf;
		for(i=0; i < nc->num_blks; i++) {
			temp1 += NX_NAND_BLK_SIZE;

#ifdef CONFIG_NX_MEMCPY
			nx_memcpy(temp1, (chip->oob_poi + (i * OOB_BYTES_PER_BLK)), OOB_BYTES_PER_BLK);
#else
			memcpy(temp1, (chip->oob_poi + (i * OOB_BYTES_PER_BLK)), OOB_BYTES_PER_BLK);
#endif			
			temp1 += OOB_BYTES_PER_BLK;	
		}
		chip->oob_poi = &temp_buf[mtd->writesize];
	
		/* Call write page raw */
		nx_nand_write_page_raw(mtd, chip, temp_buf);
	
		chip->oob_poi = oob_poi_orig;
	}

	
	/* Restore ECC,AES values & Disbale */ 	
	chip->ecc.mode = ecc_old;
	nc->aes = aes_old;

	/* Send command to program the OOB data */
	chip->cmdfunc(mtd, NAND_CMD_PAGEPROG, -1, -1);

	status = chip->waitfunc(mtd, chip);

	return status & NAND_STATUS_FAIL ? -EIO : 0;
}

/**
* nx_nand_command - Command function for small page chips
* @mtd: MTD information structure
* @cmd: Command
* @column: Column address
* @page_addr: Page address
*
* Command control function:
*/
static void nx_nand_command(struct mtd_info *mtd, unsigned int cmd,
			     int column, int page_addr)
{
	struct nand_chip 		*chip= mtd->priv;
	struct nx_nand_ctrl *nc = chip->priv;
	uint32_t page_cfg;

	/* Store the command, colmn & page address */
	nc->cur_cmd = cmd;
	if(column == -1)
		column = 0;
	nc->cur_col = column;
	if(page_addr == -1)
		page_addr = 0;
	nc->cur_page = page_addr;
	nc->mtd = mtd;	
	/*
	 * Issue the correct first command, when we write to
	 * the device.
	 */
	switch(cmd) {
	case NAND_CMD_SEQIN:
		nc->offset = 0;
		/* Address & command sent in write_page_raw */
		break;

	case NAND_CMD_PAGEPROG:
		/* command will be sent as post write command */
		break;
		
	case NAND_CMD_RESET:
	case NAND_CMD_STATUS:
		nx_nand_cmd_addr(nc, 1, cmd, 1);
		break;

	case NAND_CMD_ERASE1:
		nx_nand_erase(nc);
		break;

	case NAND_CMD_ERASE2:
		/* Already done in CMD_ERASE1 */
		break;

	case NAND_CMD_READ0:
		nc->offset = 0;
		nx_nand_read(nc);
		break;

	case NAND_CMD_READOOB:
		/* use READ0 command */
		nc->offset = mtd->writesize;
		nx_nand_read(nc);
		break;

	case NAND_CMD_READID:
		
		/* No Page operation */
		page_cfg = 0;
		if(nc->aes) {
			page_cfg |= PAGE_RW_AES_MSK;
		}
		if(chip->ecc.mode == NAND_ECC_HW) {
			page_cfg |= PAGE_RW_ECC_MSK;
		}
		writel(page_cfg, (nx_nc->ctrl_base + PAGE_RW_OFFSET));
		
		nx_nand_cmd_addr(nc, 1, cmd, 0);
		nx_nand_cmd_addr(nc, 0, column, 1);
		break;

	default:
		printk(KERN_ERR "nxnand: command not supported %d \n", cmd);
	}
}

/**
* nx_nand_command_lp - Command function for large page chips
* @mtd: MTD information structure
* @cmd: Command
* @column: Column address
* @page_addr: Page address
*
* Command control function:
*/
static void nx_nand_command_lp(struct mtd_info *mtd, unsigned int cmd,
			     int column, int page_addr)
{
	uint32_t	page_cfg;
	struct nand_chip 		*chip= mtd->priv;
	struct nx_nand_ctrl *nc = chip->priv;


	/* Store the command, colmn & page address */
	if(cmd == NAND_CMD_READOOB) {
		cmd = NAND_CMD_READ0;
		
		/* If Device OOB layout, read data in read_OOB function */ 
		if(mtd->flags & MTD_USE_DEV_OOB_LAYOUT) {
			nc->cur_cmd = cmd;
			nc->cur_col = mtd->writesize;
			nc->cur_page = page_addr;
			return;
		}
	}
	
	nc->cur_cmd = cmd;
	if(column == -1)
		column = 0;
	nc->cur_col = column;
	if(page_addr == -1)
		page_addr = 0;
	nc->cur_page =  (page_addr);
	nc->mtd = mtd;	

	/*
	 * Issue the correct first command, when we write to
	 * the device.
	 */
	switch(cmd) {
	case NAND_CMD_SEQIN:
		nc->offset = column;
		/* Address cycles & command will be sent in write_page_raw */
		break;

	case NAND_CMD_PAGEPROG:
		break;
		
	case NAND_CMD_RESET:
	case NAND_CMD_STATUS:
		nx_nand_cmd_addr(nc, 1, cmd, 1);
		break;
		
	case NAND_CMD_ERASE1:
		nx_nand_erase(nc);
		break;

	case NAND_CMD_ERASE2:
		/* Already done in CMD_ERASE1 */
		break;

	case NAND_CMD_READ0:
		nc->offset = column;
		nx_nand_read(nc);
		break;
	
	case NAND_CMD_CACHEREAD:
		/* First time, send command and return */
		nc->offset = column;
		if(chip->cached_page >= 0) {
			nx_nand_read(nc);
		}
		break;
	
	case NAND_CMD_EXIT_CACHEREAD:	
		chip->cached_page = -1;
		cmd = NAND_CMD_EXIT_CACHEREAD;
		nx_nand_cmd_addr(nc, 1, cmd, 1);
		break;	
		
	case NAND_CMD_READID:
		/* No Page operation */
		page_cfg = 0;
		if(nc->aes) {
			page_cfg |= PAGE_RW_AES_MSK;
		}
		if(chip->ecc.mode == NAND_ECC_HW) {
			page_cfg |= PAGE_RW_ECC_MSK;
		}
		writel(page_cfg, (nx_nc->ctrl_base + PAGE_RW_OFFSET));

		nx_nand_cmd_addr(nc, 1, cmd, 0);
		nx_nand_cmd_addr(nc, 0, column, 1);
		break;

	default:
		printk(KERN_ERR "nxnand: command not supported %d \n", cmd);
	}
}

/**
 * nx_nand_block_bad - Read bad block marker from the chip
 * @mtd:	MTD device structure
 * @ofs:	offset from device start
 * @getchip:	0, if the chip is already selected
 *
 * Check, if the block is bad.
 */
static int nx_nand_block_bad(struct mtd_info *mtd, loff_t ofs, int getchip)
{
	struct mtd_oob_ops ops;
	uint8_t	buf[NAND_MAX_OOBSIZE];
	int ret;
	u8 bad;
	int res = 0;
	struct nand_chip *chip = mtd->priv;
	
	mtd->flags |= MTD_USE_DEV_OOB_LAYOUT;
	/* Read OOB data */
	ops.ooblen = mtd->oobsize;
	ops.oobbuf = buf;
	ops.ooboffs = 0;
	ops.datbuf = NULL;
	ops.mode = MTD_OOB_PLACE;
	ret = mtd->read_oob(mtd, ofs, &ops);
	if (ret) {
		printk(KERN_INFO "READOOB failed 0x%x \r\n", ret);
		mtd->flags &= ~MTD_USE_DEV_OOB_LAYOUT;
		return ret;
	}
	mtd->flags &= ~MTD_USE_DEV_OOB_LAYOUT;
	
	/* Check the bad block marker */	
	bad = buf[chip->badblockpos];
	if (bad != 0xff) {
		res = 1;
	}
		
	return res;
}

/**
* nx_nand_verify_buf - Verify user buffer with read back data
* @mtd: MTD information structure
* @buf: User buffer 
* @len: Size
*
* Command control function:
*/
static int nx_nand_verify_buf(struct mtd_info *mtd, const uint8_t *buf,
		int len)
{
	int i,status = 0;
	struct nand_chip 		*chip= mtd->priv;
	struct nx_nand_ctrl *nc = chip->priv;

	/* After READ0 is sent, data will be available in driver buffer */
	invalidate_dcache_range(nc->dmabuf, (nc->dmabuf + mtd->writesize + mtd->oobsize));
	for(i=0; i < len; i++) {
		if(buf[i] != nc->dmabuf[i]){
			printk(KERN_INFO "Data mismatch 0x%x at %d W: 0x%x R: 0x%x\r\n", nc->cur_page, 
			i, buf[i], nc->dmabuf[i]);	
			status = -EIO;
			break;
		}
	}
	
	/* Stop cached read */
	if (NAND_CANCACHEDREAD(chip)) {
		chip->cmdfunc(mtd, NAND_CMD_EXIT_CACHEREAD, -1, -1);
		nand_wait_ready(mtd);
	}
	
	return status;
}

/**
 * nx_nand_block_markbad - mark a block bad
 * @mtd:  MTD device structure
 * @ofs:  offset from device start
 *
 * Mark the block as bad.
*/
static int nx_nand_block_markbad(struct mtd_info *mtd, loff_t ofs)
{
	struct nand_chip *chip = mtd->priv;
	uint8_t buf[2] = { 0, 0 };
	int block, ret=0;
	struct mtd_oob_ops ops;

	/* Get block number */
	block = (int)(ofs >> chip->bbt_erase_shift);

	/* Update bad block info in RAM BBT */
	if (chip->bbt)
		chip->bbt[block >> 2] |= 0x01 << ((block & 0x03) << 1);

	/* Update flash BBT */
	if (chip->options & NAND_USE_FLASH_BBT) {
		ret = nand_update_bbt(mtd, ofs);
	}
  	
	/* Update BBT stats */
	if (!ret) {
		mtd->ecc_stats.badblocks++;
	}

	/* Since IP_2017 overwrites bad block marker byte in Flash,
	* ECC failure is considered as criterion to find bad blocks.
	* Hence write some data in bad block marker so that
	* ECC fails
	* */
	ops.mode = MTD_OOB_RAW;
	ops.ooboffs = chip->badblockpos & ~0x01;
	ops.len = ops.ooblen = 2;
	ops.oobbuf = buf;
	ops.datbuf = NULL;

	/* Use Device layout */
	mtd->flags |= MTD_USE_DEV_OOB_LAYOUT;
	ret = mtd->write_oob(mtd, ofs, &ops);
	mtd->flags &= ~MTD_USE_DEV_OOB_LAYOUT;

	return ret;
}

/* For Testing - To be removed */
#define  DEV0_TYPE 			(0x0004C)
#define  DEV0_TIME0			(0x00050)
#define  DEV0_TIME1			(0x00054)

/**
* board_nand_init - NAND Initialisation function
* @chip: NAND chip structure
*
* Repartition a device
*/
int board_nand_init(struct nand_chip *chip)
{
	uint32_t	mod_config, dev_type, page_size;
	int page, oob;
	int ret;
	
	/* Allocate memory NAND control for structure */  
	if(!nx_nc) {
		nx_nc = kzalloc(sizeof(struct nx_nand_ctrl), GFP_KERNEL);
		if(!nx_nc) {
			printk("board_nand_init: nx_nc memory alloc failure \r\n");	
			return -ENOMEM;	
		}
	}

	/* Initialise NAND control structure */
	//nx_nc->ctrl_base = (void __iomem *) KSEG1ADDR(PNX8XXX_EFM_BASE);
	nx_nc->ctrl_base = (void __iomem *) (MMIO_BASE + 0x2B000);
	
	/* Read Module configuration */
	mod_config = readl(nx_nc->ctrl_base + MOD_CONFIG_OFFSET);
	if(mod_config & MOD_CONFIG_AES_MSK) {
		nx_nc->aes = 1;
	}
	nx_nc->slotid = 0;
	
	/* Read Flash configuration */
	mod_config = readl(nx_nc->ctrl_base + DEV_TYPE0_OFFSET);
	if(mod_config & DEV_TYPE0_CEN_DONT_MSK) {
		nx_nc->cedontcare = 1;
	}
	nx_nc->lb_chip = 0;
	
	/* NAND Chip structure */
	chip->priv = nx_nc;
	chip->chip_delay = 0;
	chip->select_chip = nx_nand_select_chip;
	chip->dev_ready = nx_nand_dev_ready;
	dev_type = mod_config & DEV_TYPE0_DATA_WIDTH_MSK;
	if (dev_type) {
		chip->options |= NAND_BUSWIDTH_16;
		chip->read_byte = nx_nand_read_byte16;
	} else {
		chip->read_byte = nx_nand_read_byte;
	}
	chip->cmdfunc = nx_nand_command;

	/* Page & OOB size calculation */
	page_size = (mod_config & DEV_TYPE0_PAGE_SIZE_MASK) >> DEV_TYPE0_PAGE_SIZE_POS;
	switch (page_size) {
		case 0:
			page = 512;
			oob = 16;
			break;
			
		case 1:
			page = 2048;
			oob = 64;
			nx_nc->lb_chip = 1;
			chip->cmdfunc = nx_nand_command_lp;
			break;
		
		case 2:
			page = 4096;
			oob = 128;
			nx_nc->lb_chip = 1;
			chip->cmdfunc = nx_nand_command_lp;
			break;
			
		default:
			printk("Invalid page configuration %d\n", page_size);
			return -ENXIO;
	}
	
	chip->ecc.read_page_raw = nx_nand_read_page_raw;
	chip->ecc.write_page_raw = nx_nand_write_page_raw;
	chip->ecc.read_oob = nx_nand_read_oob;
	chip->ecc.write_oob = nx_nand_write_oob;
	chip->read_buf = nx_nand_read_buf;
	chip->write_buf = nx_nand_write_buf;
	chip->verify_buf = nx_nand_verify_buf;
	chip->block_bad = nx_nand_block_bad;
	chip->block_markbad = nx_nand_block_markbad;
	
#ifdef CONFIG_MTD_NX_NAND_HWECC
	chip->ecc.read_page = nx_nand_read_page;
	chip->ecc.write_page = nx_nand_write_page;
	chip->ecc.mode = NAND_ECC_HW;
	chip->ecc.calculate = nx_nand_calculate_ecc;
	chip->ecc.correct = nx_nand_correct_data;
	chip->ecc.hwctl = nx_nand_hwctl;
	chip->ecc.size = 512;
	chip->ecc.bytes = 12;

	/* Use BBT decsriptors */
	chip->bbt_td = &nx_bbt_main;	
	chip->bbt_md = &nx_bbt_mirror;
	
	/* Use OOB layout */
	switch (oob) {
		case 16:
			chip->ecc.layout = &nx_nand_oob_16;
			break;
			
		case 64:
			chip->ecc.layout = &nx_nand_oob_64;
			break;
		
		case 128:
			chip->ecc.layout = &nx_nand_oob_128;
			break;
			
		default:
			printk("No oob scheme defined for oobsize %d\n", oob);
			return -ENXIO;
	}
#endif

#ifdef CONFIG_MTD_NX_NAND_SWECC
	chip->ecc.mode = NAND_ECC_SOFT;
#endif

#ifdef CONFIG_MTD_NX_NAND_NONEECC
	chip->ecc.read_page = nx_nand_read_page_raw;
	chip->ecc.write_page = nx_nand_write_page_raw;
	chip->ecc.mode = NAND_ECC_NONE;
#endif

	/* Calculate number of 512byte blocks in a page */
	nx_nc->num_blks = page >> 9;
	
	/* Allocate ECC status array */
	nx_nc->ecc_status = kzalloc(sizeof(int) * nx_nc->num_blks, GFP_KERNEL);
	if(!nx_nc->ecc_status) {
		printk(KERN_ERR "board_nand_init: ECC status alloc \r\n");
		ret = -ENOMEM;
		goto out_free1;
	}

	/* Allocate internal driver buffer */
	nx_nc->dmabuf_unalgn =  kzalloc( ((page + oob) + (CONFIG_SYS_CACHELINE_SIZE * 2)),
			      GFP_DMA | GFP_KERNEL);
	if(!nx_nc->dmabuf_unalgn) {
		printk(KERN_ERR "board_nand_init: DMA buf alloc \r\n");
		ret = -ENOMEM;
		goto out_free2;
	}
	nx_nc->dmabuf = (uint8_t *)((unsigned long)(nx_nc->dmabuf_unalgn + CONFIG_SYS_CACHELINE_SIZE - 1) 
						& (~(CONFIG_SYS_CACHELINE_SIZE - 1)));
	invalidate_dcache_range(nx_nc->dmabuf, (nx_nc->dmabuf + page + oob));
	
	//chip->options |= NAND_ALLOW_CLEAR_BBT | NAND_NO_SUBPAGE_WRITE | NAND_USE_FLASH_BBT | NAND_CACHEDREAD;
	chip->options |= NAND_ALLOW_CLEAR_BBT | NAND_NO_SUBPAGE_WRITE | NAND_USE_FLASH_BBT;

	ret = nx_dmac_init();
	if(ret) {
		printf(KERN_ERR "board_nand_init:DMAC init failed \r\n");
		return -EIO;
	}

	return 0;

out_free2:
	kfree(nx_nc->ecc_status);	
	
out_free1:	
	kfree(nx_nc);

	return ret;
}

