/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Copyright (c) 2006-2007, LIPP Alliance
 * All Rights Reserved.
 *
 *---------------------------------------------------------------------------
 * %filename:               tmbslPhy.h %
 * %pid_version:                 1.2      %
 *---------------------------------------------------------------------------
 * DESCRIPTION:  
 *
 * DOCUMENT REF: 
 *
 * NOTES:        None
 *
 *-----------------------------------------------------------------------------
 *
 */

#ifndef  TMBSLPHY_H
#define  TMBSLPHY_H
//-----------------------------------------------------------------------------
// Standard include files:
//-----------------------------------------------------------------------------
//
#include "tmNxTypes.h"  // DVP standard types/defines/structs
#include "tmNxCompId.h" // DVP system wide component IDs

//-----------------------------------------------------------------------------
// Project include files:
//-----------------------------------------------------------------------------
//

#ifdef __cplusplus
extern "C"
{
#endif

//-----------------------------------------------------------------------------
// Types and defines:
//-----------------------------------------------------------------------------
//


#define TMBSL_PHY_COMPATIBILITY_NR       1
#define TMBSL_PHY_MAJOR_VERSION_NR       1
#define TMBSL_PHY_MINOR_VERSION_NR       0



#define TMBSL_ERR_PHY_BASE          	CID_BSL_PHY
#define TMBSL_ERR_PHY_COMP          	(CID_BSL_PHY | \
                                        	 TM_ERR_COMP_UNIQUE_START)
//PHY unit not supported                                         	 
#define TMBSL_ERR_PHY_NOT_SUPPORTED 	(TMBSL_ERR_PHY_BASE+TM_ERR_NOT_SUPPORTED)
// Invalid device unit number
#define TMBSL_ERR_PHY_BAD_UNIT_NUM  	(TMBSL_ERR_PHY_BASE+TM_ERR_BAD_UNIT_NUMBER)
// AutoNegotiation Time out occured
#define TMBSL_ERR_PHY_AUTONEG_TIMEOUT   (TMBSL_ERR_PHY_BASE+TM_ERR_TIMEOUT)
//PHY unit register read failed
#define TMBSL_ERR_PHY_READ_FAILED       (TMBSL_ERR_PHY_BASE + TM_ERR_READ)
//PHY unit register write failed
#define TMBSL_ERR_PHY_WRITE_FAILED      (TMBSL_ERR_PHY_BASE + TM_ERR_WRITE)
//PHY Initialisation failed
#define TMBSL_ERR_PHY_INIT_FAILED       (TMBSL_ERR_PHY_BASE + TM_ERR_INIT_FAILED)
//PHY unit reset failed
#define TMBSL_ERR_PHY_RESET_FAILED      (TMBSL_ERR_PHY_BASE + TM_ERR_HW_RESET_FAILED)

#if 0
#define tmbslPhyRegisterComponent(unitNumber, pIntInterface)            \
         tmbslCoreRegisterComponent(TMBSL_ERR_PHY_BASE,                  \
                                    unitNumber,                          \
                                    (ptmbslCompInterface_t) pIntInterface)


#define tmbslPhyGetInterface(unitNumber, ppIntInterface)                \
         tmbslCoreGetInterface(TMBSL_ERR_PHY_BASE,                       \
                               unitNumber,                               \
                               (tmbslCompInterface_t **) ppIntInterface)
                               
#endif

// PHY DUPLEX MODE
typedef enum	_tmbslPhyDuplexMode_t
{
	tmbslPhyHalfDuplex	=0,
	tmbslPhyFullDuplex	=1
} tmbslPhyDuplexMode_t, *ptmbslPhyDuplexMode_t;

// PHY SPEED
typedef enum	_tmbslPhySpeed_t
{
	tmbslPhySpeed10Mbps	=0,
	tmbslPhySpeed100Mbps	=1,
	tmbslPhySpeed1Gbps	= 2	
} tmbslPhySpeed_t, *ptmbslPhySpeed_t;

// ENABLE DISABLE MODE
typedef enum	_tmbslPhyEnableDisable_t
{
	tmbslPhyDisable		=0,
	tmbslPhyEnable		=1
} tmbslPhyEnableDisable_t, *ptmbslPhyEnableDisable_t;

// PHY BASIC MODE CONTROL
typedef struct	_tmbslPhyBasicModeControl_t
{
	Bool					enableCollisionTest; 
	tmbslPhyDuplexMode_t	duplexMode;
	tmbslPhySpeed_t			speed;
} tmbslPhyBasicModeControl_t, *ptmbslPhyBasicModeControl_t;

// PHY BASIC MODE STATUS
typedef struct	_tmbslPhyBasicModeStatus_t
{
	Bool					jabberDetect; // This applies only in 10Mbps : value 1 - jabber condition detected
	Bool					remoteFaultDetected; 
	Bool					autoNegotiationComplete;
	Bool					receiverErrorLatch;
	Bool					invertedPolarityDetected; // Detects link polarity
	Bool					falseCarrierEverntOccured;
	Bool					linkCodeWordPageReceived;
	Bool					loopBackEnabled; //loopback status
	tmbslPhyDuplexMode_t	duplexMode; // deplex mode status
	tmbslPhySpeed_t			speed;
	Bool					linkpartnerAcknolwedged ; // Link partner acknolwedged
	Bool					parallelDetectFaultDetected ; // Parallel detect fault is deteced

} tmbslPhyBasicModeStatus_t, *ptmbslPhyBasicModeStatus_t;

// PHY PROTOCOL SELECTION
typedef enum	_tmbslPhyProtocol_t
{
	tmbslPhyProtocolNone		= 0,
	tmbslPhyIEEE8023			= 1
} tmbslPhyProtocol_t, *ptmbslPhyProtocol_t;

/* PHY auto negotiation mask 
** When the below flags are true, that particular option is disabled/not advertised
*/
typedef struct	_tmbslPhyAutoNegotiationMask_t
{
    Bool				      masknextPageDesired;
    Bool				      maskRemoteFault; // advertise remote fault detection
    Bool					mask100BaseT4;  //advertise/mask 100BaseT4 capability 
    Bool					mask100BaseTxFullDuplexSupport; 
    Bool					mask100BaseTxSupport;
    Bool					mask10BaseTFullDuplexSupport; //advertise/mask 10BaseTFullDuplex support
    Bool					mask10BaseTSupport;
    tmbslPhyProtocol_t		protocolSel;			//(0x0001 IEEE802.3 CSMA/Cd)
    Bool                             maskAsymmetricPause;
    Bool                             maskPauseFrame;    
    /* 1G related enum  start */
    Bool                            maskMSConfigEn;
    Bool                            maskMasterEn;
    Bool                            maskMultiPortEn;
    Bool                            mask1000BaseTFullDuplexSupport;
    Bool                            mask1000BaseTHalfDuplexSupport;    
    /* 1G related ends*/
} tmbslPhyAutoNegotiationMask_t, *ptmbslPhyAutoNegotitationMask_t;


// PHY IDENTIFIER
typedef struct	_tmbslPhyIdentifier_t
{
	UInt32		oui; //PHY Organizationally Unique Identifier
	UInt32		vendorModelNr; 
	UInt32		modelRevNr;
	UInt32		siliconRevisionNr; 
} tmbslPhyIdentifier_t, *ptmbslPhyIdentifier_t;


// PHY CAPABILITIES
typedef struct	_tmbslPhyCapabilities_t
{
    Bool			T4Support100Base;
    Bool			Tx_FullDuplexSupport100Base;
    Bool			Tx_HalfDuplexSupport100Base;
    Bool			Tx_FullDuplexSupport10Base;
    Bool			Tx_HalfDuplexSupport10Base;
    Bool		 	preAmbleSuppresionCapability;
    Bool		 	autoNegotiationAbility;
    Bool		 	nextPageIndication;
    Bool			force100MbpsTxOff; 
    Bool			bypassSymbolAlignment; 
    Bool			badSSDDetectionConfig; 
    Bool		 	ledStatusSupport;	   

    /* 1G related capabilities */
    Bool		 	X1000BaseFDSupport;	       
    Bool		 	X1000BaseHDSupport;	           
    Bool		 	T1000BaseFDSupport;	       
    Bool		 	T1000BaseHDSupport;	               

} tmbslPhyCapabilities_t, *ptmbslPhyCapabilities_t;


// PHY LINK PARTNER CAPABILITIES

typedef struct	_tmbslPhyLinkPartnerCapabilities_t
{

	tmbslPhyProtocol_t			protocolSel;
	Bool						TSupport10Base;
	Bool						TFullDuplexSupport10Base;
	Bool						TxSupport100Base;
	Bool						TxFullDuplexSupport100Base;
	Bool						T4Support100Base;
	Bool						flowControl;
	Bool						remoteFault;
	Bool						acknoweledges;
	Bool						nextPageIndication;
	Bool						autoNegotiation;	
} tmbslPhyLinkPartnerCapabilities_t, *ptmbslPhyLinkPartnerCapabilities_t;


// PHY PCS(PHYSICAL SUBBAND CODING
typedef struct	_tmbslPhyPcsConfig_t
{
    /* 1 - force good link condition
         0 - normal operation  */
    Bool forceGoodLink100Mbps; 
    Bool force100MbpsTxOff;
    Bool bypassSymbolAlignment;
    /* 1 - Enable bad SSD detection */
    tmbslPhyEnableDisable_t badSSDDetection; 
} tmbslPhyPcsConfig_t, *ptmbslPhyPcsConfig_t;

// PHY COUNTERS
typedef struct 	_tmbslPhyCounters_t
{
	UInt32			disconnectCounter;		
	UInt32			falseCarrierEventCounter;//Gives the number of false carrier events 
	UInt32			rxErrorCounter;	//Gives the number of Receive error occured 		
} tmbslPhyCounters_t, *ptmbslPhyCounters_t;


// PSEUDO RANDOM SEQUENCES
typedef enum	_tmbslPhyPseudoRandomSeq_t
{
	tmbslPhyPseudoRandomSequenc9bit	=0,
	tmbslPhyPseudoRandomSequenc15bit	=1
} tmbslPhyPseudoRandomSeq_t, *ptmbslPhyPseudoRandomSeq_t;


// PHY BYPASS OPTIONS
typedef struct	_tmbslPhyBypass_t
{
	Bool		bypass4B5BCodec; //Bypass 4B 5B Encoding / Decoding
	Bool		bypassNrzi;      //Bypass NRZI Encoding / Decoding
	Bool		bypassScrambler; 
	Bool		bypassDescrambler;
} tmbslPhyBypass_t, *ptmbslPhyBypass_t;  






//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetSWVersion:
//
// DESCRIPTION: This function returns the PHY device interface software version 
//				information
//
// RETURN:      TM_OK
//
// NOTES:       This API can be called anytime i.e. before initializing the PHY 
//				or in PowerOff state.
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyGetSWVersion_t) (
    ptmSWVersion_t      					pPhyVersion    
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetCapabilities:
//
// DESCRIPTION: This function returns the PHY capabilities for the specified PHY
//				unit. The function is callable at any time to return the unit's 
//				capabilities (PHY unit initialization is not necessary). 
//				Capabilities may be different among multiple PHY units.For completeness, 
//				a PHY BSL user should call this function for each PHY unit to 
//				determine its individual capabilities. 
//
// RETURN:      TM_OK
//
// NOTES:       This API can be called anytime i.e. before initializing the PHY 
//				or in PowerOff state. 
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyGetCapabilities_t) (
    tmUnitSelect_t                			phyUnitId,  
    ptmbslPhyCapabilities_t  				pPhyCaps    
    );
    
    
    
//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyInit:
//
// DESCRIPTION: This function initializes the PHY device. It should be called
//				before any access to the device is made. 
//
// RETURN:      TM_OK 

//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:       This function initializes the PHY device with the following 
//				default initial configuration.
//				1. Enable the auto negotiation. In auto-negotiation mode the 
//				highest possible performance mode is selected automatically 
//				which the Link Partner also supports. (In auto negotiation mode
//				speed and duplex mode will be selected by means of auto negotiation. 
//				Writing to speed and duplex mode does not have meaning in auto 
//				negotiation mode.)
//				2. The device abilities are programmed to Auto negotiation advertise 
//				register. None of the device abilities are masked. If the PHY 
//				device does not support Auto-negotiation mode, then this function 
//				will set the PHY to basic capabilities of the device.The function 
//				tmbslPhyGetBasicModeControl can be called after tmbsPhyInit to 
//				know what are the default configurations the tmbslPhyInit function 
//				has set to
 
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyInit_t)(
    tmUnitSelect_t  						phyUnitId   
    );

    
    
//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDeinit:
//
// DESCRIPTION: This function de-initializes the PHY device. Once the device is
//				deinitilized device will no more be available to access 
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:       
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyDeinit_t)(
    tmUnitSelect_t  						phyUnitId   
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySetPowerState:
//
// DESCRIPTION: This function will set the Power State of the PHY device to specified 
//				power state
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:       
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhySetPowerState_t)(
    tmUnitSelect_t           				phyUnitId ,  
    tmPowerState_t          				phyPowerState
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetPowerState:
//
// DESCRIPTION: This function will get the preset power state of the PHY device
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:       
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyGetPowerState_t)(
    tmUnitSelect_t                  		phyUnitId ,  
    ptmPowerState_t							phyPowerState
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetBasicModeControl:
//
// DESCRIPTION: This function will get the basic configuration of the PHY device. 
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:       PHY device is initialized using the function tmbslPhyInit.  
//				tmbslPhyInit Enable the Auto negotiation mode and will configure 
//				the PHY device for the maximum performance mode available. After 
//				the device is initialized this function can be called to know the 
//				present settings of the PHY device. If application wants to change 
//				the settings it can call tmbslPhySetBasicModeControl to change 
//				the configuration after knowing the capabilities of the PHY device 
//				and Link Partner. #define for the Basic Mode Control 
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t 
(*tmbslPhyGetBasicModeControl_t) (
    tmUnitSelect_t                   		phyUnitId,   
    ptmbslPhyBasicModeControl_t      		pPhyBasicModeControl
    );

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySetBasicModeControl:
//
// DESCRIPTION: This function will configure the PHY device for the Basic Mode. 
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t 
(*tmbslPhySetBasicModeControl_t) (
    tmUnitSelect_t                   		phyUnitId,   
    ptmbslPhyBasicModeControl_t      		pPhyBasicModeControl
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetBasicModeStatus:
//
// DESCRIPTION: This function will get the Basic Mode Status of the PHY device 
//				such as the speed, duplex mode 
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyGetBasicModeStatus_t) (
    tmUnitSelect_t                  		phyUnitId,   
    ptmbslPhyBasicModeStatus_t    			pPhyBasicModeStatus     
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyAutoNegotiate:
//
// DESCRIPTION: This function will enable the Auto negotiation of the PHY device 
//				with Link Partner. Best possible performance configuration is 
//				selected automatically during this process
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyAutoNegotiate_t) (
    tmUnitSelect_t                       	phyUnitId,   
    ptmbslPhyAutoNegotitationMask_t      	pAutoNegotiationMask
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetIdentifier:
//
// DESCRIPTION: This function gets the PHY device Identifier
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyGetIdentifier_t) (
    tmUnitSelect_t               			phyUnitId,   
    ptmbslPhyIdentifier_t     				pPhyIdentifier
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetLinkPartnerCapabilities:
//
// DESCRIPTION: This function will get the Link Partner Capabilities. 
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyGetLinkPartnerCapabilities_t) (
    tmUnitSelect_t                         phyUnitId,   
    ptmbslPhyLinkPartnerCapabilities_t     pPhyLinkPartnerCapabilities
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetCounters:
//
// DESCRIPTION: This function will get the present counter values of the PHY counters
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyGetCounters_t) (
    tmUnitSelect_t                  		phyUnitId,   
    ptmbslPhyCounters_t      		 		pPhyCounters
    );

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyPcsSetConfig:
//
// DESCRIPTION: This function will configure the Physical layer configurations. 
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyPcsSetConfig_t) (
    tmUnitSelect_t                 			phyUnitId,   
    ptmbslPhyPcsConfig_t    				pPhyPcsConfig
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyPcsGetConfig:
//
// DESCRIPTION: This function will get the present PCS configuration 
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyPcsGetConfig_t) (
    tmUnitSelect_t                 			phyUnitId,   
    ptmbslPhyPcsConfig_t    				pPhyPcsConfig
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyBist:
//
// DESCRIPTION: This function will do the Built In Self Test and the result 
//				will be indicated in the pBistState
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//              pBistState  --> result of BIST (Built in self test) Test 
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyBist_t) (		
    tmUnitSelect_t  						phyUnitId,   
    tmbslPhyPseudoRandomSeq_t    			phyPsedoRandomSeq ,
    pUInt8                              	pBistState
	);

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyConfigBypass:
//
// DESCRIPTION: This function will bypass the functional blocks within 100Base-Tx transmitter
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyConfigBypass_t) (
    tmUnitSelect_t                        	phyUnitId,   
    ptmbslPhyBypass_t                     	pPhyBypass
	);


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyLoopBack:
//
// DESCRIPTION: Function will enable or disable the PHY device in the Loopback mode.
//
// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyLoopBack_t) (
    tmUnitSelect_t                      	phyUnitId,   
    tmbslPhyEnableDisable_t           		loopbackMode
	);


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySoftReset:
//
// DESCRIPTION: Function will do the soft reset of the PHY device

// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhySoftReset_t) (
    tmUnitSelect_t                     		phyUnitId      
	);

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetLinkStatus:
//
// DESCRIPTION: Function will get the link status

// RETURN:      TM_OK
//				TMBSL_ERR_PHY_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

typedef tmErrorCode_t
(*tmbslPhyGetLinkStatus_t) (
    tmUnitSelect_t                          phyUnitId,   
    ptmbslPhyEnableDisable_t           		pLinkStatus
);

typedef struct _tmbslPhyConfig_t 
{
	char     phyName[HAL_DEVICE_NAME_LENGTH];
	UInt32   lanClkSrc; // To select Clock source
	UInt32   lanClkSpeed; // To select the speed
	
	tmbslPhyGetSWVersion_t            getSWVersionFunc; 
	tmbslPhyGetCapabilities_t         getCapabilitiesFunc;
	tmbslPhyInit_t                    initFunc;
	tmbslPhyDeinit_t                  deinitFunc;  
	tmbslPhySetPowerState_t           setPowerStateFunc;
	tmbslPhyGetPowerState_t           getPowerStateFunc;
	tmbslPhyGetBasicModeControl_t     getBasicModeControlFunc;
	tmbslPhySetBasicModeControl_t     setBasicModeControlFunc;
	tmbslPhyGetBasicModeStatus_t      getBasicModeStatusFunc;
	tmbslPhyAutoNegotiate_t           autoNegotiateFunc;
	tmbslPhyGetIdentifier_t           getIdentifier_tFunc;
	tmbslPhyGetLinkPartnerCapabilities_t  getLinkPartnerCapabilitiesFunc;
	tmbslPhyGetCounters_t                 getCountersFunc;
	tmbslPhyPcsSetConfig_t                   pcsConfigFunc;
	tmbslPhyPcsGetConfig_t                pcsGetConfigFunc;
	tmbslPhyBist_t                        bistFunc;
	tmbslPhyConfigBypass_t                configBypassFunc;
	tmbslPhyLoopBack_t                    loopBackFunc;
	tmbslPhySoftReset_t                   softResetFunc;
	tmbslPhyGetLinkStatus_t               getLinkStatusFunc;

} tmbslPhyConfig_t,*ptmbslPhyConfig_t;

void tmbslPhyGetInterface(int unitno,tmbslPhyConfig_t**pPhyInterface);

#if        defined(__cplusplus)
}
#endif  // defined(__cplusplus) 
#endif //#ifndef  TMBSLPHYDP83847_H

