/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Copyright (c) 2006-2007, LIPP Alliance
 * All Rights Reserved.
 *
 *---------------------------------------------------------------------------
 * %filename:     tmbslPhyDP83848.c %
 * %pid_version:           1.2                %
 *---------------------------------------------------------------------------
 * DESCRIPTION: API source file for DP83848 National Semiconductors PHY
 *
 * DOCUMENT REF: Datasheet DP83848 Gig PHYTER V10/100/1000 Ethernet 
 *               Physical Layer, October 2004
 *
 * NOTES:        None
 *
 *-----------------------------------------------------------------------------
 *
*/

#include "tmNxTypes.h"
#include "tmNxCompId.h"

#include "lipp_6300eth_common.h"

/*  Project include files */

#include "tmbslPhy.h"
#include "tmbslPhyDP83848.h"
#include "tmhwLIPP6100Eth_Cfg.h"
/* Timeout in case of linux */
#ifdef __LIPP_6300ETH__
#include "i2c_ip3203.h"
#define AUTO_NEG_DELAY_MULTIPLIER (4000)
#endif /* __LINUX_GMAC_DRV__*/

/* Defines */
#define PHY_UNIT_ID_COUNT  			1

#define PHY_TIMEOUT                 (0x0FFFFFFF)

#define TMBSLPHYDP83848_PHY_MMIO_ADDRESS0 (0x1be00000+0x38000) 

#define ANAR_DEFAULT_VAL (0xDE1)

/* Global Data */
 /* Initialize with base address of GMAC. This will be used to access MAC Address & MAC data register */
tmbslPhyDP83848Context_t gEthContext[PHY_UNIT_ID_COUNT] = {{TMBSLPHYDP83848_PHY_MMIO_ADDRESS0}}; 

typedef	enum	_tmbslPhyDP83848Reg_t
{
    /* Basic mode control */
    tmbslPhyDP83848Bmcr		      = 0x0,  
    /* Basic mode status */    
    tmbslPhyDP83848Bmsr		      =0x1,  

    /* PHY ID1 register */        
    tmbslPhyDP83848PhyIdr1		=0x2,

    /* PHY ID2 register */            
    tmbslPhyDP83848PhyIdr2		=0x3,

    /* Auto negotiation advertisement register */                
    tmbslPhyDP83848Anar			=0x4,

    /* Auto negotiation link partner ability register */                    
    tmbslPhyDP83848Anlpar		=0x5,

    /* Auto negotiation expansion register */                        
    tmbslPhyDP83848Aner			=0x6,

    /* Auto negotiation next page transmit register */                            
    tmbslPhyDP83848AnnPtr		=0x7,

    /* 0x8 to 0xF reserved-*/
    tmbslPhyDP83848Phystsr = 0x10,        

    /* 0x11 to 0x13 reserved */
  
   /* False carrier sense counter register */                                    
    tmbslPhyDP83848Fcscr  = 0x14,

   /* Receive Error Counter Register  */                                        
    tmbslPhyDP83848Recr = 0x15,

   /* PCS Sub-Layer Configuration and Status Register */                                            
    tmbslPhyDP83848Pcsr = 0x16,

   /* RMII and Bypass Register */                                        
    tmbslPhyDP83848Rmiir = 0x17,    

    /* LED Direct Control Register  */    
    tmbslPhyDP83848Ledcr	= 0x18,
    
    /* BIST Configuration register1 */        
    tmbslPhyDP83848Phycr = 0x19,    

    /* 10Base-T Status/Control Register */            
    tmbslPhyDP8384810Btscr = 0x1A,        

    tmbslPhyDP83848Cdctrl1 = 0x1B,

    /* 0x1C reserved */
    
    /* Energy detect control register */ 
    tmbslPhyDP83848Edcr = 0x1D,

    /* 0x1E to 0x1F reserved*/

} tmbslPhyDP83848Reg_t, *ptmbslPhyDP83848Reg_t;


/* Static functions definition */

static tmErrorCode_t 
tmbslPhyDP83848Read (
	tmUnitSelect_t				ethUnitId,
    tmbslPhyDP83848Reg_t				reg,   
    pUInt16						pVal
	);
	
	
static tmErrorCode_t	
tmbslPhyDP83848Write (
    tmUnitSelect_t				ethUnitId,
    tmbslPhyDP83848Reg_t				reg,   
    UInt16						val
	);


#ifdef __LIPP_6300ETH__
static void get_phy_out_of_rst( void);
#endif

/* Exported functions */

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848GetSWVersion:
//
// DESCRIPTION: This function returns the PHY device interface software version 
//				information
//
// RETURN:      TM_OK
//
// NOTES:       This API can be called anytime i.e. before initializing the PHY 
//				or in PowerOff state.
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhyDP83848GetSWVersion (
	ptmSWVersion_t					pPhyVersion
	)
{
	
	pPhyVersion->compatibilityNr = TMBSL_PHY_COMPATIBILITY_NR;
	pPhyVersion->majorVersionNr = TMBSL_PHY_MAJOR_VERSION_NR;
	pPhyVersion->minorVersionNr = TMBSL_PHY_MINOR_VERSION_NR;
	
	return TM_OK;

}

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848GetCapabilities:
//
// DESCRIPTION: This function returns the PHY capabilities for the specified PHY
//				unit. The function is callable at any time to return the unit's 
//				capabilities (PHY unit initialization is not necessary). 
//				Capabilities may be different among multiple PHY units.For completeness, 
//				a PHY BSL user should call this function for each PHY unit to 
//				determine its individual capabilities. 
//
// RETURN:      TM_OK
//
// NOTES:       This API can be called anytime i.e. before initializing the PHY. 
//				 
//-----------------------------------------------------------------------------


tmErrorCode_t
tmbslPhyDP83848GetCapabilities (
    tmUnitSelect_t                			ethUnitId,  
    ptmbslPhyCapabilities_t  				pPhyCaps    
    )

{

    UInt16 bmsr;

    /*  Read the PHY capabilites from the BMSR register */
    tmbslPhyDP83848Read(ethUnitId, tmbslPhyDP83848Bmsr, &bmsr);


    pPhyCaps->T4Support100Base = 
        (((bmsr & TMBSL_PHYDP83848_BMSR_T4100BASE) > 0) ? True : False);

    pPhyCaps->Tx_FullDuplexSupport100Base = 
        (((bmsr &TMBSL_PHYDP83848_BMSR_X100BASEFD) > 0) ? True : False);

    pPhyCaps->Tx_HalfDuplexSupport100Base = 
        (((bmsr &TMBSL_PHYDP83848_BMSR_X100BASEHD) > 0) ? True : False);

    pPhyCaps->Tx_FullDuplexSupport10Base = 
        (((bmsr &TMBSL_PHYDP83848_BMSR_10MBPSFD) > 0) ? True : False);        

    pPhyCaps->Tx_HalfDuplexSupport10Base = 
        (((bmsr &TMBSL_PHYDP83848_BMSR_10MBPSHD) > 0) ? True : False);

    pPhyCaps->preAmbleSuppresionCapability = 
        (((bmsr &TMBSL_PHYDP83848_BMSR_PREAMBLE_SUP) > 0) ? True : False);        

    pPhyCaps->autoNegotiationAbility = 
                (((bmsr &TMBSL_PHYDP83848_BMSR_AN_ABLE) > 0) ? True : False);

    pPhyCaps->ledStatusSupport = True ;		

    /* Other capabilites set to False */   
    pPhyCaps->nextPageIndication = False;

    pPhyCaps->force100MbpsTxOff = False;

    pPhyCaps->bypassSymbolAlignment =False;

    pPhyCaps->badSSDDetectionConfig = False;
	
    return TM_OK;

}   
    
//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848Init:
//
// DESCRIPTION: This function initializes the PHY device. It should be called
//				before any access to the device is made. 
//
// RETURN:      TM_OK 
//
// NOTES:       This function initializes the PHY device with the following 
//				default initial configuration.No Autonegotiation is done in the 
//				initialization function
//-----------------------------------------------------------------------------


tmErrorCode_t
tmbslPhyDP83848Init(
    tmUnitSelect_t  						ethUnitId   
    )
{

    tmErrorCode_t       		ethStatus = TM_OK;     		
    UInt32                         timeout=0;
    UInt16 regval=0;
    UInt16 id1, id2;

//    mdelay(10);

    #ifdef __LIPP_6300ETH__
    get_phy_out_of_rst();
    #endif

    gEthContext[ethUnitId].pRegs = MAC_BASE_ADDRESS;

    GMAC_DBG("Base addr:%x",gEthContext[ethUnitId].pRegs);

    regval = TMBSL_PHYDP83848_BMCR_RST_VAL;

    ethStatus = tmbslPhyDP83848Write(ethUnitId,tmbslPhyDP83848Bmcr,regval);

    GMAC_DBG("After write in PHY");
    
    if(ethStatus != TM_OK)
    {
        return ethStatus;
    }

    /* Check if PHY is back to normal condition */
    ethStatus = tmbslPhyDP83848Read(ethUnitId,tmbslPhyDP83848Bmcr,&regval);
    
    while( (regval & TMBSL_PHYDP83848_BMCR_RST_VAL) && (timeout < PHY_TIMEOUT) )
    {
        ethStatus = tmbslPhyDP83848Read(ethUnitId,tmbslPhyDP83848Bmcr,&regval);

        if(ethStatus != TM_OK) 
        {
            break;
        }

        timeout++;
    }  

    if(ethStatus != TM_OK)
    {
        return(ethStatus);
    }

    if( (timeout == PHY_TIMEOUT) && (regval & TMBSL_PHYDP83848_BMCR_RST_VAL) )
    {
        return(TMBSL_ERR_PHY_INIT_FAILED);	
    } 
    
#if 0

    /* Read PHY Identification Register */
    ethStatus = tmbslPhyDP83848Read(ethUnitId,tmbslPhyDP83848PhyIdr1,&id1);
    
    if (ethStatus != TM_OK)
    {
        return (ethStatus);
    }
    
    ethStatus = tmbslPhyDP83848Read(ethUnitId,tmbslPhyDP83848PhyIdr2,&id2);

    if (ethStatus != TM_OK)
    {
        return (ethStatus);
    }

  /*  if( ( id1 == 0x2000 ) && (id2 == 0x5C7A) )
    {
        // IDs are valid. Do Nothing 
    }
    else
    {
        return(TMBSL_ERR_PHY_NOT_SUPPORTED);
    }
*/
#endif

    regval =0;

    /* Set the PHY for the 10Mbps and Half Duplex mode */
    ethStatus = tmbslPhyDP83848Write(ethUnitId,tmbslPhyDP83848Bmcr,regval);

    if(ethStatus != TM_OK)
    {
        return (ethStatus);
    }

    return TM_OK;

}   

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848Deinit:
//
// DESCRIPTION: This function de-initializes the PHY device. Once the device is
//				deinitilized device will no more be available to access 
//
// RETURN:      TM_OK
//
// NOTES:       
//-----------------------------------------------------------------------------
//

tmErrorCode_t
tmbslPhyDP83848Deinit(
    tmUnitSelect_t  						ethUnitId   
    )

{

    tmErrorCode_t       		ethStatus=TM_OK;     		
    UInt16 bmcr =0;

    bmcr |= TMBSL_PHYDP83848_BMCR_RST_VAL;

    /* All the registers will be reset */
    ethStatus = tmbslPhyDP83848Write(ethUnitId,tmbslPhyDP83848Bmcr,bmcr);

    if(ethStatus != TM_OK)
    {
        return ethStatus;
    }
    else	
    {
        return TM_OK;
    }

}   


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848SetPowerState:
//
// DESCRIPTION: This function will set the Power State of the PHY device to specified 
//				power state
//
// RETURN:      TM_OK
//
// NOTES:       
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhyDP83848SetPowerState(
    tmUnitSelect_t           				ethUnitId ,  
    tmPowerState_t          				phyPowerState
    )

{
    tmErrorCode_t ethStatus=TM_OK;     		
    UInt16 bmcr = 0;

    ethStatus = tmbslPhyDP83848Read(ethUnitId, tmbslPhyDP83848Bmcr, &bmcr);

    if(ethStatus != TM_OK)
    {
        return ethStatus;
    }

    if( (phyPowerState == tmPowerOn) || (phyPowerState == tmPowerOff) )
    {
        if(phyPowerState == tmPowerOff)
        {
            bmcr |= TMBSL_PHYDP83848_BMCR_PWRDN_EN;
        }
        else
        {
            bmcr &=TMBSL_PHYDP83848_BMCR_PWRDN_CLR;
        }

        ethStatus = tmbslPhyDP83848Write(ethUnitId,tmbslPhyDP83848Bmcr,bmcr);

        return ethStatus;

    }
    else
    {
        return TMBSL_ERR_PHY_NOT_SUPPORTED;
    }

}   

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848GetPowerState:
//
// DESCRIPTION: This function will get the preset power state of the PHY device
//
// RETURN:      TM_OK
//
// NOTES:       
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhyDP83848GetPowerState(
    tmUnitSelect_t                  		ethUnitId ,  
    ptmPowerState_t				phyPowerState
    )

{

    tmErrorCode_t       		ethStatus=TM_OK;     		
    UInt16 regVal=0;

    ethStatus = tmbslPhyDP83848Read(ethUnitId,tmbslPhyDP83848Bmcr,&regVal);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    *phyPowerState = (((regVal & TMBSL_PHYDP83848_BMCR_PWRDN_EN) > 0) ? tmPowerOff : tmPowerOn);

    return TM_OK;

}   


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848GetBasicModeControl:
//
// DESCRIPTION: This function will get the basic configuration of the PHY device. 
//
// RETURN:      TM_OK
//
// NOTES:       See #define for the Basic Mode Control 
//-----------------------------------------------------------------------------

tmErrorCode_t 
tmbslPhyDP83848GetBasicModeControl (
    tmUnitSelect_t                   		ethUnitId,   
    ptmbslPhyBasicModeControl_t       pPhyBasicModeControl
    )
{
    tmErrorCode_t ethStatus=TM_OK;     		
    UInt16 bmcr;

    /* Read the present settings of the BMCR register */
    ethStatus = tmbslPhyDP83848Read(ethUnitId,tmbslPhyDP83848Bmcr,&bmcr);

    if(ethStatus != TM_OK)
    {
        return ethStatus;
    }

    pPhyBasicModeControl->enableCollisionTest = 
                (((bmcr & TMBSL_PHYDP83848_BMCR_COLTEST ) > 0) ? True : False);

    pPhyBasicModeControl->duplexMode = 
                (((bmcr & TMBSL_PHYDP83848_BMCR_FD_EN ) > 0) ? tmbslPhyFullDuplex : tmbslPhyHalfDuplex);

    if(bmcr & TMBSL_PHYDP83848_BMCR_SPEED_MSK)
    {
        pPhyBasicModeControl->speed = tmbslPhySpeed100Mbps;            
    }
    else
    {
        pPhyBasicModeControl->speed = tmbslPhySpeed10Mbps;                            
    }

    return TM_OK;

}   

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848SetBasicModeControl:
//
// DESCRIPTION: This function will configure the PHY device for the Basic Mode. 
//
// RETURN:      TM_OK
//
// NOTES:      
//-----------------------------------------------------------------------------

tmErrorCode_t 
tmbslPhyDP83848SetBasicModeControl (
    tmUnitSelect_t                           ethUnitId,   
    ptmbslPhyBasicModeControl_t        pPhyBasicModeControl
    )
{

    tmErrorCode_t       		ethStatus=TM_OK;     		
    UInt16 bmcr =0;

    if(pPhyBasicModeControl->enableCollisionTest == True)
    {
        bmcr |= TMBSL_PHYDP83848_BMCR_COLTEST;
    }

    if(pPhyBasicModeControl->duplexMode == True)
    {
        bmcr |= TMBSL_PHYDP83848_BMCR_FD_EN;
    }

    switch(pPhyBasicModeControl->speed)
    {

        case tmbslPhySpeed100Mbps :       
            bmcr |= TMBSL_PHYDP83848_BMCR_SPEED_100;                
            break;

        case tmbslPhySpeed10Mbps :       
            bmcr |= TMBSL_PHYDP83848_BMCR_SPEED_10;                
            break;

        default:
            break;

    }

    /* Write the result to the BMC register */
    ethStatus = tmbslPhyDP83848Write(ethUnitId,tmbslPhyDP83848Bmcr,bmcr);

    return ethStatus;

}   


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848GetBasicModeStatus:
//
// DESCRIPTION: This function will get the Basic Mode Status of the PHY device 
//				such as the speed, duplex mode and other statuses
//
// RETURN:      TM_OK
//
// NOTES:      
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhyDP83848GetBasicModeStatus (
    tmUnitSelect_t                  		ethUnitId,   
    ptmbslPhyBasicModeStatus_t    	pPhyBasicModeStatus     
    )

{
    tmErrorCode_t ethStatus=TM_OK;     		
    UInt16           bmsr,bmcr,phySts;

    /* Read the PHY status from the BMSR register */
    ethStatus = tmbslPhyDP83848Read(ethUnitId, tmbslPhyDP83848Bmsr, &bmsr);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    /* Read the PHY control register from the BMCR register */
    ethStatus = tmbslPhyDP83848Read(ethUnitId, tmbslPhyDP83848Bmcr, &bmcr);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    /* Read the autonegotiation status from PHY status register */
    ethStatus = tmbslPhyDP83848Read(ethUnitId, tmbslPhyDP83848Phystsr, &phySts);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    pPhyBasicModeStatus->jabberDetect = (((bmsr & TMBSL_PHYDP83848_BMSR_JAB_VAL) > 0) ? True : False);

    pPhyBasicModeStatus->remoteFaultDetected = (((bmsr & TMBSL_PHYDP83848_BMSR_RF_VAL) > 0) ? True : False);

    pPhyBasicModeStatus->autoNegotiationComplete = (((bmsr & TMBSL_PHYDP83848_BMSR_AN_VAL) > 0) ? True : False);

    pPhyBasicModeStatus->loopBackEnabled = (((bmcr & TMBSL_PHYDP83848_BMCR_LPBK_VAL) > 0) ? True : False);

    if((bmcr & TMBSL_PHYDP83848_BMCR_AN_EN) == 0) 
    {
        /* Not an auto negotiation. So read the values from BMCR */
        pPhyBasicModeStatus->duplexMode = 
                (((bmcr & TMBSL_PHYDP83848_BMCR_FD_EN ) > 0) ? tmbslPhyFullDuplex : tmbslPhyHalfDuplex);

        if(bmcr & TMBSL_PHYDP83848_BMCR_SPEED_MSK)
        {
            pPhyBasicModeStatus->speed = tmbslPhySpeed100Mbps;            
        }
        else
        {
            pPhyBasicModeStatus->speed = tmbslPhySpeed10Mbps;                        
        }

    }
    else 
    {	
        /* If autonegotiation is enabled, read from PHYSTS register */
        pPhyBasicModeStatus->duplexMode = 
                (((phySts & TMBSL_PHYDP83848_PHYSTS_DUP_MODE) > 0) ? tmbslPhyFullDuplex : tmbslPhyHalfDuplex);

        pPhyBasicModeStatus->speed= 
                (((phySts & TMBSL_PHYDP83848_PHYSTS_SPEED_10) > 0) ? tmbslPhySpeed10Mbps : tmbslPhySpeed100Mbps);



    }

    return TM_OK;

}   


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848AutoNegotiate:
//
// DESCRIPTION: This function will enable the Auto negotiation of the PHY device 
//				with Link Partner. Best possible performance configuration is 
//				selected automatically during this process
//
// RETURN:      TM_OK
//
// NOTES:      
//-----------------------------------------------------------------------------
	
tmErrorCode_t
tmbslPhyDP83848AutoNegotiate (
    tmUnitSelect_t                       	      ethUnitId,   
    ptmbslPhyAutoNegotitationMask_t      	pAutoNegotiationMask
    )
{

    tmErrorCode_t   ethStatus=TM_OK;
    UInt32             timeout=AUTO_NEG_DELAY_MULTIPLIER; /* Timeout of 800*5 msecs = 4 Sec */

    UInt16 regVal = ANAR_DEFAULT_VAL;
    UInt16 mask =0;

    ethStatus = tmbslPhyDP83848Write(ethUnitId,tmbslPhyDP83848Anar,regVal);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    if(pAutoNegotiationMask->masknextPageDesired == True)
    {
        mask |= TMBSL_PHYDP83848_ANAR_NP;        
    }

    if(pAutoNegotiationMask->maskRemoteFault == True)
    {
        mask |= TMBSL_PHYDP83848_ANAR_ADV_RF;        
    }

    if(pAutoNegotiationMask->maskAsymmetricPause == True)
    {
        mask |= TMBSL_PHYDP83848_ANAR_AP;        
    }

    if(pAutoNegotiationMask->maskPauseFrame == True)
    {
        mask |= TMBSL_PHYDP83848_ANAR_PAUSE;        
    }

    if(pAutoNegotiationMask->mask100BaseTxFullDuplexSupport == True)
    {
        mask |= TMBSL_PHYDP83848_ANAR_100B_TX_FD;        
    }

    if(pAutoNegotiationMask->mask100BaseTxSupport == True)
    {
        mask |= TMBSL_PHYDP83848_ANAR_100B_TX_HD;        
    }

    if(pAutoNegotiationMask->mask10BaseTFullDuplexSupport == True)
    {
        mask |= TMBSL_PHYDP83848_ANAR_10B_TX_FD;        
    }    

    if(pAutoNegotiationMask->mask10BaseTSupport == True)
    {
        mask |= TMBSL_PHYDP83848_ANAR_10B_TX_HD;        
    }    

    /* Clear the corresponding bits in the regVal*/
    regVal &= ~(mask);

    /* Program the Auto negotiation mask */
    tmbslPhyDP83848Write(ethUnitId,tmbslPhyDP83848Anar,regVal);

    /* Enable the Auto negotiation in the BMCR register 
    ** First clear the auto negotiate bit and then enable
    */
    tmbslPhyDP83848Read(ethUnitId,tmbslPhyDP83848Bmcr,&regVal);

    /* Clear autonegotiation bit */
    regVal &= TMBSL_PHYDP83848_BMCR_AN_CLR;

    regVal |= TMBSL_PHYDP83848_BMCR_AN_EN;
    
    tmbslPhyDP83848Write(ethUnitId,tmbslPhyDP83848Bmcr,regVal);
	
    /* Wait till the Auto negotiation is complete Or Do we need to set the Timeout */
    tmbslPhyDP83848Read(ethUnitId,tmbslPhyDP83848Bmsr,&regVal);

    /* While the Autonegotiation is not complete, stay in loop */
    while(((regVal & TMBSL_PHYDP83848_BMSR_AN_VAL) != TMBSL_PHYDP83848_BMSR_AN_VAL) && 
           (timeout != 0 ) )
    {
        mdelay(5);
        
        ethStatus = tmbslPhyDP83848Read(ethUnitId,tmbslPhyDP83848Bmsr,&regVal);

        if(ethStatus != TM_OK) 
        {
            return ethStatus;
        }

        timeout--;
    }

    if(  (timeout <= 0) && 
         ( (regVal & TMBSL_PHYDP83848_BMSR_AN_VAL) != TMBSL_PHYDP83848_BMSR_AN_VAL)
      )
    {
        return(-1);
    }

    GMAC_DBG("Autonegotiation Time:%d msec\n",(AUTO_NEG_DELAY_MULTIPLIER-timeout)*5);

    return TM_OK;

}   

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848LoopBack:
//
// DESCRIPTION: Function will enable or disable the PHY device in the Loopback 
//				mode.
//
// RETURN:      TM_OK
// NOTES:      
//-----------------------------------------------------------------------------


tmErrorCode_t
tmbslPhyDP83848LoopBack (
    tmUnitSelect_t                  ethUnitId,   
    tmbslPhyEnableDisable_t     loopbackMode
    )
{

    tmErrorCode_t ethStatus=TM_OK;     		
    UInt16 bmcr;

    /* Read the existing settings of the BMCR register */
    ethStatus = tmbslPhyDP83848Read(ethUnitId,tmbslPhyDP83848Bmcr,&bmcr);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    if(tmbslPhyEnable == loopbackMode)
    {
        bmcr |= TMBSL_PHYDP83848_BMCR_LPBK_VAL;
    }
    else if(tmbslPhyDisable == loopbackMode)
    {
        bmcr &= TMBSL_PHYDP83848_BMCR_LPBK_CLR;
    }
    else
    {
        ethStatus = TMBSL_ERR_PHY_NOT_SUPPORTED;
    }

    /* Write the Loopback setting to the BMCR register */
    ethStatus = tmbslPhyDP83848Write(ethUnitId,tmbslPhyDP83848Bmcr,bmcr);

    return ethStatus;

}   


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848SoftReset:
//
// DESCRIPTION: Function will do the soft reset of the PHY device

// RETURN:      TM_OK
//
// NOTES:      
//-----------------------------------------------------------------------------
//

tmErrorCode_t
tmbslPhyDP83848SoftReset (
    tmUnitSelect_t                     		ethUnitId      
	)

{
    tmErrorCode_t       		ethStatus=TM_OK;     		

    /* All the registers will be reset */
    ethStatus = tmbslPhyDP83848Write(ethUnitId,tmbslPhyDP83848Bmcr,TMBSL_PHYDP83848_BMCR_RST_VAL);

    return ethStatus;

}   

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848GetLinkStatus:
//
// DESCRIPTION: Function will get the link status

// RETURN:      TM_OK
//
// NOTES:      
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhyDP83848GetLinkStatus (
    tmUnitSelect_t                          ethUnitId,   
    ptmbslPhyEnableDisable_t            pLinkStatus
    )
{
    tmErrorCode_t ethStatus=TM_OK;     		
    UInt16 physts;

    /* Read the ANSR register */
    ethStatus = tmbslPhyDP83848Read(ethUnitId,tmbslPhyDP83848Phystsr,&physts);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    *pLinkStatus = 
    (((physts & TMBSL_PHYDP83848_PHYSTS_LINK_STAT) > 0) ? tmbslPhyEnable : tmbslPhyDisable);

    return(TM_OK);

}   

//-----------------------------------------------------------------------------
//	Local static functions	
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848Read:
//
// DESCRIPTION: Function will read from the Specified PHY register

// RETURN:      
//
// NOTES:      
//-----------------------------------------------------------------------------
//

static tmErrorCode_t
tmbslPhyDP83848Read (
    tmUnitSelect_t				ethUnitId,
    tmbslPhyDP83848Reg_t		reg,   
    pUInt16						pVal
	)

{

    UInt32  timeout=0;
    UInt32 pEthRegs;       
    volatile UInt32 *pAdrReg;
    volatile  UInt32 *pDataReg;
    UInt32 regValue;        

    /* get Ethernet Module Reg Pointer  */
    pEthRegs = gEthContext[ethUnitId].pRegs;

    /* Get the Address register */  
    pAdrReg = (UInt32*)(pEthRegs+TMHW_LIPP6100ETH_ADR_REG_OFFSET) ;
    pDataReg = (UInt32*)(pEthRegs+TMHW_LIPP6100ETH_DATA_REG_OFFSET) ;

    /* Program the Register address in the MII */
    regValue = *pAdrReg;
       
    /* Clear the earlier register value in Address register and write the new value */
    regValue &= TMHW_LIPP6100ETH_ADR_PHY_REG_CLR;  
    regValue |= reg << TMHW_LIPP6100ETH_ADR_PHY_REG_POS;

    /* Clear the bit GMII write for read operation */
    regValue &= TMHW_LIPP6100ETH_ADR_PHY_RD_CLR;

    regValue |= TMHW_LIPP6100ETH_ADR_PHY_EN_VAL;

    *pAdrReg = regValue;        

    /* Wait till the read operation is complete */

    do
    {
        timeout++;

    }while ( ((*pAdrReg & TMHW_LIPP6100ETH_ADR_PHY_EN_VAL) == tmbslPhyEnable) && 
                   (timeout < PHY_TIMEOUT) );

    if( (timeout == PHY_TIMEOUT) && 
        ((*pAdrReg & TMHW_LIPP6100ETH_ADR_PHY_EN_VAL) == tmbslPhyEnable) )
    {
        return(TMBSL_ERR_PHY_READ_FAILED);
    }  
	
    /* Read the data from the data register */

    *pVal = (UInt16) *pDataReg;
    
    return TM_OK;

}   


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848Write:
//
// DESCRIPTION: Function will Write to the Specified PHY register

// RETURN:      
//
// NOTES:      
//-----------------------------------------------------------------------------
//

static tmErrorCode_t
tmbslPhyDP83848Write (
    tmUnitSelect_t				ethUnitId,
    tmbslPhyDP83848Reg_t		reg,   
    UInt16						val
	)

{

    UInt32 pEthRegs;       
    volatile UInt32 *pAdrReg;
    volatile  UInt32 *pDataReg;
    UInt32 regValue;

    UInt32      timeout=0;

    /* Get the Ethernet Module Reg pointer */
    pEthRegs = gEthContext[ethUnitId].pRegs;

    /* Get the Address register */  
    pAdrReg = (UInt32*)(pEthRegs+TMHW_LIPP6100ETH_ADR_REG_OFFSET) ;
    pDataReg = (UInt32*)(pEthRegs+TMHW_LIPP6100ETH_DATA_REG_OFFSET) ;

    /* Write the data into data register  */
    *pDataReg = 0;
    *pDataReg = val;

    /* Program the Register address */
    regValue = *pAdrReg;
       
    /* Clear the earlier register value in Address register */
    regValue &= TMHW_LIPP6100ETH_ADR_PHY_REG_CLR;  
    regValue |= reg << TMHW_LIPP6100ETH_ADR_PHY_REG_POS;

    /* Set Write operation bit and enable write */
    regValue |= TMHW_LIPP6100ETH_ADR_PHY_WR_VAL | 
               TMHW_LIPP6100ETH_ADR_PHY_EN_VAL ;

    /* Write the value back to Address register */
    *pAdrReg = regValue;        

    /* wait till the write operation is complete */

    do
    {
        timeout++;
    }while ( ((*pAdrReg & TMHW_LIPP6100ETH_ADR_PHY_EN_VAL) == tmbslPhyEnable) && 
             (timeout < PHY_TIMEOUT) );
	
	if( (timeout == PHY_TIMEOUT) && 
           ((*pAdrReg & TMHW_LIPP6100ETH_ADR_PHY_EN_VAL) == tmbslPhyEnable) )
	{
	    return(TMBSL_ERR_PHY_WRITE_FAILED);
	}  

	return TM_OK;
    
}   

//-----------------------------------------------------------------------------




#ifdef __LIPP_6300ETH__
static void get_phy_out_of_rst( void)
{
    //Msg for High, to bring PHY out of reset
    unsigned char msg[]={0x00,0x04,0x00,0x02,0x07,0x01}; 

    //Msg for Low, to put PHY in reset     
//    unsigned char msg[]={0x00,0x04,0x00,0x02,0x07,0x00}; 

//    pnx8xxx_ip3203_init();
//    pnx8xxx_ip3203_write(msg, sizeof(msg));

    /*Wait till the PHY is out of reset */    
    mdelay(1);

    /* Turn the clocks to MAC from PHY */

#if 0
    /* PHY clocks */
    writel(0x00000003, (unsigned long *)(0xbbe47710));	
    writel(0x00000003, (unsigned long *)(0xbbe47714));	
    writel(0x00000003, (unsigned long *)(0xbbe47718));	
#endif 

    #ifdef __LIPP_6300ETH__
    /* PHY clocks */
    *(unsigned long *)(MMIO_BASE) = 0x00000003;
    *(unsigned long *)(MMIO_BASE) = 0x00000003;
    *(unsigned long *)(MMIO_BASE) = 0x00000003;
    #endif


    mdelay(1);
//    GMAC_DBG("\nETHERNET CLOCK1:%08x\n",*(unsigned int *)(0x1be47710));                    
//    GMAC_DBG("\nETHERNET CLOCK2:%08x\n",*(unsigned int *)(0x1be47714));                    
//    GMAC_DBG("\nETHERNET CLOCK3:%08x\n",*(unsigned int*)(0x1be47718));                       
//    GMAC_DBG("\nSMSC PHY INIT successful\n\n\n");    

    return;

}

#endif 



