/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Copyright (c) 2006-2007, LIPP Alliance
 * All Rights Reserved.
 *
 *---------------------------------------------------------------------------
 * %filename:     tmbslPhyDP83848.h %
 * %pid_version:              1.2              %
 *---------------------------------------------------------------------------
 * DESCRIPTION: Macros and function prototypes for DP83848 PHY
 *
 * DOCUMENT REF: Datasheet DP83848 Gig PHYTER V10/100/1000 Ethernet 
 *               Physical Layer, October 2004
 *
 * NOTES:        None
 *
 *-----------------------------------------------------------------------------
 *
*/

#ifndef _TMBSLPHYDP83848_H_
#define _TMBSLPHYDP83848_H_

#if        defined(__cplusplus)
extern "C"
{
#endif  // defined(__cplusplus)


#include "tmNxTypes.h"
#include "tmbslPhy.h"

/* Macros */

/* MAC Macros */
#define TMHW_LIPP6100ETH_ADR_REG_OFFSET (0x010)
#define TMHW_LIPP6100ETH_DATA_REG_OFFSET (0x014)

#define TMHW_LIPP6100ETH_ADR_PHY_REG_CLR (0xFFFFF83F)
#define TMHW_LIPP6100ETH_ADR_PHY_REG_POS (6)
#define TMHW_LIPP6100ETH_ADR_PHY_WR_VAL (0x2)
#define TMHW_LIPP6100ETH_ADR_PHY_RD_CLR (0xFFFFFFFD)
#define TMHW_LIPP6100ETH_ADR_PHY_EN_VAL (0x1)

/*PHY Macros */
/*************** Basic Mode control register ************/
#define TMBSL_PHYDP83848_BMCR_RST_VAL (0x8000)

/* Enable loopback */
#define TMBSL_PHYDP83848_BMCR_LPBK_VAL (0x4000)
#define TMBSL_PHYDP83848_BMCR_LPBK_CLR (0xBFFF)

/* Set Speed to 1Gbps */
#define TMBSL_PHYDP83848_BMCR_SPEED_1G (0x40)

/* Set Speed to 100Mbps */
#define TMBSL_PHYDP83848_BMCR_SPEED_100 (0x2000)

/* Set Speed */
#define TMBSL_PHYDP83848_BMCR_SPEED_10 (0)

/* Speed mask */
#define TMBSL_PHYDP83848_BMCR_SPEED_MSK (0x2000)

/* Enable autonegotiation */
#define TMBSL_PHYDP83848_BMCR_AN_EN (0x1000)
#define TMBSL_PHYDP83848_BMCR_AN_CLR (0xFFFFEFFF)

/* Set power down mode */
#define TMBSL_PHYDP83848_BMCR_PWRDN_EN (0x800)

/* Disable  power down mode */
#define TMBSL_PHYDP83848_BMCR_PWRDN_CLR (0xF7FF)

/* Isolate PHY enable */
#define TMBSL_PHYDP83848_BMCR_ISO_PHY (0x400)

/* Auto negotiation restart */
#define TMBSL_PHYDP83848_BMCR_AN_RESTART (0x200)

/* Collision Test enable */
#define TMBSL_PHYDP83848_BMCR_COLTEST (0x80)

/* Full duplex enable */
#define TMBSL_PHYDP83848_BMCR_FD_EN (0x100)

/****** Basic Mode status Register bits ******/

/* Autonegotiation complete value */
#define TMBSL_PHYDP83848_BMSR_T4100BASE (0x8000)

#define TMBSL_PHYDP83848_BMSR_X100BASEFD (0x4000)

#define TMBSL_PHYDP83848_BMSR_X100BASEHD (0x2000)

#define TMBSL_PHYDP83848_BMSR_10MBPSFD (0x1000)

#define TMBSL_PHYDP83848_BMSR_10MBPSHD (0x800)

#define TMBSL_PHYDP83848_BMSR_T2100BASEFD (0x400)

#define TMBSL_PHYDP83848_BMSR_T2100BASEHD (0x200)

/* Preamble suppression capability */
#define TMBSL_PHYDP83848_BMSR_PREAMBLE_SUP (0x40)

#define TMBSL_PHYDP83848_BMSR_AN_VAL (0x20)
#define TMBSL_PHYDP83848_BMCR_LPBK_VAL (0x4000)


/* Remote fault value */
#define TMBSL_PHYDP83848_BMSR_RF_VAL (0x10)

/* PHY is able to perform auto negotiation */
#define TMBSL_PHYDP83848_BMSR_AN_ABLE (0x8)

#define TMBSL_PHYDP83848_BMSR_LINK_STAT (0x4)

/* Jabber detected */
#define TMBSL_PHYDP83848_BMSR_JAB_VAL (0x2)



/****** Auto Negotiation Advertisement Register bits ******/

/* Advertise Next page desired */
#define TMBSL_PHYDP83848_ANAR_NP (0x8000)

/* Advertise remote fault */
#define TMBSL_PHYDP83848_ANAR_ADV_RF (0x2000)

/* Advertise asymmetric pause */
#define TMBSL_PHYDP83848_ANAR_AP (0x800)

/* Advertise pause frame support */
#define TMBSL_PHYDP83848_ANAR_PAUSE (0x400)

/* Advertise 100Base-TX full duplex support */
#define TMBSL_PHYDP83848_ANAR_100B_TX_FD (0x100)

/* Advertise 100Base-TX half duplex support */
#define TMBSL_PHYDP83848_ANAR_100B_TX_HD (0x80)

/* Advertise 10Base-TX full duplex support */
#define TMBSL_PHYDP83848_ANAR_10B_TX_FD (0x40)

/* Advertise 10Base-TX half duplex support */
#define TMBSL_PHYDP83848_ANAR_10B_TX_HD (0x20)


/****** 1KTCR : 1000 Base-T Master-Slave Control Register ******/

/* Maser/Slave config enable */
#define TMBSL_PHYDP83848_1KTCR_MS_CONFIG (0x1000)

/* Set PHY as master */
#define TMBSL_PHYDP83848_1KTCR_MASTER_EN (0x800)

/* Advertise device as Multiport */
#define TMBSL_PHYDP83848_1KTCR_MULTIPORT_EN (0x400)

/* 1000 Base-T Full duplex capable */
#define TMBSL_PHYDP83848_1KTCR_1000BT_FD (0x200)

/* 1000 Base-T Half duplex capable */
#define TMBSL_PHYDP83848_1KTCR_1000BT_HD (0x100)

/********1KSTSR 1000 BASE-T Master-Slave Status Register *****/
#define TMBSL_PHYDP83848_1KSTSR_MAN_FAULT (0x8000)
#define TMBSL_PHYDP83848_1KSTSR_MASTER (0x4000)
#define TMBSL_PHYDP83848_1KSTSR_LOCAL_RX_STAT (0x2000)
#define TMBSL_PHYDP83848_1KSTSR_REMOTE_RX_STAT (0x1000)
#define TMBSL_PHYDP83848_1KSTSR_PART_FD_CAP (0x800)
#define TMBSL_PHYDP83848_1KSTSR_PART_HD_CAP (0x400)

/********** EXTENDED STATUS REGISTER ******************/
#define TMBSL_PHYDP83848_1KSCR_1000BASEX_FD (0x8000)
#define TMBSL_PHYDP83848_1KSCR_1000BASEX_HD (0x4000)
#define TMBSL_PHYDP83848_1KSCR_1000BASET_FD (0x2000)
#define TMBSL_PHYDP83848_1KSCR_1000BASET_HD (0x1000)

/**************** VENDOR SPECIFIC REGISTERS **************/
/****** STRAP options register ******/
#define TMBSL_PHYDP83848_STRAP_ANE (0x8000)
#define TMBSL_PHYDP83848_STRAP_DUP (0x4000)

/* Bit 13:12  similar to bits 6:13 in basic mode control register */
#define TMBSL_PHYDP83848_STRAP_SPD_MSK (0x3000)
#define TMBSL_PHYDP83848_STRAP_SPD_1G (0x2000)
#define TMBSL_PHYDP83848_STRAP_SPD_100 (0x1000)
#define TMBSL_PHYDP83848_STRAP_SPD_10 (0x0)

#define TMBSL_PHYDP83848_PHYSTS_SPEED_10 (0x2)
#define TMBSL_PHYDP83848_PHYSTS_LINK_STAT (0x1)
#define TMBSL_PHYDP83848_PHYSTS_DUP_MODE (0x4)




//Structure Declarations
//-----------------------------------------------------------------------------
//	Typedefinition
//-----------------------------------------------------------------------------
typedef struct  _tmbslPhyDP83848Context_t
{
    UInt32    			pRegs;         // Array ETHERNET Module regs 
} tmbslPhyDP83848Context_t, *ptmbslPhyDP83848Context_t;

//-----------------------------------------------------------------------------
// Exported functions:
//-----------------------------------------------------------------------------
//

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetSWVersion:
//
// DESCRIPTION: This function returns the PHY device interface software version 
//				information
//
// RETURN:      TM_OK
//
// NOTES:       This API can be called anytime i.e. before initializing the PHY 
//				or in PowerOff state.
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhyDP83848GetSWVersion (
    ptmSWVersion_t      					pPhyVersion    
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetCapabilities:
//
// DESCRIPTION: This function returns the PHY capabilities for the specified PHY
//				unit. The function is callable at any time to return the unit's 
//				capabilities (PHY unit initialization is not necessary). 
//				Capabilities may be different among multiple PHY units.For completeness, 
//				a PHY BSL user should call this function for each PHY unit to 
//				determine its individual capabilities. 
//
// RETURN:      TM_OK
//
// NOTES:       This API can be called anytime i.e. before initializing the PHY 
//				or in PowerOff state. 
//-----------------------------------------------------------------------------
//

tmErrorCode_t
tmbslPhyDP83848GetCapabilities (
    tmUnitSelect_t                			phyUnitId,  
    ptmbslPhyCapabilities_t  				pPhyCaps    
    );

    
//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyInit:
//
// DESCRIPTION: This function initializes the PHY device. It should be called
//				before any access to the device is made. 
//
// RETURN:      TM_OK 

//				TM_BSLPHY_ERR_BAD_UNIT_NUM
//
// NOTES:       This function initializes the PHY device with the following 
//				default initial configuration.
//				1. Enable the auto negotiation. In auto-negotiation mode the 
//				highest possible performance mode is selected automatically 
//				which the Link Partner also supports. (In auto negotiation mode
//				speed and duplex mode will be selected by means of auto negotiation. 
//				Writing to speed and duplex mode does not have meaning in auto 
//				negotiation mode.)
//				2. The device abilities are programmed to Auto negotiation advertise 
//				register. None of the device abilities are masked. If the PHY 
//				device does not support Auto-negotiation mode, then this function 
//				will set the PHY to basic capabilities of the device.The function 
//				tmbslPhyGetBasicModeControl can be called after tmbsPhyInit to 
//				know what are the default configurations the tmbslPhyInit function 
//				has set to
 
//-----------------------------------------------------------------------------
//
    
tmErrorCode_t
tmbslPhyDP83848Init(
    tmUnitSelect_t  						phyUnitId   
    );


    
//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDeinit:
//
// DESCRIPTION: This function de-initializes the PHY device. Once the device is
//				deinitilized device will no more be available to access 
//
// RETURN:      TM_OK
//				TM_BSLPHY_ERR_BAD_UNIT_NUM
//
// NOTES:       
//-----------------------------------------------------------------------------
//
tmErrorCode_t
tmbslPhyDP83848Deinit(
    tmUnitSelect_t  						phyUnitId   
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySetPowerState:
//
// DESCRIPTION: This function will set the Power State of the PHY device to specified 
//				power state
//
// RETURN:      TM_OK
//				TM_BSLPHY_ERR_BAD_UNIT_NUM
//
// NOTES:       
//-----------------------------------------------------------------------------
//

tmErrorCode_t
tmbslPhyDP83848SetPowerState(
    tmUnitSelect_t           				phyUnitId ,  
    tmPowerState_t          				phyPowerState
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetPowerState:
//
// DESCRIPTION: This function will get the preset power state of the PHY device
//
// RETURN:      TM_OK
//				TM_BSLPHY_ERR_BAD_UNIT_NUM
//
// NOTES:       
//-----------------------------------------------------------------------------
//

tmErrorCode_t
tmbslPhyDP83848GetPowerState(
    tmUnitSelect_t                  		phyUnitId ,  
    ptmPowerState_t							phyPowerState
    );


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetBasicModeControl:
//
// DESCRIPTION: This function will get the basic configuration of the PHY device. 
//
// RETURN:      TM_OK
//				TM_BSLPHY_ERR_BAD_UNIT_NUM
//
// NOTES:       PHY device is initialized using the function tmbslPhyInit.  
//				tmbslPhyInit Enable the Auto negotiation mode and will configure 
//				the PHY device for the maximum performance mode available. After 
//				the device is initialized this function can be called to know the 
//				present settings of the PHY device. If application wants to change 
//				the settings it can call tmbslPhySetBasicModeControl to change 
//				the configuration after knowing the capabilities of the PHY device 
//				and Link Partner. #define for the Basic Mode Control 
//-----------------------------------------------------------------------------
//

tmErrorCode_t 
tmbslPhyDP83848GetBasicModeControl (
    tmUnitSelect_t                   		phyUnitId,   
    ptmbslPhyBasicModeControl_t      		pPhyBasicModeControl
    );

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySetBasicModeControl:
//
// DESCRIPTION: This function will configure the PHY device for the Basic Mode. 
//
// RETURN:      TM_OK
//				TM_BSLPHY_ERR_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//


tmErrorCode_t 
tmbslPhyDP83848SetBasicModeControl (
    tmUnitSelect_t                   		phyUnitId,   
    ptmbslPhyBasicModeControl_t      		pPhyBasicModeControl
    );

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetBasicModeStatus:
//
// DESCRIPTION: This function will get the Basic Mode Status of the PHY device 
//				such as the speed, duplex mode 
//
// RETURN:      TM_OK
//				TM_BSLPHY_ERR_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

tmErrorCode_t
tmbslPhyDP83848GetBasicModeStatus (
    tmUnitSelect_t                  		phyUnitId,   
    ptmbslPhyBasicModeStatus_t    			pPhyBasicModeStatus     
    );

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyAutoNegotiate:
//
// DESCRIPTION: This function will enable the Auto negotiation of the PHY device 
//				with Link Partner. Best possible performance configuration is 
//				selected automatically during this process
//
// RETURN:      TM_OK
//				TM_BSLPHY_ERR_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

tmErrorCode_t
tmbslPhyDP83848AutoNegotiate (
    tmUnitSelect_t                       	phyUnitId,   
    ptmbslPhyAutoNegotitationMask_t      	pAutoNegotiationMask
    );

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyDP83848LoopBack:
//
// DESCRIPTION: Function will enable or disable the PHY device in the Loopback mode.
//
// RETURN:      TM_OK
//				TM_BSLPHY_ERR_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

tmErrorCode_t
tmbslPhyDP83848LoopBack (
    tmUnitSelect_t                      	phyUnitId,   
    tmbslPhyEnableDisable_t           		loopbackMode
	);


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySoftReset:
//
// DESCRIPTION: Function will do the soft reset of the PHY device

// RETURN:      TM_OK
//				TM_BSLPHY_ERR_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

tmErrorCode_t
tmbslPhyDP83848SoftReset (
    tmUnitSelect_t                     		phyUnitId      
	);

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhyGetLinkStatus:
//
// DESCRIPTION: Function will get the link status

// RETURN:      TM_OK
//				TM_BSLPHY_ERR_BAD_UNIT_NUM
//
// NOTES:      
//-----------------------------------------------------------------------------
//

tmErrorCode_t
tmbslPhyDP83848GetLinkStatus (
    tmUnitSelect_t                          phyUnitId,   
    ptmbslPhyEnableDisable_t           		pLinkStatus
	);

#if        defined(__cplusplus)
}
#endif  // defined(__cplusplus)

#endif //_TMBSLPHYDP83848_H_
