/*
 * Copyright (c) 1999-2003 Petko Manolov (petkan@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */

#include <common.h>
#include <command.h>

#ifdef  CONFIG_CMD_NET

#include <usb.h>
#include <net.h>
#include <linux/mii.h>
#include <miiphy.h>

#include "usb.h"            
#include "usb_net.h"

#include "pegasus.h"

/* these are usb endpoints for urb transfer */
#define PEGASUS_CTRL_ENDPOINT       (0)
#define PEGASUS_BULK_IN_ENDPOINT    (1)
#define PEGASUS_BULK_OUT_ENDPOINT   (2)
#define PEGASUS_INTR_IN_ENDPOINT    (3)

/* these timeouts are in fact dummies _but_ the usb layer interprets them nonetheless! */
#define PEGASUS_CTRL_URB_TIMEOUT    (1)
#define PEGASUS_BULK_URB_TIMEOUT    (1)

/* move these defines to a headerfile somewhere,
 * "far.. far away.." */
#define MCS_RD_BMREQ    (0xc0)
#define MCS_WR_BMREQ    (0x40)
#define MCS_RD_BREQ (0x0e)
#define MCS_WR_BREQ (0x0d)

#define DEVICE_REV_B    (0x0100)
#define DEVICE_REV_C    (0x0200)

/* mii defines */
#define MII_DATA_REGISTER_LO    (HIF_REG_11)
#define MII_DATA_REGISTER_HI    (HIF_REG_12)
#define MII_OPCODE_REGISTER (HIF_REG_13)
#define MII_CONTROL_REGISTER    (HIF_REG_14)

#define MII_COMMAND_WRITE   (0x01)
#define MII_COMMAND_READ    (0x02)

#define MII_CONTROL_READY   (1 << 6)
#define MII_CONTROL_PENDING (1 << 7)

#define MII_PHY_CONTROL_REG     (0)
#define MII_CONTROL_RESET       (1 << 15)
#define MII_CONTROL_LOOPBACK        (1 << 14)
#define MII_CONTROL_10          (0 << 13) | (0 << 6)
#define MII_CONTROL_100         (0 << 13) | (1 << 6)
#define MII_CONTROL_ANG_EN      (1 << 12)
#define MII_CONTROL_POWERDOWN       (1 << 11)
#define MII_CONTROL_ISOLATE     (1 << 10)
#define MII_CONTROL_ANG_RESTART     (1 << 9)
#define MII_CONTROL_FULL_DUPLEX     (1 << 8)
#define MII_CONTROL_HALF_DUPLEX     (0 << 8)
#define MII_CONTROL_COLLISION_TEST  (1 << 7)
#define MII_CONTROL_RESERVED        (1 << 5) | (1 << 4) | (1 << 3) | (1 << 2) | (1 << 1) | (1 << 0)

#define MII_PHY_STATUS_REG      (1)
#define MII_STATUS_ANG_COMPLETE     (1 << 5)

/* timeout for restarting ang */
#define ANG_RETRY           (100)   /* poll completion this many times */
#define ANG_RETRY_DELAY         (10)    /* wait this many milliseconds between each poll */

/* err, info, dbg, TRACE defines */
#undef DEBUG
#undef SHOW_INFO
#undef SHOW_DBG
#undef SHOW_TRACE

/* some handy macros for printing errors, tracing function calls
 * showing info.. */
#ifdef  DEBUG
    #define err(format, arg...)                     \
        do {                            \
            printf(                     \
                "ERROR: %s;%s(%d): " format "\n",   \
                                __FILE__, __FUNCTION__, __LINE__,       \
                                ## arg);                                \
                } while (0)
#else   /* ! DEBUG */
        #define err(format, arg...) printf("ERROR: " format "\n", ## arg)
#endif  /* DEBUG */

#ifdef SHOW_INFO
    #ifdef DEBUG
        #define info(format, arg...)                    \
            do {                            \
                printf(                     \
                    "INFO: %s;%s(%d): " format "\n",    \
                    __FILE__, __FUNCTION__, __LINE__,   \
                    ## arg);                \
            } while (0)
    #else   /* ! DEBUG */
        #define info(format, arg...)            \
            do {                    \
                printf(             \
                    "INFO: " format "\n",   \
                    ## arg);        \
            } while (0)
    #endif  /* DEBUG */
#else   /* ! SHOW_INFO */
    #define info(format, arg...) do {} while(0)
#endif /* SHOW_INFO */

#ifdef  SHOW_TRACE
    #ifdef  DEBUG
        #define trace(format, arg...)                   \
            do {                            \
                printf(                     \
                    "TRACE: %s;%s(%d): " format "\n",   \
                    __FILE__, __FUNCTION__, __LINE__,   \
                    ## arg);                \
            } while (0)
    #else   /* ! DEBUG */
        #define trace(format, arg...)           \
            do {                    \
                printf(             \
                    "TRACE: " format "\n",  \
                    ## arg);        \
            } while (0)
    #endif  /* DEBUG */
#else   /* ! SHOW_TRACE */
    #define trace(format, arg...) do {} while(0)
#endif  /* SHOW_TRACE */

#define PER_USB_ETH_STATUS_LEN      (8)
#define RX_USB_ETH_RX_PACKET_LEN    (64)
#define RX_ETH_PACKET_LEN       (1518)

#define RX_ENABLED  (1 << 0)
#define TX_ENABLED  (1 << 1)

/* our administration */
struct pegasus      g_usb_eth;      /* could be an array */
struct pegasus*     g_p_usb_eth;    /* use this pegasus device */

/* eth administration */
struct eth_device   g_eth_dev;  /* use this stuff to feed `eth.c' */

typedef enum per_state {
    per_not_submitted = 0,
    per_resubmit = 1,
    per_stop = 2
} per_state_t;

/* periodically updated with usb eth status */
volatile unsigned int   g_per_usb_eth_status[PER_USB_ETH_STATUS_LEN] __attribute((aligned(CONFIG_SYS_CACHELINE_SIZE)));
volatile per_state_t    g_per_usb_eth_state;

/* intermediate usb eth packet storage */
volatile unsigned char  g_rx_usb_eth_packet[RX_USB_ETH_RX_PACKET_LEN] __attribute((aligned(CONFIG_SYS_CACHELINE_SIZE)));

/* eth packet is built here */
volatile unsigned char  g_rx_eth_packet[RX_ETH_PACKET_LEN] __attribute((aligned(CONFIG_SYS_CACHELINE_SIZE)));
volatile int        g_rx_eth_packet_offs;
volatile char       g_rx_eth_packet_available;

#define USB_ETH_DEV_MAX (1)

static int g_usb_net_total_devs = 0;    /* number of highest available usb device */

int pegasus_probe(struct usb_device* p_usb_dev, struct eth_device* p_eth_device, unsigned int ifnum, pegasus_t* p_pegasus);

int 
mcs7830_miiphy_read(
    char *devname,
    unsigned char addr,
    unsigned char reg,
    unsigned short *value);
int
mcs7830_miiphy_write(
    char *devname,
    unsigned char addr,
    unsigned char reg,
    unsigned short value);
    
/* forward declarations */
int mcs7830_read_register(pegasus_t* pegasus, __u16 indx, __u16 size, void* data);
int mcs7830_write_register(pegasus_t* pegasus, __u16 indx, __u16 size, __u8* data);
static int ANGInitializeDev(struct eth_device* p_eth_device);

#if     defined (SHOW_INFO)
static void maybe_print_receiver_status(char rcv_status);
#endif  /* defined (SHOW_INFO) */


int
ReadPhyRegisterPassive(
    pegasus_t*  p_pegasus,
    __u8        PhyRegIndex,
    __u16*      PhyRegValue) {

    int res;
    int retry;
    __u8    tmp_u8; /* temporary storage */

    res = mcs7830_write_register(   /* try to access `hif11' */
        p_pegasus,
        HIF_REG_11,
        2,
        &tmp_u8);
    if (0 > res) {
        err("failed, errorcode `%d'" , res);
        return (res);
    }

    tmp_u8 = READ_OPCODE | FARADAY_PHY_ADDRESS;
    res = mcs7830_write_register(   /* set opcode */
        p_pegasus,
        HIF_REG_13,
        1,
        &tmp_u8);
    if (0 > res) {
        err("failed, errorcode `%d'" , res);
        return (res);
    }

    tmp_u8 = HIF_REG_14_PEND_FLAG_BIT | (PhyRegIndex & 0x1F);
    res = mcs7830_write_register(   /*  */
        p_pegasus,
        HIF_REG_14,
        1,
        &tmp_u8);
    if (0 > res) {
        err("failed, errorcode `%d'" , res);
        return (res);
    }

    for (
        retry = 10;
        0 < retry;
        --retry) {
        tmp_u8 = 0x00;
        res = mcs7830_read_register(
            p_pegasus,
            HIF_REG_14,
            1,
            &tmp_u8);
        if (0 > res) {
            err("failed, errorcode `%d'" , res);
            break;
        }

        if (HIF_REG_14_READY_FLAG_BIT == (tmp_u8 & HIF_REG_14_READY_FLAG_BIT)) {
            trace("command completed");
            break;
        }
        res = -1;
    }

    if (0 > res) {
        err("failed, errorcode `%d', retry count expired", res);
        return (res);
    }

        res = mcs7830_read_register(
        p_pegasus,
        HIF_REG_11,
        2,
        (__u8*)PhyRegValue);
    if (0 > res) {
        err("failed, errorcode `%d'" , res);
    }

    trace("res=%d", res);
    return (res);
}


int
WritePhyRegisterPassive(
    pegasus_t*  p_pegasus,
    __u8        PhyRegIndex,
    __u16*      PhyRegValue) {

    int res;
    int retry;
    __u8    tmp_u8; /* temporary storage */

    trace("p_pegasus=0x%p, PhyRegIndex=0x%2.2x, PhyRegValue=0x%2.2x",
        (void*)p_pegasus, PhyRegIndex, *PhyRegValue);

    res = mcs7830_write_register(
        p_pegasus,
        HIF_REG_11,
        2,
        (__u8*)PhyRegValue);
    if (0 > res) {
        err("failed, errorcode `%d'" , res);
        return (res);
    }

    tmp_u8 = WRITE_OPCODE | FARADAY_PHY_ADDRESS;
    res = mcs7830_write_register(
        p_pegasus,
        HIF_REG_13,
        1,
        &tmp_u8);
    if (0 > res) {
        err("failed, errorcode `%d'" , res);
        return (res);
    }

    // 2. Build value for HIF_REG14
        // Set Pend Flag, Reset Ready Flag & address of Reg.with in the PHY
        // Will procesed By MAC - MIIM Hardware.
        // Msb 1000 0000 Lsb - b7 Pend, b6 Ready,
        // b5 Not used,b4b3b2b1b0 Address of Reg with in the PHY

    tmp_u8 = HIF_REG_14_PEND_FLAG_BIT | (PhyRegIndex & 0x1F);
    res = mcs7830_write_register(
        p_pegasus,
        HIF_REG_14,
        1,
        &tmp_u8);
    if (0 > res) {
        err("failed, errorcode `%d'" , res);
        return (res);
    }

    // 3. Read (completion) Status for last command
        for (retry = 10; 0 < retry; --retry) {
        tmp_u8 = 0x00;

                res = mcs7830_read_register(
            p_pegasus,
            HIF_REG_14,
            1,
            &tmp_u8);
        if (0 > res) {
            err("failed, errorcode `%d'" , res);
            break;
        }

        if (HIF_REG_14_READY_FLAG_BIT == (tmp_u8 & HIF_REG_14_READY_FLAG_BIT)) {
                        break;
        }
        res = -1;
    }

    if (0 > res) {
        err("failed, errorcode `%d'" , res);
    }

    trace("res=%d", res);
    return (res);
}

/*
 * uboot eth interface functions.
 * these funcs should map on the pegasus linux driver calls,
 * which, in turn should map on the usb_ohci instead of the fully
 * featured linux usb stack.
 */

/* forward declaration of `eth_*' functions */
static int pegasus_init(struct eth_device* dev, bd_t* bis);
// static void pegasus_reset(struct eth_device* dev, bd_t *bis);
static void pegasus_set_rx_mode(struct eth_device* p_eth_dev, int on);
static void pegasus_set_tx_mode(struct eth_device* p_eth_dev, int on);
static void pegasus_halt(struct eth_device* dev);
static int pegasus_recv(struct eth_device* dev);
static int pegasus_send(struct eth_device* dev, volatile void *packet, int length);


int sanity_check(
    struct eth_device *dev) {

    struct usb_device*  p_usb_device;
    struct pegasus*     p_pegasus;

    p_pegasus = (pegasus_t*)dev->priv;
    p_usb_device = (struct usb_device*)p_pegasus->usb;

    if ((struct usb_device*)NULL == p_usb_device) {
        err("invalid usb device reference 0x%p", (void*)p_usb_device);
        return (-1);
    }
    if ((pegasus_t*)NULL == p_pegasus) {
        err("invalid usb eth device reference 0x%p", (void*)p_pegasus);
        return (-1);
    }
    if (p_pegasus != g_p_usb_eth) {
        err("incorrect usb eth device reference 0x%p, should be 0x%p",
            (void*)p_pegasus, (void*)&g_usb_eth);
        return (-1);
    }
    if (p_pegasus->net != dev) {
        err("invalid usb eth device backreference");
        return (-1);
    }
    return (0);
}


int pegasus_initialize(
    bd_t *bis) {

    struct eth_device*  p_eth_dev = &g_eth_dev;

    g_p_usb_eth = &g_usb_eth;

    trace("bis=0x%p", (void*)bis);
    /* scan usb bus for usb-eth device */
    //if (0 < (ret = usb_net_scan(0))) {
        /*
        p_eth_dev = (struct eth_device*)malloc(sizeof(struct eth_device));
        if ((struct eth_device*)NULL == p_eth_dev) {
            err("not enough memory to allocate eth device structure!");
            return (-1);
        }
        */
    sprintf(p_eth_dev->name, "MCS7830 USB-Eth");
    p_eth_dev->iobase = 0;
    p_eth_dev->init = pegasus_init;
    p_eth_dev->halt = pegasus_halt;
    p_eth_dev->send = pegasus_send;
    p_eth_dev->recv = pegasus_recv;
    eth_register(p_eth_dev);

#if  defined(CONFIG_MII) || defined(CONFIG_CMD_MII)
    /* register mii functionality */
    miiphy_register(
            p_eth_dev->name,
            mcs7830_miiphy_read,
            mcs7830_miiphy_write);
#endif  /* defined(CONFIG_MII)      ||
     * (CONFIG_COMMANDS & CFG_CMD_MII) */
    //}

    return (1);
}

static int pegasus_init(
    struct eth_device* dev,
    bd_t* bis) {

    int         ret = 0;
    int         poll_wait;  /* actually, busy-wait, but we're polling and not just `busy' */
    int         usb_eth_packet_size;    /* used when flushing the usb eth sram/fifo */
    struct usb_device*  p_usb_device;
    pegasus_t*      p_pegasus;
    unsigned short      tmp;

    trace("dev=0x%p, bis=0x%p", (void*)dev, (void*)bis);

    p_pegasus = (pegasus_t*)dev->priv;
    p_usb_device = p_pegasus->usb;

#ifdef  DEBUG
    if (0 != (ret = sanity_check(dev))) {
        err("failed sanity check!");
        return (ret);
    }
#endif  /* DEBUG */

    /* sandbox, yeey! */

    /* back to work.. aww man! */

    if (0 > p_usb_device->irq_status)
        g_per_usb_eth_state = per_not_submitted;
    else if (0 < p_usb_device->irq_status)
        g_per_usb_eth_state = per_stop;
    else
        g_per_usb_eth_state = per_not_submitted;

    pegasus_set_tx_mode(dev, 1);

    /* power up phy and restart ang */
    mcs7830_miiphy_write(
        (char*)NULL,        /* dont care, but fill something in here */
        FARADAY_PHY_ADDRESS,
        MII_PHY_CONTROL_REG,
        MII_CONTROL_RESET |
            MII_CONTROL_ANG_EN |
            MII_CONTROL_ANG_RESTART);

    for (poll_wait = ANG_RETRY; 0 < poll_wait; --poll_wait) {
        mcs7830_miiphy_read(
            (char*)NULL,        /* dont care, but fill something in here */
            FARADAY_PHY_ADDRESS,
            MII_PHY_STATUS_REG,
            &tmp);
        if (tmp & MII_STATUS_ANG_COMPLETE)
            break;
        wait_ms(ANG_RETRY_DELAY);
    }
    if (0 == poll_wait)
                err("timed out waiting for ang completion");

        g_rx_eth_packet_offs = 0;

        for (poll_wait = 500; (0 < poll_wait) && (per_not_submitted != g_per_usb_eth_state); --poll_wait) {
        wait_ms(1);
        //OHCI usb_event_poll();
    }

    if (per_not_submitted != g_per_usb_eth_state)
        err("unable to stop resubmission of periodic urb! (%d)", (int)g_per_usb_eth_state);

#if 0
    /* check if we already have enabled the periodic `interrupt-in' endpoint */
    if (
        (-1 == p_usb_device->irq_status) ||
        (per_not_submitted == g_per_usb_eth_state)) {
        ret = usb_submit_int_msg(   /* start usb interrupt message polling */
                        p_usb_device,
                        usb_rcvintpipe(p_usb_device, 3),
                        (void*)g_per_usb_eth_status,
                        sizeof(g_per_usb_eth_status),
                        2);    /* interrupt interval */
        if (0 > ret)
            err("error %d submitting usb interrupt message", ret);
        g_per_usb_eth_state = per_resubmit;     /* running */
    }
#endif

    for (
        poll_wait = 10;
        (0 < poll_wait) &&
        (-1 != p_usb_device->irq_status);
        --poll_wait) {

        wait_ms(1);
        //OHCI usb_event_poll();
    }

    /* flush any packets that are available */
    while (0 != g_rx_eth_packet_available) {
        /* reset flag, let it be updated periodically */
        g_rx_eth_packet_available = 0;

        pegasus_set_rx_mode(dev, 1);
        ret = usb_bulk_msg(
                    p_usb_device,
                    usb_rcvbulkpipe(p_usb_device, 1),
                    (void*)g_rx_usb_eth_packet,
                    64,
                    &usb_eth_packet_size,
                    PEGASUS_BULK_URB_TIMEOUT);
        if (0 > ret) {
            err("error %d reading from usb eth bulk-in", ret);
            break;
        }

        /* to be sure that hc has processed periodic list
         * we can wait a small amount of time and then poll
         * for completion */
        for (poll_wait = 10; 0 < poll_wait; --poll_wait) {
            wait_ms(1);
            //OHCI usb_event_poll();
        }
    }
    trace("ret=%d", ret);
    return (ret);
}

// static void pegasus_reset(
    // struct eth_device *dev,
    // bd_t *bis) {

    // trace("dev=0x%p, bis=0x%p", (void*)dev, (void*)bis);
    // pegasus_set_rx_mode(dev, 1); /* disable sleep mode in order to receive some frames */
    // pegasus_set_tx_mode(dev, 1); /* enable tx mode */
    // trace("");
// }

static void pegasus_halt(
    struct eth_device *dev) {

    struct usb_device*  p_usb_device;
    pegasus_t*      p_pegasus;
    int         poll_wait;

    trace("dev=0x%p", (void*)dev);

#ifdef  DEBUG
    if (0 != sanity_check(dev)) {
        err("sanity check failed!");
        return;
    }
#endif  /* DEBUG */

    p_pegasus = (pegasus_t*)dev->priv;
    p_usb_device = p_pegasus->usb;

    /* power down phy */
    mcs7830_miiphy_write(
                (char*)NULL,
                FARADAY_PHY_ADDRESS,
                MII_PHY_CONTROL_REG,
                MII_CONTROL_POWERDOWN);

        pegasus_set_rx_mode(dev, 0);
    pegasus_set_tx_mode(dev, 0);

    if (0 > p_usb_device->irq_status)
        g_per_usb_eth_state = per_not_submitted;
    else if (0 < p_usb_device->irq_status)
        g_per_usb_eth_state = per_stop;
    else
        g_per_usb_eth_state = per_not_submitted;

    for (poll_wait = 500; (0 < poll_wait) && (per_not_submitted != g_per_usb_eth_state); --poll_wait) {
        wait_ms(1);
        //OHCI usb_event_poll();
    }

    if (per_not_submitted != g_per_usb_eth_state)
        err("unable to stop resubmission of periodic urb! (%d)", (int)g_per_usb_eth_state);

    /* sandbox, yeey! */

    #if (0)
    {
        int port;
        int portstatus;
        int j;
        int err;

        /* find the port number we're at */
        if (p_usb_device) {

            for (j = 0; j < p_usb_device->maxchild; j++) {
                if (p_usb_device->children[j] == p_usb_device) {
                    port = j;
                    break;
                }
            }
            if (port < 0) {
                err("usb_new_device: cannot locate device's port..\n");
                //printf("usb_new_device: cannot locate device's port..\n");
                //return 1;
            }

            /* reset the port for the second time */
            err = hub_port_reset(p_usb_device->parent, port, &portstatus);
            if (err < 0) {
                err("\n     Couldn't reset port %i\n", port);
                //printf("\n     Couldn't reset port %i\n", port);
                //return 1;
            }
        }
    }
    #endif  /* 0 */

    /* back to work, aww man! */

    trace("bye!");
}

static void pegasus_set_rx_mode(
    struct eth_device*  p_eth_dev,
    int         on) {

    int         ret;
    struct usb_device*  p_usb_device;
    pegasus_t*      p_pegasus;
    __u8            hif_reg15;

    trace("dev=0x%p", (void*)p_eth_dev);

#ifdef  DEBUG
    if (0 != sanity_check(p_eth_dev)) {
        err("sanity check failed!");
        return;
    }
#endif  /* DEBUG */

    p_pegasus = (pegasus_t*)p_eth_dev->priv;
    p_usb_device = p_pegasus->usb;

    if (on) {
        if (p_eth_dev->state & RX_ENABLED) {
            trace("rx already enabled");
            return;
        } else {
            p_eth_dev->state |= RX_ENABLED;
        }
    } else {
        if (~p_eth_dev->state & RX_ENABLED) {
            trace("rx already disabled");
            return;
        } else {
            p_eth_dev->state &= ~RX_ENABLED;
                }
        }

        ret = mcs7830_read_register(    /* read configuration register */
                p_pegasus,
                HIF_REG_15,
                sizeof(hif_reg15),
                &hif_reg15);
        if (0 > ret)
                err("error %d reading configuration register %d", ret, HIF_REG_15);

    if (on) {
        //hif_reg15 &= ~(1 << 2);   /* disable sleep mode */
        hif_reg15 |= (1 << 4);
    } else {
        //hif_reg15 |= (1 << 2);    /* enable sleep mode */
        hif_reg15 &= ~(1 << 4);
        }
        hif_reg15 |= (1 << 15);

        ret = mcs7830_write_register(   /* set configuration register */
                p_pegasus,
                HIF_REG_15,
                sizeof(hif_reg15),
                &hif_reg15);
        if (0 > ret)
                err("error %d writing configuration register %d", ret, HIF_REG_15);

    trace("bye!");
}

static void pegasus_set_tx_mode(
    struct eth_device*  p_eth_dev,
    int         on) {

    int         ret;
    struct usb_device*  p_usb_device;
    pegasus_t*      p_pegasus;
    __u8            hif_reg15;

    trace("dev=0x%p", (void*)p_eth_dev);

#ifdef  DEBUG
    if (0 != sanity_check(p_eth_dev)) {
        err("sanity check failed!");
        return;
    }
#endif  /* DEBUG */

    p_pegasus = (pegasus_t*)p_eth_dev->priv;
    p_usb_device = p_pegasus->usb;


    if (on) {
        if (p_eth_dev->state & TX_ENABLED) {
            trace("tx already enabled");
            return;
        } else {
            p_eth_dev->state |= TX_ENABLED;
        }
    } else {
        if (~p_eth_dev->state & TX_ENABLED) {
            trace("tx already disabled");
            return;
        } else {
            p_eth_dev->state &= ~TX_ENABLED;
        }
        }


        ret = mcs7830_read_register(    /* read configuration register */
                p_pegasus,
                HIF_REG_15,
                sizeof(hif_reg15),
                &hif_reg15);
        if (0 > ret)
                err("error %d reading configuration register %d", ret, HIF_REG_15);

    if (on)
        hif_reg15 |= (1 << 3);  /* enable usb eth tx */
        else
                hif_reg15 &= ~(1 << 3); /* disable usb eth tx */

        ret = mcs7830_write_register(   /* set configuration register */
                p_pegasus,
                HIF_REG_15,
                sizeof(hif_reg15),
                &hif_reg15);
        if (0 > ret)
                err("error %d writing configuration register %d", ret, HIF_REG_15);

    trace("bye!");
}

static int
usb_eth_on_intr_completion(
        struct usb_device *dev) {

        int     resubmit = 1;        /* resubmit intr msg by default */
        int     i;

        if (per_stop == g_per_usb_eth_state) {
                g_per_usb_eth_state = per_not_submitted;
                resubmit = 0;
        }

        /* clear before checking if there are any pending packets */
    	g_rx_eth_packet_available = 0;

        /* check if there are any pending packets */
        for (i = 0; 8 > i; ++i) {
                if (g_per_usb_eth_status[i] & (1 << 15)) {
                        g_rx_eth_packet_available = 1;
                }
                g_per_usb_eth_status[i] = 0;
        }

        return resubmit;
}


static int pegasus_recv(struct eth_device *dev)
{
    int         ret = -1;
    struct usb_device   *p_usb_device;
    pegasus_t       *p_pegasus;
    int         usb_rx_bytes;
    int         eth_rx_available;

#ifdef  DEBUG
    if (0 != (ret = sanity_check(dev))) {
        err("failed sanity check!");
        return (ret);
    }
#endif  /* DEBUG */

    p_pegasus = (pegasus_t*)dev->priv;
    p_usb_device = (struct usb_device*)p_pegasus->usb;

    if ((sizeof(g_rx_usb_eth_packet) + g_rx_eth_packet_offs) > sizeof(g_rx_eth_packet)) {
        err("no more room for usb eth packets in eth packet storage!");
        return (-1);
    }

    //if (0 != g_rx_eth_packet_available) {
        /*
         * Reset this to prevent underruns, we cannot detect a `nack'
         * condition on usb protocol level so when we try to read
         * more than the dongle can provide, ohci implementation
         * hangs and the only thing you can do then is perform a
         * `usb reset' and hope for the best.
         */
        g_rx_eth_packet_available = 0;

        pegasus_set_rx_mode(dev, 1);

        /*
         * Some counters we use to keep track of offset in ethernet
         * packet and how much we actually received from the usb
         * eth dongle.
         */
        usb_rx_bytes = 0;
        eth_rx_available = sizeof(g_rx_eth_packet) - g_rx_eth_packet_offs;

        /*
         * Get as much data as we can handle at the momemt from
         * the usb eth dongle
         */
        ret = usb_bulk_msg(
            p_usb_device,
            usb_rcvbulkpipe(p_usb_device, 1),   /* replace with something less funky.. */
            (void *)&g_rx_eth_packet[g_rx_eth_packet_offs],
            eth_rx_available,
            &usb_rx_bytes,
            PEGASUS_BULK_URB_TIMEOUT);
        if (0 > ret) {
            err("error %d reading from usb eth bulk-in", ret);
            return (ret);
        }

        /*
         * Zero usb packet means end of ethernet packet so pass
         * it up to net layer.
         */
        if (usb_rx_bytes == 0) {
            trace("got zlp! eth packet coming up!");
            if (g_rx_eth_packet_offs > 0)
                --g_rx_eth_packet_offs;

            NetReceive((void*)g_rx_eth_packet, g_rx_eth_packet_offs);

            /* reset our administration */
            g_rx_eth_packet_offs = 0;
        } else if (usb_rx_bytes > 0) {
            trace("got usb eth packet!");
            if (usb_rx_bytes < eth_rx_available)
                --usb_rx_bytes;
            g_rx_eth_packet_offs += usb_rx_bytes;

            /*
             * If usb eth dongle gave us less than we asked
             * for, it is in fact telling us * `hey i havent
             * got any more bytes, you already have the entire
             * eth packet' so pass it up to net layer.
             */
            if (usb_rx_bytes < eth_rx_available) {
                --usb_rx_bytes;
#ifdef SHOW_INFO
                maybe_print_receiver_status(
                    g_rx_usb_eth_packet[usb_rx_bytes]);
#endif
                NetReceive((void*)g_rx_eth_packet, g_rx_eth_packet_offs);
                /* reset our administration */
                g_rx_eth_packet_offs = 0;
            }
            trace("g_rx_eth_packet_offs=%d", g_rx_eth_packet_offs);
        } else {
            err("what is wrong with that dongle? "
                "received impossible amount of bytes %d", usb_rx_bytes);
            return (-1);
        }
    //}

    if (0 != ret)
        trace("ret=%d", ret);
    return (ret);
}

static int pegasus_send(
    struct eth_device*  dev,
    volatile void*      packet,
    int         length) {

    int         ret = -1;   /* assume foobar */
    struct usb_device*  p_usb_device;
    struct pegasus*     p_pegasus;

    int usb_eth_packet_number;
    int usb_eth_packet_size;
    int eth_packet_size;
    int eth_packet_sent;
    int eth_packet_total;
    char*   eth_packet_content;

    trace("dev=0x%p, packet=0x%p, length=%d", (void*)dev, (void*)packet, length);

    p_pegasus = (pegasus_t*)dev->priv;
    p_usb_device = (struct usb_device*)p_pegasus->usb;

#ifdef  DEBUG
    if (0 != (ret = sanity_check(dev))) {
        err("failed sanity check!");
        return (ret);
    }
#endif  /* DEBUG */

    //pegasus_set_tx_mode(dev, 1);  /* enable tx mode to be able to send packets via phy */

    for (   /* split packet into usb-eth packets, if necessary */
        usb_eth_packet_number = 0,
        usb_eth_packet_size = (8 << p_usb_device->maxpacketsize),
        eth_packet_content = (char*)packet,
        eth_packet_total = length;
        0 < eth_packet_total;
        ++usb_eth_packet_number,
        eth_packet_content += eth_packet_sent,
        eth_packet_total -= eth_packet_sent) {

        /* send at most packetsize bytes to usb eth device at a time */
        eth_packet_size = eth_packet_total;
        eth_packet_sent = 0;
        ret = usb_bulk_msg(     /* send stuff to usb eth device */
                p_usb_device,
            	usb_sndbulkpipe(p_usb_device, 2),   /* replace with something less funky.. */
            	(void*)eth_packet_content,
            	eth_packet_size,
            	&eth_packet_sent,
            	PEGASUS_BULK_URB_TIMEOUT);
        if (0 > ret) {
            err("error %d, still %d bytes to go", ret, eth_packet_total);
            break;
        }
        if (eth_packet_size != eth_packet_sent)
            trace("only sent %d of %d bytes", eth_packet_sent, eth_packet_size);
    }

    if (eth_packet_sent == usb_eth_packet_size) {
        /* if last usb eth packet was completely filled
         * send a zero length packet to indicate we're done */
        ret = usb_bulk_msg( /* send stuff to usb eth device */
            p_usb_device,
            usb_sndbulkpipe(p_usb_device, 2),   /* replace with something less funky.. */
            (void*)NULL,
            0,
            &eth_packet_sent,
            PEGASUS_BULK_URB_TIMEOUT);
        // if (0 > (ret = usb_eth_rx(p_usb_device, (void*)NULL, 0))) {
            // err("error %d, zlp probably not sent", ret, eth_packet_total);
        // }
        if (0 > ret)
            err("error %d, %d, zlp probably not sent", ret, eth_packet_total);
    }

    //pegasus_set_tx_mode(dev, 0);  /* no need to enable tx mode anymore */

    trace("ret=%d, usb_eth_packet_number=%d", ret, usb_eth_packet_number);
    return (ret);
}


#ifdef DEBUG
        #define PRINT_CONFIGURATION_REGISTER(x)                                         \
                printf(                                                                 \
                        "%s;%s(%d): configuration register:"                            \
            "\n\tpromiscious=%s, \n\tallmultcast=%s, "          \
            "\n\tsleepmode=%s, \n\ttx=%s, \n\trx=%s, "          \
            "\n\tfull-duplex=%s, \n\t100mbps=%s, \n\tuseang=%s\n",      \
            __FILE__, __FUNCTION__, __LINE__,               \
            x & HIF_REG_15_PROMISCIOUS ? "ENABLED" : "disabled",        \
            x & HIF_REG_15_ALLMULTICAST ? "ENABLED" : "disabled",       \
            x & HIF_REG_15_SLEEPMODE ? "ENABLED" : "disabled",      \
            x & HIF_REG_15_TXENABLE ? "ENABLED" : "disabled",       \
            x & HIF_REG_15_RXENABLE ? "ENABLED" : "disabled",       \
            x & HIF_REG_15_FULLDUPLEX_ENABLE ? "ENABLED" : "disabled",  \
            x & HIF_REG_15_SPEED100 ? "ENABLED" : "disabled",       \
            x & HIF_REG_15_CFG ? "NO" : "yes");
#else
    #define PRINT_CONFIGURATION_REGISTER(x)
#endif  /* DEBUG */


static int
AutoNegChangemode(
    pegasus_t*  p_pegasus,
    int     ptrUserPhyMode) {

    int res;
    __u16   AutoNegAdvtReg;
    __u8    ReadHifVal;

    trace("p_pegasus=0x%p, ptrUserPhyMode=%d", p_pegasus, ptrUserPhyMode);

    switch (ptrUserPhyMode) {
        case 0:
            AutoNegAdvtReg = 0
                | PHY_AUTONEGADVT_FdxPause
                | PHY_AUTONEGADVT_Fdx100TX
                | PHY_AUTONEGADVT_100TX
                | PHY_AUTONEGADVT_ieee802_3
                | PHY_AUTONEGADVT_10TFdx
                | PHY_AUTONEGADVT_10T
                ;
            break;
        case 1:
            AutoNegAdvtReg =
                PHY_AUTONEGADVT_10T         |
                    PHY_AUTONEGADVT_ieee802_3;
            break;
        case 2:
            AutoNegAdvtReg =
                PHY_AUTONEGADVT_FdxPause        |
                    PHY_AUTONEGADVT_10TFdx      |
                    PHY_AUTONEGADVT_10T     |
                    PHY_AUTONEGADVT_ieee802_3;
            break;
        case 3:
            AutoNegAdvtReg =
                PHY_AUTONEGADVT_100TX           |
                    PHY_AUTONEGADVT_ieee802_3;
            break;
        case 4:
            AutoNegAdvtReg =
                PHY_AUTONEGADVT_FdxPause        |
                    PHY_AUTONEGADVT_Fdx100TX    |
                    PHY_AUTONEGADVT_100TX       |
                    PHY_AUTONEGADVT_ieee802_3;
            break;
        case 5:
                        AutoNegAdvtReg = 0
            /*  | PHY_AUTONEGADVT_FdxPause */
            /*  | PHY_AUTONEGADVT_100T4 */
                | PHY_AUTONEGADVT_Fdx100TX
                | PHY_AUTONEGADVT_100TX
                | PHY_AUTONEGADVT_10TFdx
                | PHY_AUTONEGADVT_10T
                | PHY_AUTONEGADVT_ieee802_3;
            break;
        default:
            err("what are you doing man!?");
            return (-1);
            break;
    }

    res = WritePhyRegisterPassive(
        p_pegasus,
        PHY_AUTONEGADVT_REG_INDEX,  /* 4 */
        &AutoNegAdvtReg);
    if (0 > res) {
        err("failed, errorcode `%d'", res);
        return (res);
    }

    /* Change For Link Status Problem :Begin */

    /* First Disable All */
    p_pegasus->PhyControlReg = 0x0000;
    res = WritePhyRegisterPassive(
        p_pegasus,
        PHY_CONTROL_REG_INDEX,  /* 19 */
        &p_pegasus->PhyControlReg);
    if (0 > res) {
        err("failed, errorcode `%d'", res);
        return (res);
    }

    p_pegasus->PhyControlReg = PHY_CONTROL_AUTONEG_ENABLE;
    res = WritePhyRegisterPassive(
        p_pegasus,
        PHY_CONTROL_REG_INDEX,
        &p_pegasus->PhyControlReg);
    if (0 > res) {
        err("failed, errorcode `%d'", res);
        return (res);
    }

    /* Restart Auto Neg (Keep the Enable Auto Neg Bit Set) */
    p_pegasus->PhyControlReg =
        PHY_CONTROL_AUTONEG_ENABLE  |
            PHY_CONTROL_RESTART_AUTONEG;
        res = WritePhyRegisterPassive(
        p_pegasus,
        PHY_CONTROL_REG_INDEX,
        &p_pegasus->PhyControlReg);
    if (0 > res) {
        err("failed, errorcode `%d'", res);
        return (res);
    }
    p_pegasus->bAutoNegStarted = 1;

    /* Change For Link Status Problem :End */

    res = ReadPhyRegisterPassive(
        p_pegasus,
        PHY_CONTROL_REG_INDEX,
        &p_pegasus->TempPhyControlReg);
    if (0 > res) {
        err("failed, errorcode `%d'", res);
        return (res);
    }

    // read linkspeed & duplex from PHY & set to HIF
    AutoNegAdvtReg = 0x0;
        res = ReadPhyRegisterPassive(
        p_pegasus,
        PHY_AUTONEGADVT_REG_INDEX,
        &AutoNegAdvtReg);
    if (0 > res) {
        err("failed, errorcode `%d'", res);
        return (res);
    }

    /* reset configuration register prior to toggling bits */
    p_pegasus->HifReg15 = 0x00;

    if (PHY_AUTONEGADVT_100TX == (AutoNegAdvtReg & PHY_AUTONEGADVT_100TX))
        p_pegasus->HifReg15 |=  HIF_REG_15_SPEED100;
    else
        p_pegasus->HifReg15 &= ~HIF_REG_15_SPEED100;

    if (
        (PHY_AUTONEGADVT_10TFdx == (AutoNegAdvtReg & PHY_AUTONEGADVT_10TFdx)) ||
        (PHY_AUTONEGADVT_Fdx100TX == (AutoNegAdvtReg & PHY_AUTONEGADVT_Fdx100TX))) {
        info("HIF_REG_15_FULLDUPLEX_ENABLE");
        p_pegasus->HifReg15 |= HIF_REG_15_FULLDUPLEX_ENABLE;
    } else {
        info("!HIF_REG_15_FULLDUPLEX_ENABLE");
        p_pegasus->HifReg15 &= ~HIF_REG_15_FULLDUPLEX_ENABLE;
    }

    PRINT_CONFIGURATION_REGISTER(p_pegasus->HifReg15);


    trace("configuration register hif_reg15=0x%2.2x", p_pegasus->HifReg15);
    res = mcs7830_write_register(
        p_pegasus,
        HIF_REG_15,
        1,
        &p_pegasus->HifReg15);
    if (0 > res) {
        err("failed, errorcode `%d'", res);
        return (res);
    }

    res = mcs7830_read_register(
        p_pegasus,
        HIF_REG_15,
        1,
        &ReadHifVal);
    if (0 > res) {
        err("failed, errorcode `%d'", res);
    }
    if (p_pegasus->HifReg15 != ReadHifVal)
        err("current configuration settings do not match the ones we wanted! (0x%4x != 0x%4x",
            p_pegasus->HifReg15,
            ReadHifVal);
    trace("configuration register hif_reg15=0x%2.2x", ReadHifVal);

    PRINT_CONFIGURATION_REGISTER(ReadHifVal);

    trace("res=%d", res);
    return (res);
}


static int
ANGInitializeDev(
    struct eth_device*  p_eth_device) {

    pegasus_t*  p_pegasus = p_eth_device->priv;
    int retry;
    int res;
    __u8    tmp_u8;     /* for temporary storage */
    __u16   tmp_u16;    /* for temporary storage */

trace("devnum    = %d\n", p_pegasus->usb->devnum);
trace("idVendor  = %0x\n", (int)p_pegasus->usb->descriptor.idVendor);
trace("idProduct = %0x\n", (int)p_pegasus->usb->descriptor.idProduct);
trace("bcdDevice = %0x\n", (int)p_pegasus->usb->descriptor.bcdDevice);

    p_pegasus->DeviceReleaseNumber = p_pegasus->usb->descriptor.bcdDevice;

#if 0
    for (retry = 2; 0 < retry; --retry) {
        res = mcs7830_read_register(    /* get device revision number */
            p_pegasus,
            0x15,
            2,
            &tmp_u16);
        if (0 < res) {
            p_pegasus->DeviceReleaseNumber = DEVICE_REV_C;
            break;
        } else
        if (0 > res) {
            trace("error %d ignored", res);
        }

trace("");

        udelay(50);
    }

#endif

    switch (p_pegasus->DeviceReleaseNumber) {
        case DEVICE_REV_B:
            info("device revision `REV_B'");
            break;
        case DEVICE_REV_C:
            info("device revision `REV_C'");
            break;
        default:
            info("unknown device revision");
            break;
    }

    info("reading mac address");
    for (retry = 5; 0 < retry; --retry) {
        res = mcs7830_read_register(    /* get permanent node address */
            p_pegasus,
            HIF_REG_16,
            ETHERNET_ADDRESS_LENGTH,
            &p_pegasus->AiPermanentNodeAddress[0]);
        if (0 < res) {
            err("failed, errorcode `%d'", res);
            return (res);
        }
    }
    info("mac address `%2.2x-%2.2x-%2.2x-%2.2x-%2.2x-%2.2x'",
        p_pegasus->AiPermanentNodeAddress[0],
        p_pegasus->AiPermanentNodeAddress[1],
        p_pegasus->AiPermanentNodeAddress[2],
        p_pegasus->AiPermanentNodeAddress[3],
        p_pegasus->AiPermanentNodeAddress[4],
        p_pegasus->AiPermanentNodeAddress[5]);

    res = ReadPhyRegisterPassive(
        p_pegasus,
        PHY_AUTONEGADVT_REG_INDEX,
        &(p_pegasus->PhyAutoNegAdvReg));
    if (0 < res) {
        err("failed, errorcode `%d'", res);
        return (res);
    }
    info("PhyAutoNegAdvReg=`0x%4.4x'", p_pegasus->PhyAutoNegAdvReg);

    res = AutoNegChangemode(
        p_pegasus,
        5);
    if (0 > res) {
        err("failed, errorcode `%d'", res);
        return (res);
    }

    res = mcs7830_read_register(
        p_pegasus,
        HIF_REG_15,
        1,
        &tmp_u8);
    if (0 > res) {
        err("error %d reading configuration register (HIF_REG15)", res);
        return (res);
    }

#if (0)
    /* i have no clue as to why the following p.o.c. was present
     * (and yes, pun intended ;) */
    trace("current configuration register '%2.2x'", tmp_u8);
    tmp_u8 = (0
    //  | HIF_REG_15_PROMISCIOUS
    //  | HIF_REG_15_ALLMULTICAST
    //  | HIF_REG_15_SLEEPMODE
        | HIF_REG_15_TXENABLE
        | HIF_REG_15_RXENABLE       /* should be `reserved' according to datasheet!? */
    //  | HIF_REG_15_FULLDUPLEX_ENABLE
    //  | HIF_REG_15_SPEED100
    //  | HIF_REG_15_CFG        /* PTR_FPGA_VERSION != 0 ? | HIF_REG_15_CFG : .. */
        );
    trace("new configuration register '%2.2x'", tmp_u8);
    res = mcs7830_write_register(
        p_pegasus,
        HIF_REG_15,
        1,
        &tmp_u8);
    if (0 > res) {
        err("error %d writing configuration register (HIF_REG15)", res);
        return (res);
    }
#endif  /* 0 */

    p_pegasus->HifReg15 = tmp_u8;   /* why should we record this in our private struct!? */

    PRINT_CONFIGURATION_REGISTER(tmp_u8);

    if (    /* set `inter-packet gap' if configured for 10Mbps/half-duplex mode */
        (HIF_REG_15_FULLDUPLEX_ENABLE != (p_pegasus->HifReg15 & HIF_REG_15_FULLDUPLEX_ENABLE)) &&
        (HIF_REG_15_SPEED100 != (p_pegasus->HifReg15 & HIF_REG_15_SPEED100))) {
        info("configuring ipg");
        tmp_u8 = 40;
        res = mcs7830_write_register(   /* number of 24 clockcycles */
            p_pegasus,
            HIF_REG_09,
            1,
            &tmp_u8);
        if (0 > res)
            trace("failed, yet ignored, errorcode `%d'", res);
        tmp_u8 = 32;
        res = mcs7830_write_register(   /* number of 16 clockcycles */
            p_pegasus,
            HIF_REG_10,
            1,
            &tmp_u8);
        if (0 > res)
            trace("failed, yet ignored, errorcode `%d'", res);
    }

    p_pegasus->ptrPauseThreshold = PTR_PAUSE_THRESHOLD;
    if (DEVICE_REV_C == p_pegasus->DeviceReleaseNumber) {
        info("configuring pause threshold");
        res = mcs7830_write_register(
            p_pegasus,
            HIF_REG_23,
            1,
            &p_pegasus->ptrPauseThreshold);
        if (0 > res)
            trace("failed, yet ignored, errorcode `%d'", res);
    }

    trace("res=%d", res);
    return (res);
}


int pegasus_probe(
    struct usb_device*  p_usb_dev,
    struct eth_device*  p_eth_device,
    unsigned int        ifnum,
    pegasus_t*      p_pegasus) {

    int res = 0;

    trace("p_usb_dev=0x%p, p_eth_device=0x%p, ifnum=%d, p_pegasus=0x%p",
        (void*)p_usb_dev, (void*)p_eth_device, ifnum, (void*)p_pegasus);

    trace("idVendor = 0x%0x, idProduct = 0x%0x\n", p_usb_dev->descriptor.idVendor, p_usb_dev->descriptor.idProduct);

    if (    /* sanity check */
        (0x9710 != p_usb_dev->descriptor.idVendor) ||
        (0x7830 != p_usb_dev->descriptor.idProduct)) {
        return (0);
    }

    /* this way eth.c and we have all the info and pointers to
     * stuff that we need to `perform' (huh huh) */
    p_eth_device->priv = (void*)p_pegasus;
    p_pegasus->usb = p_usb_dev;
    p_pegasus->net = p_eth_device;

    trace("p_eth_device=0x%p, p_eth_device->priv=0x%p, p_pegasus->usb=0x%p, p_pegasus->net=0x%p",
        (void*)p_eth_device, (void*)p_eth_device->priv, (void*)p_pegasus->usb, (void*)p_pegasus->net);

    /* use auto negotiating skills */
    if (0 > (res = ANGInitializeDev(p_eth_device))) {
                err("error %d initialising usb-net device", res);
                return (res);
        }

    trace("");

    /* initialise skb stuff, i think this should be replaced with
     * the `ring' stuff thats present in the other eth drivers */
    /*
    skb_queue_head_init(&dev->rxq);
    skb_queue_head_init(&dev->txq);
    skb_queue_head_init(&dev->done);
    */

    memcpy( /* copy ethernet mac address */
        p_eth_device->enetaddr,
        p_pegasus->AiPermanentNodeAddress,
        6);

    p_pegasus->in = usb_rcvbulkpipe(p_usb_dev, 1);  /* moschip 7830 bulk ep-in */
    p_pegasus->out = usb_sndbulkpipe(p_usb_dev, 2); /*              bulk ep-out */
    //p_pegasus->intr = usb_sndintpipe(p_usb_dev, 3);   /*              interrupt ep-in */

    /* `install' usb interrupt packet handler.. */
    p_usb_dev->irq_handle = usb_eth_on_intr_completion;

    /* power down phy */
    mcs7830_miiphy_write(
        (char*)NULL,
        FARADAY_PHY_ADDRESS,
        MII_PHY_CONTROL_REG,
        MII_CONTROL_POWERDOWN);

    return (1);
}


#if     defined (SHOW_INFO)
void
maybe_print_receiver_status(
        char    rcv_status) {

    if (rcv_status & (1 << 5))
        info("received frame is a correct frame");
    else
        info("received frame is not a correct frame!");
    if (rcv_status & (1 << 4))
        info("received frame is a large frame, with length larger than 1518 bytes");
    else
        info("received frame is a normal length frame");
    if (rcv_status & (1 << 3))
        info("a crc error occured!");
    else
        info("crc ok");
    if (rcv_status & (1 << 2))
        info("there is an alignment error!");
    else
        info("an even number of nibbles has been received");
    if (rcv_status & (1 << 1))
        info("frame has an unexpected length");
    else
        info("frame length is as expected");
    if (rcv_status & (1 << 0))
        info("short frame length");
        else
                info("normal frame length");
}
#endif  /* defined (SHOW_INFO) */

#if     (0)
void
maybe_print_intr_status(
        int     intr_status) {

        //trace("intr_status=0x%4x (%d)", (unsigned int)intr_status, intr_status);
        if (intr_status & (1 << 15))
                printf("there are frames pending in single-port ram\n");
        if (~intr_status & (1 << 14))
        printf("least significant four bits do not contain valid status\n");
    if (intr_status & (1 << 14)) {
        info("miim_link=%s", intr_status & (1 << 13) ? "OK" : "NOK");
        info("miim_interrupt=%s", intr_status & (1 << 13) ? "HIGH" : "LOW");
        info("%smbps eth", intr_status & (1 << 11) ? "100" : "10");
        info("%s duplex", intr_status & (1 << 10) ? "full" : "half");
        if (intr_status & (1 << 3))
            printf("packet aborted due to excessive deferral (tx clock > 6072)\n");
        printf("collision occurred %s receiving 64 bytes\n", intr_status & (1 << 2) ? "after" : "before");
                printf("number of retries is %s than 16\n", intr_status & (1 << 1) ? "more" : "equal or less");
                printf("transmitted eth frame is %scorrect\n", intr_status & (1 << 0) ? "" : "in");
        }
}
#endif  /* 0 */


#endif  /* (CONFIG_COMMANDS & CFG_CMD_NET) &&
     * defined(CONFIG_PEGASUS) &&
     * defined(CONFIG_USB_OHCI_NEW) */
