/*
 * NXP Semiconductors IP0107 UART support
 *
 * Copyright (c) 2004	Pete Popov, Embedded Alley Solutions, Inc
 * ppopov@embeddedalley.com
 * (C) Copyright 2006
 * NXP Semiconductors,
 * Robert Delien robert.delien@nxp.com, Hans Zuidam hans.zuidam@nxp.com
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <config.h>

#ifdef CFG_IP0107_UART

#include <common.h>
#include <asm/io.h>
#include <nxp_uart_ip0107.h>

/******************************************************************************
*
* serial_init - initialize a channel
*
* This routine initializes the number of data bits, parity
* and set the selected baud rate. Interrupts are disabled.
* Set the modem control signals if the option is selected.
*
* RETURNS: N/A
*/

int serial_init (void)
{  

	int   scratch10 = (IP0126 + 0x528);
	ulong addr = readl(scratch10);
    
	if (addr == 0xAAAAAAAA)    //if Ejtag is connected then we need to initialize uart as uart initialize is done in FSBL.
	{
		/* power up the IP module */
		writel( 0,
		IP0107_PD(CFG_CONSOLE_PORT));

		/* Set bit rate generator */
		serial_setbrg();
	}
    
	return 0;
}

void serial_setbrg (void)
{
	/* set modem and line control parameters
	 * 8-bit mode & clear tx fifo */
	writel( IP0107_LCR_FCR__BIT_8 |
			(1 << 18) |
			(1 << 17),
		IP0107_LCR_FCR(CFG_CONSOLE_PORT) );

	/* set modem and status control parameters */
	writel(0x00000000, IP0107_MCR_MSR(CFG_CONSOLE_PORT));

	/* disable interrupt generation */
	writel(0x00000000, IP0107_INT_EN(CFG_CONSOLE_PORT));

	/* clear all interrupts */
	writel(0x0000003f, IP0107_INT_CLR(CFG_CONSOLE_PORT));

	/* Set bit rate */
	writel( IP0107_BAUDRATE__BAUDRATE(IP0107_BAUD2CODE(CONFIG_BAUDRATE)),
		IP0107_BAUDRATE(CFG_CONSOLE_PORT) );
}

void serial_putc (const char c)
{
	if (c == '\n')
		serial_putc ('\r');

	/* Wait for UARTA_TX register to empty */
	while( readl(IP0107_FIFOS_RBR_THR(CFG_CONSOLE_PORT)) &
	       IP0107_FIFOS_RBR_THR__TXFIFO_STA );

	/* Send the character */
	writel( IP0107_FIFOS_RBR_THR__THR(c),
		IP0107_FIFOS_RBR_THR(CFG_CONSOLE_PORT) );
}

void serial_puts (const char *s)
{
	while (*s)
		serial_putc (*s++);
}

int serial_getc (void)
{
	unsigned char ch;

	while (!serial_tstc ())
		;

	/* Read one character */
	ch = IP0107_FIFOS_RBR_THR__RBR(readl(IP0107_FIFOS_RBR_THR(CFG_CONSOLE_PORT)));

	/* Advance the RX FIFO read pointer */
	writel( readl(IP0107_LCR_FCR(CFG_CONSOLE_PORT)) | IP0107_LCR_FCR__RX_FIFO_PT_ADV,
		IP0107_LCR_FCR(CFG_CONSOLE_PORT) );

	return ch;
}

int serial_tstc (void)
{
	if ((readl(IP0107_FIFOS_RBR_THR(CFG_CONSOLE_PORT)) & IP0107_FIFOS_RBR_THR__RXFIFO_STA) >> 8)
		return 1;
	else
		return 0;
}

#endif /* CFG_IP0107_UART */
