//-----------------------------------------------------------------------------
//
//  Copyright 2002 Philips Semiconductors Limited
//
//  Philips Semiconductors - Millbrook Industrial Estate
//  Southampton - SO15 0DJ - UK
//
//  All rights are reserved. Reproduction in whole or part is prohibited
//  without the written prior consent of the copyright owner.
//
//  Company Confidential
//
//  Filename: bn.c
//  Project: Centaurus2 / Tamperproof
//
//  Rev		Date		Author		Comments
//  -------------------------------------------------------------------
//  001		05/12/2002	A.Badey		Original version for Centaurus2
//  001b    27/11/2008  T Mowle     Added BnCpy
//  002     12/01/2009	T Mowle     Added some more comments
//
//  Additionnal information:
//  This code is based on the bn.c code by Bruce Murray. Refer to this code
//  for more information on the revisions made. 
//
//  Function:
//  This module implements arithmetic operations on the Bignum_t type varaibles.
//
//----------------------------------------------------------------------------- 


//--------------------
// INCLUDE FILES
//--------------------

#include "bn.h"

//--------------------
// LOCAL MACROS 
//--------------------
#ifdef DEBUG_BN
 #define debug_write_bn debug_write
#else
 #define debug_write_bn(a,b,c,d)
#endif


//--------------------
// EXPORTED DATA
//--------------------

//--------------------
// LOCAL TYPEDEFS
//--------------------

//--------------------
// STATIC DATA
//--------------------

//--------------------
// FUNCTION PROTOTYPES
//--------------------
#ifdef REMOVE_MEMCPY  /* This macro is used in PNX85500/M0 - It causes the following function to be used instead of memcpy */

//----------------------------------------------------
//
// Name: BnCpy
//       Copy one Bignum_t to another
//
// Parameter   Flow   Description
// ----------------------------------------------
// to        in      First Bignum_t
// from      in      Second Bignum_t
//
// Additionnal information:
//
//----------------------------------------------------

void BnCpy (unsigned long *to, unsigned long *from)
{

 #ifdef EMULATOR_TEST_DEBUG_BN
   unsigned long * dest_ptr;
   unsigned long * src_ptr;
   unsigned long n;
 #endif

   int i;
   //debug_write(0xBC,  0,  (unsigned long)to, (unsigned long)from);

   for (i = 0; i<U32s_PER_BIGNUM; i++)
   {
      to[i] = from[i];
   }
   //debug_write(0xBC,  1,  (unsigned long)to, (unsigned long)from);

  #ifdef EMULATOR_TEST_DEBUG_BN
	 {
		dest_ptr = (unsigned long *) to;
	    src_ptr = (unsigned long *) from;
        for (n = 0; n<U32s_PER_BIGNUM; n++)
        {
           debug_write(0xBC,  dest_ptr[n],  src_ptr[n], n);
		}
 
     }
  #endif

}
#endif	 /* REMOVE_MEMCPY */


//----------------------------------------------------
//
// Name: BnMake
//       Set up a Bignum_t from a 32-bit unsigned number.
// Returns: void
//
// Parameter	Flow	Description
// ----------------------------------------------
// bn			out		Bignum_t to initialise
// n			in		32-bit unsigned number
//
// Additionnal information:
//
//----------------------------------------------------

void BnMake (Bignum_t * bn, unsigned int n)
{
	int i;
	bn->slice[0] = n;

	//memset implementation
	for (i = 1; i < u32s_per_bignum; i++)
	{
		bn->slice[i] = 0;
	}
}



//----------------------------------------------------
//
// Name: BnNumDigits
//       Determines num of significant digits (slices) in the Bignum_t
// Returns: Num of sig digits
//
// Parameter	Flow	Description
// ----------------------------------------------
// num			in		ptr to Bignum_t
//
// Additionnal information:
//
//----------------------------------------------------

int BnNumDigits(Bignum_t *num)
{
   int i;

   for (i = (u32s_per_bignum - 1); i >=0; i--)
   {
      if (num->slice[i] != 0)
      {
         return (i + 1);
      }
   }

   return (0);
}

//----------------------------------------------------
//
// Name: BnCmp
//       Compare two Bignum_ts
// Returns: 1 if p>q; -1 if p<q; 0 if p=q
//
// Parameter	Flow	Description
// ----------------------------------------------
// p			in		First Bignum_t
// q			in		Second Bignum_t
//
// Additionnal information:
//
//----------------------------------------------------

int BnCmp (Bignum_t *p, Bignum_t *q)
{
   int i;

   for (i = (u32s_per_bignum - 1); i >= 0; i--)
   {
      if      (p->slice[i] > q->slice[i]) return ( 1);
      else if (p->slice[i] < q->slice[i]) return (-1);
   }

   return (0);
}


//----------------------------------------------------
//
// Name: BnAcc
//       Adds two Bignum_ts with carry; p += q;
// Returns: carry
//
// Parameter	Flow	Description
// ----------------------------------------------
// p			in/out	On entry, 1st operand. On exit: p + q
// q			in		2nd operand
//
// Additionnal information:
// By default the carry is set to 0.
// We use a stored carry algorithm. This version is based on the previous one,
// just optimised for the Mips (sde-gcc) in term of space.
// The algorithm is based on the fact that if (a+b)mod(2^32)<a or (a+b)mod(2^32)<b
// then an overflow must have occured and thus the carry should be 1.
//
//----------------------------------------------------

int BnAcc (Bignum_t *p, Bignum_t *q)
{
	int i;
	unsigned long carry = 0;
	unsigned long carry_t;
	unsigned long tmp_p;

	for (i = 0; i < u32s_per_bignum; i++)		// Treat each slice separately
	{

		tmp_p = p->slice[i] + carry;			// Integrate the carry from the previous slice
		carry = ( tmp_p < carry ) ? 1 : 0;		// Calculate the new (temporary) carry

		p->slice[i] = tmp_p + q->slice[i];			// p = p+q
		carry_t = ( p->slice[i] < tmp_p ) ? 1 : 0;	// Calculate the second carry

		carry |= carry_t;		// the returned carry is set if any of the temporary ones
								// equals 1

	}
 
	return (carry);                                                                                                            
}



//----------------------------------------------------
//
// Name: BnDec
//       Subtracts with borrow a Bignum_t from another: p -=q
// Returns: void
//
// Parameter	Flow	Description
// ----------------------------------------------
// bn			in/out	On entry: 1st operand On exit: p-q
// q			in		2nd operand
//
// Additionnal information:
// On return, carry set to 0 means a borrow was generated.
// The algorithm is divided in 2 parts:
//   - we calculate q_temp = q + carry and determine if there is an overflow
//   (ie there will be a borrow)
//   - we calculate p = p - q_temp and determine if there is a borrow (based
//   on the fact that a borrow exists if p(end) is bigger than p(start) )
// The final borrow is calculated by or'ing the temporary ones, then inverted
// (and masked)
//
//----------------------------------------------------

int BnDec (Bignum_t *bn, Bignum_t *q)
{
	int i;
	unsigned long carry = 0;
	unsigned long carry_t;
	unsigned long temp_bn;
	unsigned long temp_q;

	for (i = 0; i < u32s_per_bignum; i++)			// Treat each slice successively
	{

		temp_q = q->slice[i] + carry;				// Calculate q_temp = q + carry
		carry = (temp_q < carry) ? 1 : 0;			// Determine if there is a carry (ie borrow)

		temp_bn = bn->slice[i];						// store the initial value of bn->slice[i] (for the comparison)
		bn->slice[i] -= temp_q;						// calculate bn = bn - q_temp
		carry_t = (bn->slice[i] > temp_bn) ? 1 : 0;	// determine if there is a borrow (ie result > initial)
		carry |= carry_t;							// get the final borrow

	}

	carry = ~(carry) & 0x1;		// take the opposite and mask the result (only the first
								// bit is significant as the result should be 0 or 1)

	return (carry);
}                                                                                                                               



//----------------------------------------------------
//
// Name: BnTestBit
//       Test a bit in a Bignum_t.
// Returns: non-zero if bit bit_num is set in p, else 0
//
// Parameter	Flow	Description
// ----------------------------------------------
// p			in		the Bignum_t
// bit_num		in		bit to test (between 0 and BITS_IN_Bignum_t-1)
//
// Additionnal information:
//
//----------------------------------------------------

unsigned int BnTestBit (Bignum_t *bn, int bit_num)
{
   unsigned int slice = bn->slice[bit_num / 32];
   unsigned int mask  = (1 << (bit_num % 32));
	
   return (slice & mask);
}



//----------------------------------------------------
//
// Name: BnASL
//       performs a 1 bit shift left function on Bignum
// Returns: Carry
//
// Parameter	Flow	Description
// ----------------------------------------------
// Result		out		result from the shift	
// p			in		input		
//
// Additionnal information:
//
//----------------------------------------------------

unsigned int BnASL (Bignum_t *result, Bignum_t *p)
{
   int i;
   unsigned int Carry = 0;

   for (i = 0; i < u32s_per_bignum; i++)
   {
      unsigned int Slice = p->slice[i];

      result->slice[i] = (Slice <<  1) | Carry;
      Carry            = (Slice >> 31);
   }
   
   return (Carry);
}




//----------------------------------------------------
//
// Name: Omura
//       Compute the Omura correction associated with a number m.
// Returns: void
//
// Parameter	Flow	Description
// ----------------------------------------------
// Result		out		the result (2^BITS_PER_BIGNUM - m)
// m			in		the top boundary
//
// Additionnal information:
//
//----------------------------------------------------

void Omura (Bignum_t *result, Bignum_t *m)
{
	int i;

	//memset implementation
	for (i = 0; i < u32s_per_bignum; i++)
	{
		result->slice[i] = 0;
	}

	BnDec  (result, m);
}



//----------------------------------------------------
//
// Name: BnNumBitsIn
//       Compute the number of significant bits in a Bignum_t.
// Returns: Number of significant bits in m
//
// Parameter	Flow	Description
// ----------------------------------------------
// m			in		the Bignum_t
//
// Additionnal information:
// This is a minimum code size version. It can be done without calling
// another function obviously.
//
//----------------------------------------------------

int BnNumBitsIn (Bignum_t *m)
{
   int i;

   for (i = (bits_per_bignum - 1); i >= 0; i--)
   {
      if (BnTestBit (m,i)) return (i+1);
   }

   return (0);
}

//----------------------------------------------------
//
// Name: BnCmpSwapped
//       Compare two Bignum_ts with opposite word order
//			i.e. compares p[0] to q[31] & p[31] to q[0]
// Returns: 1 if p>q; -1 if p<q; 0 if p=q
//
// Parameter	Flow	Description
// ----------------------------------------------
// p			in		First Bignum_t
// q			in		Second Bignum_t
//
// Additionnal information:
//
//----------------------------------------------------

int BnCmpSwapped (Bignum_t *p, Bignum_t *q)
{
   int i;
	int j = u32s_per_bignum - 1;
#ifdef EMULATOR_TEST_DEBUG
	  debug_write(char_H,char_A,char_S,char_H);
#endif

   for (i = j; i >= 0; i--)
   {
#ifdef EMULATOR_TEST_DEBUG
	  debug_write((unsigned long)i, (unsigned long)j, (unsigned long)p->slice[i],  (unsigned long)q->slice[j - i]);
#endif
      if      (p->slice[i] > q->slice[j - i]) return ( 1);
      else if (p->slice[i] < q->slice[j - i]) return (-1);
   }

   return (0);
}

