#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>

#include <sys/mman.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <malloc.h>
#include <errno.h>

#include <directfb.h>

#include <direct/debug.h>
#include <direct/mem.h>
#include <direct/messages.h>
#include <direct/util.h>

#include <core/coredefs.h>
#include <core/coretypes.h>

#include <core/state.h>
#include <core/gfxcard.h>
#include <core/surface.h>

#include <gfx/convert.h>

#include "pnx8550.h"
#include "pnx8550_blt.h"


D_DEBUG_DOMAIN( PNX8550_BLT, "PNX8550/BLT", "Philips PNX8550 Drawing Engine" );

/*
 * State validation flags.
 *
 * There's no prefix because of the macros below.
 */
enum {
     DESTINATION = 0x00000001,
     SOURCE      = 0x00000002,
     COLOR       = 0x00000004,
     HALPHA      = 0x00000008,
     SURFALPHA   = 0x00000010,
     SRCKEY      = 0x00000020,
     DSTKEY      = 0x00000040,

     ALL         = 0x0000007F
};

/*
 * State handling macros.
 */

#define PNX_VALIDATE(flags)        do { pdev->v_flags |=  (flags); } while (0)
#define PNX_INVALIDATE(flags)      do { pdev->v_flags &= ~(flags); } while (0)

#define PNX_CHECK_VALIDATE(flag)   do {                                              \
                                        if (! (pdev->v_flags & flag))                \
                                             pnx_validate_##flag( pdrv, pdev, state );     \
                                   } while (0)

/**************************************************************************************************/

static inline void
pnx_waitfifo( PNXBLTRegs *pblt, int num )
{
     int loop = 1000000;

     while ((pblt->HostFIFOStatus & PNXBLT_HOSTFIFOSTATUS) < num) {
          if (!--loop) {
               D_ERROR( "PNX8550/BLT: Timeout waiting for FIFO!\n" );
               return;
          }
     }
}

static inline volatile void *
pnx_prepare_packet( PNXDriverData *pdrv )
{
     int                loop   = 100;
     PNXDrawSharedArea *shared = pdrv->draw_shared;

     while (shared->write.index - shared->read.index == PNXDRAW_NUM_PACKETS) {
          if (! --loop) {
               D_ERROR( "PNX8550/BLT: Timeout waiting for free packet entry, resetting!\n" );
               ioctl( pdrv->draw_fd, PNXDRAW_IOCTL_RESET );
          }

          usleep( 10000 );
     }

     return &shared->packets[ shared->write.index & PNXDRAW_INDEX_MASK ];
}

static inline void
pnx_submit_packet( PNXDriverData *pdrv )
{
     PNXDrawSharedArea *shared = pdrv->draw_shared;

     if (! ++shared->write.index)
          shared->write.age++;

     if ((shared->write.index - shared->read.index) > (PNXDRAW_NUM_PACKETS/4) &&
         !(shared->state & PNXDRAW_STATE_ACTIVE))
          ioctl( pdrv->draw_fd, PNXDRAW_IOCTL_CONTINUE );
}

/**************************************************************************************************/

static inline void
pnx_validate_DESTINATION( PNXDriverData *pdrv,
                          PNXDeviceData *pdev,
                          CardState     *state )
{
     CoreSurfaceBuffer       *buffer = state->dst.buffer;
     PNXDrawSetupDestination *dest   = pnx_prepare_packet( pdrv );

     dest->header = PNXDRAW_HEADER_SETUP_DESTINATION;

     dest->addr   = state->dst.phys;
     dest->stride = state->dst.pitch;

     switch (buffer->format) {
          case DSPF_RGB16:
               dest->psize = PNXBLT_PSIZE_16BPP | PNXBLT_PSIZE_RGB565;
               break;

          case DSPF_ARGB4444:
               dest->psize = PNXBLT_PSIZE_16BPP | PNXBLT_PSIZE_ARGB4444;
               break;

          case DSPF_ARGB:
               dest->psize = PNXBLT_PSIZE_32BPP | PNXBLT_PSIZE_ARGB8888;
               break;

          default:
               D_BUG( "unexpected pixelformat" );
     }

     pnx_submit_packet( pdrv );

     /* Set the flag. */
     PNX_VALIDATE( DESTINATION );
}

static inline void
pnx_validate_SOURCE( PNXDriverData *pdrv,
                     PNXDeviceData *pdev,
                     CardState     *state )
{
     PNXDrawSetupSource *source = pnx_prepare_packet( pdrv );

     source->header = PNXDRAW_HEADER_SETUP_SOURCE;

     source->addr   = state->src.phys;
     source->stride = state->src.pitch;

     pnx_submit_packet( pdrv );

     /* Set the flag. */
     PNX_VALIDATE( SOURCE );
}

static inline void
pnx_validate_COLOR( PNXDriverData *pdrv,
                    PNXDeviceData *pdev,
                    CardState     *state )
{
     CoreSurfaceBuffer *buffer = state->dst.buffer;

     switch (buffer->format) {
          case DSPF_RGB16:
               pdev->MonoPatFColor = PIXEL_RGB16( state->color.r,
                                                  state->color.g,
                                                  state->color.b );
               break;

          case DSPF_ARGB4444:
               pdev->MonoPatFColor = PIXEL_ARGB4444( state->color.a,
                                                     state->color.r,
                                                     state->color.g,
                                                     state->color.b );
               break;

          case DSPF_ARGB:
               pdev->MonoPatFColor = PIXEL_ARGB( state->color.a,
                                                 state->color.r,
                                                 state->color.g,
                                                 state->color.b );
               break;

          default:
               D_BUG( "unexpected pixelformat" );
     }

     /* Set the flag. */
     PNX_VALIDATE( COLOR );
}

static inline void
pnx_validate_HALPHA( PNXDriverData *pdrv,
                     PNXDeviceData *pdev,
                     CardState     *state )
{
     pdev->HAlphaColor = PIXEL_ARGB( state->color.a,
                                     state->color.r,
                                     state->color.g,
                                     state->color.b );

     /* Set the flag. */
     PNX_VALIDATE( HALPHA );
}

static inline void
pnx_validate_SURFALPHA( PNXDriverData *pdrv,
                        PNXDeviceData *pdev,
                        CardState     *state )
{
     int r = 0xff, g = 0xff, b = 0xff, a = 0xff;

     if (state->blittingflags & DSBLIT_COLORIZE) {
          r = state->color.r;
          g = state->color.g;
          b = state->color.b;
     }

     if (state->blittingflags & DSBLIT_BLEND_COLORALPHA) {
          a = state->color.a;

          if (state->src_blend == DSBF_SRCALPHA ||
              (state->blittingflags & DSBLIT_SRC_PREMULTIPLY))
          {
               r = (r * (a + 1)) >> 8;
               g = (g * (a + 1)) >> 8;
               b = (b * (a + 1)) >> 8;
          }
     }

     if (state->blittingflags & DSBLIT_SRC_PREMULTCOLOR) {
          r = (r * (state->color.a + 1)) >> 8;
          g = (g * (state->color.a + 1)) >> 8;
          b = (b * (state->color.a + 1)) >> 8;
     }

     pdev->SurfAlpha = PIXEL_ARGB( a, r, g, b );

     /* Set the flag. */
     PNX_VALIDATE( SURFALPHA );
}

static inline void
pnx_validate_SRCKEY( PNXDriverData *pdrv,
                     PNXDeviceData *pdev,
                     CardState     *state )
{
     CoreSurfaceBuffer *buffer = state->src.buffer;
     PNXDrawSetupKey   *key    = pnx_prepare_packet( pdrv );

     key->header    = PNXDRAW_HEADER_SETUP_KEY;

     key->cccolor   = state->src_colorkey;
     key->transmask = (1 << DFB_COLOR_BITS_PER_PIXEL(buffer->format)) - 1;

     pnx_submit_packet( pdrv );

     /* Set the flag. */
     PNX_VALIDATE( SRCKEY );

     /* DST/SRCKEY is mutual exclusive. */
     PNX_INVALIDATE( DSTKEY );
}

static inline void
pnx_validate_DSTKEY( PNXDriverData *pdrv,
                     PNXDeviceData *pdev,
                     CardState     *state )
{
     CoreSurfaceBuffer *buffer = state->dst.buffer;
     PNXDrawSetupKey   *key    = pnx_prepare_packet( pdrv );

     key->header    = PNXDRAW_HEADER_SETUP_KEY;

     key->cccolor   = state->dst_colorkey;
     key->transmask = (1 << DFB_COLOR_BITS_PER_PIXEL(buffer->format)) - 1;

     pnx_submit_packet( pdrv );

     /* Set the flag. */
     PNX_VALIDATE( DSTKEY );

     /* SRC/DSTKEY is mutual exclusive. */
     PNX_INVALIDATE( SRCKEY );
}

/**************************************************************************************************/

DFBResult
pnxEngineSync( void *drv, void *dev )
{
     DFBResult          ret    = DFB_OK;
     PNXDriverData     *pdrv   = drv;
     PNXDrawSharedArea *shared = pdrv->draw_shared;

     if (shared->state & PNXDRAW_STATE_ACTIVE) {
          if (ioctl( pdrv->draw_fd, PNXDRAW_IOCTL_WAITIDLE )) {
               ret = errno2result( errno );
               D_PERROR( "pnxEngineSync(): Error in PNXDRAW_IOCTL_WAITIDLE!\n" );
          }
     }

     return ret;
}

void
pnxEngineReset( void *drv, void *dev )
{
     PNXDriverData *pdrv = drv;

     ioctl( pdrv->draw_fd, PNXDRAW_IOCTL_RESET );
}

void
pnxEmitCommands( void *drv, void *dev )
{
     PNXDriverData     *pdrv   = drv;
     PNXDrawSharedArea *shared = pdrv->draw_shared;

     if (shared->read.index != shared->write.index && !(shared->state & PNXDRAW_STATE_ACTIVE))
          ioctl( pdrv->draw_fd, PNXDRAW_IOCTL_CONTINUE );
}

void
pnxCheckState( void                *drv,
               void                *dev,
               CardState           *state,
               DFBAccelerationMask  accel )
{
     D_DEBUG_AT( PNX8550_BLT, "pnxCheckState (state %p, accel 0x%08x) <- dest %p\n",
                 state, accel, state->destination );

     /* Return if the desired function is not supported at all. */
     if (accel & ~(PNX_SUPPORTED_DRAWINGFUNCTIONS | PNX_SUPPORTED_BLITTINGFUNCTIONS))
          return;

     /* Return if the destination format is not supported. */
     switch (state->destination->config.format) {
          case DSPF_RGB16:
          case DSPF_ARGB4444:
          case DSPF_ARGB:
               break;
          default:
               return;
     }

     /* Check if drawing or blitting is requested. */
     if (DFB_DRAWING_FUNCTION( accel )) {
          /* Return if unsupported drawing flags are set. */
          if (state->drawingflags & ~PNX_SUPPORTED_DRAWINGFLAGS)
               return;

          /* Return if blending with unsupported blend functions is requested. */
          if (state->drawingflags & DSDRAW_BLEND) {
               /* Check source blend function. */
               if (state->src_blend != DSBF_SRCALPHA)
                    return;

               /* Check destination blend function. */
               if (state->dst_blend != DSBF_INVSRCALPHA)
                    return;
          }
     }
     else {
          /* Return if unsupported blitting flags are set. */
          if (state->blittingflags & ~PNX_SUPPORTED_BLITTINGFLAGS)
               return;

          /* Return if the source format doesn't match that of the destination. */
          if (state->source->config.format != state->destination->config.format)
               return;

          /* Return if blending with unsupported blend functions is requested. */
          if (state->blittingflags & (DSBLIT_BLEND_ALPHACHANNEL | DSBLIT_BLEND_COLORALPHA)) {
               /* Check source blend function. */
               switch (state->src_blend) {
                    case DSBF_SRCALPHA:
                         /* Not with premultiplication. */
                         if (state->blittingflags & DSBLIT_SRC_PREMULTIPLY)
                              return;

                    case DSBF_ONE:
                         break;

                    default:
                         return;
               }

               /* Check destination blend function. */
               if (state->dst_blend != DSBF_INVSRCALPHA)
                    return;
          }

          /* Return if source AND destination color keying is used. */
          if (D_FLAGS_ARE_SET( state->blittingflags, DSBLIT_SRC_COLORKEY | DSBLIT_DST_COLORKEY ))
               return;
     }

     /* Enable acceleration of the function. */
     state->accel |= accel;
}

/*
 * Make sure that the hardware is programmed for execution of 'accel' according to the 'state'.
 */
void
pnxSetState( void                *drv,
             void                *dev,
             GraphicsDeviceFuncs *funcs,
             CardState           *state,
             DFBAccelerationMask  accel )
{
     PNXDriverData          *pdrv     = drv;
     PNXDeviceData          *pdev     = dev;
     StateModificationFlags  modified = state->mod_hw;

     D_DEBUG_AT( PNX8550_BLT, "pnxSetState (state %p, accel 0x%08x) <- dest %p, modified 0x%08x\n",
                 state, accel, state->destination, modified );

     /*
      * 1) Invalidate hardware states
      *
      * Each modification to the hw independent state invalidates one or more hardware states.
      */

     /* Simply invalidate all? */
     if (modified == SMF_ALL) {
          PNX_INVALIDATE( ALL );
     }
     else if (modified) {
          /* Invalidate destination registers. */
          if (modified & SMF_DESTINATION)
               PNX_INVALIDATE( DESTINATION | COLOR | DSTKEY );
          else if (modified & SMF_DST_COLORKEY)
               PNX_INVALIDATE( DSTKEY );

          /* Invalidate the color register. */
          if (modified & SMF_COLOR)
               PNX_INVALIDATE( COLOR | HALPHA | SURFALPHA );
          else if (modified & SMF_BLITTING_FLAGS)
               PNX_INVALIDATE( SURFALPHA );

          /* Invalidate the source. */
          if (modified & SMF_SOURCE)
               PNX_INVALIDATE( SOURCE | SRCKEY );
          else if (modified & SMF_SRC_COLORKEY)
               PNX_INVALIDATE( SRCKEY );
     }

     /*
      * 2) Validate hardware states
      *
      * Each function has its own set of states that need to be validated.
      */

     /* Always requiring valid destination. */
     PNX_CHECK_VALIDATE( DESTINATION );

     /* Depending on the function... */
     switch (accel) {
          case DFXL_FILLRECTANGLE:
               /* ...require valid drawing color. */
               if (state->drawingflags & DSDRAW_BLEND)
                    PNX_CHECK_VALIDATE( HALPHA );
               else
                    PNX_CHECK_VALIDATE( COLOR );

               /*
                * 3) Tell which functions can be called without further validation, i.e. SetState()
                *
                * When the hw independent state is changed, this collection is reset.
                */
               state->set |= PNX_SUPPORTED_DRAWINGFUNCTIONS;
               break;

          case DFXL_BLIT:
               /* ...require valid source. */
               PNX_CHECK_VALIDATE( SOURCE );

               /* Validate source or destination color key if required. */
               if (state->blittingflags & DSBLIT_SRC_COLORKEY)
                    PNX_CHECK_VALIDATE( SRCKEY );
               else if (state->blittingflags & DSBLIT_DST_COLORKEY)
                    PNX_CHECK_VALIDATE( DSTKEY );

               /* Set blitting function depending on flags... */
               if (state->blittingflags & ~(DSBLIT_SRC_COLORKEY | DSBLIT_DST_COLORKEY)) {
                    funcs->Blit = pnxBlitBlend;

                    /* ...and validate surface alpha values. */
                    PNX_CHECK_VALIDATE( SURFALPHA );
               }
               else
                    funcs->Blit = pnxBlit;

               /*
                * 3) Tell which functions can be called without further validation, i.e. SetState()
                *
                * When the hw independent state is changed, this collection is reset.
                */
               state->set |= PNX_SUPPORTED_BLITTINGFUNCTIONS;
               break;

          default:
               D_BUG( "unexpected drawing/blitting function" );
               break;
     }

     pdev->drawingflags  = state->drawingflags;
     pdev->blittingflags = state->blittingflags;
     pdev->src_blend     = state->src_blend;

     /*
      * 4) Clear modification flags
      *
      * All flags have been evaluated in 1) and remembered for further validation.
      * If the hw independent state is not modified, this function won't get called
      * for subsequent rendering functions, unless they aren't defined by 3).
      */
     state->mod_hw = 0;
}

/*
 * Render a filled rectangle using the current hardware state.
 */
bool
pnxFillRectangle( void *drv, void *dev, DFBRectangle *rect )
{
     PNXDriverData      *pdrv = drv;
     PNXDeviceData      *pdev = dev;
     PNXDrawExecuteFill *fill = pnx_prepare_packet( pdrv );

//     printf("Fill %dx%d, ", rect->w, rect->h );

     fill->header = PNXDRAW_HEADER_EXECUTE_FILL;

     if (pdev->drawingflags & DSDRAW_BLEND) {
          fill->bltctl = PNXBLT_BLTCTL_BLEND_NORMAL | PNXBLT_BLTCTL_UPDATE_DST_A | 0x600;
          fill->color  = pdev->HAlphaColor;
     }
     else {
          fill->bltctl = PNXBLT_BLTCTL_SP | 0xF0;
          fill->color  = pdev->MonoPatFColor;
     }

     fill->xy   = (rect->y << 16) | rect->x;
     fill->size = (rect->h << 16) | rect->w;

     pnx_submit_packet( pdrv );

     return true;
}

bool
pnxBlit( void *drv, void *dev, DFBRectangle *rect, int x, int y )
{
     PNXDriverData           *pdrv  = drv;
     PNXDeviceData           *pdev  = dev;
     DFBSurfaceBlittingFlags  flags = pdev->blittingflags;
     PNXDrawExecuteBlit      *blit  = pnx_prepare_packet( pdrv );

//     printf("Blit %dx%d, ", rect->w, rect->h );

     blit->header = PNXDRAW_HEADER_EXECUTE_BLIT;
     blit->bltctl = 0xCC;

     if (flags & DSBLIT_SRC_COLORKEY)
          blit->bltctl |= PNXBLT_BLTCTL_KEY_SRC;
     else if (flags & DSBLIT_DST_COLORKEY)
          blit->bltctl |= PNXBLT_BLTCTL_KEY_DST;

     blit->dstxy = (y << 16) | x;
     blit->srcxy = (rect->y << 16) | rect->x;
     blit->size  = (rect->h << 16) | rect->w;

     pnx_submit_packet( pdrv );

     return true;
}

bool
pnxBlitBlend( void *drv, void *dev, DFBRectangle *rect, int x, int y )
{
     PNXDriverData           *pdrv  = drv;
     PNXDeviceData           *pdev  = dev;
     DFBSurfaceBlittingFlags  flags = pdev->blittingflags;
     PNXDrawExecuteBlitBlend *blit  = pnx_prepare_packet( pdrv );

//     printf("BlitBlend %dx%d, ", rect->w, rect->h );

     blit->header = PNXDRAW_HEADER_EXECUTE_BLIT_BLEND;
     blit->bltctl = PNXBLT_BLTCTL_UPDATE_DST_A;

     if (flags & DSBLIT_SRC_COLORKEY)
          blit->bltctl |= PNXBLT_BLTCTL_KEY_SRC;
     else if (flags & DSBLIT_DST_COLORKEY)
          blit->bltctl |= PNXBLT_BLTCTL_KEY_DST;

     /* Blend with the destination? */
     if (flags & (DSBLIT_BLEND_ALPHACHANNEL | DSBLIT_BLEND_COLORALPHA)) {
          /* Premultiply source? */
          if ((flags & DSBLIT_SRC_PREMULTIPLY) || pdev->src_blend == DSBF_SRCALPHA)
               blit->bltctl |= PNXBLT_BLTCTL_BLEND_NORMAL;
          else
               blit->bltctl |= PNXBLT_BLTCTL_BLEND_PREMULT;
     }
     else {
          /* Blend with black, i.e. use source only. */
          blit->bltctl |= PNXBLT_BLTCTL_BLEND_BLACK;

          /* Premultiply source? */
          if (flags & DSBLIT_SRC_PREMULTIPLY)
               blit->bltctl |= PNXBLT_BLTCTL_BLEND_NORMAL;
          else
               blit->bltctl |= PNXBLT_BLTCTL_BLEND_PREMULT;
     }

     blit->color = pdev->SurfAlpha;
     blit->dstxy = (y << 16) | x;
     blit->srcxy = (rect->y << 16) | rect->x;
     blit->size  = (rect->h << 16) | rect->w;

     pnx_submit_packet( pdrv );

     return true;
}

