#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>

#include <dfb_types.h>

#include <directfb.h>
#include <directfb_keyboard.h>

#include <core/coredefs.h>
#include <core/coretypes.h>
#include <core/input.h>
#include <core/system.h>

#include <direct/debug.h>
#include <direct/mem.h>
#include <direct/messages.h>
#include <direct/thread.h>
#include <direct/util.h>

#include <misc/conf.h>
#include <misc/util.h>

#include <core/input_driver.h>


DFB_INPUT_DRIVER( jag_input )


//D_DEBUG_DOMAIN( JAGInput, "JAG/Input", "JAG System Input" );

typedef enum {
     RS_IDLE,
     RS_START,
     RS_REPEAT
} RepeatState;

/*
 * declaration of private data
 */
typedef struct {
     CoreInputDevice         *device;

     DirectThread            *thread;

     DFBInputDeviceKeySymbol  last_symbol;

     bool                     quit_repeater;
     pthread_cond_t           repeat_cond;
     pthread_mutex_t          repeat_lock;
     RepeatState              repeat_state;
} JAGInputData;

typedef struct {
     unsigned int             command;
     DFBInputDeviceKeySymbol  symbol;
} KeySymbol;

static const KeySymbol TvCmds[] =
{
    { 0x440, DIKS_CURSOR_UP },  //  2.Up
    { 0x441, DIKS_CURSOR_DOWN },  //  3.Down
    { 0x442, DIKS_CURSOR_LEFT },  //  4.Left
    { 0x443, DIKS_CURSOR_RIGHT },  //  5.Right
    { 0x444, DIKS_OK },  //  6.OK
    { 0x3f8, DIKS_VOLUME_UP },  //  8.Volume Up
    { 0x3f9, DIKS_VOLUME_DOWN },  //  9.Volume Down
    { 0x408, DIKS_CHANNEL_UP },  // 10.Channel Up
    { 0x409, DIKS_CHANNEL_DOWN },  // 10.Channel Down
    { 0x3e8, DIKS_0 },  // 11.Digit 0
    { 0x3e9, DIKS_1 },  // 12.Digit 1
    { 0x3ea, DIKS_2 },  // 13.Digit 2
    { 0x3eb, DIKS_3 },  // 14.Digit 3
    { 0x3ec, DIKS_4 },  // 15.Digit 4
    { 0x3ed, DIKS_5 },  // 16.Digit 5
    { 0x3ee, DIKS_6 },  // 17.Digit 6
    { 0x3ef, DIKS_7 },  // 18.Digit 7
    { 0x3f0, DIKS_8 },  // 19.Digit 8
    { 0x3f1, DIKS_9 },  // 20.Digit 9
    { 0x3f7, DIKS_INFO },   // 21.Info
    { 0x455, DIKS_RED },    // 23.Red
    { 0x456, DIKS_GREEN },  // 24.Greend
    { 0x457, DIKS_YELLOW }, // 25.Yellow
    { 0x458, DIKS_BLUE },   // 26.Blue

    { 0x43c, DIKS_MENU },   //  1.Menu
    { 0x424, DIKS_TEXT },   //  6.Tele Text
    { 0x3f5, DIKS_MUTE },   //  7.Mute
    { 0x420, DIKS_CUSTOM0 },// 28.Next Source
    { 0x47f, DIKS_CUSTOM1 },// 27.Next View
    { 0x44f, DIKS_CUSTOM2 },// 22.Center
    { 0x445, DIKS_CUSTOM3 },// 29.[1]|[2]
    { 0x3f4, DIKS_POWER },  // 30.Standby
};

static JAGInputData *gData;

/*
 * Translates a TV Command into a DirectFB keycode.
 */
static DFBInputDeviceKeySymbol
translate_command( unsigned int command )
{
     int i;

     for (i=0; i<D_ARRAY_SIZE(TvCmds); i++) {
          if (TvCmds[i].command == command)
               return TvCmds[i].symbol;
     }

     return DIKS_NULL;
}

static void
send_symbol( DFBInputDeviceKeySymbol symbol, bool down )
{
     DFBInputEvent event;

     if (symbol == DIKS_NULL)
          return;

     event.type       = down ? DIET_KEYPRESS : DIET_KEYRELEASE;
     event.flags      = DIEF_KEYSYMBOL;
     event.key_symbol = symbol;

     if (event.key_symbol == DIKS_OK) {
          event.flags  |= DIEF_KEYID;
          event.key_id  = DIKI_ENTER;
     }
     else if (event.key_symbol == DIKS_MENU) {
          event.flags  |= DIEF_KEYID;
          event.key_id  = DIKI_ESCAPE;
     }

     dfb_input_dispatch( gData->device, &event );
}

static void *
repeater_thread( DirectThread *thread, void *arg )
{
     struct timeval   now;
     struct timespec  timeout;
     JAGInputData    *data = arg;

     pthread_mutex_lock( &data->repeat_lock );

     while (!data->quit_repeater) {
          switch (data->repeat_state) {
               case RS_IDLE:
                    pthread_cond_wait( &data->repeat_cond, &data->repeat_lock );
                    break;

               case RS_REPEAT:
                    send_symbol( data->last_symbol, true );

                    /* fall through */

               case RS_START:
                    gettimeofday( &now, NULL );

                    timeout.tv_sec  = now.tv_sec;
                    timeout.tv_nsec = (now.tv_usec * 1000) +
                                       ((data->repeat_state == RS_REPEAT) ? 50000000 : 300000000);

                    timeout.tv_sec  += timeout.tv_nsec / 1000000000;
                    timeout.tv_nsec %= 1000000000;

                    if (pthread_cond_timedwait( &data->repeat_cond, &data->repeat_lock, &timeout ) == ETIMEDOUT)
                         if (data->repeat_state == RS_START)
                              data->repeat_state = RS_REPEAT;

                    break;
          }

     }

     pthread_mutex_unlock( &data->repeat_lock );

     return NULL;
}

void
DirectFBJAGInputNotify( unsigned int command,
                        bool         down )
{
//     D_DEBUG_AT( JAGInput, "%s( 0x%x, %s )\n", __FUNCTION__, command, down ? "down" : "up" );
//     printf( "%s( 0x%x, %s )\n", __FUNCTION__, command, down ? "down" : "up" );

     if (!gData)
          return;

     pthread_mutex_lock( &gData->repeat_lock );

     gData->last_symbol = translate_command( command );

     send_symbol( gData->last_symbol, down );

     gData->repeat_state = down ? RS_START : RS_IDLE;

     if (!down)
          gData->last_symbol = DIKS_NULL;

     pthread_cond_signal( &gData->repeat_cond );

     pthread_mutex_unlock( &gData->repeat_lock );
}

/*
 * Return the number of available devices.
 * Called once during initialization of DirectFB.
 */
static int
driver_get_available()
{
     return 1;
}

/*
 * Fill out general information about this driver.
 * Called once during initialization of DirectFB.
 */
static void
driver_get_info( InputDriverInfo *info )
{
     /* fill driver info structure */
     snprintf ( info->name,
                DFB_INPUT_DRIVER_INFO_NAME_LENGTH, "JAG Input Driver" );
     snprintf ( info->vendor,
                DFB_INPUT_DRIVER_INFO_VENDOR_LENGTH, "Denis Oliver Kropp" );

     info->version.major = 0;
     info->version.minor = 5;
}

/*
 * Open the device, fill out information about it,
 * allocate and fill private data, start input thread.
 * Called during initialization, resuming or taking over mastership.
 */
static DFBResult
driver_open_device( CoreInputDevice  *device,
                    unsigned int      number,
                    InputDeviceInfo  *info,
                    void            **driver_data )
{
     JAGInputData *data;

     /* fill info */
     snprintf( info->desc.name, DFB_INPUT_DEVICE_DESC_NAME_LENGTH, "Philips JAG Input" );
     snprintf( info->desc.vendor, DFB_INPUT_DEVICE_DESC_VENDOR_LENGTH, "Philips" );

     info->desc.caps = DICAPS_KEYS;
     info->desc.type = DIDTF_REMOTE;

     /* allocate and fill private data */
     data = D_CALLOC( 1, sizeof(JAGInputData) );

     data->device = device;

     pthread_cond_init( &data->repeat_cond, NULL );
     pthread_mutex_init( &data->repeat_lock, NULL );

     data->thread = direct_thread_create( DTT_INPUT, repeater_thread, data, "Key Repeater" );

     /* set private data pointer */
     *driver_data = data;


     /* DIRTY */
     gData = data;

     return DFB_OK;
}


/*
 * Fetch one entry from the kernel keymap.
 */
static DFBResult
driver_get_keymap_entry( CoreInputDevice           *device,
                         void                      *driver_data,
                         DFBInputDeviceKeymapEntry *entry )
{
     return DFB_UNSUPPORTED;
}

/*
 * End thread, close device and free private data.
 */
static void
driver_close_device( void *driver_data )
{
     JAGInputData *data = driver_data;

     data->quit_repeater = true;

     pthread_cond_signal( &data->repeat_cond );

     direct_thread_join( data->thread );
     direct_thread_destroy( data->thread );

     /* free private data */
     D_FREE( data );
}

