/*
   (c) Copyright 2001-2007  The DirectFB Organization (directfb.org)
   (c) Copyright 2000-2004  Convergence (integrated media) GmbH

   All rights reserved.

   Written by Denis Oliver Kropp <dok@directfb.org>,
              Andreas Hundt <andi@fischlustig.de>,
              Sven Neumann <neo@directfb.org>,
              Ville Syrjälä <syrjala@sci.fi> and
              Claudio Ciccani <klan@users.sf.net>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <fcntl.h>
#include <sys/mman.h>

#include <direct/debug.h>
#include <direct/hash.h>
#include <direct/mem.h>

#include <core/surface_pool.h>

#include "jag.h"
#include "tmml_surface_pool.h"

D_DEBUG_DOMAIN( JAG_TMML, "JAG/Tmml", "JAG System Tmml Surface Pool" );
D_DEBUG_DOMAIN( JAG_TMML_Lock, "JAG/Tmml/Lock", "JAG System Tmml Surface Pool Locking" );

/**********************************************************************************************************************/

typedef struct {
} TmmlPoolData;

typedef struct {
     DirectHash     *hash;
} TmmlPoolLocalData;

/**********************************************************************************************************************/

__attribute__((weak))
tmErrorCode_t
tmmlRetailMalloc (
    tmmlMmspHandle_t  handle,     /* I: handle to memspace to allocate from */
    UInt32            nrOfBytes,  /* I: number of bytes to allocate         */
    pVoid             *ppAddress, /* O: address returned upon success       */
    tmmlMallocFlags_t flags       /* I: memory allocation flags             */
    )
{
     return TM_ERR_BAD_INSTANCE;
}

__attribute__((weak))
tmErrorCode_t
tmmlCreate (
    ptmmlMmspHandle_t     pHandle,   /* O: pointer to memspace handle returned */
                                     /*    upon success                        */
    UInt32                nrOfBytes, /* I: size of the memspace                */
    tmmlMmspCreateFlags_t flags      /* I: memspace creation flags             */
    )
{
     return TM_ERR_BAD_INSTANCE;
}

__attribute__((weak))
tmErrorCode_t
tmmlFree (
    pVoid    pAddress    // I: block to be freed
    )
{
     return TM_ERR_BAD_INSTANCE;
}

__attribute__((weak))
tmErrorCode_t
tmmlDelete (
    tmmlMmspHandle_t handle     // I: handle to memspace to delete
    )
{
     return TM_ERR_BAD_INSTANCE;
}

/**********************************************************************************************************************/

static DFBResult
MapPhysToVirt( void          *virtualAddress,
               unsigned long  physicalAddress,
               int            nrOfBytes )
{
     static int  fd = -1;
     void       *ptr;

     if (fd < 0) {
          fd = open( "/dev/mem", O_RDWR );
          if (fd < 0) {
               D_PERROR( "System/JAG: Couldn't open /dev/mem!\n" );
               return DFB_INIT;
          }
     }

     ptr = mmap( virtualAddress, nrOfBytes, PROT_READ | PROT_WRITE, MAP_FIXED | MAP_SHARED, fd, physicalAddress );
     if (ptr == MAP_FAILED) {
          D_PERROR( "System/JAG: Couldn't map /dev/mem (offset 0x%08lx, size 0x%x, virtual %p!\n",
                    physicalAddress, nrOfBytes, virtualAddress );
          return DFB_INIT;
     }

     return DFB_OK;
}

/**********************************************************************************************************************/

DFBResult
dfb_jag_tmml_allocate_handler( TmmlAllocationData *data )
{
     unsigned int rval;

     /* Create memory space */
     rval = tmmlCreate( &data->memHandle, data->size, tmmlMmspMpCached );
     if (rval) {
          D_ERROR( "System/JAG: Error '%u' creating memspace!\n", rval );
          return DFB_INIT;
     }

     D_DEBUG_AT( JAG_TMML, "  -> memspace handle 0x%08x\n", (unsigned int) data->memHandle );

     /* Allocate whole memspace area. */
     rval = tmmlMalloc( data->memHandle, data->size, &data->memVirt, tmmlMallocCacheAligned );
     if (rval) {
          D_ERROR( "System/JAG: Error '%u' allocating memory within memspace!\n", rval );
          return DFB_INIT;
     }

     data->memPhys = (tmmlPhysAddr_t) data->memVirt;

     return DFB_OK;
}

DFBResult
dfb_jag_tmml_deallocate_handler( TmmlAllocationData *data )
{
     unsigned int rval;

     /* Deallocate area */
     rval = tmmlFree( data->memVirt );
     if (rval) {
          D_ERROR( "System/JAG: Error '%u' deallocating memory within memspace!\n", rval );
          return DFB_INIT;
     }

     /* Destroy memory space */
     rval = tmmlDelete( data->memHandle );
     if (rval) {
          D_ERROR( "System/JAG: Error '%u' destroying memspace!\n", rval );
          return DFB_INIT;
     }

     return DFB_OK;
}

/**********************************************************************************************************************/

static int
tmmlPoolDataSize()
{
     return sizeof(TmmlPoolData);
}

static int
tmmlPoolLocalDataSize()
{
     return sizeof(TmmlPoolLocalData);
}

static int
tmmlAllocationDataSize()
{
     return sizeof(TmmlAllocationData);
}

static DFBResult
tmmlInitPool( CoreDFB                    *core,
              CoreSurfacePool            *pool,
              void                       *pool_data,
              void                       *pool_local,
              void                       *system_data,
              CoreSurfacePoolDescription *ret_desc )
{
     D_DEBUG_AT( JAG_TMML, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_ASSERT( ret_desc != NULL );

     ret_desc->caps     = CSPCAPS_NONE;
     ret_desc->access   = CSAF_CPU_READ | CSAF_CPU_WRITE | CSAF_GPU_READ | CSAF_GPU_WRITE | CSAF_SHARED;
     ret_desc->types    = CSTF_LAYER | CSTF_WINDOW | CSTF_CURSOR | CSTF_FONT | CSTF_SHARED | CSTF_EXTERNAL;
     ret_desc->priority = CSPP_PREFERED;

     snprintf( ret_desc->name, DFB_SURFACE_POOL_DESC_NAME_LENGTH, "Tmml Memory" );

     return DFB_OK;
}

static DFBResult
tmmlDestroyPool( CoreSurfacePool *pool,
                 void            *pool_data,
                 void            *pool_local )
{
     D_DEBUG_AT( JAG_TMML, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );

     return DFB_OK;
}

static DFBResult
tmmlAllocateBuffer( CoreSurfacePool       *pool,
                    void                  *pool_data,
                    void                  *pool_local,
                    CoreSurfaceBuffer     *buffer,
                    CoreSurfaceAllocation *allocation,
                    void                  *alloc_data )
{
     int                 ret;
     CoreSurface        *surface;
     TmmlAllocationData *alloc = alloc_data;

     D_DEBUG_AT( JAG_TMML, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );

     surface = buffer->surface;
     D_MAGIC_ASSERT( surface, CoreSurface );

     dfb_surface_calc_buffer_size( surface, 8, 0, &alloc->pitch, &alloc->size );

     if (fusion_call_execute( &dfb_jag->call, FCEF_NONE, JAG_TMML_ALLOCATE, alloc, &ret ))
          return DFB_FUSION;

     allocation->size   = alloc->size;
     allocation->offset = alloc->memPhys;

     return ret;
}

static DFBResult
tmmlDeallocateBuffer( CoreSurfacePool       *pool,
                      void                  *pool_data,
                      void                  *pool_local,
                      CoreSurfaceBuffer     *buffer,
                      CoreSurfaceAllocation *allocation,
                      void                  *alloc_data )
{
     int                 ret;
     TmmlAllocationData *alloc = alloc_data;

     D_DEBUG_AT( JAG_TMML, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( buffer, CoreSurfaceBuffer );

     if (fusion_call_execute( &dfb_jag->call, FCEF_NONE, JAG_TMML_DEALLOCATE, alloc, &ret ))
          return DFB_FUSION;

     return DFB_OK;
}

static DFBResult
tmmlLock( CoreSurfacePool       *pool,
          void                  *pool_data,
          void                  *pool_local,
          CoreSurfaceAllocation *allocation,
          void                  *alloc_data,
          CoreSurfaceBufferLock *lock )
{
     DFBResult           ret;
     TmmlPoolLocalData  *local = pool_local;
     TmmlAllocationData *alloc = alloc_data;

     D_DEBUG_AT( JAG_TMML_Lock, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( allocation, CoreSurfaceAllocation );
     D_MAGIC_ASSERT( lock, CoreSurfaceBufferLock );

     if (!local->hash) {
          ret = direct_hash_create( 7, &local->hash );
          if (ret) {
               D_DERROR( ret, "JAG/Tmml: Could not create local hash table!\n" );
               return ret;
          }
     }

     D_ASSERT( local->hash != NULL );

     if (!direct_hash_lookup( local->hash, alloc->memPhys )) {
          if (MapPhysToVirt( alloc->memVirt, alloc->memPhys, alloc->size ))
               return DFB_FAILURE;

          direct_hash_insert( local->hash, alloc->memPhys, alloc->memVirt );

          /* FIXME: remove/unmap? */
     }

     lock->addr   = alloc->memVirt;
     lock->phys   = alloc->memPhys;
     lock->offset = alloc->memPhys;
     lock->pitch  = alloc->pitch;
     lock->handle = (void*) alloc->memHandle;

     return DFB_OK;
}

static DFBResult
tmmlUnlock( CoreSurfacePool       *pool,
            void                  *pool_data,
            void                  *pool_local,
            CoreSurfaceAllocation *allocation,
            void                  *alloc_data,
            CoreSurfaceBufferLock *lock )
{
     D_DEBUG_AT( JAG_TMML_Lock, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( pool, CoreSurfacePool );
     D_MAGIC_ASSERT( allocation, CoreSurfaceAllocation );
     D_MAGIC_ASSERT( lock, CoreSurfaceBufferLock );

     return DFB_OK;
}

const SurfacePoolFuncs tmmlSurfacePoolFuncs = {
     PoolDataSize:       tmmlPoolDataSize,
     PoolLocalDataSize:  tmmlPoolLocalDataSize,
     AllocationDataSize: tmmlAllocationDataSize,
     InitPool:           tmmlInitPool,
     DestroyPool:        tmmlDestroyPool,

     AllocateBuffer:     tmmlAllocateBuffer,
     DeallocateBuffer:   tmmlDeallocateBuffer,

     Lock:               tmmlLock,
     Unlock:             tmmlUnlock
};

