/*
   (c) Copyright 2000-2002  convergence integrated media GmbH.
   (c) Copyright 2002-2004  convergence GmbH.

   All rights reserved.

   Written by Denis Oliver Kropp <dok@directfb.org>,
              Andreas Hundt <andi@fischlustig.de>,
              Sven Neumann <neo@directfb.org> and
              Ville Syrjl <syrjala@sci.fi>.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <stdio.h>

#include <directfb.h>

#include <fusion/fusion.h>
#include <fusion/shmalloc.h>

#include <core/core.h>
#include <core/coredefs.h>
#include <core/coretypes.h>
#include <core/layers.h>
#include <core/palette.h>
#include <core/surfaces.h>
#include <core/system.h>

#include <gfx/convert.h>

#include <misc/conf.h>
#include <misc/util.h>

#include <direct/memcpy.h>
#include <direct/messages.h>


#define MAXCLUTNR (256)


/**********************************************************************************************************************/

#include "cetvfb.h"
#include "primary.h"
#include "callbacks.h"

#define CETVFB_PRIMARY_SUPPORTED_OPTIONS   (DLOP_ALPHACHANNEL | DLOP_OPACITY | DLOP_SRC_COLORKEY)

#define DBG_PRINT	(0)


D_DEBUG_DOMAIN( CETVFB_Primary, "CETVFB/Primary", "Core System CETVFB Primary Screen & Layer" );

typedef struct _tmColor_t 
{
    unsigned char redOrY;
    unsigned char greenOrU;
    unsigned char blueOrV;
} tmColor_t;


static unsigned long dfb_ScreenWidth  = 1920;//1366;
static unsigned long dfb_ScreenHeight = 1080;//768;
static unsigned long dfb_ScreenFreq   = 50;
static int           dfb_ProgressScan = dfb_True;

/**********************************************************************************************************************/

static DFBResult dfb_cetvfb_set_region   ( CoreDFB *core, CETVFBSetRegionData    *data );
static DFBResult dfb_cetvfb_update_screen( CoreDFB *core, CETVFBUpdateScreenData *data );
static DFBResult dfb_cetvfb_set_palette  ( CoreDFB *core, CETVFBSetPaletteData   *data );
static DFBResult dfb_cetvfb_remove_region( CoreDFB *core, CETVFBLayerData        *data );
/**********************************************************************************************************************/
/*== CALLBACK FUNCTIONS ==*/

void (*pCetvfbGetOutputProperties)(		unsigned long *pwidth, unsigned long *pheight, unsigned long *pfieldrate, int /*bool*/ *pprogressive )			= NULL;
//avpgfx__mix_GetOutputProperties(      Nat32         *pwidth, Nat32         *pheight, Nat32         *pfieldrate, Bool *pprogressive );
unsigned long (*pCetvfbGetSuppNrKeys)(	int gfxlayer )																									= NULL;
void (*pCetvfbSetKey)(					int gfxlayer, unsigned long keynr, unsigned long keytype, unsigned long lowercolor, unsigned long uppercolor)	= NULL;	
void (*pCetvfbEnableKey)(				int gfxlayer, unsigned long keynr, int /*bool*/ enable )														= NULL;
int /*bool*/ (*pCetvfbSetBuffer)(		int gfxlayer, unsigned long width, unsigned long height, unsigned long stride, unsigned long fmtclass, unsigned long fmttype, void *pbuf ) = NULL;
//bool (*pCetvfbSetBuffer)(	    int gfxlayer, unsigned long width, unsigned long height, unsigned long stride, int src_ul_x, int src_ul_y, int src_lr_x, int src_lr_y, unsigned long  fmtclass, unsigned long  fmttype, void *pbuf ) = NULL;
void (*pCetvfbSetClut)(				    int gfxlayer, unsigned long nrentries, unsigned long *clut )													= NULL;
void (*pCetvfbGetMinWindowSize)(		int gfxlayer, unsigned long *pwidth, unsigned long *pheight )													= NULL;
void (*pCetvfbGetMaxLayerWindowSize)(	int gfxlayer, unsigned long *pwidth, unsigned long *pheight )													= NULL;
void (*pCetvfbSetLayerWindow)(			int gfxlayer, int ul_x, int ul_y, int lr_x, int lr_y )															= NULL;
void (*pCetvfbSetDstWindow)(			int gfxlayer, int ul_x, int ul_y, int lr_x, int lr_y )															= NULL;
void (*pCetvfbSetSrcWindow)(			int gfxlayer, int ul_x, int ul_y, int lr_x, int lr_y)															= NULL;
void (*pCetvfbSetScaleMode) (			int gfxlayer, unsigned long mode )																				= NULL;
void (*pCetvfbHideLayer)(				int gfxlayer, int /*bool*/ hide )																				= NULL;
void (*pCetvfbActivateNewWindowSettings)(int gfxlayer, int /*bool*/ sync, unsigned long millisec )														= NULL;
void (*pCetvfbSetGraphicLayerMode) (	int gfxlayer, unsigned long layermode )																			= NULL;
void * (*pCetvfbGetVideoMemory) (void)																													= NULL;
void (*pCetvfbAcceleratedCopy)(			unsigned long SrcAddr, unsigned long DstAddr, int ul_x, int ul_y, int lr_x, int lr_y, int dst_x, int dst_y )	= NULL;

/*==   ==*/

static void ConvertPalette2Clut(unsigned int num_entries, DFBColor *entries, unsigned long *pClut);


/**********************************************************************************************************************/

static unsigned long
dfb_to_pixrgbfmt( DFBSurfacePixelFormat format, unsigned long *ret_class )
{
    switch (format) 
	{
        case DSPF_ARGB:
            if (ret_class)
                *ret_class = Dfb_PixFmtClassRgb;
            return Dfb_PixFmtArgb8888;

        case DSPF_ARGB4444:
            if (ret_class)
                *ret_class = Dfb_PixFmtClassRgb;
            return Dfb_PixFmtArgb4444;

        case DSPF_RGB16:
            if (ret_class)
                *ret_class = Dfb_PixFmtClassRgb;
            return Dfb_PixFmtRgb565;

        case DSPF_LUT8:
            if (ret_class)
                *ret_class = Dfb_PixFmtClassClut;
            return Dfb_PixFmtArgbClut8bpp;

	// ADD YUV SUPPORT LATER


        default:
            D_BUG( "unexpected pixelformat '%s'", dfb_pixelformat_name( format ) );
    }

    return 0;
}

static void
get_color_minmax( DFBSurfacePixelFormat  format,
                  const DFBColor        *color,
                  unsigned int          *ret_min,  /* MGR color Nat32 */
                  unsigned int          *ret_max ) /* MGR color Nat32 */
{
     switch (format) {
          case DSPF_ARGB:
/*             ret_min->r   = color->r;
               ret_min->g   = color->g;
               ret_min->b   = color->b;
               ret_max->r   = color->r;
               ret_max->g   = color->g;
               ret_max->b   = color->b;*/

			   *ret_min = (unsigned int)((color->r <<16)+(color->g <<8)+(color->b));
			   *ret_max = *ret_min;
               break;

          case DSPF_ARGB4444:
/*             ret_min->r   = color->r & ~0xF;
               ret_min->g   = color->g & ~0xF;
               ret_min->b   = color->b & ~0xF;
               ret_max->r   = color->r |  0xF;
               ret_max->g   = color->g |  0xF;
               ret_max->b   = color->b |  0xF;*/

			   *ret_min = (unsigned int)(((color->r & ~0xF)<<16)+((color->g & ~0xF) <<8)+(color->b & ~0xF));
			   *ret_max = (unsigned int)(((color->r |  0xF)<<16)+((color->g |  0xF) <<8)+(color->b |  0xF));
               break;

          case DSPF_RGB16:
/*             ret_min->r   = color->r & ~0x7;
               ret_min->g   = color->g & ~0x3;
               ret_min->b   = color->b & ~0x7;
               ret_max->r   = color->r |  0x7;
               ret_max->g   = color->g |  0x3;
               ret_max->b   = color->b |  0x7;*/

			   *ret_min = (unsigned int)(((color->r & ~0x7)<<16)+((color->g & ~0x3) <<8)+(color->b & ~0x7));
			   *ret_max = (unsigned int)(((color->r |  0x7)<<16)+((color->g |  0x3) <<8)+(color->b |  0x7));
               break;

          case DSPF_LUT8:
               D_WARN( "color keying for LUT8 not supported" );
               break;

          default:
               D_BUG( "unexpected pixelformat '%s'", dfb_pixelformat_name( format ) );
     }
}





static void
SetSurfaceBuffer( CETVFBLayerData  *cetvld,
                  CoreSurface      *surface,
                  bool             front )
{
     bool           rval;
     unsigned long  size;
     SurfaceBuffer *buffer = front ? surface->front_buffer : surface->back_buffer;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

#if DBG_PRINT 
	 printf("\nInside primary.c:SetSurfaceBuffer()\n");
#endif

     size = DFB_PLANE_MULTIPLY( buffer->format, buffer->video.pitch * surface->height );

     D_DEBUG_AT( CETVFB_Primary, "  -> mix_SetBuffer( %dx%d, %d, Rgb %x, %lu, %lu, %p )\n",
                 surface->width, surface->height, buffer->video.pitch, cetvld->format, size,
                 dfb_cetvfb->memPhys /* Virt */ + buffer->video.offset, cetvld );


	 rval =(*pCetvfbSetBuffer)(	cetvld->layerId,										/* int   gfxlayer */
								surface->width,											/* Nat32 width    */
								surface->height,										/* Nat32 height   */
								buffer->video.pitch,									/* Nat32 stride   */
								cetvld->format_class,									/* Nat32 fmtclass */
								cetvld->format,											/* Nat32 fmttype  */	
								(void*) dfb_cetvfb->memPhys + buffer->video.offset );	/* void  *pbuf    */

}


/**********************************************************************************************************************/

static DFBResult
primaryInitScreen( CoreScreen           *screen,
                   GraphicsDevice       *device,
                   void                 *driver_data,
                   void                 *screen_data,
                   DFBScreenDescription *description )
{
     /* Set the screen capabilities. */
     description->caps = DSCCAPS_NONE;

     /* Set the screen name. */
     snprintf( description->name,
               DFB_SCREEN_DESC_NAME_LENGTH, "CETVFB Primary Screen" );


	 /* Get the screen properties */
	(*pCetvfbGetOutputProperties)( &dfb_ScreenWidth, &dfb_ScreenHeight, &dfb_ScreenFreq, &dfb_ProgressScan );  /* !!! progressScan is Bool type !!! */

	printf("\nInitScreen returned width=%d and height=%d\n",dfb_ScreenWidth,dfb_ScreenHeight);


   return DFB_OK;
}

static DFBResult
primaryGetScreenSize( CoreScreen *screen,
                      void       *driver_data,
                      void       *screen_data,
                      int        *ret_width,
                      int        *ret_height )
{
//     unsigned long    width = 1366;
//     unsigned long    height = 768;
//	 int  fieldRate = 0;
//	 int  progressScan = dfb_True;

     D_ASSERT( dfb_cetvfb != NULL );
     D_ASSERT( ret_width != NULL );
     D_ASSERT( ret_height != NULL );

     //rval = tmIVmix_GetOutputProperties( dfb_pIVmix1, &width, &height, &fieldRate, &scanType );
     //D_ASSERT( rval == TM_OK );

//	 (*pCetvfbGetOutputProperties)( &width, &height, &fieldRate, &progressScan );  /* !!! progressScan is Bool type !!! */


     *ret_width  = dfb_ScreenWidth;//1920;//width;
     *ret_height = dfb_ScreenHeight;//1080;//height;

	printf("GetScreenSize returned width = %d height = %d\n",*ret_width,*ret_height);

     return DFB_OK;
}

ScreenFuncs cetvfbPrimaryScreenFuncs = 
{
     InitScreen:    primaryInitScreen,
     GetScreenSize: primaryGetScreenSize
};

/**********************************************************************************************************************/

static int
primaryLayerDataSize()
{
     return sizeof(CETVFBLayerData);
}

static int
primaryRegionDataSize()
{
     return 0;
}

static DFBResult
primaryInitLayer( CoreLayer                  *layer,
                  void                       *driver_data,
                  void                       *layer_data,
                  DFBDisplayLayerDescription *description,
                  DFBDisplayLayerConfig      *config,
                  DFBColorAdjustment         *adjustment )
{
     CETVFBLayerData  *cetvld = layer_data;


     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

#if DBG_PRINT 
	 printf("\nInside primary.c:primaryInitLayer()\n");
#endif

#if DBG_PRINT 
	 printf("\ncetvld->layerId  = 0\n");
#endif

	 cetvld->layerId  = 0; /* FOR THE MOMENT HARCODED TO LAYER 0  */

	 /** get the output properties (screensize...) **/
	 if ( cetvld->layerId == 0 )
		 (*pCetvfbGetOutputProperties)( &dfb_ScreenWidth, &dfb_ScreenHeight, &dfb_ScreenFreq, &dfb_ProgressScan );


	 /** set the layer to scaled mode **/
//	 (*pCetvfbSetGraphicLayerMode)(cetvld->layerId, 0x20 /* LayerModeScaled */);


     /** set capabilities and type **/
     description->caps = DLCAPS_SURFACE | DLCAPS_ALPHACHANNEL | DLCAPS_OPACITY |
                         DLCAPS_SCREEN_POSITION | DLCAPS_SCREEN_SIZE;

     description->type = DLTF_GRAPHICS;


     /** set name **/
     snprintf( description->name, DFB_DISPLAY_LAYER_DESC_NAME_LENGTH,
		 "CETVFB %s Layer", (cetvld->layerId == 0) ? "Vmix1" : ( (cetvld->layerId == 1) ? "Vmix2" : "Layer3" ) );

     /* fill out the default configuration */
#if DBG_PRINT 
	 printf("\nfill out the default configuration\n");
#endif

     config->flags       = DLCONF_WIDTH       | DLCONF_HEIGHT |
                           DLCONF_PIXELFORMAT | DLCONF_BUFFERMODE | DLCONF_OPTIONS;
     config->buffermode  = DLBM_FRONTONLY;
     config->options     = DLOP_ALPHACHANNEL;

/*     if (dfb_config->mode.width)
          config->width  = dfb_config->mode.width;
     else*/
          config->width  = (cetvld->layerId == 0) ? 852 : 640;

/*     if (dfb_config->mode.height)
          config->height = dfb_config->mode.height;
     else*/
          config->height = 480;

/*     if (dfb_config->mode.format != DSPF_UNKNOWN)
          config->pixelformat = dfb_config->mode.format;
     else if (dfb_config->mode.depth > 0)
          config->pixelformat = dfb_pixelformat_for_depth( dfb_config->mode.depth );
     else*/
          config->pixelformat = DSPF_ARGB4444;

#if DBG_PRINT 
		  printf("\nCalling CetvfbGetMaxLayerWindowSize\n");
#endif

	 (*pCetvfbGetMaxLayerWindowSize)( cetvld->layerId , 
									&cetvld->maxLayerWindowWidth, 
									&cetvld->maxLayerWindowHeight );

#if DBG_PRINT 
	 printf("\nCetvfbGetMaxLayerWindowSize returned MaxWidth=%d and MaxHeight= %d\n",cetvld->maxLayerWindowWidth, cetvld->maxLayerWindowHeight);
#endif

     /* Query minimum window size */
#if DBG_PRINT 
	 printf("\nCalling CetvfbGetMinLayerWindowSize\n");
#endif

     (*pCetvfbGetMinWindowSize)( cetvld->layerId,
								&cetvld->minWindowWidth,
								&cetvld->minWindowHeight );

#if DBG_PRINT 
	 printf("\nCetvfbGetMinWindowSize returned minWidth=%d and minHeight= %d\n",cetvld->minWindowWidth, cetvld->minWindowHeight);
#endif

     D_MAGIC_SET( cetvld, CETVFBLayerData );


#if DBG_PRINT 
	 printf("\nLeaving primary : InitLayer()\n");
#endif

     return DFB_OK;
}

static DFBResult
primaryTestRegion( CoreLayer                  *layer,
                   void                       *driver_data,
                   void                       *layer_data,
                   CoreLayerRegionConfig      *config,
                   CoreLayerRegionConfigFlags *failed )
{
#if 1
     CETVFBLayerData               *cetvld  = layer_data;
     CoreLayerRegionConfigFlags  fail = 0;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

#if DBG_PRINT 
	 printf("\nInside primary.c:primaryTestRegion()\n");
#endif

     if (config->options & ~CETVFB_PRIMARY_SUPPORTED_OPTIONS)
          fail |= CLRCF_OPTIONS;

// added by Denis WP3
//     if (getenv("JAG_NOSCALE")) {
//         config->dest.w = config->width;
//         config->dest.h = config->height;
//     }

	 //JPJ
     config->dest.w = config->width;
     config->dest.h = config->height;


     D_DEBUG_AT( CETVFB_Primary, "  -> src %d, %d - %dx%d\n",
                 DFB_RECTANGLE_VALS( &config->source ) );

     D_DEBUG_AT( CETVFB_Primary, "  -> dst %d, %d - %dx%d\n",
                 DFB_RECTANGLE_VALS( &config->dest ) );


//  removed by Denis WP3
//     if (cetvld->layerId != 0) {
//          config->dest.x = 0;
//          config->dest.y = 0;
//          config->dest.w = config->source.w;
//          config->dest.h = config->source.h;
//     }


     switch (config->format) {
          case DSPF_ARGB:
          case DSPF_ARGB4444:
          case DSPF_RGB16:
          case DSPF_LUT8:
               break;

          default:
               fail |= CLRCF_FORMAT;
     }


     if (config->source.w < cetvld->minWindowWidth)
          fail |= CLRCF_SOURCE;

     if (config->source.h < cetvld->minWindowHeight)
          fail |= CLRCF_SOURCE;


     if (config->dest.w < cetvld->minWindowWidth) {
          D_DEBUG_AT( CETVFB_Primary, "  -> %d < %lu!\n", config->dest.w, cetvld->minWindowWidth );
          fail |= CLRCF_DEST;
     }

     if (config->dest.h < cetvld->minWindowHeight) {
          D_DEBUG_AT( CETVFB_Primary, "  -> %d < %lu!\n", config->dest.h, cetvld->minWindowHeight );
          fail |= CLRCF_DEST;
     }


     if (config->dest.w > cetvld->maxLayerWindowWidth) {
          D_DEBUG_AT( CETVFB_Primary, "  -> %d > %lu!\n", config->dest.w, cetvld->maxLayerWindowWidth );
          fail |= CLRCF_DEST;
     }

     if (config->dest.h > cetvld->maxLayerWindowHeight) {
          D_DEBUG_AT( CETVFB_Primary, "  -> %d > %lu!\n", config->dest.h, cetvld->maxLayerWindowHeight );
          fail |= CLRCF_DEST;
     }


// removed by Denis WP3
//     if (cetvld->layerId == 0) {
          /* Maximum scale buffer size is 852x768 for 32 bit */
          if (DFB_BYTES_PER_PIXEL(config->format) > 2 &&
              config->dest.h != config->source.h && (config->source.w > 852 || config->dest.h > 768))
          {
//               D_DEBUG_AT( CETVFB_Primary, "  -> %dx%d -> %dx%d failed\n",
//                           config->source.w, config->source.h, config->dest.w, config->dest.h );
//
//               fail |= CLRCF_SOURCE | CLRCF_DEST;
//          }
//     }
//     else if (config->dest.w != config->source.w || config->dest.h != config->source.h) {
          D_DEBUG_AT( CETVFB_Primary, "  -> %dx%d -> %dx%d failed\n",
                      config->source.w, config->source.h, config->dest.w, config->dest.h );

          fail |= CLRCF_SOURCE | CLRCF_DEST;
		  }


     if (failed)
          *failed = fail;

     if (fail) {
          D_DEBUG_AT( CETVFB_Primary, "  -> fail 0x%08x\n", fail );
          return DFB_UNSUPPORTED;
     }
#endif
     return DFB_OK;
}

static DFBResult
primaryAddRegion( CoreLayer             *layer,
                  void                  *driver_data,
                  void                  *layer_data,
                  void                  *region_data,
                  CoreLayerRegionConfig *config )
{
     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     return DFB_OK;
}

static DFBResult
primarySetRegion( CoreLayer                  *layer,
                  void                       *driver_data,
                  void                       *layer_data,
                  void                       *region_data,
                  CoreLayerRegionConfig      *config,
                  CoreLayerRegionConfigFlags  updated,
                  CoreSurface                *surface,
                  CorePalette                *palette )
{
     DFBResult         ret;
     CETVFBSetRegionData  data;
     CETVFBLayerData     *cetvld = layer_data;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

#if DBG_PRINT 
	 printf("\nInside primary.c:primarySetRegion()\n");
#endif

     data.config    = *config;
     data.updated   = updated;
     data.surface   = surface;
     data.palette   = palette;
     data.layerData = layer_data;

     ret = dfb_cetvfb_set_region( dfb_cetvfb_core, &data );
     if (ret)
          return ret;

     if (surface)
          cetvld->surface = surface;

     if (palette && (updated & CLRCF_PALETTE) && config->format == DSPF_LUT8)
	 {
          CETVFBSetPaletteData pdata;

          pdata.layerData = layer_data;
          pdata.palette   = palette;

//		  printf("\nCalling dfb_cetvfb_set_palette from primarySetRegion\n");

          dfb_cetvfb_set_palette( dfb_cetvfb_core, &pdata );
     }

#if DBG_PRINT 
	 printf("\nLeaving primarySetRegion\n");
#endif
     return DFB_OK;
}

static DFBResult
primaryRemoveRegion( CoreLayer             *layer,
                     void                  *driver_data,
                     void                  *layer_data,
                     void                  *region_data )
{
     CETVFBLayerData *cetvld = layer_data;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

     return dfb_cetvfb_remove_region( dfb_cetvfb_core, cetvld );
}

static DFBResult
primaryFlipRegion( CoreLayer           *layer,
                   void                *driver_data,
                   void                *layer_data,
                   void                *region_data,
                   CoreSurface         *surface,
                   DFBSurfaceFlipFlags  flags )
{
     CETVFBLayerData *cetvld = layer_data;
     CETVFBUpdateScreenData data;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );
//     direct_log_printf(NULL,"%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

#if DBG_PRINT 
	 printf("\nInside primary.c:primaryFlipRegion()\n");
#endif

     dfb_surface_flip_buffers( surface, false );

     data.region    = (DFBRegion){ 0, 0, surface->width - 1, surface->height - 1 };
     data.layerData = layer_data;

     cetvld->set_buffer = true;

     return dfb_cetvfb_update_screen( dfb_cetvfb_core, &data );
}

static DFBResult
primaryUpdateRegion( CoreLayer           *layer,
                     void                *driver_data,
                     void                *layer_data,
                     void                *region_data,
                     CoreSurface         *surface,
                     const DFBRegion     *update )
{
     CETVFBUpdateScreenData  data;
     CETVFBLayerData        *cetvld = layer_data;

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

#if DBG_PRINT 
	 printf("\nInside primary.c:primaryUpdateRegion()\n");
#endif

     if (update) 
	 {
          /* Temporary HACK to optimize Opera! */
          if (getenv( "OPERA_DIR" )) 
		  {
               if (update->x1 == 100 && update->y1 == 100 && update->x2 == 101 && update->y2 == 101)
                    return DFB_OK;
          }

          data.region = *update;
     }
     else
          data.region = (DFBRegion){ 0, 0, cetvld->surface->width - 1, cetvld->surface->height - 1 };

     data.layerData = layer_data;

     return dfb_cetvfb_update_screen( dfb_cetvfb_core, &data );
}

DisplayLayerFuncs cetvfbPrimaryLayerFuncs = 
{
     LayerDataSize:      primaryLayerDataSize,
     RegionDataSize:     primaryRegionDataSize,
     InitLayer:          primaryInitLayer,

     TestRegion:         primaryTestRegion,
     AddRegion:          primaryAddRegion,
     SetRegion:          primarySetRegion,
     RemoveRegion:       primaryRemoveRegion,
     FlipRegion:         primaryFlipRegion,
     UpdateRegion:       primaryUpdateRegion
};

/**********************************************************************************************************************/

static DFBResult
set_region( CETVFBLayerData            *cetvld,
            CoreLayerRegionConfig      *config,
            CoreLayerRegionConfigFlags  updated,
            CoreSurface                *surface,
            CorePalette                *palette )
{
	 int		ul_x = 0;
	 int		ul_y = 0;
	 int		lr_x = 0;
	 int		lr_y = 0;

     D_DEBUG_AT( CETVFB_Primary, "%s( %p, %p )\n", __FUNCTION__, cetvld, config );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );
     D_ASSERT( dfb_cetvfb != NULL );


#if DBG_PRINT 
	 printf("\nInside primary.c:set_region()\n");
#endif


	/** check if pixel format has changed **/
     if (updated & CLRCF_FORMAT) 
	 {
          D_ASSERT( config != NULL );
          printf("\nset_region : pixel format has changed!\n");
	

          D_DEBUG_AT( CETVFB_Primary, "  -> format %s\n", dfb_pixelformat_name(config->format) );

          cetvld->format = dfb_to_pixrgbfmt( config->format, &cetvld->format_class );
     }

	/** check if source or destination windows have changed **/
     if (updated & (CLRCF_SOURCE | CLRCF_DEST)) 
	 {
          D_ASSERT( config != NULL );
          printf("\nset_region : src or dst windows have changed!\n");

          D_DEBUG_AT( CETVFB_Primary, "  -> src %d, %d - %dx%d\n",
                      config->source.x, config->source.y, config->source.w, config->source.h );

          D_DEBUG_AT( CETVFB_Primary, "  -> dst %d, %d - %dx%d\n",
                      config->dest.x, config->dest.y, config->dest.w, config->dest.h );

		  /* Set source window. */
          ul_x = config->source.x;
          ul_y = config->source.y;
          lr_x = config->source.x + config->source.w;
          lr_y = config->source.y + config->source.h;

          D_DEBUG_AT( CETVFB_Primary, "  -> mix_SetSrcWindow( %ld, %ld, %ld, %ld )\n",
                      ul_x, ul_y, lr_x, lr_y );

		  (*pCetvfbSetSrcWindow)( cetvld->layerId, ul_x , ul_y, lr_x, lr_y);

          /* Set destination window. */
          ul_x = config->dest.x;
          ul_y = config->dest.y;
          lr_x = config->dest.x + config->dest.w;
          lr_y = config->dest.y + config->dest.h;

          D_DEBUG_AT( CETVFB_Primary, "  -> mix_SetDstWindow( %ld, %ld, %ld, %ld )\n",
                      ul_x, ul_y, lr_x, lr_y );

		  (*pCetvfbSetDstWindow)( cetvld->layerId, ul_x , ul_y, lr_x, lr_y);

          /* Set layer window. */
          ul_x = 0;
          ul_y = 0;
          lr_x = dfb_ScreenWidth;// coming from the Output Properties
          lr_y = dfb_ScreenHeight;// coming from the Output Properties

          D_DEBUG_AT( CETVFB_Primary, "  -> mix_SetLayerWindow( %ld, %ld, %ld, %ld )\n",
                      ul_x, ul_y, lr_x, lr_y );

		  (*pCetvfbSetLayerWindow)( cetvld->layerId, ul_x , ul_y, lr_x, lr_y );

          /* Activate settings. */
          D_DEBUG_AT( CETVFB_Primary, "  -> mix_ActivateNewWindowSettings( True, 0 )\n" );

		  (*pCetvfbActivateNewWindowSettings)( cetvld->layerId, dfb_True, 0 );

     }


	 /** Check if source color key has changed **/
     if ((updated & CLRCF_SRCKEY) && (config->options & DLOP_SRC_COLORKEY)) 
	 {
		 unsigned int color_min = 0;
		 unsigned int color_max = 0;

		  /* Get the min max colors for the color keying */
          get_color_minmax( config->format, &config->src_key, &color_min, &color_max );


          D_DEBUG_AT( CETVFB_Primary, "  -> CetvfbSetKey( lower=%d, upper=%d )\n", color_min, color_max );

          printf("\nset_region : src color key has changed!\n");

//          rval = tmIVmixColorKey_SetKey( jld->pIColorKey, 0, tmVmix_SrcColorKey, color_min, color_max );
//          if (rval) {
//               D_ERROR( "JAG/Primary: Could not set color key! (%lu)\n", rval );
//               return DFB_INIT;
//          }

		  (*pCetvfbSetKey)( cetvld->layerId, 0 /*keynr*/, Dfb_KeySrcColorKey /*Nat32 keytype*/, color_min /*lowercolor*/, color_max /*uppercolor*/);
		  // check if more than 1 colorkey needed ... here keynr= 0 !!
		  // check for keytypes def. as well ....
     }


	 /** check if the options have changed **/
     if (updated & CLRCF_OPTIONS) 
	 {
//          rval = tmIVmixColorKey_EnableKey( cetvld->pIColorKey, 0,
//                                            !!(config->options & DLOP_SRC_COLORKEY) );
//          if (rval) {
//               D_ERROR( "CETVFB/Primary: Could not %sable color key! (%lu)\n",
//                        (config->options & DLOP_SRC_COLORKEY) ? "en" : "dis", rval );
//               return DFB_INIT;
//          }

		 (*pCetvfbEnableKey)( cetvld->layerId, 0 /*keynr*/, (!!(config->options & DLOP_SRC_COLORKEY) ? dfb_True : dfb_False  ) /*enable*/ ); /* "!!" is used to make it a boolean */

     }

	 /** check if a new buffer needs to be passed */
     if (surface && (updated & (CLRCF_SURFACE | CLRCF_SOURCE | CLRCF_DEST | CLRCF_OPTIONS | CLRCF_SRCKEY | CLRCF_PALETTE)))
          cetvld->set_buffer = true;

     D_DEBUG_AT( CETVFB_Primary, "  -> set_buffer: %s\n", cetvld->set_buffer ? "true" : "false" );


     /** Finally show the layer. **/
     D_DEBUG_AT( CETVFB_Primary, "  -> mix_Hide( dfb_False )\n" );

	 (*pCetvfbHideLayer)( cetvld->layerId, dfb_False /*hide*/ );


     return DFB_OK;
}

static DFBResult
update_screen( CETVFBLayerData *cetvld, int x, int y, int w, int h )
{
     CoreSurface   *surface;
     SurfaceBuffer *buffer;

#if DBG_PRINT 
	 printf("\nInside primary.c:update_screen()\n");
#endif

     D_DEBUG_AT( CETVFB_Primary, "%s( %p, %d, %d - %dx%d )\n", __FUNCTION__, cetvld, x, y, w, h );

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );
     D_ASSERT( dfb_cetvfb != NULL );

     D_ASSERT( cetvld->surface != NULL );

     surface = cetvld->surface;
     buffer  = surface->front_buffer;

     D_ASSERT( buffer != NULL );

	 /** check if a new buffer is available for this layer, and exit **/

          if (cetvld->set_buffer) {
               SetSurfaceBuffer( cetvld, surface, true );

               cetvld->set_buffer = false;
          }

          return DFB_OK;
}

/**********************************************************************************************************************/

DFBResult
dfb_cetvfb_set_region_handler( CETVFBSetRegionData *data )
{
     DFBResult     ret;
     CETVFBLayerData *cetvld = data->layerData;

     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

     fusion_skirmish_prevail( &dfb_cetvfb->lock );

     ret = set_region( cetvld, &data->config, data->updated, data->surface, data->palette );

     fusion_skirmish_dismiss( &dfb_cetvfb->lock );

     return ret;
}

DFBResult
dfb_cetvfb_update_screen_handler( CETVFBUpdateScreenData *data )
{
     DFBResult ret;

     D_ASSERT( data != NULL );

     D_MAGIC_ASSERT( data->layerData, CETVFBLayerData );

     fusion_skirmish_prevail( &dfb_cetvfb->lock );

     ret = update_screen( data->layerData, DFB_RECTANGLE_VALS_FROM_REGION( &data->region ) );

     fusion_skirmish_dismiss( &dfb_cetvfb->lock );

     return ret;
}

DFBResult
dfb_cetvfb_set_palette_handler( CETVFBSetPaletteData *data )
{
     CETVFBLayerData  *cetvld  = data->layerData;
     CorePalette      *palette = data->palette;
	 unsigned long    Clut[MAXCLUTNR]   = {0};		

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );


     fusion_skirmish_prevail( &dfb_cetvfb->lock );



	 ConvertPalette2Clut(palette->num_entries, palette->entries, (unsigned long *)Clut);


	 (*pCetvfbSetClut)( cetvld->layerId,		/*gfxlayer*/
						palette->num_entries,	/*Nat32 nrentries*/
						(unsigned long *)Clut   /*Nat32 *clut */	/* !!! virtual address !!! */ 
					  );		            

     fusion_skirmish_dismiss( &dfb_cetvfb->lock );

     return DFB_OK;
}

DFBResult
dfb_cetvfb_remove_region_handler( CETVFBLayerData *cetvld )
{

     D_DEBUG_AT( CETVFB_Primary, "%s()\n", __FUNCTION__ );

     fusion_skirmish_prevail( &dfb_cetvfb->lock );

	 (*pCetvfbHideLayer)( cetvld->layerId, dfb_True /*hide*/ );

     fusion_skirmish_dismiss( &dfb_cetvfb->lock );

     return DFB_OK;
}

/**********************************************************************************************************************/

static DFBResult
dfb_cetvfb_set_region( CoreDFB *core, CETVFBSetRegionData *data )
{
     int               ret;
     CETVFBSetRegionData *tmp = NULL;

     D_ASSERT( core != NULL );
     D_ASSERT( data != NULL );

#if DBG_PRINT 
	 printf("\nInside primary.c:dfb_cetvfb_set_region()\n");
#endif

     if (dfb_core_is_master( core ))
          return dfb_cetvfb_set_region_handler( data );

     if (!fusion_is_shared( dfb_cetvfb_world, data )) {
          tmp = SHMALLOC( dfb_cetvfb->shmpool, sizeof(*tmp) );
          if (!tmp)
               return DFB_NOSYSTEMMEMORY;

          direct_memcpy( tmp, data, sizeof(*tmp) );
     }

     fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_SET_REGION,
                          tmp ? tmp : data, &ret );

     if (tmp)
          SHFREE( dfb_cetvfb->shmpool, tmp );

     return ret;
}

static DFBResult
dfb_cetvfb_update_screen( CoreDFB *core, CETVFBUpdateScreenData *data )
{
     int                  ret;
     CETVFBUpdateScreenData *tmp = NULL;

     D_ASSERT( data != NULL );
     D_MAGIC_ASSERT( data->layerData, CETVFBLayerData );

#if DBG_PRINT 
	 printf("\nInside primary.c:dfb_cetvfb_update_screen()\n");
#endif

     if (dfb_core_is_master( core ))
          return dfb_cetvfb_update_screen_handler( data );

     if (!fusion_is_shared( dfb_cetvfb_world, data )) {
          tmp = SHMALLOC( dfb_cetvfb->shmpool, sizeof(*data) );
          if (!tmp)
               return DFB_NOSYSTEMMEMORY;

          direct_memcpy( tmp, data, sizeof(*data) );

          D_MAGIC_ASSERT( tmp->layerData, CETVFBLayerData );
     }

     /* FIXME: removed ONEWAY because of SHFREE */
     fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_UPDATE_SCREEN,
                          tmp ? tmp : data, &ret );

     if (tmp)
          SHFREE( dfb_cetvfb->shmpool, tmp );

     return DFB_OK;
}

static DFBResult
dfb_cetvfb_set_palette( CoreDFB *core, CETVFBSetPaletteData *data )
{
//   int                ret;
     CETVFBSetPaletteData *tmp = NULL;

     D_ASSERT( data != NULL );
     D_MAGIC_ASSERT( data->layerData, CETVFBLayerData );

     if (dfb_core_is_master( core ))
          return dfb_cetvfb_set_palette_handler( data );

     if (!fusion_is_shared( dfb_cetvfb_world, data )) {

          tmp = SHMALLOC( dfb_cetvfb->shmpool, sizeof(*data) );
          if (!tmp)
               return DFB_NOSYSTEMMEMORY;

          direct_memcpy( tmp, data, sizeof(*data) );

          D_MAGIC_ASSERT( tmp->layerData, CETVFBLayerData );
     }

     /* FIXME: removed ONEWAY because of SHFREE */
//     fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_SET_PALETTE,
//                          tmp ? tmp : data, &ret );

     fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_SET_PALETTE,
                          tmp ? tmp : data, NULL );

     if (tmp)
          SHFREE( dfb_cetvfb->shmpool, tmp );

     return DFB_OK;
}

static void ConvertPalette2Clut(unsigned int num_entries, DFBColor *entries, unsigned long *pClut)
{
	int i=0;

	for ( i = 0; i < num_entries; i++ )
    {
		pClut[i] = (u32)((entries->a <<24)+(entries->r <<16)+(entries->g <<8)+(entries->b));
		printf("clut%u: a=%u r=%u g=%u b=%u Nat32=%lu\n",i,entries->a,entries->r,entries->g,entries->b,pClut[i]);
		entries++;
    }

  /******************************************/
  /* typedef struct {                       */
  /*    u8          a;   || alpha channel   */
  /*    u8          r;   || red channel     */
  /*    u8          g;   || green channel   */
  /*    u8          b;   || blue channel    */
  /* } DFBColor;                            */
  /******************************************/

	return;
}

//static  unsigned int ConvertDfbToMgrColor( const DFBColor dfb_Color )
//{
//    unsigned int  MgrColor;
//
//	  MgrColor = (unsigned int)((dfb_Color.r <<16)+(dfb_Color.g <<8)+(dfb_Color.b));
//
//    return( MgrColor );
//}

static DFBResult
dfb_cetvfb_remove_region( CoreDFB *core, CETVFBLayerData *cetvld )
{
     D_MAGIC_ASSERT( cetvld, CETVFBLayerData );

     if (dfb_core_is_master( core ))
          return dfb_cetvfb_remove_region_handler( cetvld );

     fusion_call_execute( &dfb_cetvfb->call, FCEF_NONE, CETVFB_REMOVE_REGION, cetvld, NULL );

     return DFB_OK;
}


// CALLBACKS REGISTRATION

void DFB_RegisterCetvCallBackTable( CETVFBCallBackTable cetvfbfct )
{
#if DBG_PRINT 
	printf("\nInside primary.c:DFB_RegisterCetvCallBackTable()\n");
#endif
	// ptrFunc = ptrFunc
	pCetvfbGetOutputProperties 			= cetvfbfct.GetOutputProperties;
	pCetvfbGetSuppNrKeys				= cetvfbfct.GetSuppNrKeys;
	pCetvfbSetKey						= cetvfbfct.SetKey;
	pCetvfbEnableKey					= cetvfbfct.EnableKey;
	pCetvfbSetBuffer					= cetvfbfct.SetBuffer;
	pCetvfbSetClut						= cetvfbfct.SetClut;
	pCetvfbGetMinWindowSize				= cetvfbfct.GetMinWindowSize;
	pCetvfbGetMaxLayerWindowSize		= cetvfbfct.GetMaxLayerWindowSize;
	pCetvfbSetLayerWindow				= cetvfbfct.SetLayerWindow;
	pCetvfbSetDstWindow					= cetvfbfct.SetDstWindow;
	pCetvfbSetSrcWindow					= cetvfbfct.SetSrcWindow;
	pCetvfbSetScaleMode					= cetvfbfct.SetScaleMode;
	pCetvfbHideLayer					= cetvfbfct.HideLayer;
	pCetvfbActivateNewWindowSettings	= cetvfbfct.ActivateNewWindowSettings;
	pCetvfbSetGraphicLayerMode			= cetvfbfct.SetGraphicLayerMode;
	pCetvfbGetVideoMemory				= cetvfbfct.GetVideoMemory;
	pCetvfbAcceleratedCopy				= cetvfbfct.AcceleratedCopy;
}

