// Filename: drawhw.c   - Drawing Engine hardware functions for pnx8550 sdk
// Author: Denis Oliver Kropp

#include <linux/module.h>

#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/wait.h>

#include <linux/poll.h>

#include <pnxdraw.h>

#include "drawhw.h"
#include "drawregs.h"


static PNXBLTRegs *pblt = (PNXBLTRegs*) (0xBBE00000 + 0x04F400);

static DECLARE_WAIT_QUEUE_HEAD( wait_idle );

/**************************************************************************************************/

static inline void
setup_destination( const PNXDrawSetupDestination *dest )
{
/*     if (dest->addr < PNX8550_GFXMEM_ADDR) {
          printk( KERN_ERR "pnxdraw: invalid destination address (0x%08x < 0x%08x)!\n",
                  dest->addr, PNX8550_GFXMEM_ADDR );
          return;
     }

     if (dest->addr >= PNX8550_GFXMEM_ADDR + PNX8550_GFXMEM_SIZE) {
          printk( KERN_ERR "pnxdraw: invalid destination address (0x%08x >= 0x%08x)!\n",
                  dest->addr, PNX8550_GFXMEM_ADDR + PNX8550_GFXMEM_SIZE );
          return;
     }
*/
     pblt->DstAddrBase = dest->addr;
     pblt->DstStride   = dest->stride;
     pblt->Psize       = dest->psize;
}

static inline void
setup_source( const PNXDrawSetupSource *source )
{
     pblt->SrcAddrBase = source->addr;
     pblt->SrcStride   = source->stride;
}

static inline void
setup_key( const PNXDrawSetupKey *key )
{
     pblt->CCColor   = key->cccolor;
     pblt->TransMask = key->transmask;
}

static inline void
execute_fill( const PNXDrawExecuteFill *fill )
{
     if (fill->bltctl & PNXBLT_BLEND_ANY) {
          pblt->SurfAlpha   = 0xFFFFFFFF;
          pblt->HAlphaColor = fill->color;
     }
     else
          pblt->MonoPatFColor = fill->color;

     pblt->DstXY   = fill->xy;
     pblt->BltCtl  = fill->bltctl;
     pblt->BltSize = fill->size;
}

static inline void
execute_blit( const PNXDrawExecuteBlit *blit )
{
     pblt->DstXY   = blit->dstxy;
     pblt->SrcXY   = blit->srcxy;
     pblt->BltCtl  = blit->bltctl;
     pblt->BltSize = blit->size;
}

static inline void
execute_blit_blend( const PNXDrawExecuteBlitBlend *blit_blend )
{
     pblt->SurfAlpha = blit_blend->color;

     pblt->DstXY     = blit_blend->dstxy;
     pblt->SrcXY     = blit_blend->srcxy;
     pblt->BltCtl    = blit_blend->bltctl;
     pblt->BltSize   = blit_blend->size;
}

static bool
decode_packet( const PNXDrawPacket *packet )
{
     switch (packet->header) {
          case PNXDRAW_HEADER_SETUP_DESTINATION:
               setup_destination( &packet->dest );
               return false;

          case PNXDRAW_HEADER_SETUP_SOURCE:
               setup_source( &packet->source );
               return false;

          case PNXDRAW_HEADER_SETUP_KEY:
               setup_key( &packet->key );
               return false;

          case PNXDRAW_HEADER_EXECUTE_FILL:
               execute_fill( &packet->fill );
               return true;

          case PNXDRAW_HEADER_EXECUTE_BLIT:
               execute_blit( &packet->blit );
               return true;

          case PNXDRAW_HEADER_EXECUTE_BLIT_BLEND:
               execute_blit_blend( &packet->blit_blend );
               return true;

          default:
               printk( "pnxdraw: Unknown header 0x%08x\n", packet->header );
               break;
     }

     return false;
}

static inline bool
packets_pending( const PNXDrawSharedArea *shared )
{
     return (shared->read.age < shared->write.age) || (shared->read.index < shared->write.index);
}

/**************************************************************************************************/

int
pnxdraw_reset( PNXDrawSharedArea *shared )
{
     pblt->PanicControl = 1;

     mdelay( 1 );

     memset( (void*) shared, 0, sizeof(PNXDrawSharedArea) );

     pblt->PanicControl = 0;
     pblt->EngineConfig = 0x100;

     return 0;
}

int
pnxdraw_continue( PNXDrawSharedArea *shared )
{
     bool exec = false;

     if (pblt->EngineStatus & 0x100) {
          printk( KERN_ERR "%s: BUSY\n", __FUNCTION__ );
          return 0;
     }

     while (packets_pending( shared )) {
          shared->state |= PNXDRAW_STATE_ACTIVE;

          exec |= decode_packet( &shared->packets[ shared->read.index & PNXDRAW_INDEX_MASK ] );

          if (! ++shared->read.index)
               shared->read.age++;

#if 0
          if ((pblt->HostFIFOStatus & 0x3f) < 6)
               break;
#else
          if (exec)
               return 0;
#endif
     }

     if (exec)
          return 0;


     shared->state &= ~PNXDRAW_STATE_ACTIVE;

     wake_up_all( &wait_idle );

     return 0;
}

int
pnxdraw_waitidle( PNXDrawSharedArea *shared )
{
     int ret = wait_event_interruptible_timeout( wait_idle,
                                                 !(shared->state & PNXDRAW_STATE_ACTIVE),
                                                 8*HZ );

     return (ret > 0) ? 0 : (ret < 0) ? ret : -ETIMEDOUT;
}

int
pnxdraw_waitdone( PNXDrawSharedArea *shared,
                  __u32              index )
{
     printk( KERN_ERR "pnxdraw: %s not implemented yet\n", __FUNCTION__ );

     return -ENOSYS;
}

int
pnxdraw_irq( PNXDrawSharedArea *shared )
{
     if (! (pblt->EngineStatus & 0x400)) {
          printk( "pnxdraw: bogus interrupt!\n" );
          return 0;
     }

     if (pblt->EngineStatus & 0x100) {
          printk( KERN_ERR "%s: BUSY\n", __FUNCTION__ );
          return 0;
     }

     /* Clear the interrupt. */
     pblt->EngineConfig |= 0x400;

     /* Process queue. */
     return pnxdraw_continue( shared );
}

