#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <direct/debug.h>
#include <direct/util.h>

#include <directfb.h>

/**************************************************************************************************/

typedef struct {
     int                  magic;

     IDirectFBWindow     *window;
     IDirectFBSurface    *surface;

     DFBWindowID          window_id;
} DemoWindow;

/**************************************************************************************************/

static IDirectFB             *m_dfb    = NULL;
static IDirectFBDisplayLayer *m_layer  = NULL;
static IDirectFBEventBuffer  *m_events = NULL;

/**************************************************************************************************/

static DemoWindow   main_window;
static DemoWindow   attached_window;

/**************************************************************************************************/

static DFBResult
init_directfb( int *pargc, char ***pargv )
{
     DFBResult ret;

     D_ASSERT( m_dfb == NULL );
     D_ASSERT( m_layer == NULL );
     D_ASSERT( m_events == NULL );

     ret = DirectFBInit( pargc, pargv );
     if (ret) {
          DirectFBError( "DirectFBInit() failed", ret );
          return ret;
     }

     ret = DirectFBCreate( &m_dfb );
     if (ret) {
          DirectFBError( "DirectFBCreate() failed", ret );
          return ret;
     }

     ret = m_dfb->GetDisplayLayer( m_dfb, DLID_PRIMARY, &m_layer );
     if (ret) {
          DirectFBError( "IDirectFB::GetDisplayLayer() failed", ret );
          goto error;
     }

     ret = m_dfb->CreateEventBuffer( m_dfb, &m_events );
     if (ret) {
          DirectFBError( "IDirectFB::CreateEventBuffer() failed", ret );
          goto error;
     }

     return DFB_OK;


error:
     if (m_layer) {
          m_layer->Release( m_layer );
          m_layer = NULL;
     }

     if (m_dfb) {
          m_dfb->Release( m_dfb );
          m_dfb = NULL;
     }

     return ret;
}

static void
destroy_directfb()
{
     D_ASSERT( m_events != NULL );
     D_ASSERT( m_layer != NULL );
     D_ASSERT( m_dfb != NULL );

     m_events->Release( m_events );
     m_layer->Release( m_layer );
     m_dfb->Release( m_dfb );

     m_events = NULL;
     m_layer  = NULL;
     m_dfb    = NULL;
}

/**************************************************************************************************/

static DFBResult
init_window( DemoWindow *dw,
             int width, int height, u32 key,
             DFBWindowStackingClass stacking,
             DFBWindowCapabilities caps,
             DFBWindowOptions options,
             DemoWindow *parent )
{
     DFBResult             ret;
     DFBWindowDescription  desc;
     IDirectFBWindow      *window;
     IDirectFBSurface     *surface = NULL;

     D_ASSERT( dw != NULL );
     D_ASSERT( width > 0 );
     D_ASSERT( height > 0 );
     D_MAGIC_ASSERT_IF( parent, DemoWindow );

     /* Fill window description. */
     desc.flags     = DWDESC_WIDTH    | DWDESC_HEIGHT | DWDESC_CAPS |
                      DWDESC_STACKING | DWDESC_PARENT | DWDESC_OPTIONS;
     desc.width     = width;
     desc.height    = height;
     desc.caps      = caps;
     desc.stacking  = stacking;
     desc.parent_id = parent ? parent->window_id : 0;
     desc.options   = options | (key ? DWOP_COLORKEYING : DWOP_NONE);

     /* Create the window. */
     ret = m_layer->CreateWindow( m_layer, &desc, &window );
     if (ret) {
          DirectFBError( "IDirectFBDisplayLayer::CreateWindow() failed", ret );
          return ret;
     }

     if (!(caps & DWCAPS_INPUTONLY)) {
          ret = window->GetSurface( window, &surface );
          if (ret) {
               DirectFBError( "IDirectFBWindow::GetSurface() failed", ret );
               window->Release( window );
               return ret;
          }

          window->SetColorKey( window, (key >> 24) & 0xff, (key >> 16) & 0xff, key & 0xff );

          surface->Clear( surface, (key >> 24) & 0xff, (key >> 16) & 0xff, key & 0xff, 0xff );
     }

     window->AttachEventBuffer( window, m_events );

     window->GetID( window, &dw->window_id );

     /* Return new interfaces. */
     dw->window  = window;
     dw->surface = surface;

     D_MAGIC_SET( dw, DemoWindow );

     return DFB_OK;
}

static void
show_window( DemoWindow *dw,
             bool        visible )
{
     IDirectFBWindow *window;

     D_MAGIC_ASSERT( dw, DemoWindow );

     window = dw->window;
     D_ASSERT( window != NULL );

     window->SetOpacity( window, visible ? 0xff : 0x00 );
}

static void
destroy_window( DemoWindow *dw )
{
     IDirectFBSurface *surface;
     IDirectFBWindow  *window;

     D_MAGIC_ASSERT( dw, DemoWindow );

     surface = dw->surface;
     D_ASSERT( surface != NULL );

     window = dw->window;
     D_ASSERT( window != NULL );

     window->DetachEventBuffer( window, m_events );

     surface->Release( surface );
     window->Release( window );

     dw->surface = NULL;
     dw->window  = NULL;

     D_MAGIC_CLEAR( dw );
}

/**************************************************************************************************/

static void
dispatch_key( DFBWindowEvent *event )
{
     static const DFBRectangle rects[] = {
          { 0, 0, 10,  1 },
          { 0, 1,  1,  9 },

          { 290, 0, 10, 1 },
          { 299, 1,  1, 9 },

          { 290, 299, 10, 1 },
          { 299, 290,  1, 9 },

          { 0, 299, 10, 1 },
          { 0, 290,  1, 9 },


          { 100,   0, 100,   1 },
          { 100, 299, 100,   1 },
          {   0, 100,   1, 100 },
          { 299, 100,   1, 100 },

          { 150,   0,   1,  70 },
          { 150, 229,   1,  70 },
          {   0, 150,  70,   1 },
          { 229, 150,  70,   1 },
     };

     if (event->type != DWET_KEYDOWN)
          return;

     switch (event->key_symbol) {
          case DIKS_CURSOR_UP:
               break;

          case DIKS_OK:
               if (attached_window.window) {
                    destroy_window( &attached_window );
               }
               else {
                    DFBWindowGeometry dst;

                    init_window( &attached_window, 300, 300, 0x000400,
                                 DWSC_MIDDLE, DWCAPS_NONE, DWOP_KEEP_ABOVE, &main_window );

                    dst.mode = DWGM_FOLLOW;

                    attached_window.window->SetDstGeometry( attached_window.window, &dst );


                    attached_window.surface->SetColor( attached_window.surface,
                                                       0xff, 0x00, 0x00, 0xff );

                    attached_window.surface->FillRectangle( attached_window.surface,
                                                            100, 100, 100, 100 );

                    attached_window.surface->SetColor( attached_window.surface,
                                                       0xff, 0xff, 0xff, 0xff );

                    attached_window.surface->FillRectangles( attached_window.surface,
                                                             rects, D_ARRAY_SIZE(rects) );

                    attached_window.surface->Flip( attached_window.surface, NULL, DSFLIP_NONE );

                    show_window( &attached_window, true );
               }
               break;

          default:
               break;
     }
}

static DFBBoolean
dispatch_events()
{
     DFBWindowEvent event;

     while (m_events->GetEvent( m_events, DFB_EVENT(&event) ) == DFB_OK) {
          switch (event.type) {
               case DWET_CLOSE:
               case DWET_DESTROYED:
                    return DFB_FALSE;

               case DWET_KEYUP:
               case DWET_KEYDOWN:
                    dispatch_key( &event );
                    break;

               default:
                    break;
          }
     }

     return DFB_TRUE;
}

/**************************************************************************************************/

int
main( int argc, char *argv[] )
{
     DFBWindowGeometry dst;

     if (init_directfb( &argc, &argv ))
          return -1;

     if (init_window( &main_window, 852, 480, 0x000400, DWSC_LOWER, DWCAPS_INPUTONLY, DWOP_NONE, NULL )) {
          destroy_directfb();
          return -2;
     }

     dst.mode = DWGM_LOCATION;
     dst.location.x = 0.2f;
     dst.location.y = 0.0f;
     dst.location.w = 0.6f;
     dst.location.h = 1.0f;

     main_window.window->SetDstGeometry( main_window.window, &dst );

     show_window( &main_window, true );

     while (dispatch_events())
          m_events->WaitForEvent( m_events );

     destroy_window( &main_window );

     destroy_directfb();

     return 0;
}

