/*-
 * Copyright (c) 2009  Atheros Communications Inc
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef _HIF_H_
#define _HIF_H_

/**
 * @file hif.h
 * HostInterFace layer.
 * Abstracts interface-dependent transmit/receive functionality.
 */ 

typedef void * hif_handle_t;

typedef enum hif_status {
    HIF_OK = 0,
    HIF_ERROR = 1
}hif_status_t;

/**
 * @brief List of registration callbacks - filled in by upper layer.
 */
typedef struct drvreg_callbacks {
    hif_status_t (* deviceInsertedHandler)(hif_handle_t hHIF);
    hif_status_t (* deviceRemovedHandler)(void *instance, uint8_t surpriseRemoved);
}drvreg_callbacks_t;

/**
 * @brief List of callbacks - filled in by upper layer.
 */ 
typedef struct completion_callbacks {
    void *Context;  /**< context meaningful to upper layer */
    hif_status_t (* txCompletionHandler)(void *Context, struct sk_buff *netbuf);
    hif_status_t (* rxCompletionHandler)(void *Context, struct sk_buff *netbuf, 
            uint8_t pipeID);
}completion_callbacks_t;


hif_status_t HIF_register(drvreg_callbacks_t *callbacks);
hif_status_t HIF_deregister(void);

/** 
 * @brief: This API is used by the upper layer to initialize the HIF layer and to
 * register different callback routines. Support for following events has
 * been captured - DSR, Read/Write completion, Device insertion/removal,
 * Device suspension/resumption/wakeup. In addition to this, the API is
 * also used to register the name and the revision of the chip. The latter
 * can be used to verify the revision of the chip read from the device
 * before reporting it to upper layer.
 * @param[in]: hHIF - HIF handle
 * @param[in]: handle - handle of the upper layer
 * @param[in]: callbacks - List of upper layer callbacks
 * @param[out]:
 * @return: 
 */

void HIFPostInit(hif_handle_t hHIF, void *handle, 
        completion_callbacks_t *callbacks);

void HIFStart(hif_handle_t hHIF);

/**
 * @brief: Send a buffer to HIF for transmission to the target.
 * @param[in]: hHIF - HIF handle
 * @param[in]: pipeID - pipe to use
 * @param[in]: netbuf - buffer to send
 * @param[out]:
 * @return: Status of the send operation.
 */ 
hif_status_t HIFSend(hif_handle_t hHIF, uint8_t PipeID, struct sk_buff *hdr_buf, 
        struct sk_buff *buf);

/**
 * @brief: Shutdown the HIF layer.
 * @param[in]: HIFHandle - opaque HIF handle.
 * @param[out]:
 * @return:
 */ 
void HIFShutDown(hif_handle_t hHIF);

void HIFGetDefaultPipe(hif_handle_t hHIF, uint8_t *ULPipe, uint8_t *DLPipe);
uint16_t HIFGetFreeQueueNumber(hif_handle_t hHIF, uint8_t PipeID);

uint8_t HIFGetULPipeNum(void);
uint8_t HIFGetDLPipeNum(void);

#endif  /* !_HIF_H_ */
