/*-
 * Copyright (c) 2009  Atheros Communications Inc
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL") version 2 as published by the Free
 * Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
#include <linux/module.h>
#include <linux/usb.h>
#include <linux/version.h>
#include <linux/timer.h>
#include <linux/spinlock.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include "hif_usb_internal.h"

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0) /* tune me! */
#define SUBMIT_URB(u,f)       usb_submit_urb(u,f)
#define USB_ALLOC_URB(u,f)    usb_alloc_urb(u,f)
#else
#define SUBMIT_URB(u,f)       usb_submit_urb(u)
#define USB_ALLOC_URB(u,f)    usb_alloc_urb(u)
#endif


#ifdef ADF_OS_MEM_DEBUG
extern void __adf_os_add_debug(void *skb, size_t size, char *func, int line);
extern void __adf_os_free_debug(void *buf);
#endif

uint32_t zfLnxUsbSubmitTxData(HIF_DEVICE_USB *macp, HIFUSBTxPipe *pipe);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) || \
    (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19))
void zfLnxUsbDataOut_callback(urb_t *urb);
#ifndef DISABLE_USB_MPHP
void zfLnxUsbHPTxCompl_callback(urb_t *urb);
void zfLnxUsbMPTxCompl_callback(urb_t *urb);
#endif
#else
void zfLnxUsbDataOut_callback(urb_t *urb, struct pt_regs *regs);
#ifndef DISABLE_USB_MPHP
void zfLnxUsbHPTxCompl_callback(urb_t *urb, struct pt_regs *regs);
void zfLnxUsbMPTxCompl_callback(urb_t *urb, struct pt_regs *regs);
#endif
#endif

extern int surprise;
static struct sk_buff *
hif_usb_alloc_skb(uint32_t size, uint32_t reserve, uint32_t align)
{
    struct sk_buff *skb = NULL;

    size += align;
    
    skb = dev_alloc_skb(size);
    if (!skb)
        return NULL;
    
#ifdef ADF_OS_MEM_DEBUG 
    __adf_os_add_debug((void *)skb, size,(char *)__FUNCTION__, __LINE__);
#endif
    reserve += (ALIGN((unsigned long)skb->data, align) - 
                (unsigned long)skb->data);

    skb_reserve(skb, reserve);
    
    hif_usb_assert(((unsigned long)skb->data % align) == 0);

    return skb;

}
static inline void
hif_usb_add_timer(struct timer_list *timer, uint32_t delay)
{   
    timer->expires = jiffies + msecs_to_jiffies(delay);
    add_timer(timer);
}
static void allocTxUrbs(HIF_DEVICE_USB *macp, HIFUSBTxPipe *pipe)
{
    int i;

    for (i = 0; i < ZM_MAX_TX_URB_NUM; i++)
    {
        pipe->TxUrbCtx[i].WlanTxDataUrb = USB_ALLOC_URB(0, GFP_KERNEL);
        pipe->TxUrbCtx[i].buf = NULL;
        pipe->TxUrbCtx[i].macp = macp;
        pipe->TxUrbCtx[i].inUse = 0;
        pipe->TxUrbCtx[i].index = i;

        if (pipe->TxUrbCtx[i].WlanTxDataUrb == 0)
        {
            int j;

            /* Free all urbs */
            for (j = 0; j < i; j++)
            {
                usb_free_urb(pipe->TxUrbCtx[j].WlanTxDataUrb);
            }

            return;
        }
    }
}

static void initCmdUrbs(HIF_DEVICE_USB *macp)
{
    int i;
    
    spin_lock_init(&(macp->CmdUrbLock));
    
    for ( i = 0; i < MAX_CMD_URB; i++ ) {
        macp->CmdUrbCtxs[i].use = 0;
        macp->CmdUrbCtxs[i].RegOutUrb = USB_ALLOC_URB(0, GFP_KERNEL);
        macp->CmdUrbCtxs[i].buf = NULL;       
        macp->CmdUrbCtxs[i].macp = macp;
    }
}

UsbCmdUrbContext* zfLnxAllocCmdUrbCtx(HIF_DEVICE_USB *macp)
{
    int i;
    unsigned long irqFlag;
    UsbCmdUrbContext *ctx = NULL;
        
    spin_lock_irqsave(&(macp->CmdUrbLock), irqFlag);
    
    for ( i = 0; i < MAX_CMD_URB; i++ ) {
        if ( macp->CmdUrbCtxs[i].use == 0 ) {
            ctx = &macp->CmdUrbCtxs[i];
            ctx->use = 1;
            break;
        }
    }
    
    spin_unlock_irqrestore(&(macp->CmdUrbLock), irqFlag);
    return ctx;
}

int zfLnxAllocAllUrbs(HIF_DEVICE_USB *macp)
{
    struct usb_interface *interface = macp->interface;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
    struct usb_interface_descriptor *iface_desc = &interface->altsetting[0];
#else
    struct usb_host_interface *iface_desc = &interface->altsetting[0];
#endif

    struct usb_endpoint_descriptor *endpoint;
    int i;

    /* descriptor matches, let's find the endpoints needed */
    /* check out the endpoints */
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))    
    for (i = 0; i < iface_desc->bNumEndpoints; ++i)
    {
        endpoint = &iface_desc->endpoint[i];
#else
    for (i = 0; i < iface_desc->desc.bNumEndpoints; ++i)
    {
        endpoint = &iface_desc->endpoint[i].desc;
#endif      
        if ((endpoint->bEndpointAddress & 0x80) &&
            ((endpoint->bmAttributes & 3) == 0x02))
        {
            /* we found a bulk in endpoint */
            printk(KERN_ERR "bulk in: wMaxPacketSize = %x\n", 
                    le16_to_cpu(endpoint->wMaxPacketSize));
        }

        if (((endpoint->bEndpointAddress & 0x80) == 0x00) &&
            ((endpoint->bmAttributes & 3) == 0x02))
        {
            /* we found a bulk out endpoint */
            printk(KERN_ERR "bulk out: wMaxPacketSize = %x\n", 
                    le16_to_cpu(endpoint->wMaxPacketSize));
        }

        if ((endpoint->bEndpointAddress & 0x80) &&
            ((endpoint->bmAttributes & 3) == 0x03))
        {
            /* we found a interrupt in endpoint */
            printk(KERN_ERR "interrupt in: wMaxPacketSize = %x\n", 
                    le16_to_cpu(endpoint->wMaxPacketSize));
            printk(KERN_ERR "interrupt in: int_interval = %d\n", 
                    endpoint->bInterval);
        }
    
        if (((endpoint->bEndpointAddress & 0x80) == 0x00) &&
            ((endpoint->bmAttributes & 3) == 0x03))
        {
            /* we found a interrupt out endpoint */
            printk(KERN_ERR "interrupt out: wMaxPacketSize = %x\n", 
                    le16_to_cpu(endpoint->wMaxPacketSize));
            printk(KERN_ERR "interrupt out: int_interval = %d\n", 
                    endpoint->bInterval);
        }
    }

    /* Allocate all Tx URBs */
    allocTxUrbs(macp, &macp->TxPipe);
#ifndef DISABLE_USB_MPHP
    allocTxUrbs(macp, &macp->HPTxPipe);
    allocTxUrbs(macp, &macp->MPTxPipe);
#endif

    /* Allocate all Rx URBs */
    for (i = 0; i < ZM_MAX_RX_URB_NUM; i++)
    {
        macp->RxUrbCtx[i].WlanRxDataUrb = USB_ALLOC_URB(0, GFP_KERNEL);

        if (macp->RxUrbCtx[i].WlanRxDataUrb == 0)
        {
            int j;

            /* Free all urbs */
            for (j = 0; j < i; j++)
            {
                usb_free_urb(macp->RxUrbCtx[j].WlanRxDataUrb);
            }

            for (j = 0; j < ZM_MAX_TX_URB_NUM; j++)
            {
                usb_free_urb(macp->TxPipe.TxUrbCtx[j].WlanTxDataUrb);
#ifndef DISABLE_USB_MPHP
                usb_free_urb(macp->HPTxPipe.TxUrbCtx[j].WlanTxDataUrb);
                usb_free_urb(macp->MPTxPipe.TxUrbCtx[j].WlanTxDataUrb);
#endif
            }

            return 0;
        }
    }

    /* Allocate Register Read/Write USB */
    //macp->RegOutUrb = USB_ALLOC_URB(0, GFP_KERNEL);
    initCmdUrbs(macp);
    
    macp->RegInUrb = USB_ALLOC_URB(0, GFP_KERNEL);

    return 1;
}

void zfLnxFreeAllUrbs(HIF_DEVICE_USB *macp)
{
    int i;

    /* Free all Tx URBs */
    for (i = 0; i < ZM_MAX_TX_URB_NUM; i++) {
        if (macp->TxPipe.TxUrbCtx[i].WlanTxDataUrb != NULL)
            usb_free_urb(macp->TxPipe.TxUrbCtx[i].WlanTxDataUrb);
#ifndef DISABLE_USB_MPHP
        if (macp->HPTxPipe.TxUrbCtx[i].WlanTxDataUrb != NULL)
            usb_free_urb(macp->HPTxPipe.TxUrbCtx[i].WlanTxDataUrb);

        if (macp->MPTxPipe.TxUrbCtx[i].WlanTxDataUrb != NULL)
            usb_free_urb(macp->MPTxPipe.TxUrbCtx[i].WlanTxDataUrb);
#endif
    }

    /* Free all Rx URBs */
    for (i = 0; i < ZM_MAX_RX_URB_NUM; i++) {
        if (macp->RxUrbCtx[i].WlanRxDataUrb != NULL)
            usb_free_urb(macp->RxUrbCtx[i].WlanRxDataUrb);
    }

    for (i = 0; i < MAX_CMD_URB; i++) {
        if (macp->CmdUrbCtxs[i].RegOutUrb != NULL)
            usb_free_urb(macp->CmdUrbCtxs[i].RegOutUrb);
    }

    /* Free USB Register Read/Write URB */
    //usb_free_urb(macp->RegOutUrb);
    usb_free_urb(macp->RegInUrb);
}

void zfLnxUnlinkAllUrbs(HIF_DEVICE_USB *macp)
{
    int i;

    /* Unlink all Tx URBs */
    for (i = 0; i < ZM_MAX_TX_URB_NUM; i++) {
        if (macp->TxPipe.TxUrbCtx[i].WlanTxDataUrb != NULL) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,15)
            macp->TxPipe.TxUrbCtx[i].WlanTxDataUrb->transfer_flags &= 
                ~URB_ASYNC_UNLINK;
#endif
            usb_kill_urb(macp->TxPipe.TxUrbCtx[i].WlanTxDataUrb);
        }
#ifndef DISABLE_USB_MPHP

        if (macp->HPTxPipe.TxUrbCtx[i].WlanTxDataUrb != NULL) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,15)
            macp->HPTxPipe.TxUrbCtx[i].WlanTxDataUrb->transfer_flags &= 
                ~URB_ASYNC_UNLINK;
#endif
            usb_kill_urb(macp->HPTxPipe.TxUrbCtx[i].WlanTxDataUrb);
        }

        if (macp->MPTxPipe.TxUrbCtx[i].WlanTxDataUrb != NULL) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,15)
            macp->MPTxPipe.TxUrbCtx[i].WlanTxDataUrb->transfer_flags &= 
                ~URB_ASYNC_UNLINK;
#endif
            usb_kill_urb(macp->MPTxPipe.TxUrbCtx[i].WlanTxDataUrb);
        }
#endif
    }

    /* Unlink all Rx URBs */
    for (i = 0; i < ZM_MAX_RX_URB_NUM; i++) {
        if (macp->RxUrbCtx[i].WlanRxDataUrb != NULL) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,15)
            macp->RxUrbCtx[i].WlanRxDataUrb->transfer_flags &= 
                ~URB_ASYNC_UNLINK;
#endif
            usb_kill_urb(macp->RxUrbCtx[i].WlanRxDataUrb);
        }
    }

    for (i = 0; i < MAX_CMD_URB; i++) {
        if (macp->CmdUrbCtxs[i].RegOutUrb != NULL) {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,15)
            macp->CmdUrbCtxs[i].RegOutUrb->transfer_flags &= ~URB_ASYNC_UNLINK;
#endif
            usb_kill_urb(macp->CmdUrbCtxs[i].RegOutUrb);
        }
    }

#if 0
    /* Unlink USB Register Read/Write URB */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,15)
    macp->RegOutUrb->transfer_flags &= ~URB_ASYNC_UNLINK;
#endif
    usb_kill_urb(macp->RegOutUrb);
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,15)
    macp->RegInUrb->transfer_flags &= ~URB_ASYNC_UNLINK;
#endif
    usb_kill_urb(macp->RegInUrb);
}

#if 0
uint32_t 
zfLnxUsbSubmitIntUrb(urb_t *urb, struct usb_device *usb, 
        uint16_t epnum, uint16_t direction,
        struct sk_buff *buf, int buffer_length, usb_complete_t complete, 
        void *context,
        uint32_t interval)
{
    uint32_t ret;
    void *transfer_buffer = (void *)skb_put(buf, ZM_USB_REG_MAX_BUF_SIZE);

    if(direction == USB_DIR_OUT)
    {
        usb_fill_int_urb(urb, usb, usb_sndbulkpipe(usb, epnum),
                transfer_buffer, buffer_length, complete, context, interval);
    }
    else
    {
        usb_fill_int_urb(urb, usb, usb_rcvbulkpipe(usb, epnum),
                transfer_buffer, buffer_length, complete, context, interval);
    }

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,15)
    urb->transfer_flags |= URB_ASYNC_UNLINK;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
    ret = usb_submit_urb(urb, GFP_ATOMIC);
#else
    ret = usb_submit_urb(urb);
#endif

    return ret;
}
#else
uint32_t zfLnxUsbSubmitIntUrb(urb_t *urb, struct usb_device *usb, 
        uint16_t epnum, uint16_t direction, void* buf, int buffer_length, 
        usb_complete_t complete, void *context, uint32_t interval)
{
    uint32_t ret;

	
    if(surprise)
	    return -1;
    if(direction == USB_DIR_OUT) {
	    void *transfer_buffer = (void*) buf;
        usb_fill_int_urb(urb, usb, usb_sndintpipe(usb, epnum),
                transfer_buffer, buffer_length, complete, context, interval);
    } else {
	    void *transfer_buffer = (void *)skb_put((struct sk_buff *)buf, 
                                        ZM_USB_REG_MAX_BUF_SIZE);

        usb_fill_int_urb(urb, usb, usb_rcvbulkpipe(usb, epnum),
                transfer_buffer, buffer_length, complete, context, interval);
    }

    if(surprise)
	    return -1;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,15)
    urb->transfer_flags |= URB_ASYNC_UNLINK;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
    ret = usb_submit_urb(urb, GFP_ATOMIC);
#else
    ret = usb_submit_urb(urb);
#endif

    return ret;
}
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) || \
    (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19))
void zfLnxUsbRegIn_callback(urb_t *urb)
#else
void zfLnxUsbRegIn_callback(urb_t *urb, struct pt_regs *regs)
#endif
{
    //zdev_t* dev = urb->context;
    //uint32_t rsp[64/4];
    int status;
    HIF_DEVICE_USB *macp = (HIF_DEVICE_USB *)urb->context;
    //int i;
    /* Check status for URB */
    if (urb->status != 0){
        printk("zfLnxUsbRegIn_callback() : status=0x%x\n", urb->status);
        if ((urb->status != -ENOENT) && (urb->status != -ECONNRESET)
            && (urb->status != -ESHUTDOWN))
        {
            #if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))    
                if (urb->status == USB_ST_INTERNALERROR)
                    status = -1;
            #else
                if (urb->status == -EPIPE){
                    /*printk(KERN_ERR "nonzero read bulk status received: \
                    -EPIPE");*/
                    status = -1;
                }

                if (urb->status == -EPROTO){
                    /*printk(KERN_ERR "nonzero read bulk status received: \
                      -EPROTO");*/
                    status = -1;
                }
            #endif
        }

        if (urb->status != -ECONNRESET) {
             if ((urb->status == -EPROTO) ||
                 (urb->status == -ETIMEDOUT)) {
                if (macp->RegInFailCnt < ZM_MAX_USB_URB_FAIL_COUNT) {
                    printk("athLnxUsbRegIn_callback() : re-submit\n");
                    macp->RegInFailCnt++;

                    dev_kfree_skb_any(macp->regUsbReadBuf);
                    
                    zfLnxSubmitRegInUrb(macp);
                    return;
                }
             }
             else {
                if (urb->status == -ESHUTDOWN) {
                    printk("%s: stop submit urb, because of shutdown\n", __func__);
                }
                else {
                     /* Re-Submit a Rx urb */
                     printk("athLnxUsbRegIn_callback() : re-submit\n");
                     macp->RegInFailCnt = 0;

                     dev_kfree_skb_any(macp->regUsbReadBuf);

                     /* Issue another USB IN URB */
                     zfLnxSubmitRegInUrb(macp);
                     return;
                }
             }
        }

        dev_kfree_skb_any(macp->regUsbReadBuf);

#ifdef ADF_OS_MEM_DEBUG
            __adf_os_free_debug ((void *) macp->regUsbReadBuf);
#endif


        //printk(KERN_ERR "urb->status: 0x%08x\n", urb->status);
        return;
    }

    if (urb->actual_length == 0)
    {
        printk(KERN_ERR "Get an URB whose length is zero");
        status = -1;
    }

    // only if 4(watch_ptn)+16(htc_rdy)    
    if (urb->actual_length == 20 || urb->actual_length == 16)
    {
        struct sk_buff *netbuf = macp->regUsbReadBuf;
        uint8_t *netdata=netbuf->data;

        // if the first 4 bytes is watchdog pattern
        if( be32_to_cpu(*(uint32_t *)netdata) == 0x00c60000 )
        {
            /*printk("#2 recv a false_alarm packet with watchdog 
            header prefixed! %p\n", *(uint32_t*)netdata);*/

            // remove the first four bytes
            skb_pull(netbuf, 4);
        }
    }

    /* Copy data into respone buffer */
    //memcpy(rsp, macp->regUsbReadBuf, urb->actual_length);
    /* Notify to upper layer */
    //zfIdlChkRsp(dev, rsp, (u16_t)urb->actual_length);
    //zfiUsbRegIn(dev, rsp, (u16_t)urb->actual_length);
    //macp->usbCbFunctions.zfcbUsbRegIn(dev, rsp, (u16_t)urb->actual_length);
    macp->Callbacks.rxCompletionHandler(macp->Callbacks.Context, 
            macp->regUsbReadBuf, USB_REG_IN_PIPE);
    macp->regUsbReadBuf = NULL;

    if(surprise)
         return;

    macp->RegInFailCnt = 0;
    /* Issue another USB IN URB */
    if(zfLnxSubmitRegInUrb(macp))
        printk("SubmitRegIn Urb failed\n");
        
}

uint32_t zfLnxSubmitRegInUrb(HIF_DEVICE_USB *macp)
{
    uint32_t ret;
     
    macp->regUsbReadBuf = hif_usb_alloc_skb(ZM_USB_REG_MAX_BUF_SIZE, 0, 4);

    if(macp->regUsbReadBuf){
        ret = zfLnxUsbSubmitIntUrb(macp->RegInUrb, macp->udev,
            USB_REG_IN_PIPE, USB_DIR_IN, macp->regUsbReadBuf,
            ZM_USB_REG_MAX_BUF_SIZE, zfLnxUsbRegIn_callback, macp, 1);
    } else {
        if (!macp->tm_rxbuf_act) {
            macp->tm_rxbuf_act = 1;
            hif_usb_add_timer(&macp->tm_rxbuf_alloc, RX_URB_BUF_ALLOC_TIMER);
        }
        return 1;
    }

    return ret;
}

int32_t 
zfLnxUsbSubmitBulkUrb(urb_t *urb, struct usb_device *usb, uint16_t epnum, 
        uint16_t direction, uint8_t *transfer_buffer, int buffer_length, 
        usb_complete_t complete, void *context)
{
    int32_t ret;

    if(surprise)
	    return -1;
    if(direction == USB_DIR_OUT) {
        usb_fill_bulk_urb(urb, usb, usb_sndbulkpipe(usb, epnum),
                transfer_buffer, buffer_length, complete, context);

        urb->transfer_flags |= URB_ZERO_PACKET;
    }
    else 
        usb_fill_bulk_urb(urb, usb, usb_rcvbulkpipe(usb, epnum),
                transfer_buffer, buffer_length, complete, context);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,15)
    urb->transfer_flags |= URB_ASYNC_UNLINK;
#endif

    if(surprise)
	    return -1;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,5,0)
    ret = usb_submit_urb(urb, GFP_ATOMIC);
#else
    ret = usb_submit_urb(urb);
#endif
    return ret;
}

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) || \
    (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19))
void zfLnxUsbRegOut_callback(urb_t *urb)
#else
void zfLnxUsbRegOut_callback(urb_t *urb, struct pt_regs *regs)
#endif
{
    //int status;
    unsigned long irqFlag;
    UsbCmdUrbContext *ctx = (UsbCmdUrbContext *)urb->context;
    HIF_DEVICE_USB *macp = (HIF_DEVICE_USB *)ctx->macp;
    struct sk_buff *buf;
            
            
    spin_lock_irqsave(&(macp->CmdUrbLock), irqFlag);
    buf = ctx->buf;
    ctx->use = 0;
    ctx->buf = NULL;
    spin_unlock_irqrestore(&(macp->CmdUrbLock), irqFlag);
     
    if (urb->status != 0) {
        printk("%s : status=0x%x\n", __FUNCTION__, urb->status);
        dev_kfree_skb_any(buf);
#ifdef ADF_OS_MEM_DEBUG
            __adf_os_free_debug ((void *) buf);
#endif

        return;
    }

    if ( buf != NULL ) 
        macp->Callbacks.txCompletionHandler(macp->Callbacks.Context,buf);
    
}

static void initTxPipe(HIFUSBTxPipe *pipe, int pipeNum)
{
    memset(pipe->UsbTxBufQ, 0, sizeof(UsbTxQ_t) * ZM_MAX_TX_URB_NUM);

    pipe->TxBufHead = 0;
    pipe->TxBufTail = 0;
    pipe->TxUrbHead = 0;
    pipe->TxUrbTail = 0;
    pipe->TxUrbCnt = ZM_MAX_TX_URB_NUM;
    pipe->TxPipeNum = pipeNum;
}

void zfLnxInitUsbTxQ(HIF_DEVICE_USB *macp)
{
    printk(KERN_ERR "zfwInitUsbTxQ\n");

    initTxPipe(&macp->TxPipe, USB_WLAN_TX_PIPE);
#ifndef DISABLE_USB_MPHP
    initTxPipe(&macp->MPTxPipe, USB_WLAN_MP_TX_PIPE);
    initTxPipe(&macp->HPTxPipe, USB_WLAN_HP_TX_PIPE);
#endif
    macp->TxPipe.TxUrbCompleteCb = zfLnxUsbDataOut_callback;
#ifndef DISABLE_USB_MPHP
    macp->HPTxPipe.TxUrbCompleteCb = zfLnxUsbHPTxCompl_callback;
    macp->MPTxPipe.TxUrbCompleteCb = zfLnxUsbMPTxCompl_callback;
#endif
}


static void RxUrbBufTimer(unsigned long param)
{
    HIF_DEVICE_USB *macp = (HIF_DEVICE_USB *) param;
    uint16_t            i;
    uint8_t             timer_act=0;

    if(!(macp->regUsbReadBuf)) {
        macp->regUsbReadBuf = hif_usb_alloc_skb(ZM_USB_REG_MAX_BUF_SIZE, 0, 4);

        if(macp->regUsbReadBuf){
            zfLnxUsbSubmitIntUrb(macp->RegInUrb, macp->udev,
                USB_REG_IN_PIPE, USB_DIR_IN, macp->regUsbReadBuf,
                ZM_USB_REG_MAX_BUF_SIZE, zfLnxUsbRegIn_callback, macp, 1);
        } else {
            printk("%s: REG_IN Buf alloc fail\n", __FUNCTION__);
            timer_act=1;
            goto error;
        }
    }

    for (i = 0; i < ZM_MAX_RX_URB_NUM; i++)
    {
        if(!(macp->RxUrbCtx[i].inUse)){
            macp->RxUrbCtx[i].buf = hif_usb_alloc_skb(ZM_MAX_RX_BUFFER_SIZE, 
                    0, 4);
            if(macp->RxUrbCtx[i].buf) {
                macp->RxUrbCtx[i].inUse = 1;
                zfLnxUsbIn(macp, &(macp->RxUrbCtx[i]));
            }
            else{
                printk("%s: DATA_IN Buf alloc fail, index=%d\n", __FUNCTION__, 
                        i);
                timer_act=1;
                break;
            }
        }
    }

error:

    if(timer_act) {
        macp->tm_rxbuf_act = 1;
        hif_usb_add_timer(&macp->tm_rxbuf_alloc, RX_URB_BUF_ALLOC_TIMER);
   }
    else{
        macp->tm_rxbuf_act = 0;
        del_timer(&macp->tm_rxbuf_alloc);
    }
}

void zfLnxInitUsbRxQ(HIF_DEVICE_USB *macp)
{
    uint16_t i;
    struct sk_buff *buf;

    /* Zero memory for UsbRxBufQ */
    //memset(macp->UsbRxBufQ, 0, sizeof(struct sk_buff) * ZM_MAX_RX_URB_NUM);
    init_timer(&macp->tm_rxbuf_alloc);
    macp->tm_rxbuf_alloc.function = RxUrbBufTimer;
    macp->tm_rxbuf_alloc.data = (unsigned long)macp;

    for (i = 0; i < ZM_MAX_RX_URB_NUM; i++)
    {
        buf = hif_usb_alloc_skb(ZM_MAX_RX_BUFFER_SIZE, 0, 4);
        macp->RxUrbCtx[i].buf = buf;
        macp->RxUrbCtx[i].failcnt = 0;
        macp->RxUrbCtx[i].macp = macp;
    }

    //macp->RxBufTail = ZM_MAX_RX_URB_NUM - 1;

    /* Submit all Rx urbs */
    for (i = 0; i < ZM_MAX_RX_URB_NUM; i++)
    {
        macp->RxUrbCtx[i].inUse = 1;
        zfLnxUsbIn(macp, &(macp->RxUrbCtx[i]));
    }
}

uint32_t zfLnxUsbIn(HIF_DEVICE_USB *macp, UsbRxUrbContext *RxUrbCtx)
{
    uint32_t ret;
    //struct usbdrv_private *macp = (struct usbdrv_private *)dev->priv;
    uint8_t *data = NULL;
    uint32_t len;
    urb_t *urb = RxUrbCtx->WlanRxDataUrb;
    struct sk_buff *buf = RxUrbCtx->buf;
    
    data= buf->data;
    len = buf->len;

    /* Submit a rx urb */
    ret = zfLnxUsbSubmitBulkUrb(urb, macp->udev, USB_WLAN_RX_PIPE,
            USB_DIR_IN, data, ZM_MAX_RX_BUFFER_SIZE,
            zfLnxUsbDataIn_callback, RxUrbCtx);

    if (ret != 0)
        printk("zfLnxUsbIn:zfwUsbSubmitBulkUrb fail, status: 0x%08x\n", 
                (int)ret);

    return ret;
}

#ifdef HIF_USB_ENABLE_STREAM_MODE
#define ZM_DONT_COPY_RX_BUFFER
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) || \
    (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19))
void zfLnxUsbDataIn_callback(urb_t *urb)
#else
void zfLnxUsbDataIn_callback(urb_t *urb, struct pt_regs *regs)
#endif
{
    //zdev_t* dev = urb->context;
    //struct usbdrv_private *macp = (struct usbdrv_private *)dev->priv;
    UsbRxUrbContext *RxUrbCtx = (UsbRxUrbContext *)urb->context;
    HIF_DEVICE_USB *macp = RxUrbCtx->macp;
    struct sk_buff *buf;
    struct sk_buff *new_buf;
    int status;


#if ZM_USB_STREAM_MODE == 1
    static int remain_len = 0, check_pad = 0, check_len = 0;
    int index = 0;
    int chk_idx;
    unsigned short int pkt_len;
    unsigned short int pkt_tag;
    int ii;
    struct sk_buff *rxBufPool[16];
    unsigned short int rxBufPoolIndex = 0;
#endif

    /* Check status for URB */
    if (urb->status != 0){
        printk("zfLnxUsbDataIn_callback() : status=0x%x\n", urb->status);
        if ((urb->status != -ENOENT) && (urb->status != -ECONNRESET)
            && (urb->status != -ESHUTDOWN))
        {
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))    
            if (urb->status == USB_ST_INTERNALERROR)
                    status = -1;
#else
            if (urb->status == -EPIPE){
                    /*printk(KERN_ERR "nonzero read bulk status received: \
                      -EPIPE");*/
                    status = -1;
                }

                if (urb->status == -EPROTO){
                    /*printk(KERN_ERR "nonzero read bulk status received: \
                      -EPROTO");*/
                    status = -1;
                }
#endif
        }

        if (urb->status != -ECONNRESET) {
            if ((urb->status == -EPROTO) ||
                (urb->status == -ETIMEDOUT)) {
                if (RxUrbCtx->failcnt < ZM_MAX_USB_URB_FAIL_COUNT) {
                    printk("zfLnxUsbDataIn_callback() : re-submit\n");
                    RxUrbCtx->failcnt ++;
                
                    /* Submit a Rx urb */
                    zfLnxUsbIn(macp, RxUrbCtx);
                
                    return;
                }
            }
            else {
                if (urb->status == -ESHUTDOWN) {
                    printk("%s: stop submit urb, because of shutdown\n", __func__);
                }
                else {
                    /* Re-Submit a Rx urb */
                    printk("zfLnxUsbDataIn_callback() : re-submit\n");
                    RxUrbCtx->failcnt = 0;
                
                    /* Submit a Rx urb */
                    zfLnxUsbIn(macp, RxUrbCtx);
                
                    return;
                }
            }
        }

        //printk(KERN_ERR "urb->status: 0x%08x\n", urb->status);

        /* Dequeue skb buffer */
        buf = RxUrbCtx->buf;
        RxUrbCtx->buf = NULL;
        dev_kfree_skb_any(buf);
#ifdef ADF_OS_MEM_DEBUG
            __adf_os_free_debug ((void *) buf);
#endif

        #if 0
        /* Enqueue skb buffer */
        zfLnxPutUsbRxBuffer(dev, buf);

        /* Submit a Rx urb */
        zfLnxUsbIn(dev, urb, buf);
        #endif

        
        return;
    }

    if (urb->actual_length == 0)
    {
        printk(KERN_ERR "Get an URB whose length is zero");
        status = -1;
    }

    /* Dequeue skb buffer */
    RxUrbCtx->failcnt = 0;
    buf = RxUrbCtx->buf;
    RxUrbCtx->buf = NULL;

    //zfwBufSetSize(dev, buf, urb->actual_length);
#if 0
#ifdef NET_SKBUFF_DATA_USES_OFFSET
    buf->tail = 0;
    buf->len = 0;
#else
    buf->tail = buf->data;
    buf->len = 0;
#endif
#endif
    skb_put(buf, urb->actual_length);

#if ZM_USB_STREAM_MODE == 1
    if (remain_len != 0)
    {
        struct sk_buff *remain_buf = macp->remain_buf;

        index = remain_len;
        remain_len -= check_pad;

        /*  Copy data */
        printk("Memcpy 1\n");
        memcpy(&(remain_buf->data[check_len]), buf->data, remain_len);
        check_len += remain_len;
        remain_len = 0;

        rxBufPool[rxBufPoolIndex++] = remain_buf;
    }

    while(index < urb->actual_length)
    {
        pkt_len = buf->data[index] + (buf->data[index+1] << 8);
        pkt_tag = buf->data[index+2] + (buf->data[index+3] << 8);

        if(pkt_tag != 0x4e00)
            printk("non Stream\n");

        if (pkt_tag == 0x4e00)
        {
            int pad_len;

            /*printk("Get a packet, index: %d, pkt_len: 0x%04x\n", index, 
              pkt_len);*/
            #if 0
            /* Dump data */
            for (ii = index; ii < pkt_len+4;)
            {
                printk("%02x ", (buf->data[ii] & 0xff));

                if ((++ii % 16) == 0)
                    printk("\n");
            }

            printk("\n");
            #endif

            pad_len = 4 - (pkt_len & 0x3);

            if(pad_len == 4)
                pad_len = 0;

            chk_idx = index;
            index = index + 4 + pkt_len + pad_len;

            if (index > ZM_MAX_RX_BUFFER_SIZE)
            {
                remain_len = index - ZM_MAX_RX_BUFFER_SIZE; // - pad_len;
                check_len = ZM_MAX_RX_BUFFER_SIZE - chk_idx - 4;
                check_pad = pad_len;

                /* Allocate a skb buffer */
                //new_buf = zfwBufAllocate(dev, ZM_MAX_RX_BUFFER_SIZE);
                new_buf = hif_usb_alloc_skb(ZM_MAX_RX_BUFFER_SIZE, 0, 4);
                if(new_buf == NULL) { 
                  printk("OOM here -- 00x\n");
                  goto alloc_fail;
                } else {
                    /* Set skb buffer length */
#ifdef NET_SKBUFF_DATA_USES_OFFSET
                    new_buf->tail = 0;
                    new_buf->len = 0;
#else
                    new_buf->tail = new_buf->data;
                    new_buf->len = 0;
#endif

                    skb_put(new_buf, pkt_len);
    
                    /* Copy the buffer */
                    printk("Memcpy 2\n");
                    memcpy(new_buf->data, &(buf->data[chk_idx+4]), check_len);
    
                    /* Record the buffer pointer */
                    macp->remain_buf = new_buf;
                }
            }
            else
            {
#ifdef ZM_DONT_COPY_RX_BUFFER
                if (rxBufPoolIndex == 0 || 1)
                {
                    new_buf = skb_clone(buf, GFP_ATOMIC);
                    
                    if(new_buf == NULL) { 
                      goto alloc_fail;
                    } else {
                        new_buf->data = &(buf->data[chk_idx+4]);
                        new_buf->len = pkt_len;
                    }
                }
                else
                {
#endif
                /* Allocate a skb buffer */
                new_buf = hif_usb_alloc_skb(ZM_MAX_RX_BUFFER_SIZE, 0, 4);
                if(new_buf == NULL) { 
                    printk("OOM here -- 002\n");
                    goto alloc_fail;
                } else {
                    /* Set skb buffer length */
                #ifdef NET_SKBUFF_DATA_USES_OFFSET
                    new_buf->tail = 0;
                    new_buf->len = 0;
                #else
                    new_buf->tail = new_buf->data;
                    new_buf->len = 0;
                #endif
    
                    skb_put(new_buf, pkt_len);
    
                    /* Copy the buffer */
                    memcpy(new_buf->data, &(buf->data[chk_idx+4]), pkt_len);
                    printk("MEMCPY 3\n");
                }


#ifdef ZM_DONT_COPY_RX_BUFFER
                }
#endif
                rxBufPool[rxBufPoolIndex++] = new_buf;
            }
        }
        else
        {
            printk(KERN_ERR "Can't find tag, pkt_len: 0x%04x, tag: 0x%04x\n", 
                    pkt_len, pkt_tag);
           
            /* Free buffer */
            dev_kfree_skb_any(buf);
#ifdef ADF_OS_MEM_DEBUG
                __adf_os_free_debug ((void *) buf);
#endif

            
            /* Allocate a skb buffer */
            new_buf = hif_usb_alloc_skb(ZM_MAX_RX_BUFFER_SIZE, 0, 0);
            if(new_buf == NULL) {
                printk("OOM here -- 003\n");
                
                RxUrbCtx->inUse = 0;
                if (!macp->tm_rxbuf_act) {
                    macp->tm_rxbuf_act = 1;
                    hif_usb_add_timer(&macp->tm_rxbuf_alloc, 
                            RX_URB_BUF_ALLOC_TIMER);
                }
            } else {
                /* Enqueue skb buffer */
                RxUrbCtx->buf = new_buf;
    
                /* Submit a Rx urb */
                zfLnxUsbIn(macp, RxUrbCtx);
            }
            
            
            return;
        }
    }
    
alloc_fail:
    /* Free buffer */
    dev_kfree_skb_any(buf);
#ifdef ADF_OS_MEM_DEBUG
        __adf_os_free_debug ((void *) buf);
#endif

#endif

    /* Allocate a skb buffer */
    //new_buf = dev_alloc_skb(ZM_MAX_RX_BUFFER_SIZE);
    new_buf = hif_usb_alloc_skb(ZM_MAX_RX_BUFFER_SIZE, 0, 4);
    if(new_buf == NULL) {
        printk("OOM here -- 004\n");

        RxUrbCtx->inUse = 0;
        if (!macp->tm_rxbuf_act) {
            macp->tm_rxbuf_act = 1;
            hif_usb_add_timer(&macp->tm_rxbuf_alloc, RX_URB_BUF_ALLOC_TIMER);
        }
    } else {
    
        /* Enqueue skb buffer */
        RxUrbCtx->buf = new_buf;
    
        /* Submit a Rx urb */
        zfLnxUsbIn(macp, RxUrbCtx);
    }

#if ZM_USB_STREAM_MODE == 1

    for(ii = 0; ii < rxBufPoolIndex; ii++)
    {
        macp->Callbacks.rxCompletionHandler(macp->Callbacks.Context, 
                rxBufPool[ii], USB_WLAN_RX_PIPE);
        //macp->usbCbFunctions.zfcbUsbRecv(dev, rxBufPool[ii]);
    }
#else
    /* pass data to upper layer */
    //macp->usbCbFunctions.zfcbUsbRecv(dev, buf);
    macp->Callbacks.rxCompletionHandler(macp->Callbacks.Context, buf, 
            USB_WLAN_RX_PIPE);
#endif
}

//uint16_t zfLnxGetFreeTxUrb(HIF_DEVICE_USB *macp, HIFUSBTxPipe *pipe)
UsbTxUrbContext* 
zfLnxGetFreeTxUrb(HIF_DEVICE_USB *macp, HIFUSBTxPipe *pipe)
{
    //struct usbdrv_private *macp = (struct usbdrv_private *)dev->priv;
    uint16_t idx;
    unsigned long irqFlag;
    UsbTxUrbContext *ctx = NULL;

    spin_lock_irqsave(&(macp->cs_lock), irqFlag);

#if 0
    //idx = ((macp->TxUrbTail + 1) & (ZM_MAX_TX_URB_NUM - 1));

    //if (idx != macp->TxUrbHead)
    if (pipe->TxUrbCnt != 0)
    {
        idx = pipe->TxUrbTail;
        pipe->TxUrbTail = ((pipe->TxUrbTail + 1) & (ZM_MAX_TX_URB_NUM - 1));
        pipe->TxUrbCnt--;
    }
    else
    {   
        //printk(KERN_ERR "macp->TxUrbCnt: %d\n", macp->TxUrbCnt); 
        idx = 0xffff;
    }
#else
    for(idx=0;idx<ZM_MAX_TX_URB_NUM;idx++) {
        if ( pipe->TxUrbCtx[idx].inUse == 0 ) {
            pipe->TxUrbCtx[idx].inUse = 1;
            ctx = &pipe->TxUrbCtx[idx];            
            break;
        }
    }
#endif

    spin_unlock_irqrestore(&(macp->cs_lock), irqFlag);
    return ctx;
    //return idx;
}

void 
zfLnxPutTxUrb(HIF_DEVICE_USB *macp, HIFUSBTxPipe *pipe, UsbTxUrbContext *ctx)
{
    //struct usbdrv_private *macp = (struct usbdrv_private *)dev->priv;
    //uint16_t idx;
    unsigned long irqFlag;
    
    spin_lock_irqsave(&(macp->cs_lock), irqFlag);

#if 0
    idx = ((pipe->TxUrbHead + 1) & (ZM_MAX_TX_URB_NUM - 1));

    //if (idx != macp->TxUrbTail)
    if (pipe->TxUrbCnt < ZM_MAX_TX_URB_NUM)
    {
        pipe->TxUrbHead = idx;
        pipe->TxUrbCnt++;
    }
    else
    {    
        printk("UsbTxUrbQ inconsistent: TxUrbHead: %d, TxUrbTail: %d\n",
                pipe->TxUrbHead, pipe->TxUrbTail);
    }
#else
    ctx->inUse = 0;           
#endif

    spin_unlock_irqrestore(&(macp->cs_lock), irqFlag);
}

uint16_t zfLnxCheckTxBufferCnt(HIF_DEVICE_USB *macp, HIFUSBTxPipe *pipe)
{
    //struct usbdrv_private *macp = (struct usbdrv_private *)dev->priv;
    uint16_t TxBufCnt;
    unsigned long irqFlag;
    
    spin_lock_irqsave(&(macp->cs_lock), irqFlag);

    TxBufCnt = pipe->TxBufCnt;

    spin_unlock_irqrestore(&(macp->cs_lock), irqFlag);
    return TxBufCnt;
}

UsbTxQ_t *zfLnxGetUsbTxBuffer(HIF_DEVICE_USB *macp, HIFUSBTxPipe *pipe)
{
    //struct usbdrv_private *macp = (struct usbdrv_private *)dev->priv;
    uint16_t idx;
    UsbTxQ_t *TxQ;
    unsigned long irqFlag;
        
    spin_lock_irqsave(&(macp->cs_lock), irqFlag);

    idx = ((pipe->TxBufHead+1) & (ZM_MAX_TX_BUF_NUM - 1));

    //if (idx != macp->TxBufTail)
    if (pipe->TxBufCnt > 0)
    {
        /*printk("CWY - zfwGetUsbTxBuffer ,macp->TxBufCnt = %d\n", 
          macp->TxBufCnt);*/
        TxQ = (UsbTxQ_t *)&(pipe->UsbTxBufQ[pipe->TxBufHead]);
        pipe->TxBufHead = ((pipe->TxBufHead+1) & (ZM_MAX_TX_BUF_NUM - 1));
        pipe->TxBufCnt--;
    }
    else
    {
        if (pipe->TxBufHead != pipe->TxBufTail)
        {
            printk(KERN_ERR "zfwGetUsbTxBuf UsbTxBufQ inconsistent: \
                    TxBufHead: %d, TxBufTail: %d\n", pipe->TxBufHead, 
                   pipe->TxBufTail);
        }

        spin_unlock_irqrestore(&(macp->cs_lock), irqFlag);
        return NULL;
    }

    spin_unlock_irqrestore(&(macp->cs_lock), irqFlag);
    return TxQ;
}

void zfLnxFreeUsbTxBuffer(HIF_DEVICE_USB *macp, HIFUSBTxPipe *pipe)
{
    UsbTxQ_t *TxQ = NULL;

    do
    {
        TxQ = zfLnxGetUsbTxBuffer(macp, pipe);
        if ( TxQ != NULL ) {
            printk("Free queued buf %p\n", TxQ->buf);
            dev_kfree_skb_any(TxQ->buf);
#ifdef ADF_OS_MEM_DEBUG
                __adf_os_free_debug ((void *) TxQ->buf);
#endif

        }       
    } while( TxQ != NULL );
}

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) || \
    (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19))
void zfLnxUsbDataOut_callback(urb_t *urb)
#else
void zfLnxUsbDataOut_callback(urb_t *urb, struct pt_regs *regs)
#endif
{
    UsbTxUrbContext *ctx = (UsbTxUrbContext *)urb->context;
    HIF_DEVICE_USB *macp = (HIF_DEVICE_USB *)ctx->macp;
    struct sk_buff *buf;
    //UsbTxQ_t *TxData;

    buf = ctx->buf;
    ctx->buf = NULL;

    /* Give the urb back */
    hif_usb_assert(ctx->WlanTxDataUrb == urb);
    zfLnxPutTxUrb(macp, &macp->TxPipe, ctx);

    if (urb->status != 0) {
        printk("%s : status=0x%x\n", __FUNCTION__, urb->status);
        dev_kfree_skb_any(buf);

#ifdef ADF_OS_MEM_DEBUG
            __adf_os_free_debug ((void *) buf);
#endif

        return;
    }

    /* Check whether there is any pending buffer needed */
    /* to be sent */
    if (zfLnxCheckTxBufferCnt(macp, &macp->TxPipe) != 0)
    {
        //TxData = zfwGetUsbTxBuffer(dev);

        //if (TxData == NULL)
        //{
        //    printk("Get a NULL buffer from zfwGetUsbTxBuffer\n");
        //    return;
        //}
        //else
        //{
            zfLnxUsbSubmitTxData(macp, &macp->TxPipe);
        //}
    }

    macp->Callbacks.txCompletionHandler(macp->Callbacks.Context, buf);
}

#if  ZM_USB_TX_STREAM_MODE == 1
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0))
void zfLnxUsbDataOut1_callback(urb_t *urb)
#else
void zfLnxUsbDataOut1_callback(urb_t *urb, struct pt_regs *regs)
#endif
{
    UsbTxUrbContext *ctx = (UsbTxUrbContext *)urb->context;
    HIF_DEVICE_USB *macp = (HIF_DEVICE_USB *)ctx->macp;
    struct sk_buff *buf;
    static long lastRecordTime = 0;
    static long lastDispCount = 0;
    static long totalCount = 0;
    //UsbTxQ_t *TxData;

    buf = ctx->buf;
    ctx->buf = NULL;

    totalCount++;
    if(jiffies - lastRecordTime > HZ)
    {
        lastRecordTime = jiffies;
        //printk("TX=%d\n",totalCount- lastDispCount);
        lastDispCount=totalCount;
    }
    if(urb->status)
        printk("urb->status !=0(%d) in %s\n", urb->status,__func__);
    /* Give the urb back */
    zfLnxPutTxUrb(macp, &macp->TxPipe,ctx);

    /* Check whether there is any pending buffer needed */
    /* to be sent */
    if (zfLnxCheckTxBufferCnt(macp, &macp->TxPipe) != 0)
    {
        //TxData = zfwGetUsbTxBuffer(dev);

        //if (TxData == NULL)
        //{
        //    printk("Get a NULL buffer from zfwGetUsbTxBuffer\n");
        //    return;
        //}
        //else
        //{
            zfLnxUsbSubmitTxData(macp, &macp->TxPipe);
        //}
    }

    //macp->htcCallbacks.txCompletionHandler(macp->htcCallbacks.Context, buf);
}
#endif

#ifndef DISABLE_USB_MPHP
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) || \
    (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19))
void zfLnxUsbHPTxCompl_callback(urb_t *urb)
#else
void zfLnxUsbHPTxCompl_callback(urb_t *urb, struct pt_regs *regs)
#endif
{
    UsbTxUrbContext *ctx = (UsbTxUrbContext *)urb->context;
    HIF_DEVICE_USB *macp = (HIF_DEVICE_USB *)ctx->macp;
    struct sk_buff *buf;
    //UsbTxQ_t *TxData;

//printk("[%s %d]\n", __FUNCTION__, __LINE__);
    buf = ctx->buf;
    ctx->buf = NULL;

    /* Give the urb back */
    hif_usb_assert(ctx->WlanTxDataUrb == urb);
    zfLnxPutTxUrb(macp, &macp->HPTxPipe, ctx);

    if (urb->status != 0) {
        printk("%s : status=0x%x\n", __FUNCTION__, urb->status);
        dev_kfree_skb_any(buf);

#ifdef ADF_OS_MEM_DEBUG
            __adf_os_free_debug ((void *) buf);
#endif

        return;
    }

    /* Check whether there is any pending buffer needed */
    /* to be sent */
    if (zfLnxCheckTxBufferCnt(macp, &macp->HPTxPipe) != 0)
    {
        //TxData = zfwGetUsbTxBuffer(dev);

        //if (TxData == NULL)
        //{
        //    printk("Get a NULL buffer from zfwGetUsbTxBuffer\n");
        //    return;
        //}
        //else
        //{
        //printk("[%s %d] buf cnd not 0\n", __FUNCTION__, __LINE__);
        zfLnxUsbSubmitTxData(macp, &macp->HPTxPipe);
        //}
    }

    macp->Callbacks.txCompletionHandler(macp->Callbacks.Context, buf);
}

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,5,0)) || \
    (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,19))
void zfLnxUsbMPTxCompl_callback(urb_t *urb)
#else
void zfLnxUsbMPTxCompl_callback(urb_t *urb, struct pt_regs *regs)
#endif
{
    UsbTxUrbContext *ctx = (UsbTxUrbContext *)urb->context;
    HIF_DEVICE_USB *macp = (HIF_DEVICE_USB *)ctx->macp;
    struct sk_buff *buf;
    //UsbTxQ_t *TxData;

//printk("[%s %d]\n", __FUNCTION__, __LINE__);
    buf = ctx->buf;
    ctx->buf = NULL;

    /* Give the urb back */
    hif_usb_assert(ctx->WlanTxDataUrb == urb);
    zfLnxPutTxUrb(macp, &macp->MPTxPipe, ctx);

    if (urb->status != 0) {
        printk("%s : status=0x%x\n", __FUNCTION__, urb->status);
        dev_kfree_skb_any(buf);

#ifdef ADF_OS_MEM_DEBUG
            __adf_os_free_debug ((void *) buf);
#endif

        return;
    }

    /* Check whether there is any pending buffer needed */
    /* to be sent */
    if (zfLnxCheckTxBufferCnt(macp, &macp->MPTxPipe) != 0)
    {
        //TxData = zfwGetUsbTxBuffer(dev);

        //if (TxData == NULL)
        //{
        //    printk("Get a NULL buffer from zfwGetUsbTxBuffer\n");
        //    return;
        //}
        //else
        //{
        //printk("[%s %d] buf cnd not 0\n", __FUNCTION__, __LINE__);
        zfLnxUsbSubmitTxData(macp, &macp->MPTxPipe);
        //}
    }

    macp->Callbacks.txCompletionHandler(macp->Callbacks.Context, buf);
    
}
#endif
int32_t zfLnxUsbSubmitTxData_NONDATA(HIF_DEVICE_USB *macp, HIFUSBTxPipe *pipe)
{
//    uint32_t i;
    int32_t ret;
    //uint16_t freeTxUrb;
    UsbTxUrbContext *freeTxUrb;
#if 0 && ZM_USB_TX_STREAM_MODE == 1
    uint8_t *puTxBuf = NULL;
#endif
    UsbTxQ_t *TxData;
    //struct usbdrv_private *macp = (struct usbdrv_private *)dev->priv;
#if 0 && ZM_USB_TX_STREAM_MODE == 1
    u8_t               ii;
    u16_t              offset = 0;
    u16_t              usbTxAggCnt;
    u16_t              *pUsbTxHdr;
    UsbTxQ_t           *TxQPool[ZM_MAX_TX_AGGREGATE_NUM];
#endif
    uint8_t *transfer_buffer;
    uint32_t len;

    /* First check whether there is a free URB */
    freeTxUrb = zfLnxGetFreeTxUrb(macp, pipe);

    /* If there is no any free Tx Urb */
    if (freeTxUrb == NULL)
    {   
        //printk(KERN_ERR "Can't get free Tx Urb\n");
        //printk("CWY - Can't get free Tx Urb\n");
        return 0xffff;
    }

#if 0 && ZM_USB_TX_STREAM_MODE == 1
    usbTxAggCnt = zfLnxCheckTxBufferCnt(dev, pipe);

    if (usbTxAggCnt >= ZM_MAX_TX_AGGREGATE_NUM)
    {
       usbTxAggCnt = ZM_MAX_TX_AGGREGATE_NUM;
    }
    else
    {
       usbTxAggCnt = 1;
    }

    //printk("usbTxAggCnt: %d\n", usbTxAggCnt);
#endif

#if 0 && ZM_USB_TX_STREAM_MODE == 1
    for(ii = 0; ii < usbTxAggCnt; ii++)
    {
#endif
    /* Dequeue the packet from UsbTxBufQ */
    TxData = zfLnxGetUsbTxBuffer(macp, pipe);
    if (TxData == NULL)
    {
        /* Give the urb back */
        zfLnxPutTxUrb(macp, pipe, freeTxUrb);
        return 0xffff;
    }

#if 0 && ZM_USB_TX_STREAM_MODE == 1
    /* Point to the freeTxUrb buffer */
    puTxBuf = freeTxUrb->txUsbBuf;

    puTxBuf += offset;
    pUsbTxHdr = (u16_t *)puTxBuf;

    /* Add the packet length and tag information */
    *pUsbTxHdr++ = TxData->hdrlen + TxData->snapLen + 
             (TxData->buf->len - TxData->offset) +  TxData->tailLen;

    *pUsbTxHdr++ = 0x697e;

    puTxBuf += 4;
#endif // #ifdef ZM_USB_TX_STREAM_MODE
    transfer_buffer = TxData->buf->data;
    len = TxData->buf->len;

#if 0 && ZM_USB_TX_STREAM_MODE == 1
    // Add the Length and Tag
    len += 4;

    //printk("%d packet, length: %d\n", ii+1, len);

    if (ii < (ZM_MAX_TX_AGGREGATE_NUM-1))
    {
        /* Pad the buffer to firmware descriptor boundary */
        offset += (((len-1) / 4) + 1) * 4;
    }

    if (ii == (ZM_MAX_TX_AGGREGATE_NUM-1))
    {
        len += offset;
    }

    TxQPool[ii] = TxData;

    //DbgPrint("%d packet, offset: %d\n", ii+1, pUsbTxTransfer->offset);

    /* free packet */
    //zfBufFree(dev, txData->buf);
    }
#endif

    freeTxUrb->buf = TxData->buf;

    /* Submit a tx urb */
    ret = zfLnxUsbSubmitBulkUrb(freeTxUrb->WlanTxDataUrb, macp->udev,
            pipe->TxPipeNum, USB_DIR_OUT, transfer_buffer,
            len, pipe->TxUrbCompleteCb, freeTxUrb);//macp);
    if (ret != 0)
    {
        printk("MM zfwUsbSubmitBulkUrb fail, status: %d\n", (int)ret);
        //zfLnxPutTxUrb(macp, pipe, freeTxUrb);
        zfLnxPutTxUrb(macp, pipe,freeTxUrb);
    }


    /* free packet */
    //dev_kfree_skb_any(TxData->buf);
#if 0 && ZM_USB_TX_STREAM_MODE == 1
    for(ii = 0; ii < usbTxAggCnt; ii++)
        macp->usbCbFunctions.zfcbUsbOutComplete(dev, TxQPool[ii]->buf, 1, TxQPool[ii]->hdr);
#else
    //macp->htcCallbacks.txCompletionHandler(macp->htcCallbacks.Context, TxData->buf);
#endif

    return ret;
}
#if ZM_USB_TX_STREAM_MODE == 1
int tm_init = 0;
unsigned long callbytimer = 0;
HIF_DEVICE_USB *tm_macp; 
HIFUSBTxPipe *tm_pipe;
uint32_t zfLnxUsbSubmitTxData(HIF_DEVICE_USB *macp, HIFUSBTxPipe *pipe);
extern int aggcnt ;
extern int aggtimer;
void txTimer(unsigned long param)
{
    //printk("TM running\n");
    callbytimer = 1;
    zfLnxUsbSubmitTxData(tm_macp, tm_pipe);
    callbytimer = 0;
    mod_timer(&tm_macp->tm_txagg, jiffies+HZ/aggtimer);
}
unsigned long enterCheck = 0;
#endif
uint32_t 
zfLnxUsbSubmitTxData(HIF_DEVICE_USB *macp, HIFUSBTxPipe *pipe)
{
    //uint32_t i;
    //struct usbdrv_private *macp = (struct usbdrv_private *)dev->priv;
#if ZM_USB_TX_STREAM_MODE == 1
    char *bufIdx;
    UsbTxQ_t *TxData;
    UsbTxUrbContext *freeTxUrb = NULL;
    uint8_t *puTxBuf = NULL;
    uint32_t ret;
    uint8_t               ii;
    uint32_t              offset = 0;
    uint16_t              usbTxAggCnt;
    uint8_t              *pUsbTxHdr;
    UsbTxQ_t           *TxQPool[ZM_MAX_TX_AGGREGATE_NUM];
    uint8_t *transfer_buffer;
    uint32_t len;
    static unsigned long lastSent = 0;
#endif



#if ZM_USB_TX_STREAM_MODE == 1
    if(pipe->TxPipeNum != 1 )
#endif
        return zfLnxUsbSubmitTxData_NONDATA(macp, pipe);
#if ZM_USB_TX_STREAM_MODE == 1

    if(test_and_set_bit(0, (void *)&enterCheck))
    {
        //printk("Reenter!!!!!!!!!\n");
        return 0xffff;
    }
    if(tm_init == 0)
    {
        tm_init = 1;
        init_timer(&macp->tm_txagg);
        macp->tm_txagg.data = 0;
        macp->tm_txagg.function = txTimer;
        tm_macp = macp;
        tm_pipe = pipe;
        mod_timer(&macp->tm_txagg, jiffies+HZ/100);
    }
    //printk("Here comes ...,%d\n",pipe->TxPipeNum);
    /* First check whether there is a free URB */
    usbTxAggCnt = zfLnxCheckTxBufferCnt(macp, pipe);
    if(usbTxAggCnt == 0)
    {
        clear_bit(0,(void *)&enterCheck);
        return 0xffff;
    }
    else if (usbTxAggCnt >= aggcnt/*ZM_MAX_TX_AGGREGATE_NUM*/ )
    {
       usbTxAggCnt = aggcnt/*ZM_MAX_TX_AGGREGATE_NUM*/;
       if(callbytimer == 0)
           mod_timer(&tm_macp->tm_txagg, jiffies+HZ/aggtimer);
            
    }
    else if (callbytimer)
    {
        if(usbTxAggCnt >= aggcnt/*ZM_MAX_TX_AGGREGATE_NUM*/  )
            usbTxAggCnt = aggcnt/*ZM_MAX_TX_AGGREGATE_NUM*/;
    }
    else
    {
       usbTxAggCnt = 1;
        clear_bit(0,(void *)&enterCheck);
       return 0xffff;
    }

    freeTxUrb = zfLnxGetFreeTxUrb(macp, pipe);
#if 0
    switch(freeTxUrb)
    {
        case 0:
            bufIdx = pipe->txbuf0;
            break;
        case 1:
            bufIdx = pipe->txbuf1;
            break;
        default:
            printk("here~~~\n");
            BUG();
    }
#endif

    /* If there is no any free Tx Urb */
    if (freeTxUrb == NULL)
    {   
        //printk(KERN_ERR "Can't get free Tx Urb\n");
        clear_bit(0,(void *)&enterCheck);
        return 0xffff;
    }
    else
    {
        if(freeTxUrb == &pipe->TxUrbCtx[0])
            bufIdx = pipe->txbuf0;
        else if(freeTxUrb == &pipe->TxUrbCtx[1])
            bufIdx = pipe->txbuf1;
        else
        {
            printk("!!!!!!!!!!!@@ErroR\n");
            clear_bit(0,(void *)&enterCheck);
            return 0xffff;
        }
    }

    //usbTxAggCnt = zfLnxCheckTxBufferCnt(macp, pipe);


    //printk("usbTxAggCnt: %d\n", usbTxAggCnt);

    lastSent = jiffies;
    for(ii = 0; ii < usbTxAggCnt; ii++)
    {
        /* Dequeue the packet from UsbTxBufQ */
        TxData = zfLnxGetUsbTxBuffer(macp, pipe);
        if (TxData == NULL)
        {
            /* Give the urb back */
            if(ii != 0)
                printk("Data is less than expection\n");
            zfLnxPutTxUrb(macp, pipe,freeTxUrb);
            clear_bit(0,(void *)&enterCheck);
            return 0xffff;
        }

        /* Point to the freeTxUrb buffer */
        puTxBuf = bufIdx;//pipe->txbuf;//pipe->txUsbBuf[freeTxUrb];

        //if(offset != 0)
        //    printk("offset !+ 0\n");
        puTxBuf += offset;
        pUsbTxHdr = (uint8_t *)puTxBuf;

        /* Add the packet length and tag information */
        //*pUsbTxHdr++ = TxData->hdrlen + TxData->snapLen + 
        //         (TxData->buf->len - TxData->offset) +  TxData->tailLen;
        *pUsbTxHdr = TxData->buf->len & 0xFF;
        pUsbTxHdr++;

        *pUsbTxHdr = (TxData->buf->len >> 8) & 0xFF;
        pUsbTxHdr++;

        *pUsbTxHdr = 0x697e & 0xFF;
        pUsbTxHdr++;

        *pUsbTxHdr = (0x697e >> 8)& 0xFF;
        pUsbTxHdr++;

        puTxBuf += 4;
        transfer_buffer = TxData->buf->data;
        len = TxData->buf->len;

        if(len > 1600)
            printk("Len > 1600 !!\n");
        memcpy(pUsbTxHdr, transfer_buffer, len);

        // Add the Length and Tag
        len += 4;

        //printk("%d packet, length: %d\n", ii+1, len);

        if (ii < (usbTxAggCnt-1))
            /* Pad the buffer to firmware descriptor boundary */
            offset += (((len-1) / 4) + 1) * 4;

        if (ii == (usbTxAggCnt-1))
            len += offset;
        pUsbTxHdr+=len;

        TxQPool[ii] = TxData;

        //printk("%d",ii);
        //DbgPrint("%d packet, offset: %d\n", ii+1, pUsbTxTransfer->offset);

        /* free packet */
        //zfBufFree(dev, txData->buf);
    }

    //pipe->TxUrbCtx[freeTxUrb].buf = TxData->buf;
    //pipe->TxUrbCtx[freeTxUrb].buf = p;

    /* Submit a tx urb */
    //printk("%p %d\n", &pipe->txbuf[0],pipe->txbuf[0] & 0xFF);
    if(len > 16384)
        printk("len = %d, usbTxAggCnt=%d \n",len, usbTxAggCnt);

    ret = zfLnxUsbSubmitBulkUrb(freeTxUrb->WlanTxDataUrb, macp->udev,
            pipe->TxPipeNum, USB_DIR_OUT, 
            bufIdx/*pipe->txbuf*//*transfer_buffer*/,
            len, zfLnxUsbDataOut1_callback, freeTxUrb);//macp);
    if(ret)
        ret = zfLnxUsbSubmitBulkUrb(freeTxUrb->WlanTxDataUrb, macp->udev,
                pipe->TxPipeNum, USB_DIR_OUT, 
                bufIdx/*pipe->txbuf*//*transfer_buffer*/,
                len, zfLnxUsbDataOut1_callback, freeTxUrb);//macp);
    {
        static long lastRecordTime = 0;
        static long lastDispCount = 0;
        static long totalCount = 0;
        //UsbTxQ_t *TxData;


        totalCount+=usbTxAggCnt;
        if(jiffies - lastRecordTime > HZ)
        {
            lastRecordTime = jiffies;
            //printk("TX=%d\n",totalCount- lastDispCount);
            lastDispCount=totalCount;
        }
    }
    if (ret != 0)
    {
        printk("zfwUsbSubmitBulkUrb fail, status: %d\n", (int)ret);
        //zfLnxPutTxUrb(macp, pipe);
        zfLnxPutTxUrb(macp, pipe,freeTxUrb);
    }

    /* free packet */
    //dev_kfree_skb_any(TxData->buf);
    for(ii = 0; ii < usbTxAggCnt; ii++)
        macp->Callbacks.txCompletionHandler(macp->Callbacks.Context, 
                TxQPool[ii]->buf);
    /*macp->usbCbFunctions.zfcbUsbOutComplete(dev, TxQPool[ii]->buf, 1, 
    TxQPool[ii]->hdr);*/

    clear_bit(0,(void *)&enterCheck);
    return ret;
#endif
}


uint16_t zfLnxGetFreeTxBuffer(HIF_DEVICE_USB *macp, HIFUSBTxPipe *pipe)
{
    return ZM_MAX_TX_BUF_NUM - pipe->TxBufCnt;
}

uint16_t
zfLnxPutUsbTxBuffer(HIF_DEVICE_USB *macp, struct sk_buff *buf, 
        HIFUSBTxPipe *pipe)
{
    //struct usbdrv_private *macp = (struct usbdrv_private *)dev->priv;
    uint16_t idx;
    UsbTxQ_t *TxQ;
    unsigned long irqFlag;
    
    spin_lock_irqsave(&(macp->cs_lock), irqFlag);

    idx = ((pipe->TxBufTail+1) & (ZM_MAX_TX_BUF_NUM - 1));

    /* For Tx debug */

    //if (idx != macp->TxBufHead)
    if (pipe->TxBufCnt < ZM_MAX_TX_BUF_NUM)
    {
        TxQ = (UsbTxQ_t *)&(pipe->UsbTxBufQ[pipe->TxBufTail]);

        TxQ->buf = buf;

        pipe->TxBufTail = ((pipe->TxBufTail+1) & (ZM_MAX_TX_BUF_NUM - 1));
        pipe->TxBufCnt++;
    }
    else
    {
#if 0
        printk(KERN_ERR "zfLnxPutUsbTxBuffer UsbTxBufQ inconsistent: \
                TxBufHead: %d, TxBufTail: %d, TxBufCnt: %d\n",
                macp->TxBufHead, macp->TxBufTail, macp->TxBufCnt);
#endif
        spin_unlock_irqrestore(&(macp->cs_lock), irqFlag);
        return 0xffff;
    }

    spin_unlock_irqrestore(&(macp->cs_lock), irqFlag);
    return 0;
}

uint32_t 
zfLnxUsbOut(HIF_DEVICE_USB *macp, struct sk_buff *buf, HIFUSBTxPipe *pipe)
{
    uint32_t ret = 0;
    //struct usbdrv_private *macp = (struct usbdrv_private *)dev->priv;

    /* Check length of tail buffer */
    //zm_assert((tailLen <= 16));

    /* Enqueue the packet into UsbTxBufQ */
    if (zfLnxPutUsbTxBuffer(macp, buf, pipe) == 0xffff)
    {
        /* free packet */
        //dev_kfree_skb_any(buf);
        //macp->usbCbFunctions.zfcbUsbOutComplete(dev, buf, 0, hdr);
        return 0xffff;
    }

    //return 0;
    zfLnxUsbSubmitTxData(macp, pipe);
    return ret;
}

int32_t 
zfLnxUsbSubmitControl(HIF_DEVICE_USB *macp, uint8_t req, uint16_t value, 
        uint16_t index, void *data, uint32_t size)
{
    int32_t result = 0;
    uint32_t ret = 0;
    //struct usbdrv_private *macp = (struct usbdrv_private *)dev->priv;
    uint8_t* buf;
    
    if (size > 0) {
        buf = kmalloc(size, GFP_KERNEL);
        
        hif_usb_assert( buf!=NULL );

        memcpy(buf, (uint8_t*)data, size);
    }
    else
        buf = NULL;
    
#if 0
    printk(KERN_ERR "req = 0x%02x\n", req);
    printk(KERN_ERR "value = 0x%04x\n", value);
    printk(KERN_ERR "index = 0x%04x\n", index);
    printk(KERN_ERR "data = 0x%lx\n", (u32_t) data);
    printk(KERN_ERR "size = %ld\n", size);
#endif

    result = usb_control_msg(macp->udev, usb_sndctrlpipe(macp->udev, 0),
            req, USB_DIR_OUT | 0x40, value, index, buf, size, HZ);

    if (result < 0) {
        printk("zfLnxUsbSubmitControl() failed, result=0x%x\n", result);
        ret = 1;
    }
    kfree(buf);
    
    return ret;
}


