// Filename: drawdev.c   - Drawing Engine device driver for pnx8550 sdk
// Author: Denis Oliver Kropp

#include <linux/module.h>

#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/miscdevice.h>
#include <linux/mm.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>

#include <linux/poll.h>

#include <asm/mach-pnx8xxx/int.h>

#include <pnx8550_misc.h>

#include <pnxdraw.h>

#include "drawhw.h"

MODULE_LICENSE("GPL v2");


static int draw_open ( struct inode          *inode,
                       struct file           *filp );

static int draw_close( struct inode          *inode,
                       struct file           *filp );

static int draw_ioctl( struct inode          *inode,
                       struct file           *filp,
                       unsigned int           cmd,
                       unsigned long          arg );

static int draw_mmap ( struct file           *file,
                       struct vm_area_struct *vma );

/**************************************************************************************************/

static struct file_operations draw_fops = {
     open:    draw_open,
     release: draw_close,
     ioctl:   draw_ioctl,
     mmap:    draw_mmap
};

static struct miscdevice draw_miscdev = {
     minor:    210,
     name:     "pnxdraw",
     fops:     &draw_fops
};

/**************************************************************************************************/

static PNXDrawSharedArea *shared;
static struct page       *shared_page;

/**************************************************************************************************/

static int
draw_open( struct inode *inode,
           struct file  *filp )
{
     return 0;
}

static int
draw_close( struct inode *inode,
            struct file  *filp )
{
     return 0;
}

static int
draw_ioctl( struct inode  *inode,
            struct file   *filp,
            unsigned int   cmd,
            unsigned long  arg )
{
     switch (cmd) {
          case PNXDRAW_IOCTL_RESET:
               return pnxdraw_reset( shared );

          case PNXDRAW_IOCTL_CONTINUE:
               if (shared->state & PNXDRAW_STATE_ACTIVE)
                    return 0;

               return pnxdraw_continue( shared, 1 );

          case PNXDRAW_IOCTL_WAITIDLE:
               return pnxdraw_waitidle( shared );

          case PNXDRAW_IOCTL_WAITDONE:
               return pnxdraw_waitdone( shared, arg );
     }

     return -ENOSYS;
}

/**************************************************************************************************/

static int
draw_mmap( struct file           *file,
           struct vm_area_struct *vma )
{
     unsigned int size;

     if (vma->vm_pgoff != 0)
          return -EINVAL;

     size = vma->vm_end - vma->vm_start;
     if (size != PAGE_ALIGN(sizeof(PNXDrawSharedArea)))
          return -EINVAL;

     /* Prevent the swapper from considering these pages for swap and touching them */
     vma->vm_flags |= (VM_RESERVED | VM_IO);
     vma->vm_page_prot = pgprot_noncached( vma->vm_page_prot );

//     return io_remap_page_range( vma, vma->vm_start,
     return io_remap_pfn_range( vma, vma->vm_start,
                                virt_to_phys( page_address(shared_page) ) >> PAGE_SHIFT,
                                size, vma->vm_page_prot );
}

/**************************************************************************************************/

static irqreturn_t
draw_irq( int irq, void *ctx, struct pt_regs *regs )
{
     pnxdraw_irq( shared );

     return IRQ_HANDLED;
}

/**************************************************************************************************/

static int __init
pnxdraw_init( void )
{
     int ret;

     /* Register a misc device called "pnxdraw". */
     ret = misc_register( &draw_miscdev );
     if (ret < 0) {
          printk( KERN_ERR "pnxdraw: can't register "
                  "misc device (minor %d)!\n", draw_miscdev.minor );
          return ret;
     }

     /* Register our interrupt handler. */
//     ret = request_irq( PNX8550_INT_2D_DRAW_ENG, draw_irq,
     ret = request_irq( PNX8XXX_INT_2D_DE, draw_irq,
                        0, "Drawing Engine", NULL );
     if (ret) {
          printk( KERN_ERR "pnxdraw: can't register interrupt "
//                  "handler for irq %d!\n", PNX8550_INT_2D_DRAW_ENG );
                  "handler for irq %d!\n", PNX8XXX_INT_2D_DE );
          misc_deregister( &draw_miscdev );
          return ret;
     }

     /* Initialize shared memory area. */
     shared_page = alloc_pages( GFP_KERNEL, 1 );
     shared      = (PNXDrawSharedArea*) ioremap( virt_to_phys( page_address(shared_page) ), PAGE_SIZE );

     SetPageReserved( shared_page );

     shared->phys = virt_to_phys( page_address(shared_page) );

     pnxdraw_reset( shared );

     printk( KERN_INFO "pnxdraw: initialized (shared page at "
             "%p using %d bytes)\n", shared, sizeof(PNXDrawSharedArea) );

     return 0;
}

module_init( pnxdraw_init );


static void __exit
pnxdraw_exit( void )
{
//     free_irq( PNX8550_INT_2D_DRAW_ENG, NULL );
     free_irq( PNX8XXX_INT_2D_DE, NULL );

     misc_deregister( &draw_miscdev );

     ClearPageReserved( shared_page );
     iounmap( shared );
     __free_page( shared_page );
}

module_exit( pnxdraw_exit );

