#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <sys/time.h>
#include <time.h>

#include <asm/types.h>

typedef struct {
     const char *name;
     void       *(*copy) (void *dst, const void *src, size_t n);
} MemcpyMethod;


static void *
loop_copy (void *dst, const void *src, size_t n)
{
     int          i;
     __u32       *d = dst;
     const __u32 *s = src;

     n >>= 2;

     for (i = 0; i < n; i++)
          d[i] = s[i];

     return dst;
}

static void *
duff_copy (void *dst, const void *src, size_t n)
{
     __u32       *d = dst;
     const __u32 *s = src;

     n >>= 2;

     while (n) {
          int l = n & 7;

          switch (l) {
               default:
                    l = 8;
                    d[7] = s[7];
               case 7:
                    d[6] = s[6];
               case 6:
                    d[5] = s[5];
               case 5:
                    d[4] = s[4];
               case 4:
                    d[3] = s[3];
               case 3:
                    d[2] = s[2];
               case 2:
                    d[1] = s[1];
               case 1:
                    d[0] = s[0];
          }

          d += l;
          s += l;
          n -= l;
     }

     return dst;
}

static void *
memset_benchmark (void *dst, const void *src, size_t n)
{
     return memset( dst, 0, n );
}

static void *
memset_loop_benchmark (void *dst, const void *src, size_t n)
{
     int    i;
     __u32 *d = dst;

     n >>= 2;

     for (i = 0; i < n; i++)
          d[i] = 0;

     return dst;
}

static MemcpyMethod methods[] = {
     { "glibc memcpy", memcpy},
     { "loop_copy", loop_copy},
     { "duff_copy", duff_copy},
     { "glibc memset", memset_benchmark},
     { "loop memset", memset_loop_benchmark},
     { NULL, NULL}
};

int
main (int argc, char *argv[])
{
     int           verify = 0;
     MemcpyMethod *method = methods;


     if (argv[1] && !strcmp(argv[1],"-v"))
          verify = 1;

     while (method->name) {
          int            i;
          char           buffer1[0x100000];
          char           buffer2[0x100000];
          struct timeval tv1, tv2;

          memset( buffer1, 0, sizeof(buffer1) );
          memset( buffer2, 0, sizeof(buffer2) );


          printf ("Benchmarking '%s'...", method->name);

          method->copy (buffer1, buffer2, 0x100000);
          method->copy (buffer2, buffer1, 0x100000);


          gettimeofday (&tv1, NULL);

          for (i=0; i<640; i++) {
               method->copy (buffer1, buffer2, 0x100000);

               if (verify) {
                    int n;

                    for (n=0; n<sizeof(buffer1); n++)
                         if (buffer1[n]) {
                              printf( "BYTE ERROR: %02x at %d\n", buffer1[n], n );
                              buffer1[n] = 0;
                         }
               }
          }

          gettimeofday (&tv2, NULL);

          printf ("%5ld Mb/sec\n",
                  640000 / (
                          (tv2.tv_sec * 1000 + tv2.tv_usec / 1000) -
                          (tv1.tv_sec * 1000 + tv1.tv_usec / 1000)));

          method++;
     }

     return 0;
}
