/*
 */

#include <common.h>
#include <asm/errno.h>
#include <malloc.h>

/*
 * MTD style partition parser for BFFS tags
 */

#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/mtd/compat.h>

#define BFFS_NAME_LEN       32
#define UBOOT_START         0x00000000

/* 
 * U_BOOT_SIZE includes some additional blocks in case of
 * bad blocks
 */
#define UBOOT_PARTITION_SIZE    (3 * 3 * 128 * 1024)
#define PUBLIC_KEY_LEN      	512

#define BLOCKSIZE           0x00040000
#define UBOOT_START_ADDR    TEXT_BASE
#define PARTTABLE_OFFSET    0x0001F400
#define TDF_KEY_OFFSET      0x0001F200

/* 
 *  bffs_part is forced into .data segment and not the .bss segment because
 *  this gets cleared
 */
static unsigned int     bffs_part[0x800/4] __attribute__ ((section (".data")));

/* Only to get the size of the partition structure */
struct bffs_tag {
    uint32_t    magic;          	/* Magic ID */
    uint32_t    partid;         	/* Partition ID */
    uint32_t    start;          	/* start address */
    uint32_t    end;            	/* End address */
    uint32_t    lastused;       	/* last used */
    uint32_t    numofblocks;        /* number of blocks */
    uint32_t    numofbadblocks;     /* number of bad blocks */
    uint32_t    numoferasedblocks;  /* number of erased blocks */
    void        *pfiles;        	/* file pointer */
    uint32_t    pnext;          	/* Next element */
    uint32_t    filesystemid;       /* File system ID */
    uint32_t    accessrule;     	/* access rules */
    uint32_t    numofopenwrfiles;   /* number of open files */
    char        volumename[BFFS_NAME_LEN];
};

static unsigned char tdf_key[PUBLIC_KEY_LEN] __attribute__ ((section (".data")));

void *mtd_part_read_key(void)
{
    return (void *)tdf_key;
}

/**
 * Routine implementing u-boot mtdparts command. Initialize/update default global
 * partition list and process user partition request (list, add, del).
 *
 * @param cmdtp command internal data
 * @param flag command flag
 * @param argc number of arguments supplied to the command
 * @param argv arguments list
 * @return 0 on success, 1 otherwise
 */
int parse_mtd_partitions(struct mtd_info *master, struct mtd_partition **pparts, unsigned long data)
{
    struct mtd_partition *parts;
    int ret, numparts = 0, realparts = 0;
    uint8_t *p8, i;
    char *names;


    /* Get a reference to the partition content */
    p8 = (uint8_t *)bffs_part;

    /* First count the number of partitions to create */
    for (i = 0; i < 20; i++) {
        uint32_t x;

        x  = *p8 * 16777216; p8++;
        x += *p8 * 65536;    p8++;
        x += *p8 * 256;      p8++;
        x += *p8;            p8++;

        if (x == 0xFFFFFFFF)
        {
            break;
        }

        numparts++;

        p8 += 4;

        p8 += 4;

        p8++;

        p8 += 4;

#ifdef DEBUG
        printk("Found: %s\n", (char *)p8);
#endif

        p8 += 12;
    }

    realparts = numparts;

#ifdef DEBUG
    printk("Found %d partitions\n", numparts);
#endif

    /* Add one additionally for the partition containing the u-boot image */
    //realparts++;
    parts = kzalloc((BFFS_NAME_LEN + sizeof(*parts)) * realparts, GFP_KERNEL);
    if (!parts) {
        ret = -ENOMEM;
        goto out;
    }

    names = (char *) &parts[realparts];

    /* Create mtd1 (uboot device) */
    //parts[0].size = UBOOT_PARTITION_SIZE;
    //parts[0].offset = 0;
    //parts[0].name = names;
    //strcpy(names, "u-boot");
    //names += BFFS_NAME_LEN;

    numparts = 0;

    /* Lets start from the beginning again and walk the walk */
    p8 = (uint8_t *)bffs_part;

    /* Walk the table and fill in the mtd parts */
    while (numparts < realparts) {
        uint32_t x;

        x  = *p8 * 16777216; p8++;
        x += *p8 * 65536;    p8++;
        x += *p8 * 256;      p8++;
        x += *p8;            p8++;

        parts[numparts].offset = x;

        //printf("parts[%d].offset = %d\n", numparts, parts[numparts].offset);

        x  = *p8 * 16777216; p8++;
        x += *p8 * 65536;    p8++;
        x += *p8 * 256;      p8++;
        x += *p8;            p8++;

        parts[numparts].size = x;

        /* Skip info from partition table not needed by the kernel */
        p8 += 4;

        if (*p8 == 0)
            parts[numparts].mask_flags = MTD_WRITEABLE;

        p8++;

        p8 += 4;

        parts[numparts].name = names;

        strncpy(names, (char *)p8, 12 - 1);

        p8 += 12;

        names += BFFS_NAME_LEN - 1;
        *names++ = 0x0;

        numparts++;
    }

    ret = numparts;
    *pparts = parts;
    
out:
    return ret;
}

/**
 * Routine implementing u-boot mtdparts command. Initialize/update default global
 * partition list and process user partition request (list, add, del).
 *
 */
#ifdef DEBUG
void bffs_dump(void)
{
    int i = 0;
    int size = 0;

    size = 8 * sizeof(struct bffs_tag);

    printf("Dump bffs\n");

    while(size) {   
        printf("[%08x] = %08x\n", &bffs_part[i], bffs_part[i]); 
        size-=4;
        i++;
    }
}
#endif

/**
 * Routine implementing copying the partition table from the u-boot image info a local 
 * structure.
 *
 */
void read_tdf_key(void)
{
    unsigned int *s = NULL;
    unsigned int *d;
    int size = PUBLIC_KEY_LEN;


    s = (unsigned int*)(UBOOT_START_ADDR + BLOCKSIZE + TDF_KEY_OFFSET);
    d = tdf_key;
 
    memcpy(d, s, size);
}

/**
 * Routine implementing copying the partition table from the u-boot image info a local 
 * structure.
 *
 */
void read_part_table(void)
{
    unsigned int* s = NULL;
    unsigned int* d = NULL;
    int size = 0;
    
    
    s = (unsigned int*)(UBOOT_START_ADDR + BLOCKSIZE + PARTTABLE_OFFSET);
    d = bffs_part;
    
    size = 8 * sizeof(struct bffs_tag);

    memcpy(d, s, size);
}

unsigned int read_part_offset(int partno)  
{
	uint8_t *p8;
	unsigned int x;


	/* Get a reference to the partition content */
	p8 = (uint8_t *)bffs_part;

	p8 += (partno * 29);     // each entry is 29 bytes long

	x  = *p8 * 16777216; p8++;
	x += *p8 * 65536;    p8++;
	x += *p8 * 256;      p8++;
	x += *p8;            p8++;

	return x;
}


