/*
 * (C) Copyright 2000, 2001
 * Wolfgang Denk, DENX Software Engineering, wd@denx.de.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 *
 */

/*
 * Support for read and write access to EEPROM like memory devices. This
 * includes regular EEPROM as well as  FRAM (ferroelectic nonvolaile RAM).
 * FRAM devices read and write data at bus speed. In particular, there is no
 * write delay. Also, there is no limit imposed on the numer of bytes that can
 * be transferred with a single read or write.
 *
 * Use the following configuration options to ensure no unneeded performance
 * degradation (typical for EEPROM) is incured for FRAM memory:
 *
 * #define CONFIG_SYS_I2C_FRAM
 * #undef CONFIG_SYS_EEPROM_PAGE_WRITE_DELAY_MS
 *
 */

#include <common.h>
#include <config.h>
#include <command.h>
#include <i2c.h>

#define UINT32 u32
#define UINT8  uchar

static UINT32 nvm_getsize(void) 
{
	return CONFIG_NVM_SIZE;
}

#if defined (CONFIG_CMD_NVRD)
static int nvm_read(UINT32 u4Offset, UINT8 au1Buf[], UINT32 u4Len);

int do_nvrd (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
    int ret;
    UINT32 u4EEPROMSize;
    UINT32 u4Offset, u4Len, u4Idx;
    UINT8  au1Buf[16];

    if (argc != 3)
    {
        printf ("FFFF: need 2 parameters\n");
        return 1;
    }

    u4EEPROMSize = nvm_getsize();
    u4Offset = (UINT32)simple_strtoul (argv [1], NULL, 16);
    u4Len = (UINT32)simple_strtoul (argv [2], NULL, 16);

    if (u4Offset > u4EEPROMSize)
    {
        printf ("FFFF: Offset is too large\n");
        return 1;
    }
    else if (u4Len == 0 || u4Len > 16)
    {
        printf ("FFFF: invalid length, not in 1..16\n");
        return 1;
    }

    DECLARE_GLOBAL_DATA_PTR;
    //gd->flags |= GD_FLG_SILENT; /* Disable EEPROM read message */
    ret = nvm_read(u4Offset, &au1Buf, u4Len);
	//gd->flags &= ~(GD_FLG_SILENT); /* Restore console */

    if (ret == 0)
    {
        printf ("%04x:", u4Offset);
        for (u4Idx = 0; u4Idx < u4Len; ++u4Idx) 
        {
            printf (" %02x", au1Buf[u4Idx]);
        }
        printf ("\n");
    }
    else 
    {
        printf ("FFFF: read error\n");
    }
    return ret;
}


static int nvm_read(UINT32 u4Offset, UINT8 au1Buf[], UINT32 u4Len)
{
	int ret;

	ret = i2c_read(CONFIG_NVM_SLAVE_ADDR, u4Offset, 2, au1Buf, u4Len);

	return ret;
}

U_BOOT_CMD(
    nvrd,    CONFIG_SYS_MAXARGS,    0,     do_nvrd,
    "nvrd    - read max. 16 bytes from NVRAM\n",
    "address length\n"
);
#endif

#if defined (CONFIG_CMD_NVWR)
static int nvm_write(UINT32 u4Offset, UINT8 au1Buf[], UINT32 u4Len);

int do_nvwr (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
    int ret;
    UINT32 u4EEPROMSize;
    UINT32 u4Idx, u4Offset, u4Len;
    UINT8 au1Buf[16];

    if (argc < 3) 
    {
        printf ("FFFF: need at least 2 parameters\n");
        return 1;
    }

    u4EEPROMSize = nvm_getsize();
    u4Offset = (UINT32)simple_strtoul (argv [1], NULL, 16);
    u4Len = argc - 2;

    if (u4Offset > u4EEPROMSize)
    {
        printf ("FFFF: Offset is too large\n");
        return 1;
    }
    else if (u4Len > 16)
    {
        printf ("FFFF: too many arguments\n");
        return 1;
    }

    for (u4Idx = 0; u4Idx < u4Len; u4Idx++)
    {
        au1Buf[u4Idx] = (UINT8) simple_strtoul (argv [u4Idx + 2], NULL, 16);
    }

    DECLARE_GLOBAL_DATA_PTR;
    //gd->flags |= GD_FLG_SILENT; /* Disable EEPROM write message */
	ret = nvm_write(u4Offset, au1Buf, u4Len);
    //gd->flags &= ~(GD_FLG_SILENT); /* Restore console */

    if (ret == 0)
    {
        printf ("%04x:", u4Offset);
        for (u4Idx = 0; u4Idx < u4Len; u4Idx++)
        {
            printf (" %02x", au1Buf[u4Idx]);
        }
        printf ("\n");
    }
    else 
    {
        printf ("FFFF: write error %d\n", ret);
    }
    return ret;
}



static int nvm_write(UINT32 u4Offset, UINT8 au1Buf[], UINT32 u4Len)
{
	int ret = 0;
	int i;
	
	for (i=0; i<u4Len; i++) {
		ret = i2c_write(CONFIG_NVM_SLAVE_ADDR, u4Offset+i, 2, &(au1Buf[i]), 1);
		if (ret != 0) {
			printf("Error at writing to addr 0x%x\n", u4Offset+i);
			break;
		}

		/*
		 * Wait for the write to complete.	The write can take
		 * up to 10mSec (we allow a little more time).
		 *
		 * On some chips, while the write is in progress, the
		 * chip doesn't respond.  This apparently isn't a
		 * universal feature so we don't take advantage of it.
		 */
		udelay(11000);
	}
	
	return ret;
}

U_BOOT_CMD(
    nvwr,    CONFIG_SYS_MAXARGS,    0,     do_nvwr,
    "nvwr    - write max. 16 bytes fo NVRAM\n",
    "address byte1 [byte2 [byte3 ...]]\n"
);

#endif


