/*
 * Device driver for NAND flash connected IP_2070 EFM.
 *
 * Copyright (C) 2008 NXP B.V.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Revision history
 * Version      Date          Remarks		
 * 0.0.1	 		20090108		Draft-Initial version
 */
/*----------------------------------------------------------------------
 * Standard include files
 *---------------------------------------------------------------------*/
#include <common.h>
#include <malloc.h>
#include <watchdog.h>
#include <linux/err.h>
#include <linux/mtd/compat.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/nand_ecc.h>

#include <asm/io.h>
#include <asm/errno.h>

#include <linux/nx_dmac_ip1902.h>

/**
* DMAC LLI structure 
*/
struct nx_dmac_lli_t {
	uint32_t						src_addr;					/* Source physical address */
	uint32_t						dst_addr;					/* Source physical address */
	uint32_t						next_lli;					/* Next LLI address */
	uint32_t						ctrl;							/* Control word for transfer */
};

/**
* DMAC channel structure 
*/
struct nx_dmac_chan_t {
 	int                 chanid;           /* channel ID */
	int									busy;							/* In use */
 	int								  chan_status;      /* DMAC channel status */
	struct nx_dmac_lli_t	*lli_ptr;				/* LLI ptr for channel */
	struct nx_dmac_lli_t	*lli_ptr_unal;	/* Unaligned LLI ptr for channel */
};

/**
* DMAC device structure 
*/
struct nx_dmac_t {
  void __iomem        		*dmac_base; 	/* DMAC base address */
	int											num_chans;		/* # of supported channels */
	int											num_ahb_mas;	/* # of AHB masters */
	int											num_reqs;			/* # of requestors */
	struct	nx_dmac_chan_t	*chans;				/* Channel structure */
};

/* DMAC device structure */
static struct nx_dmac_t	*nx_dmac=NULL;


/**
* nx_dmac_init - DMAC initialisation function
*
* Initialises the DMAC module
*/
int nx_dmac_init(void)
{
	int	i, ret = 0;
	uint32_t	per_id3;
	int	num_chans;
	struct nx_dmac_chan_t *chan;
	
	/* Allocate memory for nand control structure */
	nx_dmac = (struct nx_dmac_t *)kzalloc(sizeof(struct nx_dmac_t), GFP_KERNEL);
	if (!nx_dmac) {
		printk(KERN_ERR "nx_dmac: DMAC dev memory alloc \r\n");
		return -ENOMEM;
	}

	/* IO remap controller base */
	//nx_dmac->dmac_base =(void __iomem *) KSEG1ADDR(PNX8XXX_DMAC_BASE);
	nx_dmac->dmac_base =(void __iomem *) (MMIO_BASE + 0x2C000);
	
	/* Get HW info */
	per_id3 = readl(nx_dmac->dmac_base + NX_DMAC_PERI_ID3);

	/* # of channels */
	num_chans = per_id3 & 0x7;
	nx_dmac->num_chans = (0x2 << num_chans);
	
	/* # of AHB masters */
	if(per_id3 & 0x4) {
		nx_dmac->num_ahb_mas = 2;
	}
	else {
		nx_dmac->num_ahb_mas = 1;
	}
	
	/* # of requestors */
	if(per_id3 & 0x80) {
		nx_dmac->num_reqs = 32;
	}
	else {
		nx_dmac->num_reqs = 16;
	}
	
	/* Allocate memory for channels */
	nx_dmac->chans = kmalloc(sizeof(struct nx_dmac_chan_t) * 
			nx_dmac->num_chans, GFP_KERNEL);
	if(!nx_dmac->chans) {
		printk(KERN_ERR "nx_dmac: DMAC chan mem alloc failure \r\n");	
		ret = -ENOMEM;
		goto out1_free;
	}

	/*Initialise channel */
	for(i=0; i < nx_dmac->num_chans; i++) {
		chan = &nx_dmac->chans[i];
		chan->chanid = i;
		chan->busy = 0;
		chan->lli_ptr = NULL;
		chan->chan_status = 0;
	}

	//printk(KERN_INFO "nx_dmac: Registered with %d channels \r\n", nx_dmac->num_chans);

	return 0;

out1_free:	
	kfree(nx_dmac);

	return ret;
}

/**
* nx_dmac_tfr - DMAC transfer function 
* @tfr: DMAC transfer request structure
*
* Start the DMAC transfers
*/
int	nx_dmac_tfr(nx_dmac_tfr_t *tfr)
{
	int i, j, chanid;
	uint32_t	ctrl_wrd, chan_config;
	struct nx_dmac_chan_t *chan;
	void __iomem	*chan_start;
	
	/* Get free channel */
	i=0;
	while(i < nx_dmac->num_chans) { 
		if(!(nx_dmac->chans[i].busy)) {
			nx_dmac->chans[i].busy = 1;
			break;
		}
		i++;
	}

	/* Return if channel not free */
	if(i >= nx_dmac->num_chans) {
		printk(KERN_ERR "nx_dmac: DMAC no free chan \r\n");	
		return -EBUSY;	
	}

	/* Acquire channel */
	chan = &nx_dmac->chans[i];
	chanid = chan->chanid;
	
	/* Allocate DMAC LLI array */
	chan->lli_ptr_unal = kmalloc( (sizeof(struct nx_dmac_lli_t) + (CONFIG_SYS_CACHELINE_SIZE * 2)) * 
			tfr->num_reqs, GFP_KERNEL);
	if(!chan->lli_ptr_unal) {
		printk(KERN_ERR "nx_dmac: DMAC LLI mem failure \r\n");	
		nx_dmac->chans[i].busy = 0;
		return -ENOMEM;		
	}

	chan->lli_ptr = (struct nx_dmac_lli_t *)(((((unsigned long)chan->lli_ptr_unal) + 
						CONFIG_SYS_CACHELINE_SIZE - 1))& (~(CONFIG_SYS_CACHELINE_SIZE - 1))); 

	/* Initialise lli array */
	for(j=0; j < tfr->num_reqs; j++) {
		chan->lli_ptr[j].src_addr = tfr->req[j].src_addr;
		chan->lli_ptr[j].dst_addr = tfr->req[j].dst_addr;
		
		ctrl_wrd = 0;
		if(j == (tfr->num_reqs - 1)) {
			chan->lli_ptr[j].next_lli = (uint32_t) NULL;
			ctrl_wrd |= (1 << 31);										/* Enable TC interrupt */
		}
		else {
			chan->lli_ptr[j].next_lli = (uint32_t) (virt_to_phys) (&(chan->lli_ptr[j+1]));
		}
	
		/* Control */
		ctrl_wrd |= tfr->req[j].tfr_size & 0xFFF; /* Transfer size */
		ctrl_wrd |= (tfr->req[j].src_brst << 12); /* Source burst size */
		ctrl_wrd |= (tfr->req[j].dst_brst << 15); /* Destination burst size */
		ctrl_wrd |= (tfr->req[j].src_width << 18); /* Source width */
    ctrl_wrd |= (tfr->req[j].dst_width << 21); /* Destination width */
		ctrl_wrd |= (tfr->req[j].src_inc << 26);	/* Source increment */
		ctrl_wrd |= (tfr->req[j].dst_inc << 27);	/* Destination increment */
		ctrl_wrd |= (tfr->req[j].src_ahb << 24); 	/* Source AHB */
		ctrl_wrd |= (tfr->req[j].dst_ahb << 25); 	/* Destination AHB */
		chan->lli_ptr[j].ctrl = ctrl_wrd;
	}
	flush_dcache_range(chan->lli_ptr, (chan->lli_ptr + (sizeof(struct nx_dmac_lli_t) * tfr->num_reqs)));

	/* Write the first LLI to HW */
	chan_start = nx_dmac->dmac_base + (NX_DMAC_CHAN0_SRC + (NX_DMAC_CHAN_OFF * chanid));
	
	/* Enable DMA HW, AHB mast1, AHB mast2 to little endian */
	writel(0x1, nx_dmac->dmac_base + NX_DMAC_CONFIG);
	
	writel(chan->lli_ptr[0].src_addr, chan_start);
	writel(chan->lli_ptr[0].dst_addr, chan_start+0x4);
	writel(chan->lli_ptr[0].next_lli, chan_start+0x8);
	writel(chan->lli_ptr[0].ctrl, chan_start+0xC);

	/* Clear the interrupts */
	writel((1<<chanid), nx_dmac->dmac_base + NX_DMAC_INT_TC_CLR);
	writel((1<<chanid), nx_dmac->dmac_base + NX_DMAC_INT_ERR_CLR);

	/* Configure the channel */
	chan_config = 0;
	chan_config |= (tfr->req[0].src_per << 1);	/* Source peripheral number */
	chan_config |= (tfr->req[0].dst_per << 6);	/* Destination peripheral number */
	chan_config |= (tfr->req[0].flowctl << 11);	/* Flow control */
	chan_config |= (1 << 15);										/* Unmask TC interrupt */
	chan_config |= (1 << 14);										/* Unmask Error interrupt */
	chan_config |= 0x1;													/* Enable channel */

	chan->chan_status = 0xABCD;

	/* Write chan config to HW */
	writel(chan_config, chan_start + 0x10);

	return chan->chanid;
}

/**
* nx_dmac_complete - DMAC transfer complete function 
* @chanid: Channel id
*
* Complete the DMAC transfers
*/
int	nx_dmac_tfr_comp(int chanid)
{
	int res;
	struct nx_dmac_chan_t *chan;
	void __iomem	*chan_start;
	volatile uint32_t	int_status, tc_stat, err_stat;
	
	/* Get the free channel */
	chan = &nx_dmac->chans[chanid];
	if(!(chan->busy)) {
		printk(KERN_ERR "nx_dmac: DMAC tfr complete \r\n");
		return -EIO;
	}
	
	/* Get the channel start */
	chan_start = nx_dmac->dmac_base + (NX_DMAC_CHAN0_SRC + (NX_DMAC_CHAN_OFF * chanid));
	
	/* Wait DMAC interrupt */
	set_timeout(PNX8XXX_TIMEOUT);
	int_status = readl(nx_dmac->dmac_base + NX_DMAC_INT_STATUS);		
	while (!(int_status & (1 << chanid))) {
		if(did_timeout()) {
			printk("DMAC Transfer TIMEOUT \r\n");
			return -EIO;
		}

		int_status = readl(nx_dmac->dmac_base + NX_DMAC_INT_STATUS);		
	}
	
	/* Check & Clear TC interrupt */
	tc_stat = readl(nx_dmac->dmac_base + NX_DMAC_INT_TC_STATUS);
	if(tc_stat & (1<<chanid)) {
		writel((1<<chanid), nx_dmac->dmac_base + NX_DMAC_INT_TC_CLR);
		chan->chan_status = 0;
	}
				
	/* Check & Clear Error interrupt */
	err_stat = readl(nx_dmac->dmac_base + NX_DMAC_INT_ERR_STATUS);
	if(err_stat & (1<<chanid)) {
		writel((1<<chanid), nx_dmac->dmac_base + NX_DMAC_INT_ERR_CLR);
		chan->chan_status = -EIO;
	}
	
	/* status */
	res = chan->chan_status;
	
	/* Disable channel */
	writel(0x0, chan_start + 0x10);
	
	/* Disable DMA HW, AHB mast1, AHB mast2 to little endian */
	writel(0x0, nx_dmac->dmac_base + NX_DMAC_CONFIG);

	/* Free memory */
	kfree(chan->lli_ptr_unal);

	/* Free channel */
	chan->busy = 0;
	
	return res;
}

