/*
 * (C) Copyright 2006
 * NXP Semiconductors,
 * Robert Delien robert.delien@nxp.com, Hans Zuidam hans.zuidam@nxp.com
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 *
 */

#include <common.h>

#ifdef CONFIG_NXP_I2C_IP3203

#include <asm/io.h>
#include "nxp_i2c_ip3203.h"

#define	I2C_SLAVE	(2)
#define	I2C_WRITE	(1)
#define	I2C_READ	(0)
/*-----------------------------------------------------------------------
 * Definitions
 */


/*-----------------------------------------------------------------------
 * Local functions
 */

static int do_i2c (unsigned long	bus_addr,
		   uchar		i2c_addr,
		   uchar		i2c_write,
		   uchar		*data,
		   int			*length,
		   uchar		*ack)
{
	uchar	result = 0 ;
	uchar	state ;
	uint	data_ndx = 0 ;
	int	cntr ;
	mydebug("bus_addr=%x, i2c_addr=%x i2c_write=%x lenght=%x data[0]=%x\n",
		bus_addr, i2c_addr, i2c_write, *length, data[0]);

	state = readl(bus_addr + IP3203_STAT) & IP3203_STAT__STA ;
	if (state != 0xF8) {
		debug("\nI2C Error: Bus not in idle state\n") ;
		return -1 ;
	}

	do {
		switch (state) {
		case 0x00:  /* Bus error */
			result = -1 ;
			break ;

		case 0x08:  /* Start condition is transmitted */
		case 0x10:  /* Repeated start condition is transmitted */
			/* Clear start condition bit */
			writel( readl(bus_addr + IP3203_CON) & ~IP3203_CON__STA,
				bus_addr + IP3203_CON );

			/* Write address and R/W bit into data register */
			writel( ((i2c_addr << 1) | (i2c_write?0:1)) & IP3203_DATA__DAT,
				bus_addr + IP3203_DATA );

			/* Clear interrupt */
			writel( IP3203_INT_CLEAR__INT_CLEAR,
				bus_addr + IP3203_INT_CLEAR );
			break ;

		case 0x18:  /* Slave addres + write is transmitted, Ack received */
		case 0x20:  /* Slave addres + write is transmitted, NAck received */
		case 0x40:  /* Slave addres + read is transmitted, Ack received */
		case 0x48:  /* Slave addres + read is transmitted, NAck received */
			*ack = ((state==0x18)||(state==0x40))?1:0;
			if (i2c_write) {
				if ((*length-data_ndx > 0) && *ack) {
					/* Write the next data byte */
					writel( data[data_ndx++] & IP3203_DATA__DAT, bus_addr + IP3203_DATA );
				} else
					/* Set stop condition bit */
					writel( readl(bus_addr + IP3203_CON) | IP3203_CON__SETSTO,
						bus_addr + IP3203_CON );
			} else {
				if ((*length-data_ndx > 0) && *ack) {
					if (*length-data_ndx == 1)
						/* Indicate next byte read is the last byte by not acknowledging it */
						writel( readl(bus_addr + IP3203_CON) & ~IP3203_CON__AA,
							bus_addr + IP3203_CON );
				} else {
					/* Set stop condition bit */
					writel( readl(bus_addr + IP3203_CON) | IP3203_CON__SETSTO,
						bus_addr + IP3203_CON );
				}
			}

			/* Clear interrupt */
			writel(IP3203_INT_CLEAR__INT_CLEAR, bus_addr + IP3203_INT_CLEAR);
			break ;

		case 0x28:  /* Data byte has been transmitted, Ack received */
		case 0x30:  /* Data byte has been transmitted, NAck received */
			*ack = (state==0x28)?1:0;
			if ((*length-data_ndx > 0) && *ack) {
				writel( data[data_ndx++] & IP3203_DATA__DAT, bus_addr + IP3203_DATA );
			} else {
				/* Set stop condition bit */
				writel( readl(bus_addr + IP3203_CON) | IP3203_CON__SETSTO,
					bus_addr + IP3203_CON );
			}

			/* Clear interrupt */
			writel( IP3203_INT_CLEAR__INT_CLEAR,
				bus_addr + IP3203_INT_CLEAR );
			break ;

		case 0x50:  /* Data byte has been received, Ack was returned */
		case 0x58:  /* Data byte has been received, NAck was returned */
			*ack = (state == 0x50)?1:0;
			data[data_ndx++] = readl(bus_addr + IP3203_DATA) & IP3203_DATA__DAT ;

			switch(*length-data_ndx) {
			case 0:
				/* Set stop condition bit */
				writel( readl(bus_addr + IP3203_CON) | IP3203_CON__SETSTO,
					bus_addr + IP3203_CON );
				break;
			case 1:
				/* Indicate last byte reading by not acknowledging it */
				writel( readl(bus_addr + IP3203_CON) & ~IP3203_CON__AA,
					bus_addr + IP3203_CON );
				break;
			}

			/* Clear interrupt */
			writel( IP3203_INT_CLEAR__INT_CLEAR,
				bus_addr + IP3203_INT_CLEAR );
			break ;

		case 0x38:
		case 0x60:
		case 0x68:
		case 0x70:
		case 0x78:
		case 0x80:
		case 0x88:
		case 0x90:
		case 0x98:
		case 0xA0:
		case 0xA8:
		case 0xB0:
		case 0xB8:
		case 0xC0:
		case 0xC8:
			/* Slave mode and recovery modes not yet supported */
			result = -1 ;
			break ;

		case 0xF8:  /* Idle */
			/* Enable auto acknowledge if required */
			if( (*length-data_ndx > 1) &&
				(i2c_write == 0) )
				writel( readl(bus_addr + IP3203_CON) | IP3203_CON__AA,
					bus_addr + IP3203_CON );
			else
				writel( readl(bus_addr + IP3203_CON) & ~IP3203_CON__AA,
					bus_addr + IP3203_CON );

			/* Set start condition bit */
			writel( readl(bus_addr + IP3203_CON) | IP3203_CON__STA,
				bus_addr + IP3203_CON );

			/* Clear interrupt */
			writel( IP3203_INT_CLEAR__INT_CLEAR,
				bus_addr + IP3203_INT_CLEAR );
			break ;

		default:
			result = -2 ;
			break ;
		}

		/* Wait for a new I2C event */
		cntr = 4000 ;
		while (!(readl(bus_addr+IP3203_INT_STATUS) & IP3203_INT_STATUS__INT_STATUS) &&
			(cntr-- > 0))
			; /* nop */

		/* Read the new state */
		state = readl(bus_addr + IP3203_STAT) & IP3203_STAT__STA ;
	} while ((state != 0xF8) && (result == 0)) ;

	*length = data_ndx ;

	return(result);
}


/*=====================================================================*/
/*                         Public Functions                            */
/*=====================================================================*/

/*-----------------------------------------------------------------------
 * Initialization
 */
int nxp_i2c_ip3203_init (unsigned long bus_addr, int speed, int slaveaddr)
{
	/*
	 * WARNING: Do NOT save speed in a static variable: if the
	 * I2C routines are called before RAM is initialized (to read
	 * the DIMM SPD, for instance), RAM won't be usable and your
	 * system will crash.
	 */
	unsigned int reg ;

	/* Power up the module */
	writel( readl(bus_addr + IP3203_POWER_DOWN) & ~IP3203_POWER_DOWN__POWER_DOWN,
		bus_addr + IP3203_POWER_DOWN );

	if ( (readl(bus_addr + IP3203_MODULE_ID) & IP3203_MODULE_ID__MODULE_ID) !=
	     IP3203_MODULE_ID__MODULE_ID_ID ) {
		/* Power down the module */
		writel( readl(bus_addr + IP3203_POWER_DOWN) | IP3203_POWER_DOWN__POWER_DOWN,
			bus_addr + IP3203_POWER_DOWN );
		return(-1);
	}

	/* Configure the module */
	reg = readl(bus_addr + IP3203_CON);
	reg &= ~(IP3203_CON__EN_I2C |		/* Disable the module */
			IP3203_CON__AA |		/* Disable returning acknowledges */
			IP3203_CON__STA |		/* Clear start condition initiation */
			IP3203_CON__SETSTO );		/* Clear stop condition initiation */
	writel( reg, bus_addr + IP3203_CON);
	writel( IP3203_FSBIR__FSB_192kHz, bus_addr + IP3203_FSBIR );

	/* Disable interrupt */
	writel( readl(bus_addr + IP3203_INT_ENABLE) & ~IP3203_INT_ENABLE__INT_ENABLE,
		bus_addr + IP3203_INT_ENABLE );

	/* Clear pending interrupt */
	writel( IP3203_INT_CLEAR__INT_CLEAR,
		bus_addr + IP3203_INT_CLEAR );

	/* Enable the module */
	writel( readl(bus_addr + IP3203_CON) | IP3203_CON__EN_I2C,
		bus_addr + IP3203_CON );

	return(0);
}

/*-----------------------------------------------------------------------
 * Probe to see if a chip is present.  Also good for checking for the
 * completion of EEPROM writes since the chip stops responding until
 * the write completes (typically 10mSec).
 */
int nxp_i2c_ip3203_probe(unsigned long bus_addr, uchar addr)
{
	int	len = 0 ;
	uchar	ack ;
	int	res ;

	/*
	 * perform 1 byte write transaction with just address byte
	 * (fake write)
	 */
	res = do_i2c(bus_addr, addr, I2C_WRITE, NULL, &len, &ack);
	ack = res == 0 ? ack : 0;

	return( ack ? 0 : 1);
}

/*-----------------------------------------------------------------------
 * Read bytes
 */
int  nxp_i2c_ip3203_read(unsigned long bus_addr, uchar chip, uint addr, int alen, uchar *buffer, int len)
{
	uchar	ack ;
	uchar	addr_bytes[4];
	int	length = len ;

#ifdef CFG_I2C_EEPROM_ADDR_OVERFLOW
	/*
	 * EEPROM chips that implement "address overflow" are ones
	 * like Catalyst 24WC04/08/16 which has 9/10/11 bits of
	 * address and the extra bits end up in the "chip address"
	 * bit slots. This makes a 24WC08 (1Kbyte) chip look like
	 * four 256 byte chips.
	 *
	 * Note that we consider the length of the address field to
	 * still be one byte because the extra address bits are
	 * hidden in the chip address.
	 */
	chip |= ((addr >> (alen * 8)) & CFG_I2C_EEPROM_ADDR_OVERFLOW);

	printf("i2c_read: fix addr_overflow: chip %02X addr %02X\n",
		chip, addr);
#endif

	mydebug("bus_addr=%x, chip=%x, addr=%x, alen=%x, len=%x\n",
		bus_addr, chip, addr, alen, len);

	/* Send the address pointer first */
	switch (alen) {
	default:
		debug("\nI2C Error: Unsupported address length!\n");
		return(-1);
	case 4:
		addr_bytes[3] = (addr & 0xFF000000) >> 24 ;
		addr_bytes[2] = (addr & 0x00FF0000) >> 16 ;
		addr_bytes[1] = (addr & 0x0000FF00) >> 8 ;
		addr_bytes[0] = (addr & 0x000000FF) ;
		do_i2c (bus_addr, chip, I2C_WRITE, addr_bytes, &alen, &ack);
		break;
	case 3:
		addr_bytes[2] = (addr & 0x00FF0000) >> 16 ;
		addr_bytes[1] = (addr & 0x0000FF00) >> 8 ;
		addr_bytes[0] = (addr & 0x000000FF) ;
		do_i2c (bus_addr, chip, I2C_WRITE, addr_bytes, &alen, &ack);
		break;
	case 2:
		// EEPROM M24C64
		addr_bytes[0] = (addr & 0x0000FF00) >> 8 ;
		addr_bytes[1] = (addr & 0x000000FF) ;
		do_i2c (bus_addr, chip, I2C_WRITE, addr_bytes, &alen, &ack);
		break;
	case 1:
		addr_bytes[0] = (addr & 0x000000FF) ;
		do_i2c (bus_addr, chip, I2C_WRITE, addr_bytes, &alen, &ack);
		break ;
	case 0:
		/* Send not address pointer at all */
		break ;
	}

	/* Read the data */
	do_i2c (bus_addr, chip, I2C_READ, buffer, &len, &ack);
	if (len != length) {
		debug("\nI2C Error: %d bytes of %d bytes read from device 0x%02X\n", len, length, chip);
		return(-1);
	}

	return(0);
}

/*-----------------------------------------------------------------------
 * Write bytes
 */
int  nxp_i2c_ip3203_write(unsigned long bus_addr, uchar chip, uint addr, int alen, uchar *buffer, int len)
{
	uchar	ack ;
	uchar	addr_bytes[4];
	int	length = len ;

	mydebug("bus_addr=%x chip=%x addr=%x alen=%x, buffer[0]=%x, len=%x\n",
		bus_addr, chip, addr, alen, buffer[0], len);
	
	/* Send the address pointer first */
	switch (alen) {
	default:
		debug("\nI2C Error: Unsupported address length!\n");
		return(-1);
	case 4:
		addr_bytes[3] = (addr & 0xFF000000) >> 24 ;
		addr_bytes[2] = (addr & 0x00FF0000) >> 16 ;
		addr_bytes[1] = (addr & 0x00FF0000) >> 8 ;
		addr_bytes[0] = (addr & 0x000000FF) ;
		do_i2c (bus_addr, chip, I2C_WRITE, addr_bytes, &alen, &ack);

		/* Send the data */
		do_i2c (bus_addr, chip, I2C_WRITE, buffer, &len, &ack);
		break;
	case 3:
		addr_bytes[2] = (addr & 0x00FF0000) >> 16 ;
		addr_bytes[1] = (addr & 0x00FF0000) >> 8 ;
		addr_bytes[0] = (addr & 0x000000FF) ;
		do_i2c (bus_addr, chip, I2C_WRITE, addr_bytes, &alen, &ack);

		/* Send the data */
		do_i2c (bus_addr, chip, I2C_WRITE, buffer, &len, &ack);
		break;
	case 2:
		// EEPROm ST M24C64 requires that data directly follows address.
		// Assumes to one byte per write	
		addr_bytes[2] = buffer[0] ;
		addr_bytes[0] = (addr & 0x0000FF00) >> 8 ;
		addr_bytes[1] = (addr & 0x000000FF) ;
		mydebug("addr_bytes: [0]=%x [1]=%x\n", addr_bytes[0], addr_bytes[1]);
		alen = 3;

		/* Send the addr + data */
		do_i2c (bus_addr, chip, I2C_WRITE, addr_bytes, &alen, &ack);
		break;
	case 1:
		addr_bytes[0] = (addr & 0x000000FF) ;
		do_i2c (bus_addr, chip, 1, addr_bytes, &alen, &ack);

		/* Send the data */
		do_i2c (bus_addr, chip, I2C_WRITE, buffer, &len, &ack);
		break ;
	case 0:		
		/* Send not address pointer at all */
		do_i2c (bus_addr, chip, 1, buffer, &len, &ack);
		break ;
	}

	if (ack != 1) {
		puts("\nI2C Error: No ack is received!\n");
		return(-2);
	}

	if (len != length) {
		printf("\nI2C Error: %d bytes of %d bytes written to device 0x%02X\n", len, length, chip);
		return(-1);
	}

	return(0);
}

/*-----------------------------------------------------------------------
 * Read a register
 */
uchar nxp_i2c_ip3203_reg_read(unsigned long bus_addr, uchar i2c_addr, uchar reg)
{
	int	len = 1 ;
	uchar	buf;
	uchar	ack ;

	do_i2c (bus_addr, i2c_addr, I2C_READ, &buf, &len, &ack);
	if (len != 1)
		printf("\nI2C Error: Register read from device 0x%02X\n", i2c_addr);

	return(buf);
}

/*-----------------------------------------------------------------------
 * Write a register
 */
void nxp_i2c_ip3203_reg_write(unsigned long bus_addr, uchar i2c_addr, uchar reg, uchar val)
{
	int	len = 1 ;
	uchar	ack ;

	do_i2c (bus_addr, i2c_addr, I2C_WRITE, &val, &len, &ack);
	if (len != 1)
		printf("\nI2C Error: Register write to device 0x%02X\n", i2c_addr);
}

/*-----------------------------------------------------------------------
 * Set bus speed
 */
int nxp_i2c_ip3203_set_speed(unsigned long bus_addr, unsigned int speed)
{
	return (0) ;
}

/*-----------------------------------------------------------------------
 * Get bus speed
 */
unsigned int nxp_i2c_ip3203_get_speed(unsigned long bus_addr)
{
	printf("\nI2C Error: Setting speed is not supported yet\n");

	return (150000) ;
}

#endif	/* CONFIG_NXP_I2C_IP3203 */
