/*
 * Driver for NAND support, Rick Bronson
 * borrowed heavily from:
 * (c) 1999 Machine Vision Holdings, Inc.
 * (c) 1999, 2000 David Woodhouse <dwmw2@infradead.org>
 * PNX85xx support by Robert Delien, June 2007
 *
 * 22-08-2007, RDe:
 * The local functions 'nand_read_page_raw' and 'nand_write_page_raw'
 * always read and write whole pages. The idea behind this is that
 * the ganularity of NAND operations is always the page size because
 * ECC is calculated and validated over a whole pages.
 * BUT: That is not entirely true. ECC is calculated over chunks of
 * data of 256 bytes each. So theoretically it is possible to alter
 * data in only one of these chucks, recaculate only the ECC codes
 * of this specific chuck and write back both the chuck of data and
 * the ECC code to the device, using two separate write cycles.
 * As far as I know, this theoratical possibility isn't used
 * anywhere in U-Boot. Implementing support for this would cost
 * quite a performance penalty.
 *
 * 24-08-2007, RDe:
 * The following things still need to be improved in this legacy
 * NAND flash support file:
 * 1. Bad block detection shoudn't use DMA transfers but GPXIO
 *    cycles to read the bad-block-markers of a device. 
 * 2. Bad block detection should determine the bad-block-marker
 *    offset(S) according the bus-width, the device geometry and
 *    the manufacturer of the device.
 * 3. ECC calculation and validation should determine the offsets
 *    of the code also according the bus-width and the device
 *    geometry. The oob_config should be removed.
 * 4. The NAND flash primitives should be moved to
 *    NXP_PCIXIO_IPA051.c
 */

#include <common.h>
#include <command.h>

#ifdef CONFIG_NXP_PCIXIO_IPA051
#if defined(CONFIG_CMD_NAND) && defined(CONFIG_NAND) &&!defined(CONFIG_NAND_LEGACY) \
	&& defined (CONFIG_NAND_NXP_IPA051)

#include <malloc.h>
#include <watchdog.h>
#include <asm/io.h>
#include <asm/addrspace.h>
#include <linux/mtd/nand.h>
#include <jffs2/jffs2.h>
#include <nxp_pcixio_ipa051.h>

static void xio_nand_select_chip(struct mtd_info *mtd, int chipnr);
int do_nand_write_page_raw (struct mtd_info* mtd, size_t offset, void* data);
int do_nand_read_page_raw (struct mtd_info *mtd, size_t offset, void* data);
int do_nand_cache_read_page_raw (struct mtd_info *mtd, off_t offset, void* data);
extern int nand_check_block (struct mtd_info *mtd, unsigned long offset);
extern void set_timeout(unsigned long usec);
extern int did_timeout(void);

int board_nand_init(struct nand_chip *nand)
{
    /* Initial the IO_ADDR_R and IO_ADDR_W contains the xio profile register */
	unsigned long deviceAddr ;
	unsigned long nandId;
    unsigned long xioProfileValue = readl(IPA051 + nand->IO_ADDR_R) ;
	int           i;

    /*
     * This is a bit of an ugly function, but we need it because at this level
     * we don't know what XIO profile we're dealing with. We need to know the
     * XIO profile because it contains the bus width configuration and there's
     * no other way to determine that.
     */

    /* Check if profile is enabled and configured for NAND FLASH */
    if (  (xioProfileValue & IPA051_XIO_SELx_PROF__ENAB) &&
         ((xioProfileValue & IPA051_XIO_SELx_PROF__TYPE) == IPA051_XIO_SELx_PROF__TYPE_NAND) )
    {
        /* Calculate the offset from the profile value */
		deviceAddr = ((xioProfileValue & IPA051_XIO_SELx_PROF__OFFSET_EXT) >> 23) |
              		 ((xioProfileValue & IPA051_XIO_SELx_PROF__OFFSET) >> 5 ) ;
        deviceAddr *= 0x00800000UL ;

       	/* Add the address off XIO aperture */
       	deviceAddr += readl(IPA051 + IPA051_BASE18) & (unsigned long)IPA051_BASExx__ADDRESS ;

        mydebug("Dev: (E), profile=%08lx, deviceAddr = %08lx\n", xioProfileValue, deviceAddr);
    }  else {
        return 1;
    }                                       

    /* Correct the IO_ADDR base */
    nand->IO_ADDR_R         = (void __iomem *)deviceAddr;
    nand->IO_ADDR_W         = (void __iomem *)deviceAddr;
    
    nand->chip_delay        = 0;
	//nand->options           = NAND_NO_AUTOINCR | NAND_ALLOW_CLEAR_BBT | NAND_NO_SUBPAGE_WRITE;

    nand->ecc.mode       	      = NAND_ECC_SOFT;
    nand->select_chip         	  = xio_nand_select_chip;

	nand_reset(nand);
	if (nand_read_id(nand, &nandId) < 0)
		return (2);

	/* Check for 16 bit device */
	if (xioProfileValue & IPA051_XIO_SELx_PROF__EN_16BIT) {
		/* 16-bit device */
		nand->mafid =  nandId & 0x000000FFUL;
		nand->devid = (nandId & 0x00FF0000UL) >> 16;
		//Fix me.... todo: check spec
		nand->extid = (nandId & 0xFF000000UL) >> 24;  
		//bus16 = 1 ;
	} else {
		/* 8-bit device */
		nand->mafid =  nandId & 0x000000FFUL;
		nand->devid = (nandId & 0x0000FF00UL) >> 8;
		nand->extid = (nandId & 0xFF000000UL) >> 24;
		//bus16 = 0 ;
	}

	/* Try to identify manufacturer */
	for (i = 0; nand_flash_ids[i].id != 0x0; i++) {
		if (nand_flash_ids[i].id == nand->devid)
			break;
	}
	nand->options = nand_flash_ids[i].options;
	
    //xio_nand_scan_partitions(mtd);
    return 0;
}

/*
 * Select a chip
 */
static void xio_nand_select_chip(struct mtd_info *mtd, int chipnr)
{
#if 0   
    	struct nand_chip *nc = mtd->priv;
    	if (chipnr == -1)
        		return;
    	/*
     	* Get the XIO slot base address
     	*/
    	nc->slotbase = xn->slotbase[chipnr];
#endif  
}

int do_nand_read_page_raw (struct mtd_info *mtd, size_t offset, void* data)
{
	struct nand_chip *nand = mtd->priv;
	/*
	 * Pleas note: We dereference '*data' here beyond what may be expected:
	 * The calling function probably expects this pointer to be dereferenced
	 * by the In Band page length only, while in fact OOB is copied to this
	 * pointer as well. In normal circumstances this is no problem becuase
	 * we're just copying data to SDRAM. But when loading to variables
	 * surrounded by other variables, or loading pages out of sequence
	 * problems will arise
	 */
	 
	/* Read the page from the device using DMA */
	return( nand_dma_read( mtd,			    	     /* Device	*/
			       offset >> nand->page_shift,	     /* Row		*/
			       0,				                 /* Column	*/
				   mtd->writesize + mtd->oobsize,	 /* Length	 */
			       data, 		                     /* Destination	*/
				   0 ) );                            /* Cache read flag  */
}

int do_nand_cache_read_page_raw (struct mtd_info *mtd, off_t offset, void* data)
{
	struct nand_chip *nand = mtd->priv;
	/*
	 * Pleas note: We dereference '*data' here beyond what may be expected:
	 * The calling function probably expects this pointer to be dereferenced
	 * by the In Band page length only, while in fact OOB is copied to this
	 * pointer as well. In normal circumstances this is no problem becuase
	 * we're just copying data to SDRAM. But when loading to variables
	 * surrounded by other variables, or loading pages out of sequence
	 * problems will arise
	 */

	/* Read the page from the device using DMA */
	return( nand_dma_read( mtd,				/* Device	*/
			       offset >> nand->page_shift,	/* Row		*/
			       0,				/* Column	*/
				   mtd->writesize + mtd->oobsize,	 /* Length	 */
			       data, 			/* Destination	*/
				   NAND_CANCACHEDREAD(nand) ) );  /* Cache read flag  */
}

int do_nand_write_page_raw(struct mtd_info* mtd, size_t offset, void* data)
{
	struct nand_chip *nand = mtd->priv;
	unsigned long	nandStatus ;
	int		ret = 0;
#ifdef CONFIG_MTD_NAND_VERIFY_WRITE
	unsigned char*	compareBuffer_nonal;
	unsigned char*	compareBuffer;

	/* malloc unaligned compare buffer */
	compareBuffer_nonal = (unsigned char*)malloc(
		/* round up to cacheline to have a multiple number of cachelines in buffer */
		ROUND_UP(mtd->writesize + mtd->oobsize, CONFIG_SYS_CACHELINE_SIZE)
		/* make sure we can align the nonaligned buffer */
		+ (CONFIG_SYS_CACHELINE_SIZE - 1));
	/* align buffer pointer to nonaligned buffer */
	compareBuffer = (unsigned char*)ROUND_UP((unsigned int)compareBuffer_nonal, CONFIG_SYS_CACHELINE_SIZE);
#endif /* CONFIG_MTD_NAND_VERIFY_WRITE */

	/* Write the page to the device using DMA */
	if( nand_dma_write( mtd,				/* Device	*/
			    offset >> nand->page_shift,		/* Row		*/
			    0,					/* Column	*/
				mtd->writesize + mtd->oobsize, /* Length	*/
			    data ) ) {				/* Source	*/
		ret = -1;
		goto _cleanup;
	}

	/* Check nand status */
	nand_read_status (mtd, &nandStatus);
	if( (nandStatus & NAND_STATUS_WP) == 0 ) {
		printf( "\nError: NAND device is write protected\n" );
		ret = -3;
		goto _cleanup;
	} else if( (nandStatus & NAND_STATUS_FAIL) != 0 ) {
		printf( "\nError: NAND device reported an error (status = 0x%.8lX)\n", nandStatus );
		ret = -2;
		goto _cleanup;
	}

#ifdef CONFIG_MTD_NAND_VERIFY_WRITE
	/* Read back the page from the device using DMA */
	if( nand_dma_read( mtd,				/* Device	*/
			   offset >> nand->page_shift,		/* Row		*/
			   0,					/* Column	*/
			   mtd->writesize + mtd->oobsize,  /* Length	 */
			   compareBuffer,			/* Destination	*/
			   0 ) )	 { 		 /* Normal (uncached read)  */
		ret = -4;
		goto _cleanup;
	}

	/* Compare the read page with written page */
	if( memcmp( data, compareBuffer, mtd->writesize + mtd->oobsize ) != 0 )
	{
		printf("\nError: Verification failed (Forgotten to erase the block?)\n");
		ret = -5;
		goto _cleanup;
	}
#endif /* CONFIG_MTD_NAND_VERIFY_WRITE */

_cleanup:

#ifdef CONFIG_MTD_NAND_VERIFY_WRITE
	/* free the malloc'ed compare buffer before exit.. */
	free(compareBuffer_nonal);
#endif /* CONFIG_MTD_NAND_VERIFY_WRITE */

	return (ret);
}

#endif /* CONFIG_NAND* */
#endif /* CONFIG_NXP_PCIXIO_IPA051 */
