/*
 * Device driver for OneNAND flash connected IP_2070 ESMC.
 *
 * Copyright (C) 2008 NXP B.V.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Revision history
 * Version    Author           Date          Remarks
 * 0.0.1    Bangaragiri G     20080318    Draft-Initial version
 */

#include <common.h>
#include <linux/mtd/compat.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/onenand.h>
#include <linux/nx_onenand.h>

#include <asm/io.h>
#include <asm/errno.h>

#ifdef CONFIG_DMAC_NXP_1902
#include <linux/nx_dmac_ip1902.h>
#endif

#define CACHE_LINE_SIZE (32)
#define CACHE_LINE_MASK (CACHE_LINE_SIZE - 1)

struct nx_onenand_ctrl *nx_onenand=NULL;

static void nx_onenand_rw(void __iomem *addr, unsigned short *data, 
		int read)
{
	unsigned long misc2;
	unsigned int addr17, addr18, act_off;
	
	/* Set ENA bit */
	misc2 = readl(nx_onenand->glb_misc2);
	misc2 |= ONENAND_CFG_ENA;

	/* Get DCSN 18 bit Address */
	addr17 = ((unsigned int) addr) - ((unsigned int)nx_onenand->chip->base); 
	addr18 = addr17  << 1;
	act_off = addr18 & (~ONENAND_ADDR_MSB_BIT);
	addr = nx_onenand->chip->base + act_off;
	if(addr18 & ONENAND_ADDR_MSB_BIT) {
		misc2 |= (1 << ONENAND_ADDR_MSB_POS);
	}
	else {
		misc2 &= (~(1 << ONENAND_ADDR_MSB_POS));
	}
	writel(misc2, nx_onenand->glb_misc2);

	if(read) {
		*data = readw(addr);
	}
	else {
		writew(*data, addr);	
	}
	
	/* Reset ENA bit */
	misc2 &= ~ONENAND_CFG_ENA;
	writel(misc2, nx_onenand->glb_misc2);

	return;
}

/**
 * nx_onenand_readw - Read OneNAND register
 * @param addr    address to read
 *
 * Read OneNAND register
 */
static unsigned short nx_onenand_readw(void __iomem *addr)
{
	
	unsigned short data;

	/* Read Data */
	nx_onenand_rw(addr, &data, 1);

	return data;
}

/**
 * nx_onenand_writew - Write OneNAND register with value
 * @param value   value to write
 * @param addr    address to write
 *
 * Write OneNAND register with value
 */
static void nx_onenand_writew(unsigned short value, void __iomem *addr)
{
	nx_onenand_rw(addr, &value, 0);
}

void nx_onenand_mmcontrol(struct mtd_info *mtd, int sync_read)
{
	struct onenand_chip *chip = nx_onenand->chip;

	if(sync_read) {	
		/* Sync settings - SYSCFG1	*/
		chip->write_word(ONENAND_SYNC_SYS_CFG1, chip->base + ONENAND_REG_SYS_CFG1); // Working
 
		/* IP_2016 settings */
		writel(PNX8XXX_SYNC_MAIN_CFG, nx_onenand->main_cfg);  	// MAIN CONFIG
		writel(PNX8XXX_SYNC_READ_CFG, nx_onenand->read_cfg); 	// READ CONFIG
		writel(PNX8XXX_SYNC_BURST_CFG, nx_onenand->burst_cfg);	// BURST CONFIG
		//writel(0x00000001, (unsigned long *) (MMIO_BASE + 0x0002A200));
	}
	else {
		/* IP_2016 settings */	
		writel(nx_onenand->main_async_val, nx_onenand->main_cfg); // MAIN_CFG
		writel(nx_onenand->read_async_val, nx_onenand->read_cfg); // MAIN_CFG
		writel(nx_onenand->burst_async_val, nx_onenand->burst_cfg); // MAIN_CFG
		//writel(0x00000000, (unsigned long *) 0xBBE2A200);

		/* Async settings - SYSCFG1 */
		chip->write_word(nx_onenand->sys_cfg1_val, chip->base + ONENAND_REG_SYS_CFG1);
	}

	return;		
}

#ifdef CONFIG_NX_MEMCPY
extern void nx_memcpy( void* pDest, void* pSrc, unsigned int length);  
#endif

#ifdef CONFIG_DMAC_NXP_1902

void nx_onenand_dmac_tfr(struct mtd_info *mtd, unsigned char *buffer, 
		int offset, size_t count, int read)	
{
	struct onenand_chip *this = mtd->priv;
	nx_dmac_tfr_t       req;
	nx_dmac_stgt_t      stgt[1];
	int chanid, status;
	unsigned short word;
	unsigned char *word1;
	int i;
	int buf_offset=0;
	unsigned long misc2;
	size_t act_count=count;
	int buf = buffer;

	/* Return if count is 0 */
	if(!count) {
		return;	
	}
	
	/* Set up DMAC */
	if(read) {
		/* Word aligned transfers */
		/* If Non word-aligned , read word */
		word1 = (unsigned char *)buffer;	
		if(offset & 0x03) {
			word = this->read_word(this->base + offset);
			word1[0] = word  & 0xFF;
			word1[1] = (word >> 8)  & 0xFF;
			buf_offset += 2;
			act_count -= 2;
		}
	
		if(!act_count) {
			return;	
		}

		/* Check if size word aligned */
		if(act_count & 0x03) {
			act_count += 2; 	
		}
		
		/* Check if user buffer cache aligned */
		if (((int)(buffer + buf_offset)) & (CONFIG_SYS_CACHELINE_SIZE-1)) {
			buf = nx_onenand->dmabuf;
		}
		else {
			buf = buffer + buf_offset;
		}

		stgt[0].src_addr = (offset + nx_onenand->ahb_base + buf_offset);     
		//stgt[0].dst_addr = virt_to_phys(nx_onenand->dmabuf);
		stgt[0].dst_addr = virt_to_phys(buf);
		stgt[0].tfr_size = act_count >> 1;    
		//stgt[0].tfr_size = act_count >> 2;    
		stgt[0].flowctl = nx_dmac_mem2mem_dma;         
		//stgt[0].src_brst = nx_dmac_8;
		//stgt[0].dst_brst = nx_dmac_1;
		stgt[0].src_brst = nx_dmac_16;
		stgt[0].dst_brst = nx_dmac_16;
		stgt[0].src_width = nx_dmac_width_16;
		//stgt[0].src_width = nx_dmac_width_32;
		stgt[0].dst_width = nx_dmac_width_32;
		stgt[0].src_ahb = 1;		  /* Source AHB master 0 */
		stgt[0].dst_ahb = 0;              /* Dest AHB master 1 */
	}
	else {
		//memcpy(nx_onenand->dmabuf, buffer, count);	
#ifdef CONFIG_NX_MEMCPY
		nx_memcpy(nx_onenand->dmabuf, buffer, count);	
#else
		memcpy(nx_onenand->dmabuf, buffer, count);	
#endif		
		flush_dcache_range(nx_onenand->dmabuf, (nx_onenand->dmabuf + count));

		stgt[0].src_addr = virt_to_phys(nx_onenand->dmabuf);
		stgt[0].dst_addr = (offset + nx_onenand->ahb_base);
		stgt[0].tfr_size = count >> 2;    
		stgt[0].flowctl = nx_dmac_mem2mem_dma;         
		stgt[0].dst_brst = nx_dmac_1;
		stgt[0].src_brst = nx_dmac_8;
		stgt[0].src_width = nx_dmac_width_32;
		stgt[0].dst_width = nx_dmac_width_16;
		stgt[0].src_ahb = 0;		  /* Source AHB master 0 */
		stgt[0].dst_ahb = 1;              /* Dest AHB master 1 */
	}
 	
	stgt[0].src_per = 0;		          
 	stgt[0].dst_per = 0;
 	stgt[0].src_inc = 1;
 	stgt[0].dst_inc = 1;
	req.num_reqs = 1;
	req.req = &stgt[0];
	
	/* Reset ENA bit */
	misc2 = readl(nx_onenand->glb_misc2);
	misc2 &= ~ONENAND_CFG_ENA;
	writel(misc2, nx_onenand->glb_misc2);

	/* Start DMAC */
	chanid = nx_dmac_tfr(&req);
	if(chanid < 0) {
		printk(KERN_ERR "nx_onenand: Start DMAC \r\n");
		return;
	}

	/* Complete DMAC transfer */
	status = nx_dmac_tfr_comp(chanid);
	if(status) {
		printk(KERN_ERR "nx_onenand: COMP DMAC \r\n");	
		return;
	}

	if(read) {
		//invalidate_dcache_range(nx_onenand->dmabuf, (nx_onenand->dmabuf + count - buf_offset));
		invalidate_dcache_range((buf + buf_offset), (buf + count - buf_offset));
		//memcpy((buffer + buf_offset), nx_onenand->dmabuf, (count - buf_offset));	
		if (((int)(buffer + buf_offset)) & (CONFIG_SYS_CACHELINE_SIZE-1)) {
#ifdef CONFIG_NX_MEMCPY
			nx_memcpy((buffer + buf_offset), nx_onenand->dmabuf, (count - buf_offset));	
#else	
			memcpy((buffer + buf_offset), nx_onenand->dmabuf, (count - buf_offset));	
#endif		
		}
	}

	/* Reset ENA bit */
	misc2 = readl(nx_onenand->glb_misc2);
	misc2 |= ONENAND_CFG_ENA;
	writel(misc2, nx_onenand->glb_misc2);
}	

EXPORT_SYMBOL(nx_onenand_dmac_tfr);

#else

void nx_memcpy(void __iomem *addr, unsigned char *data, size_t count, 
		int read)
{
	unsigned long misc2;
	unsigned int addr17, addr18, act_off;
	int i;
	unsigned short *src, *dst;
	unsigned short word;	
	
	/* Set ENA bit */
	misc2 = readl(nx_onenand->glb_misc2);
	misc2 |= ONENAND_CFG_ENA;

	/* Get DCSN 18 bit Address */
	addr17 = ((unsigned int) addr) - ((unsigned int)nx_onenand->chip->base); 
	addr18 = addr17  << 1;
	act_off = addr18 & (~ONENAND_ADDR_MSB_BIT);
	addr = nx_onenand->chip->base + act_off;
	if(addr18 & ONENAND_ADDR_MSB_BIT) {
		misc2 |= (1 << ONENAND_ADDR_MSB_POS);
	}
	else {
		misc2 &= (~(1 << ONENAND_ADDR_MSB_POS));
	}
	writel(misc2, nx_onenand->glb_misc2);

	if(read) {
		src = (unsigned short *)addr;
		for(i=0; i< (count >> 1); i++) {
			word = readw(src + i);
			data[(2 * i)] = word & 0xFF;
			data[(2 * i)+1] = (word >>8) & 0xFF;
		}
	}
	else {
		dst = (unsigned short *)addr;
		for(i=0; i< (count >> 1); i++) {
			word = ((data[(2 * i)+1] << 8) | data[2 * i]);
			writew(word, (dst + i));
		}
	}
	
	/* Reset ENA bit */
	misc2 = readl(nx_onenand->glb_misc2);
	misc2 &= ~ONENAND_CFG_ENA;
	writel(misc2, nx_onenand->glb_misc2);

	return;
}

EXPORT_SYMBOL(nx_memcpy);

#endif /* CONFIG_MTD_NX_NAND_DMAC */


void nx_onenand_init(struct mtd_info *mtd, struct onenand_chip *chip)
{
	int ret, page_size, oob_size, buf_size,slot_num;
	int misc2;
	
	/* Allocate memory for onenand structure */ 
	nx_onenand = kzalloc(sizeof(struct nx_onenand_ctrl), GFP_KERNEL);
	if (!nx_onenand) {
		printk(KERN_ERR "nx_onenand: Dev Mem alloc failure \r\n");
		return;
	}
	nx_onenand->chip = chip;
	nx_onenand->mtd = mtd;
	nx_onenand->glb_misc2 = (void __iomem *) (MMIO_BASE + 0x00063600);
	
	/* Store Base address */
	nx_onenand->onenand_base_phys = CONFIG_SYS_ONENAND_BASE;

	/* Initialise chip structre */
	chip->base = (void __iomem *) KSEG1ADDR(nx_onenand->onenand_base_phys);

	/* Find AHB base address */
	slot_num = readl(nx_onenand->glb_misc2);
	slot_num >>= GLB_MISC2_ONENAND_SLOT_POS;
	slot_num &= 0x3;
	nx_onenand->ahb_base = (slot_num * ONENAND_2016_AHB_SIZE);
	nx_onenand->main_cfg = PNX8XXX_IP2016_BASE_VIRT + 
							(slot_num * PNX8XXX_DEV_CFG_OFFSET) + PNX8XXX_IP2016_MAIN_CFG; 
	nx_onenand->read_cfg = PNX8XXX_IP2016_BASE_VIRT + (slot_num * PNX8XXX_DEV_CFG_OFFSET) + 
										PNX8XXX_IP2016_READ_CFG; 
	nx_onenand->burst_cfg = PNX8XXX_IP2016_BASE_VIRT + (slot_num * PNX8XXX_DEV_CFG_OFFSET) + 
										PNX8XXX_IP2016_BURST_CFG; 

	/* Initilise the functions */
	chip->mmcontrol = nx_onenand_mmcontrol;
	//chip->mmcontrol = NULL;
	chip->read_word = nx_onenand_readw;
	chip->write_word = nx_onenand_writew;
	mtd->name = "nxonenand";
	mtd->priv = chip;

	/* Get the back of register settings for async */
	nx_onenand->main_async_val = readl(nx_onenand->main_cfg);   /* MAIN CFG reg value for async */
	nx_onenand->read_async_val = readl(nx_onenand->read_cfg);   /* READ CFG reg value for async */
	nx_onenand->burst_async_val = readl(nx_onenand->burst_cfg);  /* BUSRT CFG reg value for async */
	nx_onenand->sys_cfg1_val = chip->read_word(chip->base + ONENAND_REG_SYS_CFG1); /* SYS CFG1 reg value for async */

	/* Allocate DMA buffer */
	buf_size = MAX_ONENAND_PAGESIZE;
	nx_onenand->dmabuf_unalgn =  kzalloc( (buf_size + (CACHE_LINE_SIZE * 2)),
            GFP_DMA | GFP_KERNEL);
	if(!nx_onenand->dmabuf_unalgn) {
		printk(KERN_ERR "onenand_init: DMA buf alloc \r\n");
		return;
	}
	nx_onenand->dmabuf = (uint8_t *)((unsigned long)(nx_onenand->dmabuf_unalgn + CACHE_LINE_SIZE - 1)
		& (~CACHE_LINE_MASK));
	invalidate_dcache_range(nx_onenand->dmabuf, (nx_onenand->dmabuf + buf_size));
	
	/* DMAC init */
	ret = nx_dmac_init();
	if(ret) {
		printf(KERN_ERR "onenand_init:DMAC init failed \r\n");
		return;
	}
	
}
