/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Copyright (c) 2006-2007, LIPP Alliance
 * All Rights Reserved.
 *
 *---------------------------------------------------------------------------
 * %filename:     tmbslPhySMSC8710.c %
 * %pid_version:           1.2                %
 *---------------------------------------------------------------------------
 * DESCRIPTION: Macros and function prototypes for SMSC8710 PHY
 *
 * DOCUMENT REF: Datasheet SMSC LAN8710/LAN8710i  
 *               Revision 2.1 (03-06-09)
 *
 *
 *-----------------------------------------------------------------------------
 *
*/

#include "tmNxTypes.h"
#include "tmNxCompId.h"

#include "lipp_6300eth_common.h"

/*  Project include files */
#include "tmbslPhy.h"
#include "tmbslPhySMSC8710.h"
#include "tmhwLIPP6100Eth_Cfg.h"

/* Timeout in case of linux */
#ifdef __LIPP_6300ETH__
#include "i2c_ip3203.h"
#define AUTO_NEG_DELAY_MULTIPLIER (800)
#endif /* __LINUX_GMAC_DRV__*/

/* Defines */
#define PHY_UNIT_ID_COUNT  			1

#define PHY_TIMEOUT                 (0x0FFFFFFF)

#define TMBSLPHYSMSC8710_PHY_MMIO_ADDRESS0 ((0x1be00000+0x38000))

#define ANAR_DEFAULT_VAL (0x1E1)

/* gEthContext[] is filled with MAC base address in the tmbslPhySMSC8710Init() 
** function 
*/

tmbslPhySMSC8710Context_t gEthContext[PHY_UNIT_ID_COUNT] = {{TMBSLPHYSMSC8710_PHY_MMIO_ADDRESS0}}; 

typedef enum _tmbslPhySMSC8710Reg_t
{
    /* Basic mode control */
    tmbslPhySMSC8710Bmcr		      = 0,  

    /* Basic mode status */    
    tmbslPhySMSC8710Bmsr		      =1,  

    /* PHY ID1 register */        
    tmbslPhySMSC8710PhyIdr1		=2,

    /* PHY ID2 register */            
    tmbslPhySMSC8710PhyIdr2		=3,

    /* Auto negotiation advertisement register */                
    tmbslPhySMSC8710Anar			=4,

    /* Auto negotiation link partner ability register */                    
    tmbslPhySMSC8710Anlpar		=5,

    /* Auto negotiation expansion register */                        
    tmbslPhySMSC8710Aner			=6,

    /* 0x7 to 0xF reserved-*/

    /* Silicon Revision register */
    tmbslPhySMSC8710Srr=16,

    /* Mode control & status register */
    tmbslPhySMSC8710Mcsr=17,

    /* Special modes register */
    tmbslPhySMSC8710Smr=18,

    /* 19-25 reserved */

    /* Symbol error counter register */    
    tmbslPhySMSC8710Secr	= 26,
    
    /* Control/Status indication register */        
    tmbslPhySMSC8710Csir = 27,    

    /* Special internal testability controls register*/            
    tmbslPhySMSC8710Sitcr = 28,        
    
    /* Interrupt source register */
    tmbslPhySMSC8710Isr = 29,

    /* Interrupt mask register */
    tmbslPhySMSC8710Imr = 30,    

    /* PHY special control & status register */    
    tmbslPhySMSC8710Pscsr = 31,    

} tmbslPhySMSC8710Reg_t, *ptmbslPhySMSC8710Reg_t;


/* Static functions definition */

static tmErrorCode_t 
tmbslPhySMSC8710Read (
	tmUnitSelect_t				ethUnitId,
    tmbslPhySMSC8710Reg_t				reg,   
    pUInt16						pVal
	);
	
	
static tmErrorCode_t	
tmbslPhySMSC8710Write (
    tmUnitSelect_t				ethUnitId,
    tmbslPhySMSC8710Reg_t				reg,   
    UInt16						val
	);

#ifdef __LIPP_6300ETH__
static void get_phy_out_of_rst( void);
#endif

/* Exported functions */

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710GetSWVersion:
//
// DESCRIPTION: This function returns the PHY device interface software version 
//				information
//
// RETURN:      TM_OK
//
// NOTES:       This API can be called anytime i.e. before initializing the PHY 
//				or in PowerOff state.
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhySMSC8710GetSWVersion (
	ptmSWVersion_t					pPhyVersion
	)
{
	
	pPhyVersion->compatibilityNr = TMBSL_PHY_COMPATIBILITY_NR;
	pPhyVersion->majorVersionNr = TMBSL_PHY_MAJOR_VERSION_NR;
	pPhyVersion->minorVersionNr = TMBSL_PHY_MINOR_VERSION_NR;
	
	return TM_OK;

}

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710GetCapabilities:
//
// DESCRIPTION: This function returns the PHY capabilities for the specified PHY
//				unit. The function is callable at any time to return the unit's 
//				capabilities (PHY unit initialization is not necessary). 
//				Capabilities may be different among multiple PHY units.For completeness, 
//				a PHY BSL user should call this function for each PHY unit to 
//				determine its individual capabilities. 
//
// RETURN:      TM_OK
//
// NOTES:       This API can be called anytime i.e. before initializing the PHY. 
//				 
//-----------------------------------------------------------------------------


tmErrorCode_t
tmbslPhySMSC8710GetCapabilities (
    tmUnitSelect_t                			ethUnitId,  
    ptmbslPhyCapabilities_t  				pPhyCaps    
    )

{

    UInt16 bmsr;

    /*  Read the PHY capabilites from the BMSR register */
    tmbslPhySMSC8710Read(ethUnitId, tmbslPhySMSC8710Bmsr, &bmsr);

    pPhyCaps->T4Support100Base = 
        (((bmsr & TMBSL_PHYSMSC8710_BMSR_T4100BASE) > 0) ? True : False);

    pPhyCaps->Tx_FullDuplexSupport100Base = 
        (((bmsr &TMBSL_PHYSMSC8710_BMSR_X100BASEFD) > 0) ? True : False);

    pPhyCaps->Tx_HalfDuplexSupport100Base = 
        (((bmsr &TMBSL_PHYSMSC8710_BMSR_X100BASEHD) > 0) ? True : False);

    pPhyCaps->Tx_FullDuplexSupport10Base = 
        (((bmsr &TMBSL_PHYSMSC8710_BMSR_10MBPSFD) > 0) ? True : False);        

    pPhyCaps->Tx_HalfDuplexSupport10Base = 
        (((bmsr &TMBSL_PHYSMSC8710_BMSR_10MBPSHD) > 0) ? True : False);


    pPhyCaps->autoNegotiationAbility = 
                (((bmsr &TMBSL_PHYSMSC8710_BMSR_AN_ABLE) > 0) ? True : False);

    pPhyCaps->ledStatusSupport = True ;		

    /* Other capabilites set to False */   
    pPhyCaps->nextPageIndication = False;

    pPhyCaps->force100MbpsTxOff = False;

    pPhyCaps->bypassSymbolAlignment =False;

    pPhyCaps->badSSDDetectionConfig = False;
	
    return TM_OK;

}   
    
//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710Init:
//
// DESCRIPTION: This function initializes the PHY device. It should be called
//				before any access to the device is made. 
//
// RETURN:      TM_OK 
//
// NOTES:       This function initializes the PHY device with the following 
//				default initial configuration. No Autonegotiation is done in the 
//				initialization function
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhySMSC8710Init(
    tmUnitSelect_t  						ethUnitId   
    )
{

    tmErrorCode_t       		ethStatus = TM_OK;     		
    UInt32                         timeout=0;
    UInt16 regval=0;
    UInt16 id1, id2;

    #ifdef __LIPP_6300ETH__
    get_phy_out_of_rst();
    #endif

//    printf("\nIn SMSC Init\n");           

    gEthContext[ethUnitId].pRegs = gtmhwLIPP6100Eth_Base[ethUnitId].baseAddress;

    /* Set the mode register to default value */
    ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Smr,&regval);
//    printf("\nSMR val: %08x\n",regval);           

    regval &= 0xBF1F; /* Set to 100Mbps FD, MII mode */
    regval |= 0x60;    
    ethStatus = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Smr,regval);        
    ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Smr,&regval);
//    printf("\nSMR val: %08x\n",regval);           

    regval = TMBSL_PHYSMSC8710_BMCR_RST_VAL;

    ethStatus = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Bmcr,regval);

    if(ethStatus != TM_OK)
    {
        return ethStatus;
    }

    /* Check if PHY is back to normal condition */
    ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Bmcr,&regval);

    while( (regval & TMBSL_PHYSMSC8710_BMCR_RST_VAL) && (timeout < PHY_TIMEOUT) )
    {


        ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Bmcr,&regval);

        if(ethStatus != TM_OK)
        {
            break;
        }

        timeout++;

    }  

    if(ethStatus != TM_OK)
    {
        return(ethStatus);
    }

    if( (timeout == PHY_TIMEOUT) && (regval & TMBSL_PHYSMSC8710_BMCR_RST_VAL) )
    {
        return(TMBSL_ERR_PHY_INIT_FAILED);	
    } 

#if 0
    /* Read PHY Identification Register */
    ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710PhyIdr1,&id1);

    if (ethStatus != TM_OK)
    {
        return (ethStatus);
    }

    ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710PhyIdr2,&id2);

    if (ethStatus != TM_OK)
    {
        return (ethStatus);
    }

//    printf("\nETHERNET CLOCK1:%08x\n",*(unsigned long *)(0xbbe47710));                    
//    printf("\nETHERNET CLOCK2:%08x\n",*(unsigned long *)(0xbbe47714));                    
//    printf("\nETHERNET CLOCK3:%08x\n",*(unsigned long *)(0xbbe47718));                       
//    printf("\nSMSC PHY INIT successful\n\n\n");    
#endif

    return TM_OK;

}   

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710Deinit:
//
// DESCRIPTION: This function de-initializes the PHY device. Once the device is
//				deinitilized device will no more be available to access 
//
// RETURN:      TM_OK
//
// NOTES:       
//-----------------------------------------------------------------------------
//

tmErrorCode_t
tmbslPhySMSC8710Deinit(
    tmUnitSelect_t  						ethUnitId   
    )

{

    tmErrorCode_t       		ethStatus=TM_OK;     		
    UInt16 bmcr =0;

    bmcr |= TMBSL_PHYSMSC8710_BMCR_RST_VAL;

    /* All the registers will be reset */
    ethStatus = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Bmcr,bmcr);

    if(ethStatus != TM_OK)
    {
        return ethStatus;
    }
    else	
    {
        return TM_OK;
    }


}   

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710SetPowerState:
//
// DESCRIPTION: This function will set the Power State of the PHY device to specified 
//				power state
//
// RETURN:      TM_OK
//
// NOTES:       
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhySMSC8710SetPowerState(
    tmUnitSelect_t           				ethUnitId ,  
    tmPowerState_t          				phyPowerState
    )

{
    tmErrorCode_t ethStatus=TM_OK;     		
    UInt16 bmcr = 0;

    ethStatus = tmbslPhySMSC8710Read(ethUnitId, tmbslPhySMSC8710Bmcr, &bmcr);

    if(ethStatus != TM_OK)
    {
        return ethStatus;
    }

    if( (phyPowerState == tmPowerOn) || (phyPowerState == tmPowerOff) )
    {
        if(phyPowerState == tmPowerOff)
        {
            bmcr |= TMBSL_PHYSMSC8710_BMCR_PWRDN_EN;
        }
        else
        {
            bmcr &=TMBSL_PHYSMSC8710_BMCR_PWRDN_CLR;
        }

        ethStatus = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Bmcr,bmcr);

        return ethStatus;

    }
    else
    {
        return TMBSL_ERR_PHY_NOT_SUPPORTED;
    }

}   

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710GetPowerState:
//
// DESCRIPTION: This function will get the preset power state of the PHY device
//
// RETURN:      TM_OK
//
// NOTES:       
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhySMSC8710GetPowerState(
    tmUnitSelect_t                  		ethUnitId ,  
    ptmPowerState_t				phyPowerState
    )

{

    tmErrorCode_t       		ethStatus=TM_OK;     		
    UInt16 regVal=0;

    ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Bmcr,&regVal);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    *phyPowerState = (((regVal & TMBSL_PHYSMSC8710_BMCR_PWRDN_EN) > 0) ? tmPowerOff : tmPowerOn);

    return TM_OK;

}   


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710GetBasicModeControl:
//
// DESCRIPTION: This function will get the basic configuration of the PHY device. 
//
// RETURN:      TM_OK
//
// NOTES:       See #define for the Basic Mode Control 
//-----------------------------------------------------------------------------

tmErrorCode_t 
tmbslPhySMSC8710GetBasicModeControl (
    tmUnitSelect_t                   		ethUnitId,   
    ptmbslPhyBasicModeControl_t       pPhyBasicModeControl
    )
{
    tmErrorCode_t ethStatus=TM_OK;     		
    UInt16 bmcr;

    /* Read the present settings of the BMCR register */
    ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Bmcr,&bmcr);

    if(ethStatus != TM_OK)
    {
        return ethStatus;
    }

    pPhyBasicModeControl->enableCollisionTest = 
                (((bmcr & TMBSL_PHYSMSC8710_BMCR_COLTEST ) > 0) ? True : False);

    pPhyBasicModeControl->duplexMode = 
                (((bmcr & TMBSL_PHYSMSC8710_BMCR_FD_EN ) > 0) ? tmbslPhyFullDuplex : tmbslPhyHalfDuplex);

    if(bmcr & TMBSL_PHYSMSC8710_BMCR_SPEED_MSK)
    {
        pPhyBasicModeControl->speed = tmbslPhySpeed100Mbps;            
    }
    else
    {
        pPhyBasicModeControl->speed = tmbslPhySpeed10Mbps;                            
    }

    return TM_OK;

}   

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710SetBasicModeControl:
//
// DESCRIPTION: This function will configure the PHY device for the Basic Mode. 
//
// RETURN:      TM_OK
//
// NOTES:      
//-----------------------------------------------------------------------------

tmErrorCode_t 
tmbslPhySMSC8710SetBasicModeControl (
    tmUnitSelect_t                           ethUnitId,   
    ptmbslPhyBasicModeControl_t        pPhyBasicModeControl
    )
{

    tmErrorCode_t       		ethStatus=TM_OK;     		
    UInt16 bmcr =0;
    UInt16 regval;

    if(pPhyBasicModeControl->enableCollisionTest == True)
    {
        bmcr |= TMBSL_PHYSMSC8710_BMCR_COLTEST;
    }

    if(pPhyBasicModeControl->duplexMode == True)
    {
        bmcr |= TMBSL_PHYSMSC8710_BMCR_FD_EN;
    }

    switch(pPhyBasicModeControl->speed)
    {

        case tmbslPhySpeed100Mbps :       
//            bmcr |= TMBSL_PHYSMSC8710_BMCR_SPEED_100;                
        /* Set the mode register to default value */
        ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Smr,&regval);

        /* Clear the mode bits */
        regval &= 0xFF1F;

        regval |= 0x60;
        ethStatus = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Smr,regval);        

        ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Smr,&regval);
        printf("\nSMR val: %08x\n",regval);           
            break;

        case tmbslPhySpeed10Mbps :       
//            bmcr |= TMBSL_PHYSMSC8710_BMCR_SPEED_10;                
        ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Smr,&regval);

        /* Clear the mode bits */
        regval &= 0xFF1F;

        regval |= 0x20;

        ethStatus = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Smr,regval);        
        ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Smr,&regval);
        printf("\nSMR val: %08x\n",regval);           
            
                
            break;

        default:
            break;

    }

    ethStatus = tmbslPhySMSC8710SoftReset(ethUnitId);

    mdelay(10);
    printf("\nSPEED/MODE updated from MODE pins\n");               

    ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Pscsr,&regval);
    printf("\nPHY STS reg:%04x\n",regval);               

    if(ethStatus != TM_OK)
    {
        printf("\nRESET failed after mode change\n");           
    }

    /* Write the result to the BMC register */
//    ethStatus = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Bmcr,bmcr);

    return ethStatus;

}   


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710GetBasicModeStatus:
//
// DESCRIPTION: This function will get the Basic Mode Status of the PHY device 
//				such as the speed, duplex mode and other statuses
//
// RETURN:      TM_OK
//
// NOTES:      
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhySMSC8710GetBasicModeStatus (
    tmUnitSelect_t                  		ethUnitId,   
    ptmbslPhyBasicModeStatus_t    	pPhyBasicModeStatus     
    )

{
    tmErrorCode_t ethStatus=TM_OK;     		
    UInt16           bmsr,bmcr,phySts;

    /* Read the PHY status from the BMSR register */
    ethStatus = tmbslPhySMSC8710Read(ethUnitId, tmbslPhySMSC8710Bmsr, &bmsr);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    /* Read the PHY control register from the BMCR register */
    ethStatus = tmbslPhySMSC8710Read(ethUnitId, tmbslPhySMSC8710Bmcr, &bmcr);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    /* Read the autonegotiation status from PHY status register */
    ethStatus = tmbslPhySMSC8710Read(ethUnitId, tmbslPhySMSC8710Pscsr, &phySts);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    pPhyBasicModeStatus->jabberDetect = (((bmsr & TMBSL_PHYSMSC8710_BMSR_JAB_VAL) > 0) ? True : False);

    pPhyBasicModeStatus->remoteFaultDetected = (((bmsr & TMBSL_PHYSMSC8710_BMSR_RF_VAL) > 0) ? True : False);

    pPhyBasicModeStatus->autoNegotiationComplete = (((bmsr & TMBSL_PHYSMSC8710_BMSR_AN_VAL) > 0) ? True : False);

    pPhyBasicModeStatus->loopBackEnabled = (((bmcr & TMBSL_PHYSMSC8710_BMCR_LPBK_VAL) > 0) ? True : False);

    if((bmcr & TMBSL_PHYSMSC8710_BMCR_AN_EN) == 0) 
    {
        /* Not an auto negotiation. So read the values from BMCR */
        pPhyBasicModeStatus->duplexMode = 
                (((bmcr & TMBSL_PHYSMSC8710_BMCR_FD_EN ) > 0) ? tmbslPhyFullDuplex : tmbslPhyHalfDuplex);

        if(bmcr & TMBSL_PHYSMSC8710_BMCR_SPEED_MSK)
        {
            pPhyBasicModeStatus->speed = tmbslPhySpeed100Mbps;            
        }
        else
        {
            pPhyBasicModeStatus->speed = tmbslPhySpeed10Mbps;                        
        }

    }
    else 
    {
        /* If autonegotiation is enabled, read from PHYSTS register */
        pPhyBasicModeStatus->duplexMode = 
                (((phySts & TMBSL_PHYSMSC8710_PHYSTS_DUP_MODE) > 0) ? tmbslPhyFullDuplex : tmbslPhyHalfDuplex);

        pPhyBasicModeStatus->speed= 
                (((phySts & TMBSL_PHYSMSC8710_PHYSTS_SPEED_MSK) == 0x4) ? tmbslPhySpeed10Mbps : tmbslPhySpeed100Mbps);

    }

    return TM_OK;

}   


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710AutoNegotiate:
//
// DESCRIPTION: This function will enable the Auto negotiation of the PHY device 
//				with Link Partner. Best possible performance configuration is 
//				selected automatically during this process
//
// RETURN:      TM_OK
//
// NOTES:      
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhySMSC8710AutoNegotiate (
    tmUnitSelect_t                       	      ethUnitId,   
    ptmbslPhyAutoNegotitationMask_t      	pAutoNegotiationMask
    )
{

    tmErrorCode_t   ethStatus=TM_OK;
    UInt32             timeout=AUTO_NEG_DELAY_MULTIPLIER; 

    UInt16 regVal = ANAR_DEFAULT_VAL;
    UInt16 mask =0;

    ethStatus = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Anar,regVal);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    if(pAutoNegotiationMask->masknextPageDesired == True)
    {
        mask |= TMBSL_PHYSMSC8710_ANAR_NP;        
    }

    if(pAutoNegotiationMask->maskRemoteFault == True)
    {
        mask |= TMBSL_PHYSMSC8710_ANAR_ADV_RF;        
    }

    if(pAutoNegotiationMask->maskAsymmetricPause == True)
    {
        mask |= TMBSL_PHYSMSC8710_ANAR_AP;        
    }

    if(pAutoNegotiationMask->maskPauseFrame == True)
    {
        mask |= TMBSL_PHYSMSC8710_ANAR_PAUSE;        
    }

    if(pAutoNegotiationMask->mask100BaseTxFullDuplexSupport == True)
    {
        mask |= TMBSL_PHYSMSC8710_ANAR_100B_TX_FD;        
    }

    if(pAutoNegotiationMask->mask100BaseTxSupport == True)
    {
        mask |= TMBSL_PHYSMSC8710_ANAR_100B_TX_HD;        
    }

    if(pAutoNegotiationMask->mask10BaseTFullDuplexSupport == True)
    {
        mask |= TMBSL_PHYSMSC8710_ANAR_10B_TX_FD;        
    }    

    if(pAutoNegotiationMask->mask10BaseTSupport == True)
    {
        mask |= TMBSL_PHYSMSC8710_ANAR_10B_TX_HD;        
    }    

    /* Clear the corresponding bits in the regVal*/
    regVal &= ~(mask);

//    printf("\nANAR val: %08x\n",regVal);
   
    /* Program the Auto negotiation mask */
    tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Anar,regVal);

    /* Enable the Auto negotiation in the BMCR register 
    ** First clear the auto negotiate bit and then enable
    */
    tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Bmcr,&regVal);

    /* Clear autonegotiation bit */
    regVal &= TMBSL_PHYSMSC8710_BMCR_AN_CLR;

    regVal |= TMBSL_PHYSMSC8710_BMCR_AN_EN |
              TMBSL_PHYSMSC8710_BMCR_AN_RESTART;
    
//    printf("\nBMCR val: %08x\n",regVal);
   
    tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Bmcr,regVal);

    /* Wait till the Auto negotiation is complete */
    tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Bmsr,&regVal);

    /* While the Autonegotiation is not complete, stay in loop */
    while(((regVal & TMBSL_PHYSMSC8710_BMSR_AN_VAL) != TMBSL_PHYSMSC8710_BMSR_AN_VAL) && 
           (timeout != 0 ) )
    {
        mdelay(5);
        
        ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Bmsr,&regVal);

        if(ethStatus != TM_OK) 
        {
            return ethStatus;
        }

        timeout--;
    }

    if(  (timeout <= 0) && 
         ( (regVal & TMBSL_PHYSMSC8710_BMSR_AN_VAL) != TMBSL_PHYSMSC8710_BMSR_AN_VAL)
      )
    {
        return(-1);
    }

    GMAC_DBG("Autonegotiation Time:%d msec\n",(AUTO_NEG_DELAY_MULTIPLIER-timeout)*5);
    
    return TM_OK;

}   

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710LoopBack:
//
// DESCRIPTION: Function will enable or disable the PHY device in the Loopback 
//				mode.
//
// RETURN:      TM_OK
// NOTES:      
//-----------------------------------------------------------------------------


tmErrorCode_t
tmbslPhySMSC8710LoopBack (
    tmUnitSelect_t                  ethUnitId,   
    tmbslPhyEnableDisable_t     loopbackMode
    )
{

    tmErrorCode_t ethStatus=TM_OK;     		
    UInt16 bmcr;

    /* Read the existing settings of the BMCR register */
    ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Bmcr,&bmcr);

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    if(tmbslPhyEnable == loopbackMode)
    {
        bmcr |= TMBSL_PHYSMSC8710_BMCR_LPBK_VAL;
    }
    else if(tmbslPhyDisable == loopbackMode)
    {
        bmcr &= TMBSL_PHYSMSC8710_BMCR_LPBK_CLR;
    }
    else
    {
        ethStatus = TMBSL_ERR_PHY_NOT_SUPPORTED;
    }

    /* Write the Loopback setting to the BMCR register */
    ethStatus = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Bmcr,bmcr);

    return ethStatus;

}   


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710SoftReset:
//
// DESCRIPTION: Function will do the soft reset of the PHY device

// RETURN:      TM_OK
//
// NOTES:      
//-----------------------------------------------------------------------------
//

tmErrorCode_t
tmbslPhySMSC8710SoftReset (
    tmUnitSelect_t                     		ethUnitId      
	)

{
    tmErrorCode_t       		ethStatus=TM_OK;     		

    /* All the registers will be reset */
    ethStatus = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Bmcr,TMBSL_PHYSMSC8710_BMCR_RST_VAL);

    return ethStatus;

}   

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710GetLinkStatus:
//
// DESCRIPTION: Function will get the link status

// RETURN:      TM_OK
//
// NOTES:      
//-----------------------------------------------------------------------------

tmErrorCode_t
tmbslPhySMSC8710GetLinkStatus (
    tmUnitSelect_t                          ethUnitId,   
    ptmbslPhyEnableDisable_t            pLinkStatus
    )
{
    tmErrorCode_t ethStatus=TM_OK;     		
    UInt16 physts;

    /* Read the BMSR register twice, as per datasheet */
    ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Bmsr,&physts);
    ethStatus = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Bmsr,&physts);    

    if(ethStatus != TM_OK)
    {
        return ethStatus ;
    }

    *pLinkStatus =
        (((physts & TMBSL_PHYSMSC8710_PHYSTS_LINK_STAT) > 0) ? tmbslPhyEnable : tmbslPhyDisable);

    return(TM_OK);

}   

//-----------------------------------------------------------------------------
//	Local static functions	
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710Read:
//
// DESCRIPTION: Function will read from the Specified PHY register

// RETURN:      
//
// NOTES:      
//-----------------------------------------------------------------------------
//

static tmErrorCode_t
tmbslPhySMSC8710Read (
    tmUnitSelect_t				ethUnitId,
    tmbslPhySMSC8710Reg_t		reg,   
    pUInt16						pVal
	)

{

    UInt32  timeout=0;
    UInt32 pEthRegs;       
    volatile UInt32 *pAdrReg;
    volatile  UInt32 *pDataReg;
    UInt32 regValue;        

    /* get Ethernet Module Reg Pointer  */
    pEthRegs = gEthContext[ethUnitId].pRegs;

    /* Get the Address register */  
    pAdrReg = (UInt32*)(pEthRegs+TMHW_LIPP6100ETH_ADR_REG_OFFSET) ;
    pDataReg = (UInt32*)(pEthRegs+TMHW_LIPP6100ETH_DATA_REG_OFFSET) ;

    /* Program the Register address in the MII */
    regValue = *pAdrReg;
       
    /* Clear the earlier register value in Address register and write the new value */
    regValue &= TMHW_LIPP6100ETH_ADR_PHY_REG_CLR;  
    regValue |= reg << TMHW_LIPP6100ETH_ADR_PHY_REG_POS;

    /* Clear the bit GMII write for read operation */
    regValue &= TMHW_LIPP6100ETH_ADR_PHY_RD_CLR;

    regValue |= TMHW_LIPP6100ETH_ADR_PHY_EN_VAL;

    *pAdrReg = regValue;        

    /* Wait till the read operation is complete */

    do
    {
        timeout++;

    }while ( ((*pAdrReg & TMHW_LIPP6100ETH_ADR_PHY_EN_VAL) == tmbslPhyEnable) && 
                   (timeout < PHY_TIMEOUT) );

    if( (timeout == PHY_TIMEOUT) && 
        ((*pAdrReg & TMHW_LIPP6100ETH_ADR_PHY_EN_VAL) == tmbslPhyEnable) )
    {
        printf("\n&*&*&*PHY READ FAILED\n");    
        return(TMBSL_ERR_PHY_READ_FAILED);
    }  
	
    /* Read the data from the data register */

    *pVal = (UInt16) *pDataReg;
    
    return TM_OK;

}   


//-----------------------------------------------------------------------------
// FUNCTION:    tmbslPhySMSC8710Write:
//
// DESCRIPTION: Function will Write to the Specified PHY register

// RETURN:      
//
// NOTES:      
//-----------------------------------------------------------------------------
//

static tmErrorCode_t
tmbslPhySMSC8710Write (
    tmUnitSelect_t				ethUnitId,
    tmbslPhySMSC8710Reg_t		reg,   
    UInt16						val
	)

{

    UInt32 pEthRegs;       
    volatile UInt32 *pAdrReg;
    volatile  UInt32 *pDataReg;
    UInt32 regValue;

    UInt32      timeout=0;

    /* Get the Ethernet Module Reg pointer */
    pEthRegs = gEthContext[ethUnitId].pRegs;

    /* Get the Address register */  
    pAdrReg = (UInt32*)(pEthRegs+TMHW_LIPP6100ETH_ADR_REG_OFFSET) ;
    pDataReg = (UInt32*)(pEthRegs+TMHW_LIPP6100ETH_DATA_REG_OFFSET) ;

    /* Write the data into data register  */
    *pDataReg = 0;
    *pDataReg = val;

    /* Program the Register address */
    regValue = *pAdrReg;
       
    /* Clear the earlier register value in Address register */
    regValue &= TMHW_LIPP6100ETH_ADR_PHY_REG_CLR;  
    regValue |= reg << TMHW_LIPP6100ETH_ADR_PHY_REG_POS;

    /* Set Write operation bit and enable write */
    regValue |= TMHW_LIPP6100ETH_ADR_PHY_WR_VAL | 
               TMHW_LIPP6100ETH_ADR_PHY_EN_VAL ;

    /* Write the value back to Address register */
    *pAdrReg = regValue;        

    /* wait till the write operation is complete */

    do
    {
        timeout++;

    }while ( ((*pAdrReg & TMHW_LIPP6100ETH_ADR_PHY_EN_VAL) == tmbslPhyEnable) && 
             (timeout < PHY_TIMEOUT) );
	
	if( (timeout == PHY_TIMEOUT) && 
           ((*pAdrReg & TMHW_LIPP6100ETH_ADR_PHY_EN_VAL) == tmbslPhyEnable) )
	{
        printf("\n&*&*&*PHY WRITE FAILED\n");
	    return(TMBSL_ERR_PHY_WRITE_FAILED);
	}  

	return TM_OK;
    
}   

//-----------------------------------------------------------------------------

#if 0
static tmErrorCode_t
set_regs (void );

static tmErrorCode_t
    set_regs (void )
{

    tmErrorCode_t err = TM_OK;
    UInt16 regVal=0;
    tmUnitSelect_t ethUnitId = 0;

    regVal =0x0;
    err = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Mcsr,regVal);
    err = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Mcsr,&regVal);
    printf("\nMCSR val: %08x\n",regVal);           

    regVal = 0xF6;
    err = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Smr,regVal);        
    err = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Smr,&regVal);
    printf("\nSMR val: %08x\n",regVal);           

    regVal =0;
    err = tmbslPhySMSC8710Write(ethUnitId,tmbslPhySMSC8710Csir,regVal);        
    err = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Csir,&regVal);
    printf("\nCSIR val: %08x\n",regVal);           

    err = tmbslPhySMSC8710Read(ethUnitId,tmbslPhySMSC8710Pscsr,&regVal);
    printf("\nPSCSR val: %08x\n",regVal);  

    return err;
}           
#endif

#ifdef __LIPP_6300ETH__
static void get_phy_out_of_rst( void)
{
    //Msg for High, to bring PHY out of reset
//    unsigned char msg[]={0x00,0x04,0x00,0x02,0x07,0x01}; 

    //Msg for Low, to put PHY in reset     
//    unsigned char msg[]={0x00,0x04,0x00,0x02,0x07,0x00}; 

//    pnx8xxx_ip3203_init();
//    pnx8xxx_ip3203_write(msg, sizeof(msg));

    /*Wait till the PHY is out of reset */    
    mdelay(1);

    /* Turn the clocks to MAC from PHY */

#if 0
    /* PHY clocks */
    writel(0x00000003, (unsigned long *)(0xbbe47710));	
    writel(0x00000003, (unsigned long *)(0xbbe47714));	
    writel(0x00000003, (unsigned long *)(0xbbe47718));	
#endif 

    #ifdef __LIPP_6300ETH__
    /* PHY clocks */
    *(unsigned long *)(MMIO_BASE) = 0x00000003;
    *(unsigned long *)(MMIO_BASE) = 0x00000003;
    *(unsigned long *)(MMIO_BASE) = 0x00000003;
    #endif


    mdelay(1);
//    GMAC_DBG("\nETHERNET CLOCK1:%08x\n",*(unsigned int *)(0x1be47710));                    
//    GMAC_DBG("\nETHERNET CLOCK2:%08x\n",*(unsigned int *)(0x1be47714));                    
//    GMAC_DBG("\nETHERNET CLOCK3:%08x\n",*(unsigned int*)(0x1be47718));                       
//    GMAC_DBG("\nSMSC PHY INIT successful\n\n\n");    

    return;

}

#endif 



