/*
 * (C) Copyright 2006
 * NXP Semiconductors,
 * Robert Delien robert.delien@nxp.com, Hans Zuidam hans.zuidam@nxp.com
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 *
 */

#include <common.h>
#include <pci.h>
#include <nxp_pcixio_ipa051.h>
#include <asm/io.h>
#include <asm/addrspace.h>

#undef SHOW_INFO
#ifdef SHOW_INFO
    #define info(format, arg...) printf("INFO: " format "\n", ## arg)
#else
    #define info(format, arg...) do {} while(0)
#endif /* SHOW_INFO */

#ifdef	__TCS__
	#define	sync()
#else	/* ! __TCS__ */
	#define sync() __asm__ __volatile__("sync\n");
#endif	/* __TCS__ */

#define DEFAULT_ADDRESS_PHASES	4

#define inl(x)    readl(x)
#define outl(x,y) writel(x,y)

extern void set_timeout(unsigned long usec);
extern int  did_timeout(void);
extern void flush_dcache_range(ulong start_addr, ulong stop);
extern void invalidate_dcache_range(ulong start_addr, ulong stop);

static unsigned long nand_get_addr_phases(struct nand_chip *nand);

#if defined(CONFIG_NXP_PCIXIO_IPA051)
#if defined(CONFIG_PCI)
/******************************************************************************/
/*** PCI								    ***/
/******************************************************************************/
static inline void nxp_ipa051_pci_clear_status( void )
{
	unsigned long pci_stat;

	pci_stat = inl(IPA051 + IPA051_GPPM_INT_STATUS);
	outl(pci_stat, IPA051 + IPA051_GPPM_INT_CLR);
}

static inline unsigned int nxp_ipa051_pci_calc_cfg_addr( pci_dev_t	dev,
							 int		where )
{
	unsigned int	addr;
	u32		bus;
	u32		device;

	bus    = ((dev & 0x00FF0000) >> 16);
	device = ((dev & 0x0000F800) >> 11);

	addr  = ((bus > 0) ? ((bus << PCI_CFG_BUS_SHIFT) | 1) : 0);
	addr |= (((dev >> 8) & 0xFF) << PCI_CFG_FUNC_SHIFT) | (where & 0xFC);

	return addr;
}

static int nxp_ipa051_pci_io ( unsigned int	ioaddr,
			       unsigned int	pci_mode,
			       unsigned int	pci_cmd,
			       unsigned int	*val )
{
	unsigned long retries = 0;

	/* Clear pending interrupt status */
	if (readl(IPA051 + IPA051_GPPM_INT_STATUS)) {
		nxp_ipa051_pci_clear_status();
		while (!(inl(IPA051 + IPA051_GPPM_INT_STATUS) == 0)) ;
	}

	//outl(ioaddr, IPA051 + IPA051_GPPM_ADDR);
	writel(ioaddr, IPA051 + IPA051_GPPM_ADDR);

	if ((pci_cmd == IPA051_GPPM_CTRL__GPPM_CMD_IO_WR) ||
	    (pci_cmd == IPA051_GPPM_CTRL__GPPM_CMD_CFG_WR))
		outl(*val, IPA051 + IPA051_GPPM_WDAT);

	outl(IPA051_GPPM_CTRL__INIT_PCI_CYCLE | pci_cmd | (pci_mode & PCI_BYTE_ENABLE_MASK),
	     IPA051 + IPA051_GPPM_CTRL);

	while (1) {
		if (inl(IPA051 + IPA051_GPPM_INT_STATUS) & IPA051_GPPM_INT__GPPM_DONE) {
			if ((pci_cmd == IPA051_GPPM_CTRL__GPPM_CMD_IO_RD) ||
			    (pci_cmd == IPA051_GPPM_CTRL__GPPM_CMD_CFG_RD))
				*val = inl(IPA051 + IPA051_GPPM_RDAT);
			nxp_ipa051_pci_clear_status();
			return 0;
		} else if (inl(IPA051 + IPA051_GPPM_INT_STATUS) & IPA051_GPPM_INT__GPPM_R_MABORT) {
			break;
		}

		if (retries > (PCI_IO_TIMEOUT_US/PCI_IO_RETRYTIME_US)) {
			printf("\n%s : Arbiter Locked.\n", __FUNCTION__);
			return -1;
		}
		retries ++ ;
		udelay(PCI_IO_RETRYTIME_US);
	}

	nxp_ipa051_pci_clear_status();
	if ((pci_cmd == IPA051_GPPM_CTRL__GPPM_CMD_IO_RD) ||
	    (pci_cmd == IPA051_GPPM_CTRL__GPPM_CMD_IO_WR)) {
		printf("\n%s timeout (GPPM_CTRL=%X) ioaddr %X pci_cmd %X\n",
		       __FUNCTION__, inl(IPA051 + IPA051_GPPM_CTRL), ioaddr,
		       pci_cmd);
	}

	if ((pci_cmd == IPA051_GPPM_CTRL__GPPM_CMD_IO_RD) ||
	    (pci_cmd == IPA051_GPPM_CTRL__GPPM_CMD_CFG_RD))
		*val = 0xFFFFFFFF;
	return 0;
}

/*
 * We can't address 8 and 16 bit words directly.  Instead we have to
 * read/write a 32bit word and mask/modify the data we actually want.
 */
static int nxp_ipa051_pcibios_read_config_byte( struct pci_controller	*hose,
						pci_dev_t		dev,
						int			where,
						u8			*val )
{
	unsigned int	data = 0;
	int		err;

	debug("dev = %p\n", dev);

	if (dev == 0)
		return -1;

	err = nxp_ipa051_pci_io( nxp_ipa051_pci_calc_cfg_addr(dev, where),
				 ~(1 << (where & 3)),
				 IPA051_GPPM_CTRL__GPPM_CMD_CFG_RD,
				 &data );

	switch (where & 0x03) {
	case 0:
		*val = (unsigned char)(data & 0x000000FF);
		break;
	case 1:
		*val = (unsigned char)((data & 0x0000FF00) >> 8);
		break;
	case 2:
		*val = (unsigned char)((data & 0x00FF0000) >> 16);
		break;
	case 3:
		*val = (unsigned char)((data & 0xFF000000) >> 24);
		break;
	}

	return err;
}

static int nxp_ipa051_pcibios_read_config_word( struct pci_controller	*hose,
						pci_dev_t		dev,
						int			where,
						u16			*val )
{
	unsigned int	data = 0;
	int		err;

	if (dev == 0)
		return -1;

	if (where & 0x01)
		return -1;

	err = nxp_ipa051_pci_io( nxp_ipa051_pci_calc_cfg_addr(dev, where),
				 ~(3 << (where & 3)),
				 IPA051_GPPM_CTRL__GPPM_CMD_CFG_RD,
				 &data );

	switch (where & 0x02) {
	case 0:
		*val = (unsigned short)(data & 0x0000FFFF);
		break;
	case 2:
		*val = (unsigned short)((data & 0xFFFF0000) >> 16);
		break;
	}

	return err;
}

static int nxp_ipa051_pcibios_read_config_dword( struct pci_controller	*hose,
						 pci_dev_t		dev,
						 int			where,
						 u32			*val )
{
	int err;

	if (dev == 0)
		return -1;

	if (where & 0x03)
		return -1;

	err = nxp_ipa051_pci_io( nxp_ipa051_pci_calc_cfg_addr(dev, where),
				 0,
				 IPA051_GPPM_CTRL__GPPM_CMD_CFG_RD,
				 val );

	return err;
}

static int nxp_ipa051_pcibios_write_config_byte( struct pci_controller	*hose,
						 pci_dev_t		dev,
						 int			where,
						 u8			val )
{
	unsigned int	data = (unsigned int)val;
	int		err;

	if (dev == 0)
		return -1;

	switch (where & 0x03) {
	case 1:
		data = (data << 8);
		break;
	case 2:
		data = (data << 16);
		break;
	case 3:
		data = (data << 24);
		break;
	default:
		break;
	}

	err = nxp_ipa051_pci_io( nxp_ipa051_pci_calc_cfg_addr(dev, where),
				 ~(1 << (where & 3)),
				 IPA051_GPPM_CTRL__GPPM_CMD_CFG_WR,
				 &data );

	return err;
}

static int nxp_ipa051_pcibios_write_config_word( struct pci_controller	*hose,
						 pci_dev_t		dev,
						 int			where,
						 u16			val )
{
	unsigned int	data = (unsigned int)val;
	int		err;

	if (dev == 0)
		return -1;

	if (where & 0x01)
		return -1;

	switch (where & 0x02) {
	case 2:
		data = (data << 16);
		break;
	default:
		break;
	}

	err = nxp_ipa051_pci_io( nxp_ipa051_pci_calc_cfg_addr(dev, where),
				 ~(3 << (where & 3)),
				 IPA051_GPPM_CTRL__GPPM_CMD_CFG_WR,
				 &data );

	return err;
}

static int nxp_ipa051_pcibios_write_config_dword( struct pci_controller	*hose,
						  pci_dev_t		dev,
						  int			where,
						  u32			val )
{
	int err;

	if (dev == 0)
		return -1;

	if (where & 0x03)
		return -1;

	err = nxp_ipa051_pci_io( nxp_ipa051_pci_calc_cfg_addr(dev, where),
				 0,
				 IPA051_GPPM_CTRL__GPPM_CMD_CFG_WR, &val );

	return err;
}


/*
 *	Initialize Module
 */
void init_nxp_ipa051_pci( struct pci_controller	*hose )
{
	DECLARE_GLOBAL_DATA_PTR;

        debug("E: %s\n", __FUNCTION__);

	/* this will configure the nxp a051 pci (xio) bus bridge itself
	 * when it's addressed as a pci slave
	 * set the total pci mem aperture,
	 * set the total pci io aperture and 
	 * set the total pci sdam aperture */
	
	hose->first_busno = 0x00;
	hose->last_busno  = 0xFF;

	/* PCI memory space #1 */
	pci_set_region( hose->regions + 0,
			PCIMEM_BASE,
			KSEG1ADDR(PCIMEM_BASE),
			PCIMEM_SIZE,
			PCI_REGION_MEM );

        debug("1: %s\n", __FUNCTION__);

	/* PCI I/O space #2 */
	pci_set_region( hose->regions + 1,
			PCIIO_BASE,
			KSEG1ADDR(PCIIO_BASE),
			PCIIO_SIZE,
			PCI_REGION_IO );

        debug("2: %s\n", __FUNCTION__);

	/* set the sdram aperture in pci space at the same location 
	 * as where is resides in mips virtual addressing space 
	 * while we're at it, map the whole shabang so that there is
	 * a literal mapping of the whole KSEG0 segment in pci addressing
	 * space */
	pci_set_region( hose->regions + 2,
			SDRAM_BASE,
			KSEG1ADDR(SDRAM_BASE),
			gd->ram_size,
			PCI_REGION_MEM | PCI_REGION_MEMORY );

        debug("3: %s\n", __FUNCTION__);

	hose->region_count = 3;

	hose->read_byte   = nxp_ipa051_pcibios_read_config_byte;
	hose->read_word   = nxp_ipa051_pcibios_read_config_word;
	hose->read_dword  = nxp_ipa051_pcibios_read_config_dword;
	hose->write_byte  = nxp_ipa051_pcibios_write_config_byte;
	hose->write_word  = nxp_ipa051_pcibios_write_config_word;
	hose->write_dword = nxp_ipa051_pcibios_write_config_dword;

	pci_register_hose(hose);

        debug("4: %s\n", __FUNCTION__);

	hose->last_busno = pci_hose_scan(hose);

        debug("X: %s\n", __FUNCTION__);
}
#endif /* CONFIG_PCI */


#if (defined(CONFIG_NAND_LEGACY) || defined(CONFIG_NAND)) && defined(CONFIG_CMD_NAND) \
	&& defined(CONFIG_NAND_NXP_IPA051)
/******************************************************************************/
/*** NAND flash								    ***/
/******************************************************************************/
#if defined(CONFIG_NAND)
#include <linux/mtd/nand.h>
#endif
#if defined(CONFIG_NAND_LEGACY)
#include <linux/mtd/nand_legacy.h>
#endif

#define ROUND_DOWN(value,boundary)	((value) & (~((boundary)-1)))
#define ROUND_UP(value,boundary)	(ROUND_DOWN((value)+(boundary)-1, (boundary)))


#define GPXIO_TIMEOUT_US	1000000		/* 1000000 us == 1 second */
#define DMA_TIMEOUT_US		1000000		/* 1000000 us == 1 second */

/* Required for 64 MiB XIO limit bug work-around */
#define XIO_LIMIT		0x08000000

/* Moved to nxp_pcixio_ipa051.h */
/* Required for alignment buffer */
#define MAX_PAGE_SIZE		2112

//static unsigned char	aligned_buf[(MAX_PAGE_SIZE + CFG_CACHELINE_SIZE)&~CFG_CACHELINE_SIZE] __attribute__ ((aligned(CFG_CACHELINE_SIZE))) ;
#ifdef	__TCS__
#define	ALIGNED_BUF		ROUND_UP(MAX_PAGE_SIZE, CONFIG_SYS_CACHELINE_SIZE)
#define	ALIGNED_BUF_NONAL	(ALIGNED_BUF + (CONFIG_SYS_CACHELINE_SIZE - 1))
static unsigned char*	aligned_buf_nonal;
static unsigned char*	aligned_buf;
#else	/* __TCS__ */
#define	ALIGNED_BUF	((MAX_PAGE_SIZE + (CONFIG_SYS_CACHELINE_SIZE - 1)) & ~(CONFIG_SYS_CACHELINE_SIZE - 1))
static unsigned char	aligned_buf[ALIGNED_BUF] __attribute__ ((aligned(CONFIG_SYS_CACHELINE_SIZE))) ;
#endif	/* __TCS__ */

#define EMPTY_BYTE		0xFF

#define GPXIO_NODATA		0
#define GPXIO_READ		1
#define GPXIO_WRITE		2

#define GPXIO_NO_ACK		0
#define GPXIO_ACK		1

#define DMA_READ		0
#define DMA_WRITE		1

static int do_dma( unsigned long	xio_addr,
		   unsigned long	ram_addr,
		   unsigned long	length,
		   unsigned char	transfer_type )
{
	int	result = 0 ;

	mydebug("xio_addr=%lx, ram_addr=%lx, length=%lx, transfer_type=%x\n",
		xio_addr, ram_addr, length, transfer_type);
	
	/* Clear the DMA status bit */
	writel( IPA051_DMA_INT__DMA_DONE,
		IPA051_DMA_INT_CLR + IPA051 );
	
	/* Wait for a pending DMA transfer to be finished */
	set_timeout(DMA_TIMEOUT_US) ;
	while(readl(IPA051_DMA_INT_STATUS + IPA051) & IPA051_DMA_INT__DMA_DONE) {
		if (did_timeout()) {
			printf("\nError: DMA engine still engaged in previous transfer\n");
			return (-1) ;
		}
	}

	/* Flush the caches */
	if (transfer_type == DMA_WRITE) {	
		flush_dcache_range(ram_addr, ram_addr + length) ;
	}
	
	/* Set DMA paramters */
	writel( xio_addr,
		IPA051_DMA_EADDR + IPA051);
	writel( CPHYSADDR(ram_addr),
		IPA051_DMA_IADDR + IPA051);
	writel( length >> 2,
		IPA051_DMA_LENGTH + IPA051);

#ifdef XIO_LIMIT
	/* 64 MiB XIO limit bug work-around */
	if (xio_addr >= (XIO_BASE + XIO_LIMIT))
		writel(XIO_BASE + XIO_LIMIT, IPA051 + IPA051_BASE18);
#endif /* XIO_LIMIT */

	/* Start the DMA cycle */
	writel( ((transfer_type == DMA_READ )?IPA051_DMA_CTRL__CMD_READ :0) |
		((transfer_type == DMA_WRITE)?IPA051_DMA_CTRL__CMD_WRITE:0) |
		IPA051_DMA_CTRL__SND2XIO |
		IPA051_DMA_CTRL__MAX_BURST_SIZE_NORES |
		IPA051_DMA_CTRL__INIT_DMA ,
		IPA051_DMA_CTRL + IPA051 );

	/* Wait for the DMA engine to start the cycle */
	set_timeout(DMA_TIMEOUT_US) ;
	while(readl(IPA051 + IPA051_DMA_CTRL) & IPA051_DMA_CTRL__INIT_DMA) {
		if (did_timeout()) {
			printf("\nError: DMA engine timeout while initiating read cycle\n");

			result = -1 ;
			goto cleanup ;
		}
	}

	/* Wait for the DMA engine to finish the cycle */
	set_timeout(DMA_TIMEOUT_US) ;
	while((readl(IPA051 + IPA051_DMA_INT_STATUS) & IPA051_DMA_INT__DMA_DONE) != IPA051_DMA_INT__DMA_DONE) {
		if (did_timeout()) {
			if (transfer_type == DMA_READ) {
				printf( "Error: DMA Timeout while trying to read %lu bytes from address 0x%.8lX to address 0x%.8lX\n",
					length,
					xio_addr,
					ram_addr );
			} else {
				printf( "Error: DMA Timeout while trying to write %lu bytes from address 0x%.8lX to address 0x%.8lX\n",
					length,
					ram_addr,
					xio_addr );
			}

			result = -1 ;
			goto cleanup ;
		}
	}

	/* Invalidate the caches (instructions too; we may have loaded code) */
	if (transfer_type == DMA_READ) {
		invalidate_dcache_range(ram_addr, ram_addr + length) ;
	}

cleanup:
#ifdef XIO_LIMIT
	/* 64 MiB XIO limit bug work-around */
	if (xio_addr >= (XIO_BASE + XIO_LIMIT))
		writel(XIO_BASE, IPA051 + IPA051_BASE18);
#endif /* XIO_LIMIT */

 	return (result) ;
}


static int do_gpxio( unsigned long	xio_addr,
		     unsigned char	transfer_type,
		     unsigned char	wait_for_ack)
{
	int		result = 0 ;
	unsigned long	status ;
	unsigned long	int_mask = IPA051_GPXIO_INT__GPXIO_DONE ;

	//mydebug("....xio_addr=%lx\n", xio_addr);
	if (wait_for_ack == GPXIO_ACK)
		int_mask |= IPA051_GPXIO_INT__GPXIO_XIO_ACK_DONE ;

	/* Clear pending interrupts */
	writel( int_mask,
		IPA051_GPXIO_INT_CLR + IPA051 );

	/* Set the XIO address */
	writel( xio_addr,
		IPA051_GPXIO_ADDR + IPA051);

#ifdef XIO_LIMIT
	/* 64 MiB XIO limit bug work-around */
	if (xio_addr >= (XIO_BASE + XIO_LIMIT))
		writel( XIO_BASE + XIO_LIMIT,
			IPA051_BASE18 + IPA051);
#endif /* XIO_LIMIT */
	sync();

	/* Start the GPXIO cycle */
	writel(	((transfer_type != GPXIO_WRITE)?IPA051_GPXIO_CTRL__GPXIO_RD:0) |
		IPA051_GPXIO_CTRL__CLR_GPXIO_DONE |
		IPA051_GPXIO_CTRL__GPXIO_INIT,
		IPA051_GPXIO_CTRL + IPA051 );
	sync();

	/* Wait for the GPXIO engine to finish the cycle */
	set_timeout(GPXIO_TIMEOUT_US);
	while((status = readl(IPA051 + IPA051_GPXIO_INT_STATUS) & int_mask) != int_mask) {
		if (did_timeout()) {
			printf("\nError: GPXIO Timeout while ");
			switch (transfer_type) {
			case GPXIO_READ:
				printf( "reading data (0x%.8lX) from address 0x%.8lX",
					(long unsigned int)readl(IPA051 + IPA051_GPXIO_RDATA), (long unsigned int)xio_addr );
				break;
			case GPXIO_WRITE:
				printf( "writing data (0x%.8lX) to address 0x%.8lX",
					(long unsigned int)readl(IPA051 + IPA051_GPXIO_WDATA), (long unsigned int)xio_addr );
				break;
			default:
				printf( "sending command(s) (0x%.2X & 0x%.2X)",
					(unsigned int)readl(IPA051_NAND_CTRLS + IPA051) & 0x000000FF,
					(unsigned int)(readl(IPA051_NAND_CTRLS + IPA051) & 0x0000FF00) >> 8);
				break;
			}
			if (status & IPA051_GPXIO_INT__GPXIO_DONE)
				printf(" (GPXIO engine locked)");
			if (status & IPA051_GPXIO_INT__GPXIO_XIO_ACK_DONE)
				printf(" (No ack received)");
			printf("\n");

			result = -1 ;
			goto cleanup ;
		}
	}

	/* check if error bit is set.. warn if so! */
	if (status & IPA051_GPXIO_INT__GPXIO_ERR) {
		printf("\nError: Non-supported GPXIO command attempt\n");
		result = -1;
	}
cleanup:
#ifdef XIO_LIMIT
	/* 64 MiB XIO limit bug work-around */
	if (xio_addr >= (XIO_BASE + XIO_LIMIT))
		writel(XIO_BASE, IPA051 + IPA051_BASE18);
#endif /* XIO_LIMIT */

	return (result) ;
}


void nand_reset (struct nand_chip* nand)
{
	/* Configure NAND controls */
	writel( IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_RESET) |
		IPA051_NAND_CTRLS__CTRLS_CMD_PH(1),
		IPA051_NAND_CTRLS + IPA051);

	do_gpxio( (unsigned long)(nand->IO_ADDR_W),
			  GPXIO_NODATA,
			  GPXIO_ACK );
}

void nand_stop_cache_read(struct mtd_info* mtd)
{
	struct nand_chip *nand = (struct nand_chip *)mtd->priv;
	
	/* Configure NAND controls */
	writel( IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_EXIT_CACHEREAD) |
			IPA051_NAND_CTRLS__CTRLS_CMD_PH(1),
			IPA051_NAND_CTRLS + IPA051);

	do_gpxio( (unsigned long)(nand->IO_ADDR_W), GPXIO_NODATA, GPXIO_NO_ACK );	
}

int nand_read_id (struct nand_chip* nand, unsigned long *id)
{
	int result ;

	/* Configure NAND controls */
	writel( IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_READID) |
		IPA051_NAND_CTRLS__CTRLS_CMD_PH(1) |
		IPA051_NAND_CTRLS__CTRLS_ADR_PH(1) |
		IPA051_NAND_CTRLS__CTRLS_INC_DATA,
		IPA051_NAND_CTRLS + IPA051);

	result = do_gpxio((unsigned long)(nand->IO_ADDR_W),
			   GPXIO_READ,
			   GPXIO_NO_ACK );

	*id = readl(IPA051 + IPA051_GPXIO_RDATA);
	return (result) ;
}

int nand_read_status (struct mtd_info* mtd, unsigned long *status)
{
	int result ;

	/* Configure NAND controls */
	writel( IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_STATUS) |
		IPA051_NAND_CTRLS__CTRLS_CMD_PH(1) |
		IPA051_NAND_CTRLS__CTRLS_INC_DATA,
		IPA051_NAND_CTRLS + IPA051);

	result = do_gpxio( (unsigned long)(((struct nand_chip *)(mtd->priv))->IO_ADDR_W),
			   GPXIO_READ,
			   GPXIO_NO_ACK );

	*status = readl(IPA051 + IPA051_GPXIO_RDATA);
	return (result) ;
}

int nand_erase_block (struct mtd_info* mtd, size_t offset)
{
	unsigned long	nandOffset ;
	struct nand_chip *nand = mtd->priv;

	/* Do not allow reads past end of device */
	if ((offset + mtd->writesize) > mtd->size) {
		printf ("\n%s: Attempt to erase beyond end of device 0x%X 0x%X 0x%X\n",
			__FUNCTION__,
			offset,
			mtd->writesize,
			mtd->size);
		return (-1) ;
	}

	/* Do not allow non block-aligned addresses */
	if (offset & (mtd->erasesize - 1)) {
		printf ("\n%s: Attempt to erase non block-aligned address 0x%X 0x%X\n",
			__FUNCTION__,
			(uint) offset,
			(uint) mtd->erasesize);
		return (-1);
	}

	/* Check device geometry */
	if (nand->page_shift < 11) {
		/* We're dealing with a Small or Tiny Page device */
		unsigned long nandCtrls ;

		/* Configure NAND controls */
		nandCtrls = IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_ERASE1) |	/* First command is 'Erase' */
			    IPA051_NAND_CTRLS__COMMAND_B(NAND_CMD_ERASE2) |	/* First command is 'Erase confirm' */
			    IPA051_NAND_CTRLS__CTRLS_CMD_PH(2) ;		/* Two command cycles */


#if 0
		if (nand->nand->chip_shift < 26)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_ADR_PH(nand->pageadrlen-1) ;
		else
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_ADR_PH(nand->pageadrlen-2) ;
#endif
		nandCtrls |= nand_get_addr_phases(nand);

		if (nand->chip_shift == 26)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_64MB ;
		if (nand->chip_shift == 27)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_128MB ;

		writel( nandCtrls,
			IPA051_NAND_CTRLS + IPA051 );

		nandOffset = offset ;
	} else {
		/* We're dealing with a Large Page device */
		unsigned long row ;
		unsigned long col ;

		/* Configure NAND controls */
		writel( IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_ERASE1) |	/* First command is 'Erase' */
			IPA051_NAND_CTRLS__COMMAND_B(NAND_CMD_ERASE2) |	/* First command is 'Erase confirm' */
			IPA051_NAND_CTRLS__CTRLS_CMD_PH(2) |		/* Two command cycles */
			IPA051_NAND_CTRLS__CTRLS_ADR_PH(2) |		/* Two address cycle */
			IPA051_NAND_CTRLS__CTRLS_ROW_ADDR |		/* Include row address */
			IPA051_NAND_CTRLS__CTRLS_LB |			/* Use large page addressing mode */
			IPA051_NAND_CTRLS__CTRLS_128MB,
			IPA051_NAND_CTRLS + IPA051 );

		row = offset & ~((1 << nand->page_shift) - 1) ;	/* Row address within the device */
		col = offset & ((1 << nand->page_shift) - 1) ;	/* Column address within the device */

		nandOffset = col |		/* Byte within pages (won't be sent to the device) */
			     (row << 1) ;	/* Row address, shift up to make room for A11 */
	}

	/* erase */
	return( do_gpxio( (unsigned long)(nand->IO_ADDR_W) + nandOffset,
			  GPXIO_NODATA,
			  GPXIO_ACK ) );
}

int nand_dma_read( struct mtd_info *mtd,
		   off_t		row,
		   off_t		column,
		   size_t		length,
		   void			*dest,
		   int			cached)
{
	off_t		aligned_column ;
	size_t		aligned_length ;
	void		*aligned_dest ;
	unsigned long	nandCtrls ;
	unsigned long	nandOffset ;
	struct nand_chip *nand = mtd->priv;

	mydebug("row=%lx, column=%lx, length=%x, cached=%d\n", row, column, length, cached);
	
	/* Do not allow to read past end of device */
	if ((row + column + length) > mtd->size) {
		printf( "Error: Attempt to read beyond end of device (row = 0x%.8lX, column = 0x%.8lX, length = 0x%.8lX, device = 0x%.8lX)\n",
				(long unsigned int)row, (long unsigned int)column, (long unsigned int)length, (long unsigned int)mtd->size );
		return (-1);
	}

	/*
	 * Alignment here is a bit tricky because there are a couple of rules
	 * that must be taken into account:
	 * 1. The XIO bus is 32 bits wide and doesn't allow unaligned transfers,
	 *    so both SDRAM address, XIO address and transfer length must be a
	 *    modulo of 4.
	 * 2. The SDRAM address used for DMA transfers must be cache line size
	 *    aligned because other variables sharing the same cache line(s) may
	 *    corrupt the transfer buffer when written back.
	 * 3. We must make sure that '*dest' is not dereferenced beyond 'length'.
	 * This function will still work if the caller supplies parameters that
	 * don't meet these rules, but an intermediate buffer will be used
	 * resulting in a performance penalty.
	 */

	/* Round down column address for word alignment */
	aligned_column = ROUND_DOWN(column, sizeof(uint32_t));

	/* Round up transfer length, compensate for column alignment correction */
	aligned_length = ROUND_UP(length + (column - aligned_column), sizeof(uint32_t));
	
	/* Check if we need the alignment buffer */
	if( (length != aligned_length) ||
	    (column != aligned_column) ||
	    ((off_t)dest & (CONFIG_SYS_CACHELINE_SIZE-1)) ) {
		if (aligned_length > sizeof(aligned_buf)) {
			printf( "Error: Alignment buffer too small for DMA transfer (transfer = 0x%.8lX, buffer = 0x%.8lX)\n",
				(long unsigned int)aligned_length, (long unsigned int)sizeof(aligned_buf)) ;
			return (-1);
		}
		aligned_dest = aligned_buf ;
	} else
		aligned_dest = dest ;

	/* Check device geometry */
	if (nand->page_shift < 11) {
		/* We're dealing with a Small or Tiny Page device */

		/* Check what kind of data is to be transfered */
		if (aligned_column < (mtd->writesize >> 1)) {
			/* Lower half of In Band data is to be transfered */
			nandCtrls  = IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_READ0) ;
			nandOffset = aligned_column & ((1 << (nand->page_shift-1))-1) ;
		}
		else if (aligned_column < mtd->writesize) {
			/* Upper half of In Band data is to be transfered */
			nandCtrls  = IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_READ1) ;
			nandOffset = aligned_column & ((1 << (nand->page_shift-1))-1) ;
		}
		else if (aligned_column < (mtd->writesize + mtd->oobsize)) {
			/* Out Of Band data is to be transfered */
			nandCtrls  = IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_READOOB) ;
			nandOffset = aligned_column & (mtd->oobsize-1) ;
		}
		else {
			printf( "Error: Attempt to send address beyond end of row (column = 0x%.8lX, IB = 0x%.8lX, OOB = 0x%.8lX)\n",
				(long unsigned int)aligned_column, (long unsigned int)mtd->writesize, (long unsigned int)mtd->oobsize );
			return (-1);
		}

		/* Add offset address with row address */
		nandOffset |= row << nand->page_shift ;

		/* Add the NAND controls */
		nandCtrls |= IPA051_NAND_CTRLS__CTRLS_CMD_PH(1) |	/* One command cycle */
			     IPA051_NAND_CTRLS__CTRLS_INC_DATA ;	/* Command includes data */

#if 0
		/* Set the number of address cycles */
		/*
		if (nand->chip_shift < 26)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_ADR_PH(nand->pageadrlen) ;
		else
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_ADR_PH(nand->pageadrlen-1) ;
		*/
		/// when it comes here, it only for SP flash
		if (nand->chip_shift < 26)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_ADR_PH(3) ;
		else
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_ADR_PH(4 - 1) ;
#endif
		nandCtrls |=  nand_get_addr_phases(nand);			

		/* Set the 64MiB and 128MiB device compatibility bits */
		if (nand->chip_shift == 26)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_64MB ;
		if (nand->chip_shift == 27)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_128MB ;
	} else {
		int nandcmd_b;
		/* We're dealing with a Large Page device */
		if (aligned_column >= (mtd->writesize + mtd->oobsize)) {
			printf( "Error: Attempt to send address beyond end of row (column = 0x%.8lX, IB = 0x%.8lX, OOB = 0x%.8lX)\n",
				(long unsigned int)aligned_column, (long unsigned int)mtd->writesize, (long unsigned int)mtd->oobsize );
			return (-1);
		}

		nandcmd_b = ( cached ? NAND_CMD_CACHEREAD : NAND_CMD_READSTART);	
		nandCtrls = IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_READ0) |	/* First command is 'Read' */
			    IPA051_NAND_CTRLS__COMMAND_B(nandcmd_b) |	/* First command is 'Read confirmed' */
			    IPA051_NAND_CTRLS__CTRLS_CMD_PH(2) |		/* Two command cycles */
			    IPA051_NAND_CTRLS__CTRLS_ADR_PH(2) |		/* Two address cycle (Yes, this is confusing, but so is the manual) */
			    IPA051_NAND_CTRLS__CTRLS_2ND_CMD |			/* Insert second command before the data */
			    IPA051_NAND_CTRLS__CTRLS_ROW_ADDR |			/* Include row address */
			    IPA051_NAND_CTRLS__CTRLS_COL_ADDR |			/* Include column address */
			    IPA051_NAND_CTRLS__CTRLS_LB |			/* Use large page addressing mode */
			    IPA051_NAND_CTRLS__CTRLS_128MB |
			    IPA051_NAND_CTRLS__CTRLS_INC_DATA ;			/* Command includes data */

		nandOffset = (row << (nand->page_shift + 1)) | aligned_column ;
	}

	/* Set NAND control parameters */
	writel(nandCtrls, IPA051 + IPA051_NAND_CTRLS);

	/* Execute the DMA transfer */
	if( do_dma( (unsigned long)(nand->IO_ADDR_W) + nandOffset,
		    (unsigned long)aligned_dest,
		    aligned_length,
		    DMA_READ ) )
		return (-1) ;

	if (aligned_dest != dest) {
		memcpy(dest, ((unsigned char*)aligned_dest) + (column - aligned_column), length);
		mydebug("aligned_dest (%x) != dest (%x)\n", aligned_dest, dest);
	}
	return (0) ;
}

int nand_dma_write( struct mtd_info* mtd,
		    off_t		row,
		    off_t		column,
		    size_t		length,
		    void		*source	)
{
	off_t		aligned_column ;
	size_t		aligned_length ;
	void		*aligned_src = 0 ;
	unsigned long	nandCtrls ;
	unsigned long	nandOffset ;
	struct nand_chip *nand = mtd->priv;

	mydebug("row=%lx, column=%lx\n", row, column);
	
	/* Do not allow to write past end of device */
	if ((row + column + length) > mtd->size) {
		printf( "Error: Attempt to write beyond end of device (row = 0x%.8lX, column = 0x%.8lX, length = 0x%.8lX, device = 0x%.8lX)\n",
			(long unsigned int)row, (long unsigned int)column, (long unsigned int)length, (long unsigned int)mtd->size );
		return (-1);
	}

	/* Round down column address for word alignment */
	aligned_column = ROUND_DOWN(column, sizeof(uint32_t));

	/* Round up transfer length, compensate for column alignment correction */
	aligned_length = ROUND_UP(length + (column - aligned_column), sizeof(uint32_t));

	/* Check if we need the alignment buffer */
	if( (length != aligned_length) ||
	    (column != aligned_column) ) {
		if (aligned_length > sizeof(aligned_buf)) {
			printf( "Error: Alignment buffer too small for DMA transfer (transfer = 0x%.8lX, buffer = 0x%.8lX)\n",
				(long unsigned int)aligned_length, (long unsigned int)sizeof(aligned_buf)) ;
			return (-1);
		}
		aligned_src = aligned_buf ;

		/* Pad the alignment buffer with empty bytes before the actual data */
		memset (aligned_buf, EMPTY_BYTE, column-aligned_column);
		/* Copy the data into the alignment buffer */
		memcpy (((unsigned char*)aligned_src) + (column - aligned_column), source, length);
		/* Pad the alignment buffer with empty bytes behind the actual data */
		memset (((unsigned char*)aligned_src) + (column - aligned_column) + length, EMPTY_BYTE,
			aligned_length - (column - aligned_column) - length);
	} else
		aligned_src = source ;

	/* Check device geometry */
	if (nand->page_shift < 11) {
		/* We're dealing with a Small or Tiny Page device */

		/* Check what kind of data is to be transfered */
		if (aligned_column < (mtd->writesize >> 1)) {
			/* Lower half of In Band data is to be transfered */

			/* Configure NAND controls */
			writel( IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_READ0) |
				IPA051_NAND_CTRLS__CTRLS_CMD_PH(1),
				IPA051_NAND_CTRLS + IPA051);

			nandOffset = aligned_column & ((1 << (nand->page_shift-1))-1) ;
		}
		else if (aligned_column < mtd->writesize) {
			/* Upper half of In Band data is to be transfered */

			/* Configure NAND controls */
			writel( IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_READ1) |
				IPA051_NAND_CTRLS__CTRLS_CMD_PH(1),
				IPA051_NAND_CTRLS + IPA051);

			nandOffset = aligned_column & ((1 << (nand->page_shift-1))-1) ;
		}
		else if (aligned_column < (mtd->writesize + mtd->oobsize)) {
			/* Out Of Band data is to be transfered */

			/* Configure NAND controls */
			writel( IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_READOOB) |
				IPA051_NAND_CTRLS__CTRLS_CMD_PH(1),
				IPA051_NAND_CTRLS + IPA051);

			nandOffset = aligned_column & (mtd->oobsize-1) ;
		}
		else {
			printf( "Attempt to send address beyond end of row (column = 0x%.8lX, IB = 0x%.8lX, OOB = 0x%.8lX)\n",
				(long unsigned int)aligned_column, (long unsigned int)mtd->writesize, (long unsigned int)mtd->oobsize );
			return (-1);
		}

		/* Execute the GPXIO cycle the with the set NAND controls to select the proper region */
		do_gpxio( (unsigned long)(nand->IO_ADDR_W),
			  GPXIO_NODATA,
			  GPXIO_NO_ACK );

		/* Add offset address with row address */
		nandOffset |= (row << nand->page_shift) ;

		/* Set the NAND controls */
		nandCtrls = IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_SEQIN) |	/* First command is 'Page in' */
			    IPA051_NAND_CTRLS__COMMAND_B(NAND_CMD_PAGEPROG) |	/* First command is 'Program' */
			    IPA051_NAND_CTRLS__CTRLS_CMD_PH(2) |		/* Two command cycles */
			    IPA051_NAND_CTRLS__CTRLS_INC_DATA ;			/* Command includes data */

#if 0
		/* Set the number of address cycles */
		if (nand->chip_shift < 26)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_ADR_PH(nand->pageadrlen) ;
		else
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_ADR_PH(nand->pageadrlen-1) ;
#endif
		nandCtrls |= nand_get_addr_phases(nand);

		/* Set the 64MiB and 128MiB device compatibility bits */
		if (nand->chip_shift == 26)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_64MB ;
		if (nand->chip_shift == 27)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_128MB ;
	} else {
		/* We're dealing with a Large Page device */
		if (aligned_column >= (mtd->writesize + mtd->oobsize)) {
			printf( "Error: Attempt to send address beyond end of row (column = 0x%.8lX, IB = 0x%.8lX, OOB = 0x%.8lX)\n",
				(long unsigned int)aligned_column, (long unsigned int)mtd->writesize, (long unsigned int)mtd->oobsize );
			return (-1);
		}

		nandCtrls = IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_SEQIN) |	/* First command is 'Page in' */
			    IPA051_NAND_CTRLS__COMMAND_B(NAND_CMD_PAGEPROG) |	/* First command is 'Program' */
			    IPA051_NAND_CTRLS__CTRLS_CMD_PH(2) |		/* Two command cycles */
			    IPA051_NAND_CTRLS__CTRLS_ADR_PH(2) |		/* Two address cycle (Yes, this is confusing, but so is the manual) */
			    IPA051_NAND_CTRLS__CTRLS_ROW_ADDR |			/* Include row address */
			    IPA051_NAND_CTRLS__CTRLS_COL_ADDR |			/* Include column address */
			    IPA051_NAND_CTRLS__CTRLS_LB |			/* Use large page addressing mode */
			    IPA051_NAND_CTRLS__CTRLS_128MB |
			    IPA051_NAND_CTRLS__CTRLS_INC_DATA ;			/* Command includes data */

		nandOffset = (row << (nand->page_shift + 1)) | aligned_column ;
	}

	/* Set NAND control parameters */
	writel(nandCtrls, IPA051 + IPA051_NAND_CTRLS);

	/* Execute the DMA transfer */
	return( do_dma( (unsigned long)(nand->IO_ADDR_W) + nandOffset,
			(unsigned long)aligned_src,
			aligned_length,
			DMA_WRITE ) );
}

int nand_gpxio_read( struct mtd_info	*mtd,
		     off_t		row,
		     off_t		column,
		     uint32_t		*dest )
{
	struct nand_chip *nand = mtd->priv;
	unsigned long	nandCtrls ;
	unsigned long	nandOffset ;

	//mydebug("IO_ADDR=%x row=%lx column=%lx\n", (uint)nand->IO_ADDR_W, row, column);
	
	/* Do not allow to read past end of device */
	if ((row + column + sizeof(uint32_t)) > mtd->size) {
		printf( "Error: Attempt to read beyond end of device (row = 0x%.8lX, column = 0x%.8lX, device = 0x%.8lX)\n",
			(long unsigned int)row, (long unsigned int)column, (long unsigned int)mtd->size );
		return (-1);
	}

	/* Check alignment */
	if (column != ROUND_DOWN(column, sizeof(uint32_t))) {
		printf( "\nError: Attempt to read unaligned column address (row = 0x%.8lX, column = 0x%.8lX)\n",
			row, column );
		return (-1);
	}

	/* Check device geometry */
	if (nand->page_shift < 11) {
		/* We're dealing with a Small or Tiny Page device */

		/* Check what kind of data is to be transfered */
		if (column < (mtd->writesize >> 1)) {
			/* Lower half of In Band data is to be transfered */
			nandCtrls  = IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_READ0) ;
			nandOffset = column & ((1 << (nand->page_shift-1))-1) ;
		}
		else if (column < mtd->writesize) {
			/* Upper half of In Band data is to be transfered */
			nandCtrls  = IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_READ1) ;
			nandOffset = column & ((1 << (nand->page_shift-1))-1) ;
		}
		else if (column < (mtd->writesize + mtd->oobsize)) {
			/* Out Of Band data is to be transfered */
			nandCtrls  = IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_READOOB) ;
			nandOffset = column & (mtd->oobsize-1) ;
		}
		else {
			printf( "Error: Attempt to send address beyond end of row (column = 0x%.8lX, IB = 0x%.8lX, OOB = 0x%.8lX)\n",
				(long unsigned int)column, (long unsigned int)mtd->writesize, (long unsigned int)mtd->oobsize );
			return (-1);
		}

		/* Add offset address with row address */
		nandOffset |= row << nand->page_shift ;

		/* Add the NAND controls */
		nandCtrls |= IPA051_NAND_CTRLS__CTRLS_CMD_PH(1) |	/* One command cycle */
			     IPA051_NAND_CTRLS__CTRLS_INC_DATA ;	/* Command includes data */


#if 0
		/* Set the number of address cycles */
		if (nand->chip_shift < 26)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_ADR_PH(nand->pageadrlen) ;
		else
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_ADR_PH(nand->pageadrlen-1) ;
#endif
		nandCtrls |= nand_get_addr_phases(nand);

		/* Set the 64MiB and 128MiB device compatibility bits */
		if (nand->chip_shift == 26)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_64MB ;
		if (nand->chip_shift == 27)
			nandCtrls |= IPA051_NAND_CTRLS__CTRLS_128MB ;
	} else {
		/* We're dealing with a Large Page device */
		if (column >= (mtd->writesize + mtd->oobsize)) {
			printf( "Error: Attempt to send address beyond end of row (column = 0x%.8lX, IB = 0x%.8lX, OOB = 0x%.8lX)\n",
				(long unsigned int)column, (long unsigned int)mtd->writesize, (long unsigned int)mtd->oobsize );
			return (-1);
		}

		nandCtrls = IPA051_NAND_CTRLS__COMMAND_A(NAND_CMD_READ0) |	/* First command is 'Read' */
			    IPA051_NAND_CTRLS__COMMAND_B(NAND_CMD_READSTART) |	/* First command is 'Read confirmed' */
			    IPA051_NAND_CTRLS__CTRLS_CMD_PH(2) |		/* Two command cycles */
			    IPA051_NAND_CTRLS__CTRLS_ADR_PH(2) |		/* Two address cycle (Yes, this is confusing, but so is the manual) */
			    IPA051_NAND_CTRLS__CTRLS_2ND_CMD |			/* Insert second command before the data */
			    IPA051_NAND_CTRLS__CTRLS_ROW_ADDR |			/* Include row address */
			    IPA051_NAND_CTRLS__CTRLS_COL_ADDR |			/* Include column address */
			    IPA051_NAND_CTRLS__CTRLS_LB |			/* Use large page addressing mode */
			    IPA051_NAND_CTRLS__CTRLS_128MB |
			    IPA051_NAND_CTRLS__CTRLS_INC_DATA ;			/* Command includes data */

		nandOffset = (row << (nand->page_shift + 1)) | column ;
	}

	/* Set NAND control parameters */
	writel(nandCtrls, IPA051 + IPA051_NAND_CTRLS);

	//mydebug("nandOffset=%lx\n", nandOffset);
	if( do_gpxio( (unsigned long)(nand->IO_ADDR_W) + nandOffset,
		      GPXIO_READ,
		      GPXIO_ACK ) )
		return (-1) ;

	*dest = readl(IPA051 + IPA051_GPXIO_RDATA);

	return (0) ;
}

int nand_gpxio_write( struct nand_chip	*nand,
		      off_t		row,
		      off_t		column,
		      uint32_t		*source )
{
	return (0);
}

static unsigned long nand_get_addr_phases(struct nand_chip *nand) 
{
	/* Set the number of address cycles */
	/*
	if (nand->chip_shift < 26)
		nandCtrls |= IPA051_NAND_CTRLS__CTRLS_ADR_PH(nand->pageadrlen) ;
	else
		nandCtrls |= IPA051_NAND_CTRLS__CTRLS_ADR_PH(nand->pageadrlen-1) ;
	*/

	/// when it comes here, it only for SP flash
	/*
	unsigned long addrcycles = 0;

	if (nand->chip_shift < 26)
		addrcycles = IPA051_NAND_CTRLS__CTRLS_ADR_PH(3) ;
	else
		addrcycles = IPA051_NAND_CTRLS__CTRLS_ADR_PH(4 - 1) ;

	return(addrcycles);
	*/

	// The legacy code defines pageadrlen per nand flash. The mtd interface
	// does not have this information. On the other hand, for most SP nand 
	// flashes, if size < 64 mb, 3 address phases are used; for 64/128mb flash,
	// 4 phases are used. 
	// In combination with IP051, we set the phases as DEFAULT_ADDRESS_PHASES - 1.	
	return IPA051_NAND_CTRLS__CTRLS_ADR_PH(DEFAULT_ADDRESS_PHASES - 1) ;	
}

#endif /* CONFIG_NAND* */
#endif /* CONFIG_NXP_PCIXIO_IPA051 */
